import React, { useEffect, useMemo } from "react";
import { Message, Segment } from "semantic-ui-react";
import { AutoSizer, CellMeasurer, CellMeasurerCache, Column, Table } from "react-virtualized";
import { useCameraLogQuery, CameraLog, LogSeverity } from "@generated/graphql";
import { UUID } from "@solid/types";
import { __ } from "@solid/libs/i18n";
import CameraLogCell from "./CameraLogCell";

import "./style.css";

type CameraLogsProps = {
  deviceId: UUID;
};

const severityStyle: Record<LogSeverity, string> = {
  [LogSeverity.Info]: "CameraLogMessage-Info",
  [LogSeverity.Debug]: "CameraLogMessage-Debug",
  [LogSeverity.Error]: "CameraLogMessage-Error",
  [LogSeverity.Fatal]: "CameraLogMessage-Fatal",
  [LogSeverity.Warn]: "CameraLogMessage-Warn",
};

const CameraLogs = ({ deviceId }: CameraLogsProps) => {
  const { data, error, loading } = useCameraLogQuery({ variables: { deviceId }, fetchPolicy: "no-cache" });
  const cameraLog: CameraLog[] = useMemo(() => data?.cameraLog?.map(log => log) ?? [], [data]);

  const cache = new CellMeasurerCache({
    defaultHeight: 32,
    minHeight: 32,
    fixedWidth: true
  });

  useEffect(() => {
    return () => {
      cache && cache.clearAll();
    };
  }, []);

  function cellRenderer({ rowData, rowIndex, dataKey, logKey, parent}: {rowData: CameraLog, rowIndex: number, dataKey: string, logKey: keyof CameraLog, parent: any}) {
    return (
      <CellMeasurer
        cache={cache}
        key={dataKey}
        parent={parent}
        rowIndex={rowIndex}
      >
        <div className="CameraLogCell-Message">
          <CameraLogCell rowData={rowData} logKey={logKey} />
        </div>
      </CellMeasurer>
    );
  }

  return (
    <Segment loading={loading} className="CameraLog-Wrapper">
      {error &&
        <Message warning>
          <Message.Header>{error.message}</Message.Header>
        </Message>
      }
      {!loading && !error &&
        <div className="CameraLog-Grid">
          <AutoSizer style={{ width: "100%", height: "100%" }}>
            {({ width, height }) => (
              <Table
                className="CameraLog-Table"
                width={width}
                height={height}
                rowHeight={cache.rowHeight}
                headerHeight={32}
                rowCount={cameraLog.length}
                rowGetter={({ index }) => cameraLog[index]}
                deferredMeasurementCache={cache}
                rowClassName={({ index }) => { return severityStyle[cameraLog[index]?.severity]; }}
              >
                <Column
                  dataKey="time"
                  label={__("Date Time")}
                  width={200}
                  cellRenderer={({ rowData }) => <CameraLogCell rowData={rowData} logKey="time" />}
                />
                <Column
                  dataKey="severity"
                  label={__("Severity")}
                  width={100}
                  cellRenderer={({ rowData }) => <CameraLogCell rowData={rowData} logKey="severity" />}
                />
                <Column
                  dataKey="message"
                  label={__("Message")}
                  width={500}
                  flexGrow={1}
                  cellRenderer={({ rowData, rowIndex, dataKey, parent }) => cellRenderer({rowData, rowIndex, dataKey, logKey: "message", parent})}
                />
              </Table>
            )}
          </AutoSizer>
        </div>}
    </Segment>
  );
};

export default CameraLogs;
