import React, { useEffect, useState } from "react";
import { useNavigate } from "react-router-dom";
import { Button, Form, Header, Segment} from "semantic-ui-react";
import { useActivateAvatarMutation } from "@generated/graphql";
import { UUID } from "@solid/types";
import { Log } from "@solid/libs/log";
import { __ } from "@solid/libs/i18n";
import { useDeviceActions } from "@core/actions";
import markdown from "./help.md";
import Help from "components/Help";

import "./style.css";

const AvatarActivation = () => {
  const [activateAvatar, { loading }] = useActivateAvatarMutation();
  const [avatar, setAvatar] = useState<UUID>("");
  const [isAvatar, setIsAvatar] = useState(false);
  const { updateCachedDevice } = useDeviceActions({skipSubscription: true});
  const navigate = useNavigate();

  useEffect(() => {
    if (avatar.length > 0) {
      setIsAvatar(true);
    } else {
      setIsAvatar(false);
    }
  }, [avatar]);

  const activate = async (avatarId: UUID) => {
    try {
      const { data, errors } = await activateAvatar({ variables: { avatarId } });
      if (data?.activateAvatar) {
        await updateCachedDevice(avatarId);
        navigate(`/view/link/edit/${avatarId}`);
      }
      if (errors) {
        const error = errors.map(error => error.message).join(".\n");
        console.error("Avatar activation error:", error);
        Log.error(__("Avatar activation error: ") + error);
      }
    }
    catch (e: any) {
      console.error("Avatar activation error:", e);
      Log.error(__("Avatar activation error: ") + e.message);
    }
  };

  return (
    <Segment className="AvatarActivation" loading={loading}>
      <Form onSubmit={e => e.preventDefault()}>
        <Header as="h3">
          {__("Activate Avatar")}
        </Header>
        <Form.Field>
          <label>{__("Enter Avatar GUID")}</label>
          <input type="text" id="uuidAvatar" tabIndex={1} autoFocus
            onInput={(e) => { setAvatar(e.currentTarget.value); }}/>
        </Form.Field>
        <Button disabled={!isAvatar} onClick={() => activate(avatar)}>{__("Activate")}</Button>
        <Button className="discovery-button" icon="globe" content={__("Discover")} onClick={() => navigate("/view/LinkProvision")} />
      </Form>
      <Help markdown={markdown}/>
    </Segment>
  );
};

export default AvatarActivation;
