import React, { useState, useEffect } from "react";
import { LocalStorageInput } from "@generated/graphql";
import { Checkbox, Dropdown, Form, Input } from "semantic-ui-react";
import { __ } from "@solid/libs";
import { OnChangeLocalStorageValueProp, PoolSizeUnit } from "../LinkLocalStorageConfig";

import "./style.css";

type LinkLocalStorageFormProps = {
  input: LocalStorageInput;
  onChangeLocalStorageValue: (value: OnChangeLocalStorageValueProp) => void;
};

const LinkLocalStorageForm = ({ input, onChangeLocalStorageValue}: LinkLocalStorageFormProps) => {
  const [isLocalStorageEnable, setIsLocalStorageEnable] = useState<boolean>();
  const [poolSizeValue, setPoolSizeValue] = useState<number>();
  const [poolSizeUnit, setPoolSizeUnit] = useState<string>();
  const [path, setPath] = useState<string>();

  const { enabled, pool } = input;
  const poolSizeUnitOptions = [
    {key: PoolSizeUnit.K, text: PoolSizeUnit.K, value: "K"},
    {key: PoolSizeUnit.M, text: PoolSizeUnit.M, value: "M"},
    {key: PoolSizeUnit.G, text: PoolSizeUnit.G, value: "G"},
  ];

  useEffect(() => {
    const match = pool.size.match(/([0-9]+)([a-zA-Z]+)/);
    if (!match) {
      return;
    }

    setPoolSizeValue(Number(match[1]));
    setPoolSizeUnit(match[2] ? match[2] : "G");
    setPath(pool.path);
    setIsLocalStorageEnable(enabled);
  }, [enabled, pool]);

  const changeLocalStorageEnabled = (value: boolean) => {
    setIsLocalStorageEnable(value);
    onChangeLocalStorageValue({enabled: value});
  };

  const changePoolSizeValue = (value: string) => {
    let val = Number(value);
    if (isNaN(val) || val < 0) {
      val = 0;
    }

    setPoolSizeValue(val);
    onChangeLocalStorageValue({
      pool: {
        enforce: pool.enforce,
        path: path || pool.path,
        size: `${val}${poolSizeUnit}`
      },
    });
  };

  const changePoolSizeUnit = (value: string) => {
    setPoolSizeUnit(value);
    onChangeLocalStorageValue({
      pool: {
        enforce: pool.enforce,
        path: path || pool.path,
        size: `${poolSizeValue}${value}`
      },
    });
  };

  const changePoolPath = (value: string) => {
    setPath(value);
    onChangeLocalStorageValue({
      pool: {
        enforce: pool.enforce,
        path: value,
        size: `${poolSizeValue}${poolSizeUnit}`
      },
    });
  };

  return (
    <Form onSubmit={(e) => e.preventDefault()} className="LinkLocalStorage-Form">
      <Form.Field inline className="LinkLocalStorage-Field">
        <label className="LinkLocalStorage-Label">{__("Enable Local Storage")}</label>
        <Checkbox
          toggle
          checked={isLocalStorageEnable}
          onChange={(e, data) => changeLocalStorageEnabled(!!data.checked)}
        />
      </Form.Field>
      <Form.Field inline className="LinkLocalStorage-Field">
        <label className="LinkLocalStorage-Label">{__("Storage Size")}</label>
        <Input
          className="LinkLocalStorage-StorageSize-Input"
          disabled={!isLocalStorageEnable}
          value={poolSizeValue?.toString() || ""}
          onChange={e => changePoolSizeValue(e.currentTarget.value)}
        />
        <Dropdown
          className="LinkLocalStorage-Dropdown"
          disabled={!isLocalStorageEnable}
          compact
          selection
          options={poolSizeUnitOptions || ""}
          value={poolSizeUnit}
          onChange={(e, { value }) => changePoolSizeUnit(typeof value === "string" ? value : "")}
        />
      </Form.Field>
      <Form.Field inline className="LinkLocalStorage-Field">
        <label className="LinkLocalStorage-Label">{__("Storage Directory")}</label>
        <Input
          disabled={!isLocalStorageEnable}
          value={path || ""}
          onChange={e => changePoolPath(e.currentTarget.value)}
        />
      </Form.Field>
    </Form>
  );
};

export default LinkLocalStorageForm;
