import React, { useEffect, useState } from "react";
import { Button, Header, Icon, List, Modal, Popup } from "semantic-ui-react";

import { __ } from "@solid/libs";

import "./style.css";

enum ManageKeyAction {
  Regenerate = "regenerate",
  Remove = "remove"
}

const actionWarningMap: Record<ManageKeyAction, string> = {
  [ManageKeyAction.Regenerate]: __("Old API key will be invalid."),
  [ManageKeyAction.Remove]: __("All clients that used this API key will lost access to Avatar.")
};

const actionMap: Record<ManageKeyAction, string> = {
  [ManageKeyAction.Regenerate]: __("Regenerate"),
  [ManageKeyAction.Remove]: __("Remove")
};

type ApiKeyProps = {
  apiKey?: string | null;
  onCreate: () => void;
  onRegenerate: () => void;
  onRemove: () => void;
  closePopup?: boolean;
  onOpen?: (state: boolean) => void;
};

const ApiKey = ({ apiKey, onCreate, onRegenerate, onRemove, closePopup, onOpen }: ApiKeyProps) => {
  const [action, setAction] = useState<ManageKeyAction>();
  const [popupOpen, setPopupOpen] = useState<boolean>(false);

  useEffect(() => {
    if (popupOpen && closePopup) {
      setPopupOpen(false);
      onOpen?.(false);
    }
  }, [closePopup]);

  return (
    <>
      <Popup
        open={popupOpen}
        content={
          <List>
            {!apiKey ?
              <List.Item className="ApiKey-ActionItem" onClick={() => {
                setPopupOpen(false);
                onOpen?.(false);
                onCreate();
              }}>
                <List.Icon name="plus" />
                <List.Content>{__("Create")}</List.Content>
              </List.Item> :
              <>
                <List.Item className="ApiKey-ActionItem" onClick={() => {
                  setPopupOpen(false);
                  onOpen?.(false);
                  setAction(ManageKeyAction.Remove);
                }}>
                  <List.Icon name="trash" />
                  <List.Content>{__("Remove")}</List.Content>
                </List.Item>
                <List.Item className="ApiKey-ActionItem" onClick={() => {
                  setPopupOpen(false);
                  onOpen?.(false);
                  setAction(ManageKeyAction.Regenerate);
                }}>
                  <List.Icon name="sync alternate" />
                  <List.Content>{__("Regenerate")}</List.Content>
                </List.Item>
              </> }
          </List>
        }
        trigger={
          <Icon name="ellipsis horizontal" className="ApiKey-ActionIcon" onClick={() => {
            setPopupOpen(!popupOpen);
            onOpen?.(!popupOpen);
          }} />
        }
      />
      {action &&
        <Modal open onClose={() => setAction(undefined)}>
          <Header>{__("{{action}} API key", { action: actionMap[action] })}</Header>
          <Modal.Content>{__("Are you sure you want to {{action}} current API key? {{warning}}", { action, warning: actionWarningMap[action] })}</Modal.Content>
          <Modal.Actions>
            <Button negative onClick={() => {
              action === ManageKeyAction.Remove ? onRemove() : onRegenerate();
              setAction(undefined);
            }}>
              <Icon name="check"/>{ actionMap[action] }
            </Button>
            <Button onClick={() => setAction(undefined)}>
              <Icon name="cancel"/>{__("Cancel")}
            </Button>
          </Modal.Actions>
        </Modal>}
    </>
  );
};

export default ApiKey;
