import React, { useState, useEffect, useMemo } from "react";
import { Segment, Button, Tab, Table, Checkbox, Message, Form, DropdownItemProps, List, Icon } from "semantic-ui-react";
import {
  DeviceFunctionalAspectType,
  DeviceFunctionalAspectInput,
  useVaeTemplatesQuery,
  useUpdateDeviceMutation,
  DeviceEventType,
  useDeviceLazyQuery,
  HealthStatus
} from "@generated/graphql";
import { Device, deviceToDeviceInput, aspectToAspectInput, useDeviceActions } from "@core/actions";
import WithQueryStatus from "components/WithQueryStatus";
import Loading from "components/Loading";
import VaeConfig from "../VaeConfig";
import {__} from "@solid/libs/i18n";

import "./style.css";

const eventTypeOptions: DropdownItemProps[] = [
  {key: DeviceEventType.Alert, value: DeviceEventType.Alert, text: __("ALERT")},
  {key: DeviceEventType.Information, value: DeviceEventType.Information, text: __("INFO")}
];

export type VaeManagementProps = {
  device: Device;
  onBack?: () => void;
};

const VaeManagement = ({ device, onBack }: VaeManagementProps) => {
  const { data: templData, error: templError, loading: templLoading } = useVaeTemplatesQuery({
    variables: { deviceId: device.id },
    fetchPolicy: "no-cache"
  });
  const [getAvatar, {data: avatarData, error: avatarError, loading: avatarLoading}] = useDeviceLazyQuery();
  const [avatarStatus, setAvatarStatus] = useState<HealthStatus | undefined>();
  const [vaeAspects, setVaeAspects] = useState<DeviceFunctionalAspectInput[]>([]);
  const vaeTemplates = useMemo(() => Array.from(templData?.vaeTemplates ?? [])
    .filter(template => template.id !== "vae_anpr")
    .sort((a, b) => a.order - b.order), [templData]);
  const mediaAspectOptions = useMemo(() => device.aspects
    .filter(aspect => aspect.enabled && aspect.template.id === "video")
    .map(({ id, name }) => ({ key: id, value: id, text: name })), [device]);
  const [vaeInputId, setVaeInputId] = useState("");
  const [updateDevice, { data: updateData, error: updateError, loading: updateLoading }] = useUpdateDeviceMutation();
  const { updateCachedDevice } = useDeviceActions({ skipSubscription: true });
  const [refetching, setRefetching] = useState(false);
  const [editEngine, setEditEngine] = useState("");
  const [eventType, setEventType] = useState<DeviceEventType>(DeviceEventType.Alert);
  const [acknowledgeRequired, setAcknowledgeRequired] = useState<boolean>(false);

  useEffect(() => {
    if (!device.platform?.id) {
      return;
    }
    getAvatar({variables: {id: device.platform?.id}});
  }, [device]);

  useEffect(() => {
    if (avatarError || avatarLoading) {
      return;
    }
    if (avatarData && avatarData.device?.healthStatus) {
      setAvatarStatus(avatarData.device?.healthStatus);
    }
  }, [avatarData, avatarError, avatarLoading]);

  useEffect(() => {
    const vaeAspects = device.aspects
      .filter(aspect => aspect.type === DeviceFunctionalAspectType.Vae)
      .map(aspect => aspectToAspectInput(aspect));
    setVaeAspects(vaeAspects);
    const mediaAspects = device.aspects.filter(aspect => aspect.enabled && aspect.template.id === "video");
    setVaeInputId(
      vaeAspects.find(aspect => aspect.inputAspectIDs.length > 0)?.inputAspectIDs[0] ??
      (mediaAspects.length > 0 ? mediaAspects[0].id : "")
    );
    if (device.eventType) {
      setEventType(device.eventType);
    }
    if (typeof device.acknowledgeRequired === "boolean") {
      setAcknowledgeRequired(device.acknowledgeRequired);
    }
  }, [device]);

  useEffect(() => {
    if (updateData) {
      setRefetching(true);
      updateCachedDevice(device.id).finally(() => setRefetching(false));
    }
  }, [updateData]);

  function onVaeInputChange(id: string | number | boolean | (string | number | boolean)[] | undefined): void {
    if (typeof id !== "string") {
      return;
    }
    setVaeInputId(id);
    const newAspects = Array.from(vaeAspects);
    for (let i = 0; i < newAspects.length; i++) {
      newAspects[i] = { ...newAspects[i], inputAspectIDs: [id] };
    }
    updateVae(newAspects);
  }

  function enableDisableVaeAspect(templateId: string, enable: boolean): void {
    const newAspects = Array.from(vaeAspects);
    if (!enable) {
      const index = newAspects.findIndex(aspect => aspect.templateId === templateId);
      if (index >= 0) {
        newAspects[index] = { ...newAspects[index], enabled: false };
      }
      else {
        return;
      }
    }
    else {
      const template = vaeTemplates.find(templ => templ.id === templateId);
      if (!template) {
        return;
      }
      const index = newAspects.findIndex(aspect => aspect.templateId === templateId);
      if (index >= 0) {
        newAspects[index] = { ...newAspects[index], enabled: true };
      }
      else {
        const input = device.aspects.find(aspect => aspect.id === vaeInputId);
        const aspect = {
          id: `${device.id}-${template.engine}`,
          name: template.name,
          templateId: template.id,
          inputAspectIDs: input ? [input.id] : [],
          enabled: true
        };
        newAspects.push(aspect);
      }
    }
    updateVae(newAspects);
  }

  function updateVae(vaeAspects: DeviceFunctionalAspectInput[]): void {
    const dev: Device = { ...device, aspects: device.aspects.filter(aspect => aspect.type !== DeviceFunctionalAspectType.Vae) };
    const input = deviceToDeviceInput(dev);
    input.aspects = (input.aspects ?? []).concat(vaeAspects);
    updateDevice({ variables: { id: device.id, device: input } });
  }

  function updateEventType(eventType: DeviceEventType) {
    const input = deviceToDeviceInput(device);
    input.eventType = eventType;
    updateDevice({ variables: { id: device.id, device: input } });
  }

  function updateAcknowledgeRequired(isAck: boolean) {
    const input = deviceToDeviceInput(device);
    input.acknowledgeRequired = isAck;
    updateDevice({ variables: { id: device.id, device: input } });
  }

  const panes = [
    {
      menuItem: __("Analytics"),
      render: () => (
        <Tab.Pane key="vae">
          <div className="VaeManagement-Vae">
            <Form className="VaeManagement-VaeInput">
              <Form.Field
                control={Form.Input}
                label={__("Device Name")}
                value={device.name}
                readOnly
              />
              <Form.Select
                label={__("Stream")}
                options={mediaAspectOptions}
                value={vaeInputId}
                onChange={(e, { value }) => onVaeInputChange(value)}
              />
              <Form.Select
                label={__("Event type")}
                options={eventTypeOptions}
                value={eventType}
                onChange={(e, { value }) => updateEventType(value as DeviceEventType)}
              />
              <Form.Field
                label={
                  <>
                    <label htmlFor="acknowledgeRequired">{__("Acknowledge Required")}</label>
                    <Checkbox
                      name="acknowledgeRequired"
                      toggle
                      checked={acknowledgeRequired}
                      onChange={(e, data) => updateAcknowledgeRequired(!!data.checked)}
                    />
                  </>
                }
              />
            </Form>
            <div className="VaeManagement-VaeAspects">
              {avatarStatus && avatarStatus !== HealthStatus.Normal &&
              <List.Description className="AvatarStatus-Message">
                <span>
                  {__("Avatar status - ")}
                  <span className="AvatarStatus-Red">
                    {__("OFFLINE")}
                  </span>
                  {__(". Analytics cannot be configured while avatar '{{name}}' status is not 'NORMAL'.", {name: device.platform?.name})}
                </span>
              </List.Description>}
              <Table collapsing compact>
                <Table.Body>
                  {vaeTemplates.map(template =>
                    <Table.Row key={template.id}>
                      <Table.Cell>
                        <Checkbox
                          label={template.name}
                          toggle
                          checked={vaeAspects.some(aspect => aspect.templateId === template.id && aspect.enabled)}
                          onChange={(e, data) => enableDisableVaeAspect(template.id, !!data.checked)}
                        />
                      </Table.Cell>
                      <Table.Cell>
                        <Button
                          disabled={!vaeAspects.some(aspect => aspect.templateId === template.id && aspect.enabled) || avatarStatus !== HealthStatus.Normal}
                          content={__("Configure")}
                          onClick={() => template.engine && setEditEngine(template.engine)}
                        />
                      </Table.Cell>
                    </Table.Row>)}
                </Table.Body>
              </Table>
            </div>
          </div>
        </Tab.Pane>
      )
    },
  ];

  return !editEngine ?
    <Segment className="VaeManagement">
      <div className="VaeManagement-TopButtons">
        <Button onClick={() => onBack && onBack()}>
          <Icon name="arrow left"/>{__("Back")}
        </Button>
      </div>

      <WithQueryStatus error={templError} loading={templLoading || avatarLoading}>
        {(!!updateError || !!updateData?.updateDevice.warning) && <Message error content={updateError?.message || updateData?.updateDevice.warning}/>}

        <Tab
          panes={panes}
          className="VaeManagement-Tab"
        />

        {(updateLoading || refetching) && <Loading text={__("Updating...")}/>}
      </WithQueryStatus>
    </Segment> :
    <VaeConfig device={device} engine={editEngine} onBack={() => setEditEngine("")}/>;
};

export default VaeManagement;
