import React, { useState, useEffect, useRef, useMemo, useCallback } from "react";
import { Segment, Button, Tab, Dropdown, DropdownProps, Message, Input, Ref, Icon, Modal, Header, Label, Checkbox, CheckboxProps, Popup } from "semantic-ui-react";
import {
  useVaeConfigsQuery,
  useUpdateDeviceMutation,
  useCreateVaeConfigMutation,
  useRenameVaeConfigMutation,
  useDeleteVaeConfigMutation,
  VaeConfigsDocument,
  DeviceFunctionalAspectInput,
  HealthStatus,
  useDeviceLazyQuery
} from "@generated/graphql";
import { Device, deviceToDeviceInput, useDeviceActions } from "@core/actions";
import WithQueryStatus from "components/WithQueryStatus";
import Loading from "components/Loading";
import IconButton from "components/IconButton";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";
import VaeConfigIframe from "./VaeConfigIframe";

import "./style.css";

type CheckFeatureMapValue = {
  name: string;
  checked: boolean;
};

type VaeConfigProps = {
  device: Device;
  engine: string;
  onBack?: () => void;
};

const VaeConfig = ({ device, engine, onBack }: VaeConfigProps) => {
  const aspect = useMemo(() => device.aspects.find(aspect => aspect.template.engine === engine), [device, engine]);
  const aspectError = useMemo(() => !aspect ? new Error(__("Requested device analytics is not defined")) :
    !aspect.enabled ? new Error(__("{{name}} is disabled", {name: aspect.template.name})) : undefined, [aspect]);
  const { data: confData, error: confError, loading: confLoading } = useVaeConfigsQuery({ variables: { deviceId: device.id, engine }, skip: !aspect });
  const confOptions = useMemo(() => {
    const vaeConfigs = confData?.vaeConfigs ? [...confData?.vaeConfigs] : [];
    const confOptions = vaeConfigs
      .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }))
      .map(({ name }) => ({ key: name, value: name, text: name }));
    return confOptions;
  }, [confData]);
  const activeConfName = aspect?.__typename === "DFA_VAE" ? aspect.vaeConfig?.name ?? "" : "";
  const [confValue, setConfValue] = useState(activeConfName);
  const [updateDevice, { data: updateData, error: updateError, loading: updateLoading }] = useUpdateDeviceMutation();
  const [getAvatar, { data: avatarData }] = useDeviceLazyQuery();
  const { updateCachedDevice } = useDeviceActions({ skipSubscription: true });
  const [refetching, setRefetching] = useState(false);
  const [isConfAdd, setIsConfAdd] = useState(false);
  const [isConfEdit, setIsConfEdit] = useState(false);
  const [confName, setConfName] = useState("");
  const [confNameOk, setConfNameOk] = useState(false);
  const [deleteConfirmOpen, setDeleteConfirmOpen] = useState(false);
  const inputRef = useRef<Input>(null);
  const okButtonRef = useRef<HTMLButtonElement>(null);
  const [createConf, { data: createConfData, error: createConfError, loading: createConfLoading }] = useCreateVaeConfigMutation();
  const [renameConf, { data: renameConfData, error: renameConfError, loading: renameConfLoading }] = useRenameVaeConfigMutation();
  const [deleteConf, { data: deleteConfData, error: deleteConfError, loading: deleteConfLoading }] = useDeleteVaeConfigMutation();
  const [error, setError] = useState<Error | undefined>();
  const [reloadIframeKey, setReloadIframeKey] = useState(Math.random());
  const radioFeatures = useMemo(() => aspect?.template.features
    ?.filter(feature => feature.exclusive)
    ?.sort((a, b) => a.order - b.order) ?? [], [aspect]);
  const [radioFeature, setRadioFeature] = useState("");
  const checkFeaturesList = useMemo(() => aspect?.template.features
    ?.filter(feature => !feature.exclusive)
    ?.sort((a, b) => a.order - b.order) ?? [], [aspect]);
  const [checkFeatures, setCheckFeatures] = useState<Map<string, CheckFeatureMapValue>>(new Map(checkFeaturesList.map(feature =>
    [feature.id, { name: feature.name, checked: isFeatureChecked(feature.id) }]) ?? []
  ));

  const avatarStatus: HealthStatus | undefined = useMemo(() => {
    return avatarData?.device?.healthStatus;
  }, [avatarData]);

  const featuresChanged: boolean = useMemo(() => {
    if (!aspect || aspect.__typename !== "DFA_VAE" || !aspect.vaeFeatures) {
      return false;
    }

    const aspectTemplateFeatures = aspect.template.features?.map(feature => feature.id);
    if (aspectTemplateFeatures) {
      const existFeatures = aspect.vaeFeatures.filter(feature => aspectTemplateFeatures.includes(feature));
      const checkedFeatures = getCheckedFeatures(checkFeatures);

      return (radioFeatures.length > 0 && !existFeatures.includes(radioFeature)) ||
              (checkFeaturesList.length > 0 && (checkedFeatures.length !== existFeatures.length ||
              !!(existFeatures.find(feature => !checkedFeatures.includes(feature)))));
    }

    return false;
  }, [radioFeature, radioFeatures, checkFeatures, aspect]);

  useEffect(() => {
    if (!device.platform?.id) {
      return;
    }
    getAvatar({variables: {id: device.platform.id}});
    updateIframe();
  }, [device]);

  useEffect(() => {
    updateIframe();
  }, [device.id, engine, confValue]);

  useEffect(() => {
    setConfValue(activeConfName);
  }, [activeConfName]);

  useEffect(() => {
    if (radioFeatures.length === 0 || !aspect || aspect.__typename !== "DFA_VAE" || !aspect.vaeFeatures) {
      setRadioFeature("");
      return;
    }
    const feature = aspect.vaeFeatures.find(id => radioFeatures.some(feature => feature.id === id)) ?? "";
    setRadioFeature(feature);
  }, [aspect, radioFeatures]);

  useEffect(() => {
    if (isConfAdd || isConfEdit) {
      window.requestAnimationFrame(() => {
        if (inputRef.current) {
          inputRef.current.focus();
        }
      });
    }
  }, [isConfAdd, isConfEdit]);

  useEffect(() => {
    if (updateData) {
      if (updateData.updateDevice.warning) {
        Log.error(updateData.updateDevice.warning);
      }
      setRefetching(true);
      updateCachedDevice(device.id).finally(() => setRefetching(false));
    }
  }, [updateData]);

  useEffect(() => {
    if (updateError) {
      setError(updateError);
    }
  }, [updateError]);

  useEffect(() => {
    if (createConfData?.createVaeConfig) {
      setConfValue(confName);
    }
  }, [createConfData]);

  useEffect(() => {
    if (createConfError) {
      setError(createConfError);
    }
  }, [createConfError]);

  useEffect(() => {
    if (renameConfData?.renameVaeConfig) {
      setRefetching(true);
      updateCachedDevice(device.id).finally(() => {
        setRefetching(false);
        setConfValue(confName);
      });
    }
  }, [renameConfData]);

  useEffect(() => {
    if (renameConfError) {
      setError(renameConfError);
    }
  }, [renameConfError]);

  useEffect(() => {
    if (deleteConfData?.deleteVaeConfig) {
      setConfValue(activeConfName);
    }
  }, [deleteConfData]);

  useEffect(() => {
    if (deleteConfError) {
      setError(deleteConfError);
    }
  }, [deleteConfError]);

  const applyFeatures = useCallback(() => {
    if (!aspect || aspect.__typename !== "DFA_VAE" || !aspect.vaeFeatures) {
      return;
    }

    const aspectTemplateFeatures = aspect.template.features?.map(feature => feature.id);
    if (aspectTemplateFeatures) {
      const existFeatures = aspect.vaeFeatures.filter(feature => !aspectTemplateFeatures.includes(feature));
      const checkedFeatures = getCheckedFeatures(checkFeatures);

      let newVaeFeatures: string[] = [];

      if (existFeatures.length > 0) {
        newVaeFeatures = [...existFeatures];
      }
      if (checkedFeatures.length > 0) {
        newVaeFeatures = newVaeFeatures.concat(checkedFeatures);
      }
      if (radioFeature) {
        newVaeFeatures.push(radioFeature);
      }

      updateDeviceAspect(aspectInput => {
        aspectInput.vaeConfigChangedNames = [confValue];
        aspectInput.vaeFeatures = newVaeFeatures;
      });
    }

  }, [radioFeature, checkFeatures, aspect]);

  function getCheckedFeatures(features: Map<string, CheckFeatureMapValue>): string[] {
    const checkedFeatures = [];
    for (const [key, value] of features.entries()) {
      value.checked && checkedFeatures.push(key);
    }

    return checkedFeatures;
  }

  function onConfValueChange(e: React.SyntheticEvent, { value }: DropdownProps): void {
    if (typeof value !== "string") {
      return;
    }
    setConfValue(value);
  }

  function onAddConfigClick(): void {
    setIsConfAdd(true);
    setIsConfEdit(false);
    setConfName("");
  }

  function onEditConfigClick(): void {
    setIsConfAdd(false);
    setIsConfEdit(true);
    setConfName(confValue);
  }

  function onCancelAddEditConfig(): void {
    setIsConfAdd(false);
    setIsConfEdit(false);
  }

  function onInputChange(e: React.ChangeEvent<HTMLInputElement>): void {
    const name = e.currentTarget.value;
    setConfName(name);
    const confIndex = confData?.vaeConfigs.findIndex(conf => conf.name === confValue);
    setConfNameOk(!!name && !confData?.vaeConfigs.some((conf, index) => index !== confIndex && conf.name === name));
  }

  function onInputKeyDown(e: React.KeyboardEvent): void {
    if (e.key === "Enter" && confNameOk) {
      isConfAdd ? onCreateConfig() : onRenameConfig();
    }
    if (e.key === "Escape") {
      onCancelAddEditConfig();
    }
  }

  function onCreateConfig(): void {
    setIsConfAdd(false);
    setError(undefined);
    createConf({
      variables: { deviceId: device.id, engine, name: confName },
      refetchQueries: [
        {
          query: VaeConfigsDocument,
          variables: { deviceId: device.id, engine }
        }
      ],
      awaitRefetchQueries: true
    });
  }

  function onRenameConfig(): void {
    setIsConfEdit(false);
    setError(undefined);
    renameConf({
      variables: { deviceId: device.id, engine, name: confValue, newName: confName },
      refetchQueries: [
        {
          query: VaeConfigsDocument,
          variables: { deviceId: device.id, engine }
        }
      ],
      awaitRefetchQueries: true
    });
  }

  function onDeleteConfig(): void {
    setError(undefined);
    deleteConf({
      variables: { deviceId: device.id, engine, name: confValue },
      refetchQueries: [
        {
          query: VaeConfigsDocument,
          variables: { deviceId: device.id, engine }
        }
      ],
      awaitRefetchQueries: true
    });
  }

  function updateDeviceAspect(changeAspect: (aspectInput: DeviceFunctionalAspectInput) => void): void {
    if (!aspect) {
      return;
    }
    const devInput = deviceToDeviceInput(device);
    const aspectInput = devInput.aspects?.find(asp => asp.templateId === aspect.template.id);
    if (!aspectInput) {
      return;
    }
    changeAspect(aspectInput);
    setError(undefined);
    updateDevice({ variables: { id: device.id, device: devInput } });
  }

  function onActivateConfig(): void {
    updateDeviceAspect(aspect => { aspect.vaeConfigName = confValue; });
  }

  function isFeatureChecked(featureId: string): boolean {
    return !!aspect && aspect.__typename === "DFA_VAE" && !!aspect.vaeFeatures && aspect.vaeFeatures.includes(featureId);
  }

  function onCheckFeatureChange(e: React.FormEvent<HTMLInputElement>, { value, checked }: CheckboxProps): void {
    if (typeof value !== "string") {
      return;
    }
    setCheckFeatures(checkFeatures => {
      const newMap = new Map(checkFeatures);
      const newValue = newMap.get(value);
      newValue && newMap.set(value, {...newValue, checked: !!checked});
      return newMap;
    });
  }

  function updateIframe() {
    setReloadIframeKey(Math.random());
  }

  const panes = [
    {
      menuItem: __("{{name}} Configuration", {name: `${device.name} - ${aspect?.template.name}`}),
      render: () => (
        <Tab.Pane key="vae">
          {avatarStatus && avatarStatus !== HealthStatus.Normal &&
          <div className="VaeConfig-AvatarStatusNotification">
            <span>
              {__("Avatar status - ")}
              <span className="AvatarStatus-Red">
                {__("OFFLINE")}
              </span>
              {__(". Analytics cannot be configured while avatar '{{name}}' status is not 'NORMAL'.", {name: device.platform?.name})}
            </span>
            <Popup
              content={__("Changes in the analytics configurator will not be saved")}
              trigger={
                <Icon name="sync alternate" onClick={() => updateIframe()} className="VaeConfig-ReloadFrameIcon" />
              }
            />
          </div>
          }
          <div className="VaeConfig-ConfigManage">
            {!isConfAdd && !isConfEdit ?
              <>
                <Dropdown
                  className="VaeConfig-ConfigDropdown"
                  fluid
                  selection
                  options={confOptions}
                  value={confValue}
                  onChange={onConfValueChange}
                          />
                <div className="VaeConfig-ConfigButtons">
                  <IconButton icon="plus" hint={__("Create Configuration")} onClick={onAddConfigClick}/>
                  <IconButton icon="edit" hint={__("Rename Configuration")} disabled={!confValue} onClick={onEditConfigClick}/>
                  <IconButton icon="trash alternate" hint={__("Delete Configuration")} disabled={!confValue || confValue === activeConfName} onClick={() => setDeleteConfirmOpen(true)}/>
                </div>
                {confValue === activeConfName ?
                  <Label content={__("This is the active configuration")} color="blue"/> :
                  <Button content={__("Make Configuration Active")} onClick={onActivateConfig}/>}
              </> :
              <Input
                ref={inputRef}
                placeholder={__("Name")}
                fluid
                action
                value={confName}
                onChange={onInputChange}
                onBlur={(e: React.FocusEvent) => {
                  if (e.nativeEvent.relatedTarget !== okButtonRef.current) {
                    onCancelAddEditConfig();
                  }
                }}
                onKeyDown={onInputKeyDown}>
                <input/>
                <Ref innerRef={okButtonRef}>
                  <Button positive icon disabled={!confNameOk} onClick={() => isConfAdd ? onCreateConfig() : onRenameConfig()}>
                    <Icon name={isConfAdd ? "plus" : "check"}/>
                  </Button>
                </Ref>
                <Button negative icon onClick={() => onCancelAddEditConfig()}>
                  <Icon name="cancel"/>
                </Button>
              </Input>}
          </div>

          <VaeConfigIframe
            key={reloadIframeKey}
            deviceId={device.id}
            confValue={confValue}
            engine={engine}
            />
        </Tab.Pane>
      )
    },
  ];

  return (
    <Segment className="VaeConfig">
      <div className="VaeConfig-TopButtons">
        <Button onClick={() => onBack && onBack()}>
          <Icon name="arrow left"/>{__("Back")}
        </Button>
        {(radioFeatures.length > 0 || checkFeatures.size > 0) &&
        <Button disabled={!featuresChanged} positive onClick={() => applyFeatures()}>
          <Icon name="check"/>{__("Apply Features")}
        </Button>}
      </div>

      {(radioFeatures.length > 0 || checkFeatures.size > 0) &&
      <div className="VaeConfig-Features">
        <span className="VaeConfig-FeaturesLabel">{__("Features:")}</span>
        <div className="VaeConfig-RadioFeatures">
          {radioFeatures.map(feature =>
            <Checkbox
              key={feature.id}
              radio
              label={feature.name}
              name="radioFeatures"
              value={feature.id}
              checked={feature.id === radioFeature}
              onChange={(e, {value}) => typeof value === "string" && setRadioFeature(value)}
                />)}
        </div>
        <div className="VaeConfig-CheckFeatures">
          {[...checkFeatures.keys()].map(featureId =>
            <Checkbox
              key={featureId}
              toggle
              label={checkFeatures.get(featureId)?.name}
              value={featureId}
              checked={checkFeatures.get(featureId)?.checked}
              onChange={onCheckFeatureChange}
          />)}
        </div>
      </div>}

      <WithQueryStatus error={aspectError || confError} loading={confLoading}>
        {!!error && <Message error content={error.message}/>}

        <Tab
          panes={panes}
          className="VaeConfig-Tab"
        />

        {(updateLoading || createConfLoading || renameConfLoading || deleteConfLoading || refetching) && <Loading text={__("Updating...")}/>}

        {deleteConfirmOpen &&
          <Modal open={deleteConfirmOpen} onClose={() => setDeleteConfirmOpen(false)}>
            <Header>{__("Delete Configuration")}</Header>
            <Modal.Content>{__("Are you sure you want to delete configuration '{{name}}'?", {name: confValue})}</Modal.Content>
            <Modal.Actions>
              <Button
                negative
                onClick={() => {
                  onDeleteConfig();
                  setDeleteConfirmOpen(false);
                }}>
                <Icon name="trash"/>{__("Delete")}
              </Button>
              <Button onClick={() => setDeleteConfirmOpen(false)}>
                <Icon name="cancel"/>{__("Cancel")}
              </Button>
            </Modal.Actions>
          </Modal>}
      </WithQueryStatus>
    </Segment>
  );
};

export default VaeConfig;
