import React, { useMemo, useEffect, useState } from "react";
import { Checkbox, Form, Icon, Popup, Table } from "semantic-ui-react";

import {
  StoragePool,
  MediaStreamType,
  ProbeDeviceQuery
} from "@generated/graphql";
import { __ } from "@solid/libs/i18n";
import { DeviceShort} from "@core/actions";

import { IsCorrectType, ImportCameraEntry, DeviceConfigType, indexShift } from "../ImportCameraList";

import "./style.css";

type ImportDataRenderProps = {
  listItem: ImportCameraEntry,
  deviceList: DeviceShort[],
  zonesList: string[],
  storageList: StoragePool[],
  startIndex: number,
  handleChangeItem: (index: number, listItem: ImportCameraEntry) => void,
  isCorrect: (item: ImportCameraEntry) => IsCorrectType,
  getStorageId: (name?: string) => string,
  aspects?: ProbeDeviceQuery["probeDevice"]["aspects"]
};

export type OptionType = {
  key: string,
  text: string,
  value: string
};

const ImportDataRender = ({
  listItem,
  deviceList,
  zonesList,
  storageList,
  startIndex,
  handleChangeItem,
  isCorrect,
  getStorageId,
  aspects
}: ImportDataRenderProps) => {
  const isValid = listItem.valid;
  const [definitions, setDefinitions] = useState<ImportCameraEntry>({
    deviceZone: listItem.deviceZone ?? "",
    deviceName: listItem.deviceName ?? "",
    deviceLocation: listItem.deviceLocation ?? "",
    cloudDirect: listItem.cloudDirect === "1" || !!listItem.cloudDirect,
    avatarId: listItem.avatarId ?? "",
    storagePool: storageList.some(storage => storage.id === listItem.storagePool) ? listItem.storagePool : getStorageId(listItem.storagePool),
    deviceAddress: listItem.deviceAddress ?? "",
    username: listItem.username ?? "",
    password: listItem.password ?? "",
    deviceType: listItem.deviceType ?? "",
    deviceUrl: listItem.deviceUrl ?? "",
    onvifProfile: listItem.onvifProfile ?? "",
    httpPort: isNaN(Number(listItem.httpPort)) ? 0 : Number(listItem.httpPort),
    rtcpPort: isNaN(Number(listItem.rtcpPort)) ? 0 : Number(listItem.rtcpPort),
    correct: listItem.correct,
    id: listItem.id,
    valid: listItem.valid
  });
  const [correctMessage, setCorrectMessage] = useState<string>("");

  const onvifProfilesList = useMemo(() => {
    const profileList: string[] = [];
    aspects?.forEach(aspect =>
      aspect.__typename === "DFA_Media" &&
      aspect.streamType === MediaStreamType.Video &&
      aspect.onvifProfile &&
      profileList.push(aspect.onvifProfile)
    );
    return profileList;
  }, [aspects]);

  const storagePoolOptions = useMemo(() => {
    const options: OptionType[] = [];
    options.push({key: "default", text: "", value: ""});
    storageList.forEach((item: StoragePool) => {
      const option = {};
      option["key"] = item.name;
      option["text"] = item.name;
      option["value"] = item.id;
      options.push(option as OptionType);
    });
    return options;
  }, [storageList]);

  const avatarOptions = useMemo(() => {
    const options: OptionType[] = [];
    options.push({key: "default", text: "", value: ""});
    deviceList.forEach((item: DeviceShort) => {
      const option = {};
      option["key"] = item.id;
      option["text"] = `[${item.id}] ${item.name}`;
      option["value"] = item.id;
      options.push(option as OptionType);
    });
    return options;
  }, [deviceList]);

  const deviceType = [DeviceConfigType.Onvif, DeviceConfigType.Url];
  const itemIndex = startIndex + indexShift;

  const listToOptions = (list: string[]) => {
    const options: OptionType[] = [];
    options.push({key: "default", text: "", value: ""});
    list.forEach((item: string) => {
      const option = {};
      option["key"] = item;
      option["text"] = item;
      option["value"] = item;
      options.push(option as OptionType);
    });
    return options;
  };

  const handleChange = (name: string, value: any) => {
    const newDefinitions = {
      ...definitions,
      [name]: value
    };

    setDefinitions({
      ...newDefinitions,
      correct: isCorrect(newDefinitions).correct
    });

    handleChangeItem(startIndex, {
      ...newDefinitions,
      correct: isCorrect(newDefinitions).correct
    });
  };

  useEffect(() => {
    const definitions: ImportCameraEntry = {
      ...listItem,
      deviceZone: zonesList.some(zone => zone === listItem.deviceZone) ? listItem.deviceZone : "",
      avatarId: deviceList.some(dev => dev.id === listItem.avatarId) ? listItem.avatarId : "",
      storagePool: storageList.some(storage => storage.id === listItem.storagePool) ? listItem.storagePool : getStorageId(listItem.storagePool),
      deviceType: deviceType.includes((listItem.deviceType as DeviceConfigType) ?? "") ? listItem.deviceType : "",
      deviceUrl: listItem.deviceUrl ?? "",
      onvifProfile: listItem.onvifProfile ?? "",
      httpPort: isNaN(Number(listItem.httpPort)) ? 0 : Number(listItem.httpPort),
      rtcpPort: isNaN(Number(listItem.rtcpPort)) ? 0 : Number(listItem.rtcpPort),
      username: listItem.username ?? "",
      password: listItem.password ?? "",
      deviceAddress: listItem.deviceAddress ?? "",
      deviceLocation: listItem.deviceLocation ?? "",
      deviceName: listItem.deviceName ?? "",
      cloudDirect: listItem.cloudDirect === "1" || !!listItem.cloudDirect,
    };
    setDefinitions(definitions);
  },  [listItem]);

  useEffect(() => {
    if (!isValid) {
      setCorrectMessage(__("Error camera definitions"));
      return;
    }
    const isCorrectDef = isCorrect(definitions);
    updateCorrectMessage(definitions, isCorrectDef);
  }, [
    definitions.deviceName,
    definitions.deviceZone,
    definitions.cloudDirect,
    definitions.deviceAddress,
    definitions.avatarId,
    definitions.storagePool,
    definitions.deviceType,
    definitions.deviceUrl,
    definitions.onvifProfile
  ]);

  useEffect(() => {
    if (onvifProfilesList.length === 0) return;

    if (onvifProfilesList.some(profile => profile === definitions.onvifProfile)) {
      setDefinitions(prevDefinitions => ({
        ...prevDefinitions,
        correct: isCorrect(prevDefinitions).correct
      }));
      handleChangeItem(startIndex, {
        ...definitions,
        correct: isCorrect(definitions).correct
      });
    }
    else {
      const newDefinitions = { ...definitions, onvifProfile: "" };
      setDefinitions(newDefinitions);
      handleChangeItem(startIndex, {
        ...newDefinitions,
        correct: isCorrect(newDefinitions).correct
      });
    }
  }, [definitions.onvifProfile, onvifProfilesList]);

  function updateCorrectMessage(definitions: ImportCameraEntry, isCorrectDef: IsCorrectType) {
    if (!isCorrectDef.correct) {
      if (!isCorrectDef.isCorrectZone) {
        setCorrectMessage(__("Zone not specified"));
        return;
      }
      if (!isCorrectDef.isCloudDirect) {
        setCorrectMessage(__("Avatar ID not specified"));
        return;
      }
      if (!isCorrectDef.isName) {
        setCorrectMessage(__("Name not specified"));
        return;
      }
      if (!isCorrectDef.isAddress) {
        setCorrectMessage(__("Address not specified"));
        return;
      }
      if (!isCorrectDef.isCorrectType) {
        if (definitions.deviceType === DeviceConfigType.Onvif && definitions.onvifProfile === "") {
          setCorrectMessage(__("ONVIF profile not specified"));
          return;
        }
        else if (definitions.deviceType === DeviceConfigType.Url && definitions.deviceUrl === "") {
          setCorrectMessage(__("Device URL not specified"));
          return;
        }
        else if (definitions.onvifProfile && definitions.deviceUrl) {
          setCorrectMessage(__("Choose only Device URL or ONVIF profile"));
          return;
        }
        setCorrectMessage(__("Device type not specified"));
        return;
      }
      if (!isCorrectDef.isCorrectStorage) {
        setCorrectMessage(__("Storage pool not specified"));
        return;
      }
    }
    else {
      setCorrectMessage("");
    }
  }

  return (
    <Table.Row>
      <Table.Cell collapsing textAlign="center">
        {itemIndex}
      </Table.Cell>
      <Table.Cell collapsing>
        {!isValid || !definitions.correct ?
          <div className="Import-Action-Icon" onChange={() => handleChangeItem(startIndex, {...definitions})}>
            <Popup
              content={correctMessage}
              trigger={
                <Icon name="warning sign"/>
            }/>
          </div> :
          definitions.id ?
            <div className="Import-Action-Icon">
              <Popup
                content={__("Created successfully")}
                trigger={
                  <Icon name="check" color="green" size="large"/>
                }/>
            </div> :
            null}
      </Table.Cell>
      <Table.Cell collapsing>
        {isValid &&
        <label>
          <input
            id={`deviceName_${itemIndex}`}
            disabled={!!definitions.id}
            type="text"
            placeholder={__("Device name")}
            value={definitions.deviceName}
            className="Import-Input-FullWidth"
            onChange={(e) => handleChange("deviceName", e.target.value)}
          />
        </label>}
      </Table.Cell>
      <Table.Cell collapsing>
        {isValid &&
        <label>
          <input
            id={`deviceLocation_${itemIndex}`}
            disabled={!!definitions.id}
            type="text"
            placeholder={__("Location name")}
            value={definitions.deviceLocation}
            className="Import-Input"
            onChange={(e) => handleChange("deviceLocation", e.target.value)}
          />
        </label>}
      </Table.Cell>
      <Table.Cell collapsing>
        {isValid &&
          <Form.Select
            id={`selectZone_${itemIndex}`}
            search
            disabled={!!definitions.id}
            options={listToOptions(zonesList)}
            placeholder={__("Select zone")}
            value={definitions.deviceZone}
            className="Import-Select"
            onChange={(e: any, {value}: any) => handleChange("deviceZone", value)}
          />}
      </Table.Cell>
      <Table.Cell collapsing>
        {isValid &&
          <div className="Import-Checkbox">
            <Checkbox
              disabled={!!definitions.id}
              checked={!!definitions.cloudDirect}
              onChange={(e, {checked}) => handleChange("cloudDirect", !!checked)}
              toggle/>
          </div>}
      </Table.Cell>
      <Table.Cell collapsing>
        {isValid &&
          <Form.Select
            id={`selectStorage_${itemIndex}`}
            search
            disabled={!!definitions.id}
            options={storagePoolOptions}
            placeholder={__("Select storage pool")}
            value={definitions.storagePool}
            className="Import-Select"
            onChange={(e, {value}) => handleChange("storagePool", value)}
          />}
      </Table.Cell>
      <Table.Cell collapsing>
        {isValid &&
          <Form.Select
            id={`selectAvatar_${itemIndex}`}
            search
            disabled={!!definitions.id}
            options={avatarOptions}
            placeholder={__("Select avatar ID")}
            value={definitions.avatarId}
            className="Import-Select"
            onChange={(e, {value}) => handleChange("avatarId", value)}
          />}
      </Table.Cell>
      <Table.Cell collapsing>
        {isValid &&
          <label>
            <input
              id={`deviceAddress_${itemIndex}`}
              disabled={!!definitions.id}
              type="text"
              placeholder={__("Device address")}
              value={definitions.deviceAddress}
              className="Import-Input-FullWidth"
              onChange={(e) => handleChange("deviceAddress", e.target.value)}
            />
          </label>}
      </Table.Cell>
      <Table.Cell collapsing>
        {isValid &&
        <label>
          <input
            id={`username_${itemIndex}`}
            disabled={!!definitions.id}
            type="text"
            placeholder={__("User name")}
            value={definitions.username}
            className="Import-Input"
            onChange={(e) => handleChange("username", e.target.value)}
          />
        </label>}
      </Table.Cell>
      <Table.Cell collapsing>
        {isValid &&
        <label>
          <input
            id={`password_${itemIndex}`}
            disabled={!!definitions.id}
            type="text"
            placeholder={__("User password")}
            value={definitions.password}
            className="Import-Input"
            onChange={(e) => handleChange("password", e.target.value)}
          />
        </label>}
      </Table.Cell>
      <Table.Cell collapsing>
        {isValid &&
          <Form.Select
            id={`selectDeviceType_${itemIndex}`}
            search
            disabled={!!definitions.id}
            options={listToOptions(deviceType)}
            placeholder={__("Select device type")}
            value={definitions.deviceType}
            className="Import-Select"
            onChange={(e, {value}) => handleChange("deviceType", value)}
            />}
      </Table.Cell>
      <Table.Cell collapsing>
        {isValid &&
        <label>
          <input
            id={`deviceUrl_${itemIndex}`}
            disabled={!!definitions.id}
            type="text"
            placeholder={__("Device url")}
            value={definitions.deviceUrl}
            className="Import-Input"
            onChange={(e) => handleChange("deviceUrl", e.target.value)}
          />
        </label>}
      </Table.Cell>
      <Table.Cell collapsing>
        {isValid ?
          onvifProfilesList.length > 0 ?
            <Form.Select
              id={`selectOnvifProfile_${itemIndex}`}
              search
              disabled={!!definitions.id}
              options={listToOptions(onvifProfilesList)}
              placeholder={__("Select onvif profile")}
              value={definitions.onvifProfile}
              className="Import-Select"
              onChange={(e, {value}) => handleChange("onvifProfile", value)}
            /> :
            <label>
              <input
                disabled={!!definitions.id}
                type="text"
                placeholder={__("Device profile")}
                value={definitions.onvifProfile}
                className="Import-Input"
                onChange={(e) => handleChange("onvifProfile", e.target.value)}
              />
            </label> :
          null}
      </Table.Cell>
      <Table.Cell collapsing>
        {isValid &&
        <label>
          <input
            id={`httpPort_${itemIndex}`}
            disabled={!!definitions.id}
            type="text"
            placeholder={__("Http Port")}
            value={definitions.httpPort}
            className="Import-Input number"
            onChange={(e) => {
              const value = isNaN(Number(e.target.value)) ? undefined : Number(e.target.value);
              typeof value === "number" && handleChange("httpPort", value);
            }}
          />
        </label>}
      </Table.Cell>
      <Table.Cell collapsing disabled={!!definitions.id}>
        {isValid &&
        <label>
          <input
            id={`rtcpPort_${itemIndex}`}
            disabled={!!definitions.id}
            type="text"
            placeholder={__("Rtcp Port")}
            value={definitions.rtcpPort}
            className="Import-Input number"
            onChange={(e) => {
              const value = isNaN(Number(e.target.value)) ? undefined : Number(e.target.value);
              typeof value === "number" && handleChange("rtcpPort", value);
            }}
          />
        </label>}
      </Table.Cell>
    </Table.Row>
  );
};

export default ImportDataRender;
