import React, { useEffect, useRef, useMemo } from "react";
import { Card, Icon, Input, Checkbox, Label, Message, Dropdown } from "semantic-ui-react";
import classNames from "classnames";
import { useSnapshot } from "@core/actions";
import { DiscoveredDevice, DeviceUserInput } from "../CameraDiscovery";
import Loading from "components/Loading";
import {__} from "@solid/libs/i18n";

type CameraOnboardProps = {
  device: DiscoveredDevice;
  onUserInput: (device: DiscoveredDevice, input: DeviceUserInput) => void;
};

const CameraOnboard = ({ device, onUserInput }: CameraOnboardProps) => {
  const snapshot = device.snapshots.length > 0 ? device.snapshots[0] : "";
  const imageRef = useRef<HTMLImageElement>(null);
  const { imageSrc, loading: snapshotLoading } = useSnapshot({ load: !!device.device, deviceId: device.device?.id ?? "", imageRef });
  const snapshotRef = useRef<HTMLImageElement>(null);
  const streamOptions = useMemo(() => device.aspects
    .filter(aspect => aspect.template.id === "video")
    .map(({ name }) => ({ key: name, value: name, text: name })), [device.aspects]);

  useEffect(() => {
    setTimeout(() => {
      if (snapshotRef.current) {
        snapshotRef.current.src = "";
        snapshotRef.current.src = `data:image/jpeg;base64, ${snapshot}`;
      }
    }, 250);
  }, [snapshot]);

  return (
    <Card className="CameraDiscovery-Camera" color={device.device ? "green" : undefined}>
      <Card.Content>
        {!device.device &&
        <Checkbox checked={device.userInput.isSelected} onChange={(e, { checked }) => onUserInput(device, { ...device.userInput, isSelected: !!checked })}/>}

        {!!device.device &&
        <Label color="blue" content={<>{__("Registered as")}<br/>{`'${device.device.name}'`}</>} className="first"/>}
      </Card.Content>

      <div className="CameraDiscovery-CameraImage">
        {!!snapshot && <img ref={snapshotRef} src={`data:image/jpeg;base64, ${snapshot}`} alt=""/>}

        {!snapshot &&
        <>
          <img ref={imageRef} src={imageSrc} alt="" className={classNames({ "hidden": !imageSrc })}/>

          {snapshotLoading && <Loading/>}

          {!imageSrc &&
          <>
            <Icon name="image outline" size="huge"/>
            <div>{__("Image is not available")}</div>
          </>}
        </>}
      </div>

      {!device.device &&
      <Card.Content>
        {!!device.userInput.validationError &&
        <Message error content={device.userInput.validationError}/>}

        <div className="CameraDiscovery-CameraFields">
          <div className="CameraDiscovery-CameraFieldLabel">{__("Name")}</div>
          <Input fluid value={device.userInput.name} onChange={e => onUserInput(device, { ...device.userInput, name: e.currentTarget.value })}/>

          <div className="CameraDiscovery-CameraFieldLabel">{__("Stream")}</div>
          <Dropdown
            fluid
            selection
            options={streamOptions}
            value={device.userInput.streamName}
            disabled={streamOptions.length === 0}
            onChange={(e, { value }) => typeof value === "string" && onUserInput(device, { ...device.userInput, streamName: value })}
          />
        </div>
      </Card.Content>}
    </Card>
  );
};

export default CameraOnboard;
