import React, {useEffect, useState, useCallback} from "react";
import ActivityBar from "./ActivityBar";
import ActivityItem, {ActivityItemProps} from "./ActivityItem";
import SideBar from "./SideBar";
import {ACTIVITY_BAR_WIDTH, SIDE_PANEL_WIDTH} from "components/SidePanel/info";

import "./style.css";

type ActivityProps = {
  children?: React.ReactElement<ActivityItemProps, typeof ActivityItem>[]
  onResize?: (size: {width: number}) => void,
  currentIndex?: number,
  isSideBarVisible?: boolean,
  onChange?: (index: number) => void,
  onSideBarShow?: (visible: boolean) => void
};

const Activity = (props: ActivityProps) => {
  const [currentActivityItem, setCurrentActivityItem] = useState(0);
  const [isSideBarVisible, setIsSideBarVisible] = useState(true);
  const [holdSideBarIndexes, setHoldSideBarIndexes] = useState<number[]>([]);

  const currentIndex = props.currentIndex ?? currentActivityItem;
  const sideBarVisible = props.isSideBarVisible ?? isSideBarVisible;

  useEffect(() => {
    if (props.children) {
      props.children?.forEach((child, index) => {
        if (child && child.props.holdSideBar) {
          holdSideBarIndexes.push(index);
        }
        setHoldSideBarIndexes(holdSideBarIndexes);
      });
    }
  }, []);

  useEffect(useCallback(() => {
    const width = sideBarVisible ? SIDE_PANEL_WIDTH : ACTIVITY_BAR_WIDTH;
    const widget = document.getElementById("main_left");
    if (widget) {
      widget.style.width = width + "px";
      widget.style.minWidth = width + "px";
      widget.style.maxWidth = width + "px";
    }
    props.onResize && props.onResize({width});
  }, [sideBarVisible, props.onResize]), [sideBarVisible]);

  const showSideBar = useCallback((visible: boolean) => {
    props.onSideBarShow && props.onSideBarShow(visible);
    props.isSideBarVisible === undefined && setIsSideBarVisible(visible);
  }, [props.isSideBarVisible, props.onSideBarShow]);

  const onActivityBarChange = useCallback((index: number) => {
    if (holdSideBarIndexes.includes(index)) {
      return;
    }
    if (index !== currentIndex) {
      showSideBar(true);
      props.onChange && props.onChange(index);
      props.currentIndex === undefined && setCurrentActivityItem(index);
    } else {
      showSideBar(!sideBarVisible);
    }
  }, [currentIndex, sideBarVisible, props.currentIndex, props.onChange, showSideBar]);

  return (
    <div className="activity">
      <ActivityBar onChange={onActivityBarChange} current={currentIndex} inactive={!sideBarVisible}>
        {props.children}
      </ActivityBar>
      {sideBarVisible &&
      <SideBar showSideBar={showSideBar}>
        {props.children && typeof props.children[currentIndex] !== "undefined" ? props.children[currentIndex] : undefined}
      </SideBar>}
    </div>);
};

Activity.propTypes = {
  children: (props: ActivityProps, propName: string, componentName: string) => {
    const prop = props[propName];

    let error = null;
    React.Children.forEach(prop, (child) => {
      if (child.type !== ActivityItem) {
        error = new Error("`" + componentName + "` children should be of type `ActivityItem`.");
      }
    });
    return error;
  }
};

export default Activity;
