import { ESLint } from "eslint";
import { FuseLog } from "fuse-box-typechecker/dist/fuseLogger/fuseLog";

export type InspectorOptions = {
  patterns: string[];
  formatter?: string;
  cache?: boolean;
  cacheLocation?: string;
  name?: string;
};

const defaultOptions: InspectorOptions = {
  patterns: ["."]
};

const logger = new FuseLog();

class Inspector {
  private options = defaultOptions;
  private results: ESLint.LintResult[] = [];

  private createESLint(): ESLint {
    return new ESLint({
      cache: this.options.cache,
      cacheLocation: this.options.cacheLocation
    });
  }

  private mergeOptions(options?: Partial<InspectorOptions>): void {
    if (options) {
      this.options = { ...defaultOptions, ...options };
    }
  }

  async inspect(options?: Partial<InspectorOptions>): Promise<void> {
    this.mergeOptions(options);
    const eslint = this.createESLint();
    this.results = await eslint.lintFiles(this.options.patterns);
  }

  async print(options?: Partial<InspectorOptions>): Promise<void> {
    this.mergeOptions(options);
    const eslint = this.createESLint();
    const formatter = await eslint.loadFormatter(this.options.formatter);
    const messages = formatter.format(this.results);
    if (messages) {
      console.log(messages);
    }
    else {
      logger.info(`<bgGreen> SUCCESS </bgGreen> <green>ESLint - (${this.options.name}): No Errors found</green>`);
    }
  }
}

export default Inspector;
