/* eslint-disable import/no-extraneous-dependencies */
import {loadENV} from "./env";
import {fusebox, pluginLess, pluginPostCSS, sparky} from "@libs/fuse-box";
import {pluginTypeChecker} from "fuse-box-typechecker";
import fetch, {FetchError} from "node-fetch";
import * as path from "path";
import * as fs from "fs";
import * as electronWinInstaller from "electron-winstaller";
import electronDmgInstaller from "electron-installer-dmg";
import {IRunResponse} from "@libs/fuse-box/core/IRunResponse";
import {SquirrelWindowsOptions} from "electron-winstaller/lib/options";
import electronPackager from "electron-packager";
import {pluginGraphQL} from "./plugins/plugin_graphql";
import {pluginESLint} from "./plugins/eslint";
import * as png2icons from "png2icons";
import * as new_exec from "child_process";
import util from "util";

loadENV();

enum Mode {
  PRODUCTION = "production",
  DEVELOPMENT = "development"
}

// @ts-ignore
function isMode(arg: string): arg is Mode {
  return arg === Mode.DEVELOPMENT || arg === Mode.PRODUCTION;
}

enum Target {
  BROWSER = "browser",
  ELECTRON = "electron"
}

function isTarget(arg: string): arg is Target {
  return arg === Target.BROWSER || arg === Target.ELECTRON;
}

class Context {
  getAppFuse(mode: Mode, target: Target, runServer: boolean = false, isElectronPackage: boolean = false) {
    const isDev = mode !== Mode.PRODUCTION;
    const isBrowser = target !== Target.ELECTRON;

    const env: Record<string, string> = {};
    if (isDev) {
      env.API_GRAPHQL = process.env.API_GRAPHQL;
      env.VERSION = "latest";
    } else {
      env.VERSION = process.env.VERSION ?? "latest";
    }

    return fusebox({
      target: isBrowser ? "browser" : "electron",
      electron: {
        nodeIntegration: true
      },
      entry: "src/index.ts",
      env,
      webIndex: {
        publicPath: "./",
        template: "src/index.html"
      },
      sourceMap: {
        project: true,
        vendor: true,
        css: !isDev
      },
      devServer: runServer && !isElectronPackage ?
        (process.env.API_BACKEND ? {
          httpServer: isBrowser ? {port: 5555} : false,
          proxy: [
            {path: "/api", options: {target: process.env.API_BACKEND, changeOrigin: true, ws: true, /*followRedirects: true,*/ proxyTimeout: 60000, timeout: 60000}},
            {path: "/sdi", options: {target: process.env.API_BACKEND, changeOrigin: true, ws: false, /*followRedirects: true,*/ proxyTimeout: 60000, timeout: 60000}},
            {path: "/wetty", options: {target: process.env.API_BACKEND, changeOrigin: true, ws: false, /*followRedirects: true,*/ proxyTimeout: 60000, timeout: 60000}},
            {path: "/elog-restful", options: {target: process.env.API_BACKEND, changeOrigin: true, ws: false, /*followRedirects: true,*/ proxyTimeout: 60000, timeout: 60000}}
          ],
          hmrServer: { port: 5556 }
        } : true) : false,
      cache: {
        enabled: true,
        root: isBrowser ? "build/.cache/browser" : "build/.cache/renderer"
      },
      hmr: {
        enabled: true
      },
      watcher: runServer && !isElectronPackage ? {
        root: path.resolve(__dirname, "../"),
        chokidarOptions: process.platform === "darwin" ? {
          followSymlinks: false,
          useFsEvents: false,
          usePolling: true,
          interval: 2000
        } : undefined
      } : false,
      logging: {level: "succinct"},
      stylesheet: {
        groupResourcesFilesByType: true,
        macros: {
          "~@lumino": path.resolve(__dirname, "../../", "node_modules", "@lumino")
        }
      },
      resources: {
        resourceFolder: "./resources",
        resourcePublicRoot: runServer || isElectronPackage ? "resources" : "/solid/resources"
      },
      link: {resourcePublicRoot: "resources"},
      // use commonjs react-virtualized build instead of wrong esnext build
      alias: {
        "react-virtualized-auto-sizer": "react-virtualized-auto-sizer",
        "react-virtualized/dist": "react-virtualized/dist",
        "react-virtualized/styles.css": "react-virtualized/styles.css",
        "react-virtualized": "react-virtualized/dist/commonjs"
      },
      modules: [
        "./modules"
      ],
      plugins: [
        pluginPostCSS("*.css"),
        pluginLess("*.less", {
          stylesheet: {
            macros: {
              "../../theme.config": path.resolve(__dirname, "src/semantic-ui/theme.config"),
              "themes/grey": path.resolve(__dirname, "src/semantic-ui/themes/grey"),
              "themes/purple": path.resolve(__dirname, "src/semantic-ui/themes/purple"),
              "~semantic-ui-less": path.resolve(__dirname, "../../", "node_modules/semantic-ui-less"),
              "@spritePath": path.resolve(__dirname, "../../", "node_modules/semantic-ui-less/themes/default/assets/images/flags.png"),
              "@{fontPath}/@{fontName}": path.resolve(__dirname, "../../", "node_modules/semantic-ui-less/themes/default/assets/fonts/icons"),
              "@{fontPath}/@{outlineFontName}": path.resolve(__dirname, "../../", "node_modules/semantic-ui-less/themes/default/assets/fonts/outline-icons"),
              "@{fontPath}/@{brandFontName}": path.resolve(__dirname, "../../", "node_modules/semantic-ui-less/themes/default/assets/fonts/brand-icons")
            }
          }
        }),
        pluginGraphQL("*.graphql"),
        ...(
          runServer || !isDev
            ? [
              pluginTypeChecker({
                tsConfig: isBrowser ? "./tsconfig.json" : "./tsconfig.renderer.json",
                name: "TypeChecker App"
              }),
              pluginESLint({
                name: "ESLint App",
                cache: true,
                cacheLocation: "build/"
              })
            ] : []
        )
      ]
    });
  }

  getElectronFuse(mode: Mode, isElectronPackage: boolean = false) {
    const isDev = mode !== Mode.PRODUCTION;

    return fusebox({
      target: "server",
      entry: "src/electron/main.ts",
      dependencies: { serverIgnoreExternals: true },
      env: {
        VERSION: !isDev ? require("./src/electron/package.json").version : "latest"
      },
      sourceMap: {
        project: true,
        vendor: true,
        css: !isDev
      },
      hmr: false,
      watcher: isDev && !isElectronPackage,
      logging: { level: "succinct" },
      cache: {
        enabled: true,
        root: "build/.cache/electron"
      },
      plugins: [
        pluginTypeChecker({
          tsConfig: "./tsconfig.electron.json",
          name: "TypeChecker Electron"
        })
      ]
    });
  }
}

const { task, src, rm, exec } = sparky<Context>(Context);

async function build(context: Context, mode: Mode, target: Target, runServer: boolean = false, isElectronPackage: boolean = false) {
  const isDev = mode !== Mode.PRODUCTION;
  const isBrowser = target !== Target.ELECTRON;
  if (isBrowser) {
    // clean previous browser build
    if (!isDev) {
      rm("build/.cache");
      rm("build/browser");
    }

    await src("./src/loading/loading.css")
      .dest("build/browser", "loading/")
      .exec();

    await src("./src/generated/locales/ar-AE/translation.json")
      .dest("build/browser", "generated/")
      .exec();

    await src("./src/components/Admin/AdminCameraList/ImportCameraList/cameras.csv")
      .dest("build/browser", "ImportCameraList/")
      .exec();

    const pdfjsDistPath = path.dirname(require.resolve("pdfjs-dist/package.json"));
    const pdfWorkerPath = path.join(pdfjsDistPath, "legacy", "build", "pdf.worker.min.js");
    await src(pdfWorkerPath)
      .dest("build/browser", "build/")
      .exec();

    if (isDev) {
      const app = context.getAppFuse(Mode.DEVELOPMENT, Target.BROWSER, runServer);
      await app.runDev({
        uglify: false,
        bundles: {
          distRoot: "build/browser",
          cssSplitting: true,
          app: "app.js",
          vendor: "vendor.js",
        }
      });
    } else {
      const app = context.getAppFuse(Mode.PRODUCTION, Target.BROWSER, runServer);
      await app.runProd({
        uglify: true,
        bundles: {
          distRoot: "build/browser",
          cssSplitting: true,
          app: "app.$hash.js",
          vendor: "vendor.$hash.js",
          mapping: [
            { matching: "semantic*", target: "./vendor.semantic.$hash.js" },
            { matching: "@fortawesome*", target: "./vendor.fortawesome.$hash.js" },
            { matching: "@libs*", target: "./vendor.libs.$hash.js" },
            { matching: "@lumino*", target: "./vendor.lumino.$hash.js" },
            { matching: "moment*", target: "./vendor.moment.$hash.js" },
            { matching: "(graphql*|@apollo*)", target: "./vendor.graphql.$hash.js" },
            { matching: "vis*", target: "./vendor.vis.$hash.js" },
            { matching: "(react-pdf*|pdfjs*)", target: "./vendor.pdf.$hash.js" },
            { matching: ".*react.*", target: "./vendor.react.$hash.js" },
          ]
        }
      });
    }
  } else {
    if (!process.env.API_BACKEND) {
      console.error("API_BACKEND is not defined");
      process.exit(1);
    }

    // clean previous electron build
    if (!isDev) {
      rm("build/.cache");
      rm("build/electron");
    }

    process.env?.PLAYER !== "mse" && await exec("copy:plugin");

    await src("./src/loading/loading.css")
      .dest("build/electron/renderer", "loading/")
      .exec();

    // copy pdf viewer
    const pdfjsDistPath = path.dirname(require.resolve("pdfjs-dist/package.json"));
    const pdfWorkerPath = path.join(pdfjsDistPath, "legacy", "build", "pdf.worker.min.js");
    await src(pdfWorkerPath)
      .dest("build/electron/renderer", "build/")
      .exec();

    if (isDev) {
      const app = context.getAppFuse(Mode.DEVELOPMENT, Target.ELECTRON, runServer, isElectronPackage);
      await app.runDev({
        uglify: false,
        bundles: {
          distRoot: "build/electron/renderer",
          cssSplitting: true,
          app: "app.js",
          vendor: "vendor.js"
        }
      });
    } else {
      const app = context.getAppFuse(Mode.PRODUCTION, Target.ELECTRON, runServer, isElectronPackage);
      await app.runProd({
        uglify: true,
        bundles: {
          distRoot: "build/electron/renderer",
          cssSplitting: true,
          app: "app.js",
          vendor: "vendor.js",
          mapping: [
            { matching: "@libs*", target: "./vendor.libs.js" },
          ]
        }
      });
    }

    await src("./src/electron/preload_main.js")
      .dest("build/electron/main", "electron")
      .exec();

    await src("./src/electron/preload_child.js")
      .dest("build/electron/main", "electron")
      .exec();

    // To avoid warning in console coming from some electron packages looking for package.json.
    await src("package.json")
      .dest("build/electron/renderer", "client")
      .exec();

    await src("./src/generated/locales/ar-AE/translation.json")
      .dest("build/electron/renderer", "generated/")
      .exec();

    await src("./src/components/Admin/AdminCameraList/ImportCameraList/cameras.csv")
      .dest("build/electron/renderer", "ImportCameraList/")
      .exec();

    await src("./src/electron/img/cancel.png")
      .dest("build/electron/main", "electron")
      .exec();

    await src("./src/electron/img/resume.png")
      .dest("build/electron/main", "electron")
      .exec();

    await src("./src/electron/img/pause.png")
      .dest("build/electron/main", "electron")
      .exec();

    const isElectronDev = isDev && !isElectronPackage;

    let onComplete: IRunResponse["onComplete"];
    if (isElectronDev) {
      const electron = context.getElectronFuse(Mode.DEVELOPMENT, isElectronPackage);
      const response = await electron.runDev({
        bundles: {
          distRoot: "build/electron/main",
          cssSplitting: true,
          app: "app.js",
          vendor: "vendor.js"
        }
      });
      onComplete = response.onComplete;
    } else {
      const electron = context.getElectronFuse(Mode.PRODUCTION, isElectronPackage);
      const response = await electron.runProd({
        manifest: true,
        bundles: {
          distRoot: "build/electron/main",
          cssSplitting: true,
          app: "app.js",
          vendor: "vendor.js"
        }
      });
      onComplete = response.onComplete;
    }

    if (isElectronPackage) {
      await src("./src/electron/package.json")
        .dest("build/electron", "electron")
        .exec();
      await src("./src/electron/yarn.lock")
        .dest("build/electron", "electron")
        .exec();

      const env = {
        API_BACKEND: process.env.API_BACKEND,
        API_MINIDUMP: process.env.API_MINIDUMP
      };
      const dotENV = Object.entries(env)
        .map(([key, value]) => `${key}=${value}`)
        .join("\n");
      await fs.promises.writeFile("build/electron/main/.env", dotENV);
    } else {
      onComplete(({electron}) => {
        if (!isElectronPackage) {
          electron?.start();
        }
      });
    }
  }
}

task("default", async (context) => {
  throw new Error("Build task is not specified");
});

task("start", async (context) => {
  if (!isTarget(process.env.BUILD_TARGET)) {
    console.error("Incorrect process.env.BUILD_TARGET");
    process.exit(1);
  }

  await build(context, Mode.DEVELOPMENT, process.env.BUILD_TARGET, true);
});

task("build", async (context) => {
  if (!isTarget(process.env.BUILD_TARGET)) {
    console.error("Incorrect process.env.BUILD_TARGET");
    process.exit(1);
  }

  await build(context, Mode.PRODUCTION, process.env.BUILD_TARGET);
});

task("build:dev", async (context) => {
  if (!isTarget(process.env.BUILD_TARGET)) {
    console.error("Incorrect process.env.BUILD_TARGET");
    process.exit(1);
  }

  await build(context, Mode.DEVELOPMENT, process.env.BUILD_TARGET);
});

task("build:prod", async (context) => {
  if (!isTarget(process.env.BUILD_TARGET)) {
    console.error("Incorrect process.env.BUILD_TARGET");
    process.exit(1);
  }

  await build(context, Mode.PRODUCTION, process.env.BUILD_TARGET);
});

task("build:browser", async (context) => {
  await build(context, Mode.PRODUCTION, Target.BROWSER);
});

task("build:electron", async (context) => {
  await build(context, Mode.DEVELOPMENT, Target.ELECTRON);
});

task("build:electron-package", async (context) => {
  await build(context, Mode.DEVELOPMENT, Target.ELECTRON, false, true);
});

task("sentry-upload", async () => {
  if (process.env.VERSION === "latest") {
    return;
  }

  try {
    await util.promisify(new_exec.exec)(`yarn sentry-cli releases -o titan-systems -p solid-react files solid-react@${process.env.VERSION} upload-sourcemaps ./build/browser`);
  } catch (e: any) {
    console.error(e.message);
    process.exit(1);
  }
});

task("sentry-upload-electron", async () => {
  if (process.env.VERSION === "latest") {
    return;
  }

  try {
    const version = getVersion();
    await util.promisify(new_exec.exec)(`yarn sentry-cli releases -o titan-systems -p solid-react files solid-react@${version} upload-sourcemaps ./build/electron`);
  } catch (e: any) {
    console.error(e.message);
    process.exit(1);
  }
});

task("clean", async () => {
  console.log("Cleaning build ...");
  rm("build/.cache");
  rm("build/browser");
  rm("build/electron");

  // TODO: remove file like .cache_psun5a from build
});

task("clean:full", async () => {
  console.log("Full cleaning build ...");
  rm("build");
});

task("copy:plugin", async () => {
/*
  console.log("Copying plugin ...");

  // package.json
  await src("../vn-player/package.json")
    .dest('build/electron/renderer/vn-player', "vn-player")
    .exec();

  // js files
  await src("../vn-player/!*.js")
    .dest('build/electron/renderer/vn-player', "vn-player")
  await src("../vn-player/*.js")
    .dest('build/electron/renderer/vn-player', "vn-player")
    .exec();

  // binding
  await src("../vn-player/binding/node-win32-x64-Release/!*.*")
    .dest('build/electron/renderer/vn-player', "vn-player")
  await src("../vn-player/binding/node-win32-x64-Release/*.*")
    .dest('build/electron/renderer/vn-player', "vn-player")
    .exec();

  // binding
  await src("../vn-player/binding/node-win32-x64-Release/!*.*")
    .dest('build/renderer/node-native', "node-win32-x64-Release")
    .exec();
*/
});

const getIcon = async () => {
  const theme = process.env.THEME ?? "";
  const themeDir = path.resolve(__dirname, "src/semantic-ui/themes/", theme);
  const iconUrl = `${process.env.API_BACKEND}/sdi/branding/icon.ico`;
  try {
    console.log(`Get icon from ${iconUrl}`);
    const response = await fetch(iconUrl);
    const buffer = await response.buffer();
    if (response.ok && buffer) {
      await fs.promises.writeFile("./build/electron/renderer/resources/images/icon.ico", buffer);
      return path.resolve(__dirname, "build/electron/renderer/resources/images/icon.ico");
    }
  }
  catch (e) {
    if (e instanceof FetchError) {
      console.error(`Cannot get icon: [${e.code}] ${e.message}`);
    } else {
      console.error(e);
    }
  }

  console.log("Use default icon");
  return path.resolve(themeDir, "assets/icon/icon.ico");
};

const getIcnsIcon = async () => {
  const theme = process.env.THEME ?? "";
  const themeDir = path.resolve(__dirname, "src/semantic-ui/themes/", theme);
  const iconUrl = `${process.env.API_BACKEND}/sdi/branding/logo.png`;
  try {
    console.log(`Get icon from ${iconUrl}`);
    const response = await fetch(iconUrl);
    const buffer = await response.buffer();
    if (response.ok && buffer) {
      const icon =  png2icons.createICNS(buffer, 1, 0);
      if (icon) {
        await fs.promises.writeFile("./build/electron/renderer/resources/images/icon.icns", icon);
        return path.resolve(__dirname, "build/electron/renderer/resources/images/icon.icns");
      }
    }
  }
  catch (e) {
    if (e instanceof FetchError) {
      console.error(`Cannot get icon: [${e.code}] ${e.message}`);
    } else {
      console.error(e);
    }
  }

  console.log("Use default icon");
  return path.resolve(themeDir, "assets/icon/icon.ico");
};

const getInfo = async () => {
  const theme = process.env.THEME ?? "";
  const themeDir = path.resolve(__dirname, "src/semantic-ui/themes/", theme);
  const icon = await getIcon();
  const iconIcns = await getIcnsIcon();

  const API_BACKEND = process.env.API_BACKEND;
  if (!API_BACKEND) {
    throw new Error("API_BACKEND is not defined");
  }
  const server = (new URL(API_BACKEND)).host;
  const shortName = "Solid";
  const fullName = `Solid-${server}`;
  const installDirName = fullName.replace(/\.|_|-|\s/g, "").toLowerCase(); // electronWinInstaller not allow some characters in dir name

  return {
    theme,
    themeDir,
    icon,
    iconIcns,
    shortName,
    fullName,
    installDirName,
    server,
    exe: shortName
  };
};

const getVersion = () => {
  return require("./build/electron/package.json").version;
};

task("build:installer-win", async (context) => {
  console.log("Build electron package");

  // name in package.json will be used by electronWinInstaller as directory name for electron configuration in AppData\Roaming\
  const packagePath = "./build/electron/package.json";
  const electronPackage = require(packagePath);
  const info = await getInfo();
  console.log({info});
  electronPackage.name = info.installDirName;
  await fs.promises.writeFile(packagePath, JSON.stringify(electronPackage, undefined, "  "));

  try {
    const options: electronPackager.Options = {
      name: info.fullName, // will be used as name in Start menu
      executableName: info.exe,
      icon: info.icon,
      platform: "win32",
      arch: "x64",
      out: "build",
      dir: "build/electron",
      overwrite: true,
      tmpdir: false
    };
    const appPaths = await electronPackager(options);
    console.log("Package is created in: ", appPaths);

    await buildInstaller(appPaths[0]);
  }
  catch (e) {
    console.error(e);
  }
});

const buildInstaller = async (appDirectory: string) => {
  console.log("Build installer from " + appDirectory);

  const info = await getInfo();

  const iconUrl = info.icon; // control panel
  const setupIcon = info.icon; // setup file
  // const loadingGif = path.resolve(__dirname, "src/electron/img/installing.gif"); // installing indicator

  const name = info.fullName;
  const version = getVersion();

  try {
    const options: SquirrelWindowsOptions = {
      appDirectory,
      outputDirectory: path.resolve(__dirname, `build/${info.server}`),
      name: info.installDirName, // will be used as directory name in AppData\Local\
      title: name, // will be used as name Control Panel program list
      description: name,
      authors: "videoNEXT",
      exe: `${info.exe}.exe`,
      setupExe: `${name}-${version}.exe`,
      noMsi: true,
      iconUrl,
      setupIcon,
      //loadingGif
    };

    const certificateFile = "./sign/certificate.pfx";
    const certificatePasswordFile = "./sign/pass.txt";

    if (fs.existsSync(certificateFile)) {
      options.certificateFile = certificateFile;

      if (fs.existsSync(certificatePasswordFile)) {
        options.certificatePassword = String(await fs.promises.readFile("./sign/pass.txt"));
      }
    }

    await electronWinInstaller.createWindowsInstaller(options);
    console.log("Windows installer is created");
  }
  catch (e: any) {
    console.error("Error creating Windows installer:", e.message);
  }
};

task("build:installer-mac", async () => {
  console.log("Build electron package mac");

  const packagePath = "./build/electron/package.json";
  const electronPackage = require(packagePath);
  const info = await getInfo();
  console.log({ info });
  electronPackage.name = info.installDirName;
  await fs.promises.writeFile(packagePath, JSON.stringify(electronPackage, undefined, "  "));

  try {
    const options: electronPackager.Options = {
      name: info.fullName, // will be used as name in Start menu
      executableName: info.exe,
      icon: info.iconIcns,
      platform: "darwin",
      arch: "x64",
      out: "build",
      dir: "build/electron",
      overwrite: true,
      tmpdir: false,
    };
    const appPaths = await electronPackager(options);
    console.log("Package is created in: ", appPaths);

    await buildInstallerOnMac(appPaths[0]);
  }
  catch (e) {
    console.error(e);
  }
});

const buildInstallerOnMac = async (appPath: string) => {
  const info = await getInfo();
  console.log(info);
  const icon = info.iconIcns;
  const version = getVersion();

  const name = `${info.fullName}-${version}`;

  try {
    const options: electronDmgInstaller.CreateOptions = {
      appPath,
      out: path.resolve(__dirname, `build/${info.server}`),
      name,
      icon,
      title: info.fullName,
      overwrite: true
    };

    await electronDmgInstaller(options);
    console.log("MacOs installer is created");

  } catch (e: any) {
    console.error("Error creating MacOs installer:", e.message);
  }
};

// @ts-ignore
const monitor = async () => {
  setInterval(() => {
    const MB = 1000000;
    const usage = process.memoryUsage();
    console.log(Math.round(usage.rss / MB), Math.round(usage.heapTotal / MB), Math.round(usage.heapUsed / MB), Math.round(usage.external / MB));
  }, 1000);
};
