import {fusebox, pluginPostCSS, sparky} from "fuse-box";
import {pluginTypeChecker} from "fuse-box-typechecker";
import {loadENV} from "./env";

loadENV();

class Context {
  getAppFuse() {
    return fusebox({
      target: "electron",
      electron: {
        nodeIntegration: true
      },
      entry: "src/index.ts",
      webIndex: {
        publicPath: "./",
        template: "src/index.html"
      },
      sourceMap: {
        project: true,
        vendor: true,
        css: false
      },
      devServer:
        process.env.API_BACKEND ? {
          httpServer: false,
          proxy: [
            {path: "/api", options: {target: process.env.API_BACKEND, changeOrigin: true, ws: true, /*followRedirects: true,*/ proxyTimeout: 60000, timeout: 60000}},
            {path: "/sdi", options: {target: process.env.API_BACKEND, changeOrigin: true, ws: false, /*followRedirects: true,*/ proxyTimeout: 60000, timeout: 60000}},
            {path: "/elog-restful", options: {target: process.env.API_BACKEND, changeOrigin: true, ws: false, /*followRedirects: true,*/ proxyTimeout: 60000, timeout: 60000}}
          ],
          hmrServer: { port: 5556 }
        } : true,
      cache: {
        enabled: true,
        root: "build/.cache/renderer"
      },
      hmr: {
        enabled: true
      },
      watcher: process.platform !== "darwin" ? {
        enabled: true,
        root: "../"
      } : false,
      logging: {level: "succinct"},
      stylesheet: {
        groupResourcesFilesByType: true,
      },
      resources: {
        resourceFolder: "./resources",
        resourcePublicRoot: "resources"
      },
      link: {resourcePublicRoot: "resources"},
      plugins: [
        pluginPostCSS("*.css"),
        ...(
            [
              pluginTypeChecker({
                tsConfig: "./tsconfig.renderer.json",
                name: "TypeChecker App"
              })
            ]
        )
      ]
    });
  }

  getElectronFuse() {
    return fusebox({
      target: "server",
      entry: "src/electron/main.ts",
      sourceMap: {
        project: true,
        vendor: true,
        css: false
      },
      dependencies: { serverIgnoreExternals: true },
      hmr: false,
      watcher: {
        enabled: true,
        root: "./src/electron"
      },
      logging: { level: "succinct" },
      cache: {
        enabled: true,
        root: "build/.cache/electron"
      },
      plugins: [
        pluginTypeChecker({
          tsConfig: "./tsconfig.electron.json",
          name: "TypeChecker Electron"
        })
      ]
    });
  }
}

const { task, src, rm } = sparky<Context>(Context);

task("clean", async (context) => {
  console.log("Removing build ...");
  rm("build");
});

task("start", async (context) => {
  const app = context.getAppFuse();
  await app.runDev({
    bundles: {
      distRoot: "build/electron/renderer",
      cssSplitting: true,
      app: "app.js",
      vendor: "vendor.js"
    }
  });

  await src("./src/electron/preload_main.js")
    .dest("build/electron/main", "electron")
    .exec();

  // To avoid warning in console coming from some electron packages looking for package.json
  await src("./src/electron/package.json")
    .dest("build/electron/renderer", "electron")
    .exec();
  const electron = context.getElectronFuse();
  const response = await electron.runDev({
    bundles: {
      distRoot: "build/electron/main",
      cssSplitting: true,
      app: "app.js",
      vendor: "vendor.js",
      codeSplitting: {
        path: "other/$name.$hash.js"
      }
    }
  });

  response.onComplete(({electron}) => {
    electron?.start();
  });
});
