/// <summary>Hue shift</summary>
/// <minValue>0</minValue>
/// <maxValue>1</maxValue>
/// <defaultValue>0</defaultValue>
float HueShift : register(c0);

/// <summary>The brightness offset.</summary>
/// <minValue>-1</minValue>
/// <maxValue>1</maxValue>
/// <defaultValue>0</defaultValue>
float Brightness : register(c1);

/// <summary>The contrast multiplier.</summary>
/// <minValue>0</minValue>
/// <maxValue>2</maxValue>
/// <defaultValue>1.0</defaultValue>
float Contrast : register(c2);

/// <summary>The saturation offset.</summary>
/// <minValue>1</minValue>
/// <maxValue>5</maxValue>
/// <defaultValue>1</defaultValue>
float Saturation : register(C3);

/// <summary>Zoom</summary>
/// <minValue>1</minValue>
/// <maxValue>5</maxValue>
/// <defaultValue>1</defaultValue>
float Zoom : register(C4);

/// <summary>Pan</summary>
/// <minValue>0.0</minValue>
/// <maxValue>1.0</maxValue>
/// <defaultValue>0.5</defaultValue>
float PanX : register(C5);
float PanY : register(C6);

bool ApplyZoom : register(C7);
bool ApplyAdjust : register(C8);

int AspectLeft   : register(C9);
int AspectTop    : register(C10);
int AspectRight  : register(C11);
int AspectBottom : register(C12);

struct VS_OUTPUT
{
    float4 Position   : POSITION;
    float2 Texture    : TEXCOORD;
    float2 ScreenPos  : VPOS;    
};

sampler2D Samp : register(S0);

float3 rgb_to_hsv_no_clip(float3 RGB)
{
    float3 HSV;
    
	float minChannel, maxChannel;
	if (RGB.x > RGB.y) 
	{
		maxChannel = RGB.x;
		minChannel = RGB.y;
	}
	else 
	{
		maxChannel = RGB.y;
		minChannel = RGB.x;
	}
	
	if (RGB.z > maxChannel) maxChannel = RGB.z;
	if (RGB.z < minChannel) minChannel = RGB.z;
    
    HSV.xy = 0;
    HSV.z = maxChannel;
    float delta = maxChannel - minChannel;             //Delta RGB value 
    if (delta != 0) 
    {                    // If gray, leave H & S at zero
       HSV.y = delta / HSV.z;
       float3 delRGB;
       delRGB = (HSV.zzz - RGB + 3*delta) / (6.0*delta);
       if      ( RGB.x == HSV.z ) HSV.x = delRGB.z - delRGB.y;
       else if ( RGB.y == HSV.z ) HSV.x = ( 1.0/3.0) + delRGB.x - delRGB.z;
       else if ( RGB.z == HSV.z ) HSV.x = ( 2.0/3.0) + delRGB.y - delRGB.x;
    }
    return (HSV);
}

float3 hsv_to_rgb(float3 HSV)
{
    float3 RGB = HSV.z;
    //if ( HSV.y != 0 ) { // we don't really need this since it just adds an obsolete instruction slot
       float var_h = HSV.x * 6;
       float var_i = floor(var_h);   // Or ... var_i = floor( var_h )
       float var_1 = HSV.z * (1.0 - HSV.y);
       float var_2 = HSV.z * (1.0 - HSV.y * (var_h-var_i));
       float var_3 = HSV.z * (1.0 - HSV.y * (1-(var_h-var_i)));
       if      (var_i == 0) { RGB = float3(HSV.z, var_3, var_1); }
       else if (var_i == 1) { RGB = float3(var_2, HSV.z, var_1); }
       else if (var_i == 2) { RGB = float3(var_1, HSV.z, var_3); }
       else if (var_i == 3) { RGB = float3(var_1, var_2, HSV.z); }
       else if (var_i == 4) { RGB = float3(var_3, var_1, HSV.z); }
       else                 { RGB = float3(HSV.z, var_1, var_2); }
   //}
   return (RGB);
}

float4 ApplyAdjustments(float4 color)
{
    color.rgb /= color.a;
    
    // Apply contrast.
    color.rgb =((color.rgb - 0.5f) * max(Contrast, 0)) + 0.5f;
    
    // Apply brightness.
    color.rgb += Brightness;
    
    // Return final pixel color.
    color.rgb *= color.a;
	
    // Apply saturation
    float3 luminanceWeights = float3(0.299, 0.587, 0.114);
    float luminance = dot(color.rgb, luminanceWeights);
    float3 pixel = lerp(luminance, color.rgb, Saturation);
	
    // Apply hsv
    float3 hsv = rgb_to_hsv_no_clip(pixel.xyz);
    hsv.x += HueShift;
    
    if ( hsv.x > 1.0 ) 
    { 
    	hsv.x -= 1.0; 
    }
    return float4(hsv_to_rgb(hsv), color.w);
}

float4 main( in VS_OUTPUT In ) : COLOR0
{
    float2 samplePoint;
    if (ApplyZoom)
    {
//        float2 center = float2(0.5, 0.5);	
        float2 center = float2(PanX, PanY);	

        float2 centerToPixel = In.Texture.xy - center;
        samplePoint = center + centerToPixel / Zoom;
    }
    else
    {
        samplePoint = In.Texture.xy;
    }
	
    float4 color = tex2D(Samp, samplePoint);

    if (ApplyAdjust)
    {
        // do not ajust color of fields of ration
        if (In.ScreenPos.x >= AspectLeft
            && In.ScreenPos.x <= AspectRight
            && In.ScreenPos.y >= AspectTop
            && In.ScreenPos.y <= AspectBottom)
        {
            color = ApplyAdjustments(color);
        }
    }
        
    return color;
}
