#!/bin/bash

set -x

FFMPEG_VERSION=5.1
XML2_VERSION=2.9.9
OPENSSL_VERSION=1.1.1k
CURL_VERSION=7.65.1
ACE_VERSION=6.5.13
SDL2_VERSION=2.24.0
FREETYPE_VERSION=2.4.12
LIBWEBSOCKETS_VERSION=2.4.2
ICONV_VERSION=1.16
ZLIB_VERSION=1.2.13

DEPS=("x264-master.tar.bz2                         => https://code.videolan.org/videolan/x264/-/archive/master/x264-master.tar.bz2"
      "ffmpeg-$FFMPEG_VERSION.tar.bz2              => http://ffmpeg.org/releases/ffmpeg-$FFMPEG_VERSION.tar.bz2"
      "libxml2-$XML2_VERSION.tar.gz                => ftp://xmlsoft.org/libxml2/libxml2-sources-$XML2_VERSION.tar.gz"
      "openssl-$OPENSSL_VERSION.tar.gz             => http://www.openssl.org/source/openssl-$OPENSSL_VERSION.tar.gz"
      "curl-$CURL_VERSION.tar.bz2                  => http://curl.haxx.se/download/curl-$CURL_VERSION.tar.bz2"
      "ACE-$ACE_VERSION.tar.bz2                    => https://github.com/DOCGroup/ACE_TAO/releases/download/ACE%2BTAO-6_5_13/ACE-$ACE_VERSION.tar.bz2"
      "SDL2-$SDL2_VERSION.tar.gz                   => https://github.com/libsdl-org/SDL/releases/download/release-2.24.0/SDL2-$SDL2_VERSION.tar.gz"
      "libwebsockets-$LIBWEBSOCKETS_VERSION.tar.gz => http://github.com/warmcat/libwebsockets/archive/v$LIBWEBSOCKETS_VERSION.tar.gz"
      "libiconv-$ICONV_VERSION.tar.gz              => http://ftp.gnu.org/pub/gnu/libiconv/libiconv-$ICONV_VERSION.tar.gz"
      "zlib-$ZLIB_VERSION.tar.gz                   => http://zlib.net/zlib-$ZLIB_VERSION.tar.gz"
)

export FFMPEG_SPECIFIC_CONFIG=
export PREBUILD=`pwd`/../prebuild
export BUILD=`pwd`/../../build
export ROOT=`pwd`
export BUILD_HOST=`uname`

if [ -z "$1" ]; then
    echo "Usage: $0 arch [package]"
    echo "Architecture list:"
    echo "x86_64-linux"
    echo "x86_64-ubuntu20.04"
    echo "x86_64-darwin"
    echo "arm64-darwin-iphoneos"
    echo "x86_64-darwin-iphonesimulator"
    echo "x86_64-mingw32"
    echo "arm64-android"
    echo "x86_64-android"
    echo ""
    echo "Optional packages list:"
    echo "ffmpeg"
    echo "live555"
    echo "x264"
    echo "openssl"
    echo "xml2"
    echo "curl"
    echo "libwebsockets"
    echo "ACE"
    echo "SDL2"
    echo "iconv"
    echo "zlib"
    exit
fi

function update_deps
{
    [ -d .packages ] || mkdir .packages

    local file="$1"
    for dep in "${DEPS[@]}" ; do
        f=`echo "${dep%%=>*}"| xargs echo -n`
        url=`echo "${dep##*=>}"| xargs echo -n`

        if [ ! -z "$file" ]; then
            if [ "$file" = "$f" ]; then
                echo "Check './.packages/$f'"
                wget -c -O .packages/$f $url || exit
                break
            else
                continue
            fi
        fi

        echo "Check './.packages/$f'"
        wget -c -O .packages/$f $url || exit
    done
}

function build_x264
{
    echo "Build x264"
    update_deps "x264-master.tar.bz2"

    cd $ROOT/.stage
    if [ -e x264 ]; then rm -rf x264; fi
    tar -xf ../.packages/x264-master.tar.bz2 && mv x264-master* x264

    cd x264
    ./configure --host="$HOST" --enable-static --disable-opencl --disable-lavf --disable-swscale --disable-avs --disable-ffms $X264_SPECIFIC_CONFIG \
        && make || exit
    cd ..
    rm -rf $PREBUILD/x264
    mkdir $PREBUILD/x264
    mkdir $PREBUILD/x264/include
    mkdir $PREBUILD/x264/lib
    cp x264/x264.h x264/x264_config.h x264/config.h $PREBUILD/x264/include
    cp x264/libx264.a $PREBUILD/x264/lib
}

function copy_ffmpeg_sub_folder
{
    [ ! -d $PREBUILD/ffmpeg/include/$1 ] && mkdir $PREBUILD/ffmpeg/include/$1
    cp -v $1/*.h $PREBUILD/ffmpeg/include/$1/
    if [ "$2" != "" ]; then
        [ ! -d $PREBUILD/ffmpeg/include/$1/$2 ] && mkdir $PREBUILD/ffmpeg/include/$1/$2
        cp -v $1/$2/*.h $PREBUILD/ffmpeg/include/$1/$2/
    fi
}

function build_ffmpeg
{
    echo "Build ffmpeg"
    update_deps "ffmpeg-$FFMPEG_VERSION.tar.bz2"

    cd $ROOT/.stage
    if [ -e ffmpeg ]; then rm -rf ffmpeg; fi
    tar -xf ../.packages/ffmpeg-$FFMPEG_VERSION.tar.bz2 && mv ffmpeg-$FFMPEG_VERSION ffmpeg

    cd ffmpeg
    patch -p0 < ../../ffmpeg.patch
    ffbuild/version.sh . version.h
    ./configure --cc="$CC" \
                --disable-sdl2 --disable-libxcb --disable-xlib \
                --disable-ffmpeg --disable-ffprobe --disable-ffplay --disable-doc --disable-shared --enable-static \
                --enable-swscale \
                --disable-hwaccels --disable-decoders --disable-encoders --disable-protocols --disable-muxers --disable-demuxers --disable-filters \
                --enable-decoder=h264 --enable-decoder=hevc --enable-decoder=h263 --enable-decoder=h263i --enable-decoder=mpegvideo --enable-decoder=mjpeg \
                --enable-decoder=mpeg2video --enable-decoder=mpeg1video --enable-decoder=mpeg4 --enable-decoder=aac \
                --enable-encoder=aac --enable-encoder=mjpeg \
                --enable-muxer=mov --enable-muxer=mp4 --enable-muxer=hevc \
                --enable-demuxer=mov --enable-demuxer=mpegts --enable-demuxer=hevc --enable-protocol=file --disable-iconv \
                --disable-stripping --ar=${AR} --enable-pic \
                --prefix=$PREBUILD/ffmpeg \
                $FFMPEG_SPECIFIC_CONFIG \
        && make V=1 || exit
    make V=1 install || exit

    copy_ffmpeg_sub_folder libavcodec $FFMPEG_CODEC_ARCH
    copy_ffmpeg_sub_folder libavdevice
    copy_ffmpeg_sub_folder libavfilter
    copy_ffmpeg_sub_folder libavformat
    copy_ffmpeg_sub_folder libavutil $FFMPEG_CODEC_ARCH
    copy_ffmpeg_sub_folder libpostproc
    copy_ffmpeg_sub_folder libswresample
    copy_ffmpeg_sub_folder libswscale

    cp config.h version.h $PREBUILD/ffmpeg/include/
}

function copy_live555_sub_folder
{
    [ ! -d "$PREBUILD/live555/include/$1" ] && mkdir $PREBUILD/live555/include/$1
    cp live555/$1/include/*.h* $PREBUILD/live555/include/$1/
    cp live555/$1/lib$1.a $PREBUILD/live555/lib/
}

function build_live555
{
    echo "Build live555"
    cd $ROOT
    cd live555
    make clean
    make -f Makefile.$BUILD_TARGET  || exit
    cd ..
    [ ! -d "$PREBUILD/live555" ] && mkdir $PREBUILD/live555
    [ ! -d "$PREBUILD/live555/include" ] && mkdir $PREBUILD/live555/include
    [ ! -d "$PREBUILD/live555/lib" ] && mkdir $PREBUILD/live555/lib
    copy_live555_sub_folder BasicUsageEnvironment
    copy_live555_sub_folder UsageEnvironment
    copy_live555_sub_folder groupsock
    copy_live555_sub_folder liveMedia
}

function build_xml2
{
    echo "Build xml2"
    update_deps "libxml2-$XML2_VERSION.tar.gz"

    cd $ROOT/.stage
    if [ -e libxml2 ]; then rm -rf libxml2; fi
    tar -zxf ../.packages/libxml2-$XML2_VERSION.tar.gz && mv libxml2-$XML2_VERSION libxml2

    h=$HOST
    if [ "$BUILD_TARGET" = "arm64-android" ]; then
        h="arm-linux"
    fi

    if [ "$BUILD_TARGET" = "x86_64-mingw32" ]; then
        XML2_OPTS+=" --with-zlib=$PREBUILD/zlib"
        export CFLAGS+=" -I$PREBUILD/zlib/include"
    fi

    XML2_OPTS="--disable-shared --host=$h --with-pic --without-lzma --without-http --without-ftp --without-iconv --without-threads --without-catalog"

    export CFLAGS+=" -O3 "
    cd libxml2
    ./configure $XML2_OPTS
    make V=1 libxml2.la || exit
    cd ..
    rm -rf $PREBUILD/libxml2
    mkdir $PREBUILD/libxml2
    mkdir $PREBUILD/libxml2/include
    mkdir $PREBUILD/libxml2/include/libxml
    mkdir $PREBUILD/libxml2/lib
    cp libxml2/*.h libxml2/include/*.h $PREBUILD/libxml2/include/
    cp libxml2/include/libxml/*.h $PREBUILD/libxml2/include/libxml/
    cp libxml2/.libs/libxml2.a $PREBUILD/libxml2/lib/
}

function build_openssl
{
    echo "Build openssl"
    update_deps "openssl-$OPENSSL_VERSION.tar.gz"

    cd $ROOT/.stage

    if [ -e openssl ]; then rm -rf openssl; fi
    tar -zxf ../.packages/openssl-$OPENSSL_VERSION.tar.gz && mv openssl-$OPENSSL_VERSION openssl

    cd openssl

    if [ "$BUILD_TARGET" = "arm64-darwin-iphoneos" ]; then
        CC="$CC $CFLAGS -O3" ./Configure ios64-cross no-shared no-async --prefix=$PREBUILD/openssl
#       sed -ie "s!^CFLAG=!CFLAG=-isysroot $SDK_PATH -arch $ARCH !" "Makefile"
#       sed -ie "s!static volatile sig_atomic_t intr_signal;!static volatile intr_signal;!" "crypto/ui/ui_openssl.c"
    fi

    if [ "$BUILD_TARGET" = "x86_64-darwin-iphonesimulator" ]; then
        CC="$CC $CFLAGS -O3" ./Configure darwin64-x86_64-cc no-shared --prefix=$PREBUILD/openssl
#       sed -ie "s!^CFLAG=!CFLAG=$CFLAGS !" "Makefile"
#       sed -ie "s!^CFLAG=!CFLAG=-isysroot $SDK_PATH -arch $ARCH !" "Makefile"
    fi

    if [ "$BUILD_TARGET" = "arm64-android" ]; then
        CC="$CC $CFLAGS -O3" ./Configure android-arm64 no-shared --prefix=$PREBUILD/openssl
        sed -ie "s!^CFLAG=!CFLAG=$CFLAGS !" "Makefile"
        sed -ie "s!^CROSS_COMPILE=.*!CROSS_COMPILE=!" "Makefile"
    fi

    if [ "$BUILD_TARGET" = "x86_64-android" ]; then
        CC="$CC $CFLAGS -O3" ./Configure android-x86_64 no-shared --prefix=$PREBUILD/openssl
        sed -ie "s!^CFLAG=!CFLAG=$CFLAGS !" "Makefile"
        sed -ie "s!^CROSS_COMPILE=.*!CROSS_COMPILE=!" "Makefile"
    fi

    if [ "$BUILD_TARGET" = "x86_64-darwin" ]; then
        CC="gcc $CFLAGS -O3" ./Configure darwin64-x86_64-cc no-shared no-hw no-gost --prefix=$PREBUILD/openssl
    fi

    if [ "$BUILD_TARGET" = "x86_64-linux" ]; then
        CC="$CC $CFLAGS -O3" ./Configure linux-generic64 no-shared no-gost --prefix=$PREBUILD/openssl
        #sed -ie "s!^CFLAG=!CFLAG=$CFLAGS !" "Makefile"
    fi

    if [ "$BUILD_TARGET" = "x86_64-mingw32" ]; then
        CC="$CC $CFLAGS -O3" ./Configure mingw64 no-shared --prefix=$PREBUILD/openssl
    fi

    rm -rf $PREBUILD/openssl

    make build_libs install_dev || exit
    #make build_libs || exit

    #make all install_sw || exit

    # cd ..
    # rm -rf $PREBUILD/openssl
    # mkdir -p $PREBUILD/openssl/lib/pkgconfig
    # mkdir -p $PREBUILD/openssl/include/openssl

    # cp openssl/libcrypto.a $PREBUILD/openssl/lib
    # cp openssl/libssl.a  $PREBUILD/openssl/lib
    # cp -L openssl/include/openssl/* $PREBUILD/openssl/include/openssl/

    # cp libcrypto.pc $PREBUILD/openssl/lib/pkgconfig
    # chmod 644 $PREBUILD/openssl/lib/pkgconfig/libcrypto.pc
    # cp libssl.pc $PREBUILD/openssl/lib/pkgconfig
    # chmod 644 $PREBUILD/openssl/lib/pkgconfig/libssl.pc
    # cp openssl.pc $PREBUILD/openssl/lib/pkgconfig
    # chmod 644 $PREBUILD/openssl/lib/pkgconfig/openssl.pc
}

function build_curl
{
    echo "Build curl"
    update_deps "curl-$CURL_VERSION.tar.bz2"

    cd $ROOT/.stage
    if [ -e curl ]; then rm -rf curl; fi
    tar -xf ../.packages/curl-$CURL_VERSION.tar.bz2 && mv curl-$CURL_VERSION curl

    CURL_OPTS="--enable-ipv6 --disable-ftp --disable-file --disable-ldap --disable-ldaps --disable-rtsp --without-nghttp2 --without-libidn2 --without-brotli
              --disable-proxy --disable-dict --disable-telnet --disable-tftp --disable-pop3 --disable-imap
              --disable-smtp --disable-gopher --disable-sspi --disable-manual --disable-shared --enable-static
              --with-ssl=$PREBUILD/openssl --without-ca-bundle --without-ca-path --without-libidn2 --without-libpsl --without-libssh2 --without-librtmp --prefix=$PREBUILD/curl"

    if [ "$BUILD_TARGET" = "arm64-android" ]; then
        CURL_OPTS+=" --with-random=/dev/urandom --host=arm-linux-androideabi --without-zlib"
    fi

    if [ "$BUILD_TARGET" = "x86_64-android" ]; then
        CURL_OPTS+=" --with-random=/dev/urandom --host=$HOST --without-zlib"
    fi

    if [ "$BUILD_TARGET" = "arm64-darwin-iphoneos" -o "$BUILD_TARGET" = "x86_64-darwin-iphonesimulator" ]; then
      CURL_OPTS+=" --disable-dependency-tracking --host=$HOST --with-zlib"
    fi

    if [ "$BUILD_TARGET" = "x86_64-mingw32" ]; then
#      perl -p -i -e "s/#if \!defined\(HAVE_STRUCT_POLLFD\)/#if 0 && \!defined\(HAVE_STRUCT_POLLFD\)/g" curl/lib/select.h
      CURL_OPTS+=" --host=$HOST --with-zlib=$PREBUILD/zlib"
      cd curl
      LIBS=-lcrypt32 ./configure $CURL_OPTS || exit
      make V=1 && make install || exit
      return
    fi

    if [ "$BUILD_TARGET" = "x86_64-darwin" ]; then
      CURL_OPTS+=" --host=$HOST --with-zlib"
    fi

    cd curl
    ./configure $CURL_OPTS $CURL_SPECIFIC_CONFIG || exit
    if [ "$BUILD_TARGET" = "arm64-darwin-iphoneos" -o "$BUILD_TARGET" = "x86_64-darwin-iphonesimulator" ]; then
      echo "#undef HAVE_CLOCK_GETTIME_MONOTONIC" >> lib/curl_config.h
    fi
    make V=1 && make install || exit
}

function build_iconv
{
    echo "Build iconv"
    update_deps "libiconv-$ICONV_VERSION.tar.gz"

    cd $ROOT/.stage
    if [ -e iconv ]; then rm -rf iconv; fi
    tar -xf ../.packages/libiconv-$ICONV_VERSION.tar.gz && mv libiconv-$ICONV_VERSION libiconv

    ICONV_OPTS="--disable-shared --enable-static --prefix=$PREBUILD/iconv"

    if [ "$BUILD_TARGET" = "arm64-android" ]; then
        ICONV_OPTS+=" --host=arm-linux-androideabi --with-ssl=$PREBUILD/openssl"
    fi

    if [ "$BUILD_TARGET" = "arm64-darwin-iphoneos" -o "$BUILD_TARGET" = "x86_64-darwin-iphonesimulator" ]; then
      ICONV_OPTS+=" --host=$HOST "
    fi

    if [ "$BUILD_TARGET" = "x86_64-mingw32" ]; then
      ICONV_OPTS+=" --host=$HOST"
    fi

    if [ "$BUILD_TARGET" = "x86_64-darwin" ]; then
      ICONV_OPTS+=" --host=$HOST"
    fi

    cd libiconv
    ./configure $ICONV_OPTS || exit
    make V=1 && make install || exit
}

function build_SDL2
{
    echo "Build SDL2"
    update_deps "SDL2-$SDL2_VERSION.tar.gz"

    cd $ROOT/.stage
    if [ -e SDL2 ]; then rm -rf SDL2; fi
    tar -xf ../.packages/SDL2-$SDL2_VERSION.tar.gz
    mv SDL2-$SDL2_VERSION SDL2

    h=$HOST
    if [ "$BUILD_TARGET" = "x86_64-darwin-iphonesimulator" ]; then
        h="x86_64-ios-darwin"
        CFLAGS+=" -fobjc-arc -DSDL_VIDEO_DRIVER_UIKIT"
    fi
    if [ "$BUILD_TARGET" = "arm64-darwin-iphoneos" ]; then
        h="arm-ios-darwin"
        CFLAGS+=" -fobjc-arc -DSDL_VIDEO_DRIVER_UIKIT"
    fi

    #   SDL2_OPTS="--disable-joystick --disable-haptic --disable-sensor --enable-static --disable-shared --host=$h --prefix=$PREBUILD/SDL2 "
    SDL2_OPTS="--enable-static --disable-shared --host=$h --disable-video-wayland --prefix=$PREBUILD/SDL2 "

    cd SDL2
    mkdir build

    if [ "$BUILD_TARGET" = "x86_64-darwin" ]; then
        SDL2_OPTS+=" --without-x --disable-haptic --disable-sensor --disable-joystick"
        #    CC+=" -g3 -O0"
    fi

    # if [ "$BUILD_TARGET" = "x86_64-darwin-iphonesimulator" ]; then
    #     SDL2_OPTS+=" --disable-joystick --disable-haptic --disable-sensor "
    #     CFLAGS+=" -fobjc-arc"
    # fi

    if [ "$BUILD_TARGET" = "arm64-android" -o "$BUILD_TARGET" = "x86_64-android" ]; then
#        SDL2_OPTS+=" --without-x --enable-libudev=no --enable-render=no --enable-events=no --enable-joystick=no --enable-haptic=no --enable-hidapi=no --enable-sensor=no"
        SDL2_OPTS+=" --without-x --enable-hidapi=no "
        CFLAGS+=" -DGL_GLEXT_PROTOTYPES -O3"
        patch -p0 < ../../sdl-android.patch
        perl -p -i -e 's/ifndef EVIOCGNAME/ifndef EVIOCGNAME_DISABLED/g' configure
        ./configure $SDL2_OPTS
        make V=1 all install-hdrs install-lib || exit
        cp src/core/android/SDL_android.h $PREBUILD/SDL2/include/SDL2/ || exit
        perl -p -i -e 's/#include "..\/..\/SDL_internal.h"//g' $PREBUILD/SDL2/include/SDL2/SDL_android.h
    else
        CFLAGS="$CFLAGS -O3 -I$PREBUILD/iconv/include" LDFLAGS="$LDFLAGS -L$PREBUILD/iconv/lib" ./configure $SDL2_OPTS
        # if [ "$BUILD_TARGET" = "arm64-darwin-iphoneos" ]; then
        #     cp include/SDL_config_iphoneos.h include/SDL_config.h
        # fi
        make V=1 all install || exit
    fi
}

function build_ACE
{
    echo "Build ACE"
    update_deps "ACE-$ACE_VERSION.tar.bz2"

    cd $ROOT/.stage
    if [ -e ACE_wrappers ]; then rm -rf ACE_wrappers; fi
    tar -xf ../.packages/ACE-$ACE_VERSION.tar.bz2
    cd ACE_wrappers
    patch -p0 < ../../ACE.patch

    if [ "$BUILD_TARGET" = "x86_64-mingw32" ]; then
        cat > ace/config.h <<EOF
            #include "ace/config-win32.h"
            //#undef ACE_LACKS_USECONDS_T
            //#define ACE_LACKS_SIGACTION
            //#define ACE_HAS_POSIX_TIME
            //#define ACE_LACKS_SIGSET_T
            //#define ACE_LACKS_TIMESPEC_T
            //#define ACE_LACKS_STRTOD
            //#define ACE_LACKS_WCSTOD
            //#undef ACE_HAS_PTHREAD_SIGMASK_PROTOTYPE
            //#define ACE_LACKS_SIGSET
            ///////////// 64bit related /////////////
            //#undef ACE_HAS_WCHAR
            //#define bswap64(a)              __bswapq(a)
            //#define ACE_HAS_BSWAP64
EOF
        cat > include/makeinclude/platform_macros.GNU <<EOF
            include \$(ACE_ROOT)/include/makeinclude/platform_mingw32.GNU
            COMPARCH=x86_64-w64-mingw32-
            TCPU=generic
            CPPFLAGS += -fpermissive
            DCFLAGS    += -O3
            DCCFLAGS   += -O3
            INSTALL_PREFIX=$PREBUILD/ace
EOF
    fi

    if [ "$BUILD_TARGET" = "arm64-darwin-iphoneos" ]; then
        cat > ace/config.h <<EOF
            #include "ace/config-macosx-iOS.h"
            #undef ACE_LACKS_CLOCKID_T
            #undef ACE_LACKS_CLOCK_REALTIME
            #undef ACE_LACKS_CLOCK_MONOTONIC
            #define ACE_LACKS_SYSTEM
            #define ACE_LACKS_CLOCK_GETTIME
EOF
        cat > include/makeinclude/platform_macros.GNU <<EOF
            include \$(ACE_ROOT)/include/makeinclude/platform_macosx_iOS.GNU
            IPHONE_HARDWARE_ARCHITECTURE=$ARCH
            INSTALL_PREFIX=$PREBUILD/ace
            CC:=$CC
            CXX:=$CXX
            AR:=$AR
            RANLIB:=$RANLIB
            DLD = $CXX
            LD  = $CXX
            CFLAGS  = $CFLAGS -O3
            CCFLAGS = $CXXFLAGS -O3
EOF
        export IPHONE_TARGET=HARDWARE
        export IPHONE_VERSION=$IOS_SDK_VERSION
    fi

    if [ "$BUILD_TARGET" = "x86_64-darwin-iphonesimulator" ]; then
        cat > ace/config.h <<EOF
            #include "ace/config-macosx-iOS.h"
            #undef ACE_LACKS_CLOCKID_T
            #undef ACE_LACKS_CLOCK_REALTIME
            #undef ACE_LACKS_CLOCK_MONOTONIC
            #undef ACE_HAS_CLOCK_GETTIME
            #define ACE_LACKS_SYSTEM
EOF
        cat > include/makeinclude/platform_macros.GNU <<EOF
            include \$(ACE_ROOT)/include/makeinclude/platform_macosx_iOS.GNU
            INSTALL_PREFIX=$PREBUILD/ace
            CC:=$CC
            CXX:=$CXX
            AR:=$AR
            RANLIB:=$RANLIB
            DLD = $CXX
            LD  = $CXX
            CFLAGS  = $CFLAGS -O3
            CCFLAGS = $CXXFLAGS -O3
EOF
        export IPHONE_TARGET=SIMULATOR
        export IPHONE_VERSION=$IOS_SDK_VERSION
    fi

    if [ "$BUILD_TARGET" = "x86_64-darwin" ]; then
        cat > ace/config.h <<EOF
            #include "ace/config-macosx-snowleopard.h"
EOF
        cat > include/makeinclude/platform_macros.GNU <<EOF
            include \$(ACE_ROOT)/include/makeinclude/platform_macosx_snowleopard.GNU
            INSTALL_PREFIX=$PREBUILD/ace
            CFLAGS = -g -O3
            CCFLAGS = -g -O3
EOF
    fi

    if [ "$BUILD_TARGET" = "arm64-android" ]; then
        cat > ace/config.h <<EOF
            #define __ANDROID_API__ $MIN_ANDROID_API
            #define __NDK_MINOR__ $MIN_ANDROID_API
            #define __NDK_MAJOR__ $TARGET_ANDROID_API
            #define ACE_ANDROID_NDK_HAS_NDK_VERSION_H
            #include "ace/config-android.h"
            #define ACE_HAS_UCONTEXT_T
            #define ACE_DISABLE_MKTEMP
            #define ACE_DISABLE_TEMPNAM
            #define ACE_LACKS_CONDATTR_SETCLOCK
            #define ACE_HAS_CPU_SET_T
            #undef ACE_HAS_BIG_FD_SET
            #define ACE_LACKS_FD_MASK
            #define ACE_HAS_NONCONST_FD_ISSET
EOF
        cat > include/makeinclude/platform_macros.GNU <<EOF
            ANDROID_ABI:=arm64-v8a
            android_force_clang:=0
            include \$(ACE_ROOT)/include/makeinclude/platform_android.GNU
            INSTALL_PREFIX=$PREBUILD/ace

            CC:=$CC
            CXX:=$CXX
            AR:=$AR
            RANLIB:=$RANLIB
            CFLAGS = $CFLAGS -O3 -fPIE -fPIC
            CCFLAGS = $CXXFLAGS -O3
            LDFLAGS:=$LDFLAGS -lstdc++ -lsupc++
            PLATFORM_LARGEFILE_CFLAGS =
EOF
    fi

    if [ "$BUILD_TARGET" = "x86_64-android" ]; then
        cat > ace/config.h <<EOF
            #define __ANDROID_API__ $MIN_ANDROID_API
            #define __NDK_MINOR__ $MIN_ANDROID_API
            #define __NDK_MAJOR__ $TARGET_ANDROID_API
            #define ACE_ANDROID_NDK_HAS_NDK_VERSION_H
            #include "ace/config-android.h"
            #define ACE_HAS_UCONTEXT_T
            #define ACE_DISABLE_MKTEMP
            #define ACE_DISABLE_TEMPNAM
            #define ACE_LACKS_CONDATTR_SETCLOCK
            #define ACE_HAS_CPU_SET_T
            #undef ACE_HAS_BIG_FD_SET
            #define ACE_LACKS_FD_MASK
            #define ACE_HAS_NONCONST_FD_ISSET
EOF
        cat > include/makeinclude/platform_macros.GNU <<EOF
            ANDROID_ABI:=x86_64
            android_force_clang:=0
            include \$(ACE_ROOT)/include/makeinclude/platform_android.GNU
            INSTALL_PREFIX=$PREBUILD/ace

            CC:=$CC
            CXX:=$CXX
            AR:=$AR
            RANLIB:=$RANLIB
            CFLAGS = $CFLAGS -O3 -fPIE -fPIC
            CCFLAGS = $CXXFLAGS -O3
            LDFLAGS:=$LDFLAGS -lstdc++ -lsupc++
            PLATFORM_LARGEFILE_CFLAGS =
EOF
    fi

    if [ "$BUILD_TARGET" = "x86_64-linux" ]; then
        cat > ace/config.h <<EOF
            #include "ace/config-linux.h"
EOF
        cat > include/makeinclude/platform_macros.GNU <<EOF
            include \$(ACE_ROOT)/include/makeinclude/platform_linux.GNU
            CC:=$CC
            CXX:=$CXX
            AR:=$AR
            RANLIB:=$RANLIB
            CFLAGS    += $CFLAGS -fPIC -O3
            CCFLAGS   += $CXXFLAGS -fPIC -O3
            #DCFLAGS  += -fPIC -O3
            #DCCFLAGS += -fPIC -O3
            LDFLAGS:=$LDFLAGS # -lstdc++ -lsupc++
            INSTALL_PREFIX=$PREBUILD/ace
EOF
    fi

    export ACE_ROOT=`pwd`
    export SSL_ROOT=$PREBUILD/openssl

    cd ace
    make static_libs=1 static_libs_only=1 ssl=1  install || exit
}

function build_libwebsockets
{
    echo "Build libwebsockets"
    update_deps "libwebsockets-$LIBWEBSOCKETS_VERSION.tar.gz"

    cd $ROOT/.stage
    if [ -e libwebsockets-$LIBWEBSOCKETS_VERSION ]; then rm -rf libwebsockets-$LIBWEBSOCKETS_VERSION; fi
    tar -xf ../.packages/libwebsockets-$LIBWEBSOCKETS_VERSION.tar.gz
    cd libwebsockets-$LIBWEBSOCKETS_VERSION
    patch -p0 < ../../libwebsockets.patch
    mkdir build;cd build

    if [ "$BUILD_TARGET" = "x86_64-mingw32" ]; then
        cmake -DCMAKE_SYSTEM_NAME=Windows -DLWS_WITHOUT_TESTAPPS=ON -DLWS_IPV6=ON -DLWS_WITH_STATIC=ON -DLWS_WITH_SHARED=OFF \
          -DLWS_WITH_BUNDLED_ZLIB=OFF -DLWS_ZLIB_INCLUDE_DIRS=$PREBUILD/zlib/include -DLWS_ZLIB_LIBRARIES=$PREBUILD/zlib/lib/libz.a \
          -DCMAKE_C_COMPILER="$CC" -DCMAKE_C_FLAGS="$CFLAGS -DWINVER=0x0600 -D_WIN32_WINNT=0x0600" -DCMAKE_EXE_LINKER_FLAGS="-L$PREBUILD/zlib/lib" \
          -DLWS_OPENSSL_INCLUDE_DIRS="$PREBUILD/openssl/include" -DLWS_OPENSSL_LIBRARIES="$PREBUILD/openssl/lib/libssl.a;$PREBUILD/openssl/lib/libcrypto.a" \
          -DCMAKE_INSTALL_PREFIX:PATH=$PREBUILD/libwebsockets ..
        make VERBOSE=1 all install || exit
        return
    fi

    if [ "$BUILD_TARGET" = "arm64-darwin-iphoneos" -o "$BUILD_TARGET" = "x86_64-darwin-iphonesimulator" ]; then
        cmake_opts+=" -DCMAKE_OSX_SYSROOT=$SDK_PATH"
    fi

    if [ "$BUILD_TARGET" = "arm64-android" -o "$BUILD_TARGET" = "x86_64-android" ]; then
        cmake_opts+=" -DCMAKE_SYSTEM_NAME=Linux -DLWS_HAVE_GETIFADDRS=0 -DLWS_ZLIB_INCLUDE_DIRS=$ANDROID_NDK/sysroot/usr/include"
    fi

    if [ "$BUILD_TARGET" = "arm64-android" ]; then
        cmake_opts+=" -DCMAKE_SYSTEM_PROCESSOR=aarch64 -DLWS_ZLIB_LIBRARIES=$ANDROID_NDK/platforms/android-$MIN_ANDROID_API/arch-arm64"
    elif [ "$BUILD_TARGET" = "x86_64-android" ]; then
        cmake_opts+=" -DLWS_ZLIB_LIBRARIES=$ANDROID_NDK/platforms/android-$MIN_ANDROID_API/arch-x86_64"
    fi

    if [ "$BUILD_TARGET" = "x86_64-darwin" ]; then
        CC=gcc
    fi

    if [ "$BUILD_TARGET" = "x86_64-darwin" -o "$BUILD_TARGET" = "arm64-darwin-iphoneos" -o "$BUILD_TARGET" = "x86_64-darwin-iphonesimulator" ]; then
        CFLAGS="$CFLAGS -Wno-unused-but-set-variable -Wno-unknown-warning-option"
    fi

    cmake -DLWS_WITHOUT_TESTAPPS=ON -DLWS_IPV6=ON -DLWS_WITH_STATIC=ON -DLWS_WITH_SHARED=OFF \
          -DCMAKE_C_COMPILER="$CC" -DCMAKE_C_FLAGS="$CFLAGS" \
          -DLWS_OPENSSL_LIBRARIES="$PREBUILD/openssl/lib/libcrypto.a;$PREBUILD/openssl/lib/libssl.a" -DCMAKE_EXE_LINKER_FLAGS="-L$PREBUILD/openssl/lib" \
          -DLWS_OPENSSL_INCLUDE_DIRS="$PREBUILD/openssl/include" -DCMAKE_INSTALL_PREFIX:PATH=$PREBUILD/libwebsockets $cmake_opts $LIBWEBSOCKETS_SPECIFIC_CONFIG ..

    make VERBOSE=1 all install || exit
    if [ "$BUILD_TARGET" = "arm-android" -o "$BUILD_TARGET" = "x86-android" -o "$BUILD_TARGET" = "arm64-android" -o "$BUILD_TARGET" = "x86_64-android" ]; then
        $RANLIB $PREBUILD/libwebsockets/lib/libwebsockets.a
    fi
}

function build_boost
{
    echo "Build boost"
    rm -rf $PREBUILD/boost
    # just make symlink to /usr/include/boost
    if [ -d /usr/include/boost ]; then
        ln -s /usr/include/boost $PREBUILD/boost
    elif [ -d /opt/local/include/boost ]; then
        ln -s /opt/local/include/boost $PREBUILD/boost
    elif [ -d /usr/local/include/boost ]; then
        ln -s /usr/local/include/boost $PREBUILD/boost
    fi
}

function build_zlib
{
    echo "Build zlib"
    update_deps "zlib-$ZLIB_VERSION.tar.gz"

    cd $ROOT/.stage
    if [ -e zlib ]; then rm -rf zlib; fi
    tar -xf ../.packages/zlib-$ZLIB_VERSION.tar.gz && mv zlib-$ZLIB_VERSION zlib

    cd zlib
    if [ "$BUILD_TARGET" = "x86_64-mingw32" ]; then
      sed -ie "s!^PREFIX =!PREFIX = $MINGW_PREFIX!" "win32/Makefile.gcc"
      make -f win32/Makefile.gcc || exit
      BINARY_PATH=$PREBUILD/zlib/bin \
        INCLUDE_PATH=$PREBUILD/zlib/include \
        LIBRARY_PATH=$PREBUILD/zlib/lib \
        make -f win32/Makefile.gcc install || exit
      return
    fi

    #./configure --host="$HOST" --prefix=$PREBUILD/zlib --disable-shared --enable-static \
    #    && make || exit
    #make install
}

[ ! -d "$PREBUILD" ] && mkdir $PREBUILD
[ ! -d .stage ] && mkdir .stage
[ ! -d .packages ] && update_deps

export CFLAGS="-pipe "
export LDFLAGS=""
export FFMPEG_CODEC_ARCH=x86
export BUILD_LIBXML2="no"
export BUILD_TARGET="$1"
export BUILD_LIBX264="no"
export BUILD_ICONV="no"
export BUILD_ZLIB="no"

case "$1" in
x86_64-linux)
    echo "Linux (64)"
    export HOST="x86_64-linux"
    export CC="gcc"
    export CXX="g++"
    export CFLAGS="$CFLAGS -m64 -fPIC"
    export CXXFLAGS="$CFLAGS"
    export CPP="cpp"
    #export CPPFLAGS=""
    export AS="as"
    export AR="ar"
    export RANLIB="ranlib"
    #export LDFLAGS=""
    export PREBUILD="$PREBUILD/$BUILD_TARGET"
    if [ \! -e $PREBUILD ]; then mkdir $PREBUILD; fi
    export FFMPEG_SPECIFIC_CONFIG="--extra-cflags=-fPIC"
    export BUILD_LIBXML2="yes"
    ;;

x86_64-ubuntu20.04)
    echo "Linux Ubuntu 20.04 (64)"
    export HOST="x86_64-linux"
    export BUILD_TARGET=x86_64-linux
    export UBUNTU2004_PREFIX=~/x-tools/x86_64-ubuntu20.04-linux-gnu/bin/x86_64-ubuntu20.04-linux-gnu-
    export CC="${UBUNTU2004_PREFIX}gcc"
    export CXX="${UBUNTU2004_PREFIX}g++"
    export CFLAGS="$CFLAGS -m64 -fPIC -Wno-write-strings"
    export CXXFLAGS="$CFLAGS"
    export CPP="${UBUNTU2004_PREFIX}cpp"
    #export CPPFLAGS=""
    export AS="${UBUNTU2004_PREFIX}as"
    export AR="${UBUNTU2004_PREFIX}ar"
    export RANLIB="${UBUNTU2004_PREFIX}ranlib"
    export LD="${UBUNTU2004_PREFIX}ld"
    #export LDFLAGS=""
    export PREBUILD="$PREBUILD/$BUILD_TARGET"
    export PKG_CONFIG_PATH=$HOME/x-tools/x86_64-ubuntu20.04-linux-gnu/x86_64-ubuntu20.04-linux-gnu/sysroot/usr/lib/pkgconfig
    if [ \! -e $PREBUILD ]; then mkdir $PREBUILD; fi
    export FFMPEG_SPECIFIC_CONFIG="--extra-cflags=-fPIC --enable-cross-compile --target-os=linux --arch=x86_64 --ranlib=$RANLIB --disable-libdrm --disable-vaapi"
    export CURL_SPECIFIC_CONFIG="-host=x86_64-unknown-linux"
    export LIBWEBSOCKETS_SPECIFIC_CONFIG="-DCMAKE_SYSTEM_NAME=Linux"
    export BUILD_LIBXML2="yes"
    ;;

x86_64-darwin)
    echo "MacOS X (64)"
    export HOST="x86_64-apple-darwin"
    # export CC="gcc "
    # export CXX="g++ "
    # export CFLAGS+="-mmacosx-version-min=10.14"
    # export CXXFLAGS+="$CFLAGS"
    export MACOSX_DEPLOYMENT_TARGET="10.14"
    # export SYSTEM_VERSION_COMPAT=1
    # export AS="as"
    # export AR="ar"
    # export RANLIB="ranlib"

    export PREBUILD="$PREBUILD/$BUILD_TARGET"
    if [ \! -e $PREBUILD ]; then mkdir $PREBUILD; fi

    export FFMPEG_SPECIFIC_CONFIG="--target-os=darwin --arch=x86_64 --enable-hwaccel=h264_videotoolbox"
    export BUILD_LIBXML2="yes"
    export BUILD_ICONV="yes"
    ;;

arm64-android | x86_64-android)
    # WARNING:
    # env var $ANDROID_NDK shoud be set to NDK root
    if [ -z "$ANDROID_NDK" ]; then
       echo "Please set ANDROID_NDK environment variable"
       exit
    fi
    export MIN_ANDROID_API=21
    export TARGET_ANDROID_API=30

    case "$1" in
    arm64-android)
        echo "Android (arm64)"
        if [ "$BUILD_HOST" == "Darwin" ]
        then
            export ANDROID_TOOLCHAIN=$ANDROID_NDK/toolchains/aarch64-linux-android-4.9/prebuilt/darwin-x86_64
        else
            export ANDROID_TOOLCHAIN=$ANDROID_NDK/toolchains/aarch64-linux-android-4.9/prebuilt/linux-x86_64
        fi

        export MIN_ANDROID_API=21
        export ARCH=aarch64
        export HOST="arm-linux-android"
        TARGET_HOST="aarch64-linux-android"
        SYSROOT=$ANDROID_NDK/platforms/android-$MIN_ANDROID_API/arch-arm64
        export CXXFLAGS+="-isystem $ANDROID_NDK/sources/cxx-stl/gnu-libstdc++/4.9/include/backward \
            -isystem $ANDROID_NDK/sources/cxx-stl/gnu-libstdc++/4.9/include \
            -isystem $ANDROID_NDK/sources/cxx-stl/gnu-libstdc++/4.9/libs/arm64-v8a/include"
        export FFMPEG_CODEC_ARCH=aarch64
        export X264_SPECIFIC_CONFIG="--enable-pic --disable-asm"
        export FFMPEG_SPECIFIC_CONFIG="--enable-cross-compile --target-os=android --arch=$ARCH \
            --enable-pic --enable-thumb --ranlib=$TARGET_HOST-ranlib \
            --ar=$TARGET_HOST-ar --extra-cflags=-I../x264 --extra-ldflags=-L../x264"
        ;;
    x86_64-android)
        echo "Android (x86_64)"
        if [ "$BUILD_HOST" == "Darwin" ]
        then
            export ANDROID_TOOLCHAIN=$ANDROID_NDK/toolchains/x86_64-4.9/prebuilt/darwin-x86_64
            LLVM_TOOLCHAIN="$ANDROID_NDK/toolchains/llvm/prebuilt/darwin-x86_64/bin"
        else
            export ANDROID_TOOLCHAIN=$ANDROID_NDK/toolchains/x86_64-4.9/prebuilt/linux-x86_64
            LLVM_TOOLCHAIN="$ANDROID_NDK/toolchains/llvm/prebuilt/linux-x86_64/bin"
        fi

        export MIN_ANDROID_API=21
        export ARCH=x86_64
        export HOST="x86_64-linux-android"
        TARGET_HOST="x86_64-linux-android"
        SYSROOT=$ANDROID_NDK/platforms/android-$MIN_ANDROID_API/arch-x86_64
        export CXXFLAGS+="-isystem $ANDROID_NDK/sources/cxx-stl/gnu-libstdc++/4.9/include/backward \
            -isystem $ANDROID_NDK/sources/cxx-stl/gnu-libstdc++/4.9/include \
            -isystem $ANDROID_NDK/sources/cxx-stl/gnu-libstdc++/4.9/libs/$ARCH/include"
        export FFMPEG_CODEC_ARCH=x86
        export X264_SPECIFIC_CONFIG="--enable-pic --disable-asm"
        export FFMPEG_SPECIFIC_CONFIG="--enable-cross-compile --target-os=android --arch=x86-64 --cc=$HOST-gcc --cxx=$HOST-g++ \
            --enable-pic --ranlib=$HOST-ranlib --ar=$HOST-ar --as=$HOST-as \
            --extra-cflags=-march=x86-64 --extra-cflags=-msse4.2 --extra-cflags=-mpopcnt --extra-cflags=-m64 --extra-cflags=-mtune=intel --extra-cflags=-I../x264 \
            --extra-ldflags=-L../x264" # --logfile=/dev/stdout"
        ;;
    *)
	# sanity check;)
	exit 1
	;;
    esac

    # Add the standalone toolchain to the search path.
    export PATH=$ANDROID_TOOLCHAIN/bin:$PATH
    export BUILD_LIBX264="no"
    export BUILD_LIBXML2="yes"

    echo "ANDROID_NDK: $ANDROID_NDK"
    echo "ANDROID_TOOLCHAIN: $ANDROID_TOOLCHAIN"

    export AR=$ANDROID_TOOLCHAIN/bin/$TARGET_HOST-ar
    export AS=$ANDROID_TOOLCHAIN/bin/$TARGET_HOST-gcc
    export CC=$ANDROID_TOOLCHAIN/bin/$TARGET_HOST-gcc
    export CXX=$ANDROID_TOOLCHAIN/bin/$TARGET_HOST-g++
    export CPP="$ANDROID_TOOLCHAIN/bin/$TARGET_HOST-gcc -E"
    export CXXCPP="$ANDROID_TOOLCHAIN/bin/$TARGET_HOST-g++ -E"
    export LD=$ANDROID_TOOLCHAIN/bin/$TARGET_HOST-ld
    export STRIP=$ANDROID_TOOLCHAIN/bin/$TARGET_HOST-strip
    export RANLIB=$ANDROID_TOOLCHAIN/bin/$TARGET_HOST-ranlib

    # Tell configure what flags Android requires.
    export CPPFLAGS="-D__ANDROID_API__=$MIN_ANDROID_API -DANDROID -isysroot $ANDROID_NDK/sysroot -isystem $ANDROID_NDK/sysroot/usr/include/$TARGET_HOST $CFLAGS $CXXFLAGS"
    export CFLAGS+=" -D__ANDROID_API__=$MIN_ANDROID_API -DANDROID -fPIC -Wformat -isysroot $ANDROID_NDK/sysroot -isystem $ANDROID_NDK/sysroot/usr/include/$TARGET_HOST"
    export CXXFLAGS+=" $CFLAGS -ffunction-sections -funwind-tables"
    export LDFLAGS+=" -Wl,--gc-sections -Wl,-z,nocopyreloc -Wl,--no-undefined -Wl,-z,noexecstack --sysroot=$SYSROOT -lc -lm"
    #export LDFLAGS+=" --gc-sections -z nocopyreloc --no-undefined -z noexecstack -lc -lm" # --sysroot=$SYSROOT -lc -lm" # clang version
    export PREBUILD="$PREBUILD/$BUILD_TARGET"
    if [ \! -e $PREBUILD ]; then mkdir $PREBUILD; fi

    ;;

arm64-darwin-iphoneos | x86_64-darwin-iphonesimulator)
    SDK=iphoneos
    IOS_MIN_VERSION=9.0
    MIN_VERSION_FLAG="-mios-version-min=$IOS_MIN_VERSION"
    export SDK_PATH=`xcrun -sdk $SDK --show-sdk-path`
    export IOS_SDK_VERSION=`xcrun -sdk $SDK --show-sdk-version`
    export BUILD_LIBX264="yes"
    export BUILD_LIBXML2="yes"
    export CROSS_TOP=`xcode-select --print-path`/Platforms/iPhoneOS.platform/Developer
    export CROSS_SDK=iPhoneOS.sdk

    case "$1" in
    arm64-darwin-iphoneos)
	echo "iOS (arm 64)"
	export ARCH=arm64
        export BUILD_LIBX264="no"
	export HOST="arm-apple-darwin"
	export CFLAGS+="-fembed-bitcode-marker -fembed-bitcode "
	export X264_SPECIFIC_CONFIG="--enable-pic --disable-asm --sysroot=$SDK_PATH " #--extra-cflags=-arch --extra-cflags=$ARCH --extra-ldflags=-arch --extra-ldflags=$ARCH" --extra-cflags=-no-integrated-as
	export FFMPEG_SPECIFIC_CONFIG="--disable-outdevs --extra-cflags=-I../x264 --extra-cflags=-arch --extra-cflags=$ARCH --extra-ldflags=-L../x264 --extra-ldflags=-arch --extra-ldflags=$ARCH --enable-cross-compile --target-os=darwin --arch=$ARCH --sysroot=$SDK_PATH"
        export FFMPEG_CODEC_ARCH=aarch64
	;;
    x86_64-darwin-iphonesimulator)
	echo "64bit iOS Simulator"
        export BUILD_LIBX264="no"
	export ARCH=x86_64
	export HOST=x86_64-apple-darwin
	SDK=iphonesimulator
	export SDK_PATH=`xcrun -sdk $SDK --show-sdk-path`
	export IOS_SDK_VERSION=`xcrun -sdk $SDK --show-sdk-version`
	MIN_VERSION_FLAG="-mios-simulator-version-min=$IOS_MIN_VERSION"
	export X264_SPECIFIC_CONFIG="--enable-pic --disable-asm --sysroot=$SDK_PATH " #--extra-cflags=-arch --extra-cflags=$ARCH --extra-ldflags=-arch --extra-ldflags=$ARCH"
	export FFMPEG_SPECIFIC_CONFIG="--disable-outdevs --extra-cflags=-I../x264 --extra-cflags=-arch --extra-cflags=$ARCH --extra-ldflags=-L../x264 --extra-ldflags=-arch --extra-ldflags=$ARCH --enable-cross-compile --target-os=darwin --arch=$ARCH --sysroot=$SDK_PATH"
	;;
    *)
	# sanity check;)
	exit 1
	;;
    esac

    export XCODE_ROOT="/Applications/Xcode.app/Contents/Developer"
    export CC=`xcrun -f -sdk $SDK clang`
    export CXX=`xcrun -f -sdk $SDK clang++`
    export CFLAGS+="-pipe -DNO_STRSTREAM -arch $ARCH -isysroot $SDK_PATH $MIN_VERSION_FLAG "
    export CXXFLAGS="$CFLAGS -stdlib=libc++ "
    export AS=`xcrun -f -sdk $SDK as`
    export AR=`xcrun -f -sdk $SDK ar`
    export RANLIB=`xcrun -f -sdk $SDK ranlib`
    export CPP=`xcrun -f -sdk $SDK clang`" -E"
    export CPPFLAGS="-arch $ARCH -isysroot $SDK_PATH"
    export CXXCPP=`xcrun -f -sdk $SDK clang++`" -E"
    export LDFLAGS="$LDFLAGS -v -arch $ARCH $MIN_VERSION_FLAG -L$SDK_PATH/usr/lib "

    export PREBUILD="$PREBUILD/$BUILD_TARGET"
    if [ \! -e $PREBUILD ]; then mkdir $PREBUILD; fi

    export PATH=$BUILD/scripts:$PATH
    ;;

x86_64-mingw32)
    echo "MinGW (64)"
    export HOST="x86_64-mingw32"
    export MINGW_PREFIX="x86_64-w64-mingw32-"
    export CC="${MINGW_PREFIX}gcc"
    export CXX="${MINGW_PREFIX}g++"
    GCC_GTEQ_10=`expr $($CC -dumpversion | cut -f1 -d.) \>= 10`

    if [ "$GCC_GTEQ_10" = "1" ]; then
        export CFLAGS="$CFLAGS -g -fcommon"
        export CXXFLAGS="$CFLAGS -fweak"
    else
        export CFLAGS="$CFLAGS -g"
        export CXXFLAGS="$CFLAGS"
    fi

    export CPP="${MINGW_PREFIX}cpp"
    #export CPPFLAGS=""
    export AS="${MINGW_PREFIX}as"
    export AR="${MINGW_PREFIX}ar"
    export RANLIB="${MINGW_PREFIX}ranlib"
    export LD="${MINGW_PREFIX}ld"
    export RC="${MINGW_PREFIX}windres"
    #export LDFLAGS="$LDFLAGS"
    export STRIP="${MINGW_PREFIX}strip"

    export PREBUILD="$PREBUILD/$BUILD_TARGET"
    if [ \! -e $PREBUILD ]; then mkdir $PREBUILD; fi

    export X264_SPECIFIC_CONFIG="--enable-pic --enable-win32thread"
    export FFMPEG_SPECIFIC_CONFIG="--extra-cflags=-I../x264 --extra-ldflags=-L../x264 --enable-cross-compile --target-os=mingw32 --arch=x86_64 --cross-prefix=$MINGW_PREFIX \
                 --enable-w32threads --extra-cflags=-msse3 \
                 --enable-dxva2 --enable-hwaccel=h264_dxva2 --enable-hwaccel=mpeg2_dxva2 --enable-indev=dshow" # --enable-memalign-hack
    export BUILD_LIBXML2="yes"
    #export BUILD_LIBX264="yes"
    export BUILD_ZLIB="yes"
    ;;
update)
    update_deps
    exit 0
    ;;

*)
    echo "Architecture is not supported"
    exit 1
esac

if [ -z "$2" ]
then
    if [ "$BUILD_LIBX264" == "yes" ]; then build_x264; fi
    if [ "$BUILD_ICONV" == "yes" ]; then build_iconv; fi
    build_ffmpeg
    build_live555
    build_openssl
    build_boost
    if [ "$BUILD_ZLIB" == "yes" ]; then  build_zlib; fi
    build_curl
    build_libwebsockets
    build_ACE
    build_SDL2
    if [ "$BUILD_LIBXML2" == "yes" ]; then build_xml2; fi
    echo "Build done"
else
    build_$2
fi
