SET search_path TO confdb;

-------------------------------------------------------------------
----- watchlist_types

CREATE TABLE watchlist_type ( 
    watchlist_type_id   INTEGER     PRIMARY KEY, 
    name                VARCHAR     NOT NULL,
    description         VARCHAR     NOT NULL,
    entries_mime_type   VARCHAR     NULL
);

INSERT INTO watchlist_type VALUES ( 1, 'LPR', 'License Plate Recognition watch list', 'text/plain' );
INSERT INTO watchlist_type VALUES ( 2, 'Facial', 'Facial Recognition subject list', 'vnd/facial-template' );

REVOKE ALL ON TABLE watchlist_type FROM PUBLIC;
GRANT SELECT ON TABLE watchlist_type TO apl;

-------------------------------------------------------------------
----- watchlist

CREATE TABLE watchlist ( 
    watchlist_id        UUID        PRIMARY KEY DEFAULT uuid_generate_v1(),
    watchlist_type_id   INTEGER     NOT NULL REFERENCES watchlist_type(watchlist_type_id),
    realm_id            UUID        NOT NULL REFERENCES _objs(obj),
    name                VARCHAR     NOT NULL,
    description         VARCHAR     NOT NULL,
    far_threshold       DECIMAL(6,6) NOT NULL,
    action              VARCHAR     NOT NULL CHECK (action in ('ignore', 'event', 'alert')),
    updated_at          TIMESTAMP WITHOUT TIME ZONE DEFAULT (now() AT TIME ZONE 'UTC'),
    protected           INTEGER     DEFAULT 0
);

CREATE INDEX watchlist_realm_id ON watchlist(realm_id);

REVOKE ALL ON TABLE watchlist FROM PUBLIC;
GRANT SELECT,INSERT,UPDATE,DELETE ON TABLE watchlist TO apl;

-------------------------------------------------------------------
----- watchlist_entries

CREATE TABLE watchlist_entries ( 
    watchlist_id        UUID        REFERENCES watchlist(watchlist_id),
    watchlist_entry_id  UUID        DEFAULT uuid_generate_v1(),
    description         VARCHAR     NOT NULL,
    entry               BYTEA       NOT NULL,
    PRIMARY KEY (watchlist_id, watchlist_entry_id)
);
-- note 1: access to entire list by the leading part of index :)
-- note 2: access to individual entries will require both parts of compound PK :(

REVOKE ALL ON TABLE watchlist_entries FROM PUBLIC;
GRANT SELECT,INSERT,UPDATE,DELETE ON TABLE watchlist_entries TO apl;

-------------------------------------------------------------------
----- watchlist_realm_miss_actions

CREATE TABLE watchlist_realm_miss_actions ( 
    realm_id            UUID        REFERENCES _objs(obj),
    watchlist_type_id   INTEGER     REFERENCES watchlist_type(watchlist_type_id),
    action              VARCHAR     NOT NULL CHECK (action in ('ignore', 'event', 'alert')),
    updated_at          TIMESTAMP WITHOUT TIME ZONE DEFAULT (now() AT TIME ZONE 'UTC'),
    PRIMARY KEY (realm_id, watchlist_type_id)
);

REVOKE ALL ON TABLE watchlist_realm_miss_actions FROM PUBLIC;
GRANT SELECT,INSERT,UPDATE,DELETE ON TABLE watchlist_realm_miss_actions TO apl;

-------------------------------------------------------------------
----- triggers

SET search_path TO confdb;

CREATE OR REPLACE FUNCTION update_wl_time() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
    if TG_OP = 'DELETE' THEN
        UPDATE watchlist SET updated_at = now() AT TIME ZONE 'UTC'
            WHERE watchlist_id = OLD.watchlist_id;
        RETURN OLD;
    ELSE
        UPDATE watchlist SET updated_at = now() AT TIME ZONE 'UTC'
            WHERE watchlist_id = NEW.watchlist_id;
        RETURN NEW;
    END IF;
END;
$$;

CREATE TRIGGER trg_wl_upd_time BEFORE UPDATE ON watchlist FOR EACH ROW EXECUTE PROCEDURE upd_timestamp();
CREATE TRIGGER trg_wle_upd_time BEFORE UPDATE OR INSERT OR DELETE ON watchlist_entries FOR EACH ROW EXECUTE PROCEDURE update_wl_time();

-------------------------------------------------------------------
----- public interface

SET search_path TO apl;

CREATE OR REPLACE VIEW watchlist_type AS SELECT * FROM confdb.watchlist_type;
REVOKE ALL ON TABLE watchlist_type FROM PUBLIC;
GRANT SELECT ON TABLE watchlist_type TO apl;

CREATE OR REPLACE VIEW watchlist AS SELECT * FROM confdb.watchlist;
REVOKE ALL ON TABLE watchlist FROM PUBLIC;
GRANT SELECT,INSERT,UPDATE,DELETE ON TABLE watchlist TO apl;

CREATE OR REPLACE VIEW watchlist_entries AS SELECT * FROM confdb.watchlist_entries;
REVOKE ALL ON TABLE watchlist_entries FROM PUBLIC;
GRANT SELECT,INSERT,UPDATE,DELETE ON TABLE watchlist_entries TO apl;

CREATE OR REPLACE VIEW watchlist_realm_miss_actions AS SELECT * FROM confdb.watchlist_realm_miss_actions;
REVOKE ALL ON TABLE watchlist_realm_miss_actions FROM PUBLIC;
GRANT SELECT,INSERT,UPDATE,DELETE ON TABLE watchlist_realm_miss_actions TO apl;
