SET search_path TO apl;

--
-- Name: count_expired_objects(integer); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION count_expired_objects(integer) RETURNS integer
    LANGUAGE plpgsql
    AS $_$
DECLARE
    preserve_hours ALIAS FOR $1;
    result integer;
BEGIN

    SELECT count(obj) INTO result FROM _objs WHERE ((otype='D' and subtype <> 'N') OR otype='X')
	AND deleted=1 AND protected=0 AND (now() at time zone 'UTC' - (preserve_hours || 'hour')::interval) > rtime
	AND obj > 99 AND NOT EXISTS (SELECT 1 FROM event WHERE objid=obj);

    RETURN result;

END;
$_$;


--
-- Name: el_get_object_info(uuid); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION el_get_object_info(uuid) RETURNS SETOF el_object_info
    LANGUAGE plpgsql
    AS $_$
DECLARE
   obj_id   ALIAS FOR $1;
   result   EL_Object_Info;
BEGIN
  FOR result in
      SELECT o.obj, o.name, o.type, o.otype, o.subtype, COALESCE(a1.val, 'UTC'), 0, COALESCE(a2.val, '0')
        FROM _objs o
        LEFT OUTER JOIN _obj_attr a1 ON a1.obj=o.obj AND a1.attr='TIME_ZONE'
        LEFT OUTER JOIN _obj_attr a2 ON a2.obj=o.obj AND a2.attr='STAT_VL_RATE_LAST'
       WHERE o.obj = obj_id
  LOOP
     RETURN NEXT result;
  END LOOP;

RETURN;

END;
$_$;


--
-- Name: el_get_workflow(uuid, integer, integer); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION el_get_workflow(uuid, integer, integer) RETURNS SETOF el_workflow
    LANGUAGE plpgsql
    AS $_$
DECLARE
   obj_id     ALIAS FOR $1;
   source_id  ALIAS FOR $2;
   event_type ALIAS FOR $3;
   result EL_Workflow;
BEGIN
    --
    -- Return:
    --  -1 - do not create event policy, 0 - workflow is not found, 1+ - workflow
    --

    FOR result in
        SELECT CASE q.workflow WHEN -1 THEN -1 ELSE COALESCE(w.workflow, 0) END as workflow, q.parameters
          FROM (SELECT p.parameters,
                       CASE event_type
                           WHEN -1 THEN
                               CASE substring(p.parameters from 'eventtype=([[:digit:]])')
                                   WHEN '2' THEN -1
                                   WHEN '0' THEN p.infworkflow
                                   ELSE p.workflow
                               END
                           WHEN 2 THEN -1
                           WHEN 0 THEN p.infworkflow
                           ELSE p.workflow
                       END as workflow
                  FROM _obj_attr a, eventpolicycontent p
                 WHERE a.obj=obj_id AND a.attr='EVENT_POLICY'
                   AND p.eventsource=source_id AND CAST(p.eventpolicy AS TEXT)=a.val) q
        LEFT JOIN workflow w ON w.workflow = q.workflow
  LOOP
     RETURN NEXT result;
  END LOOP;

RETURN;

END;
$_$;


--
-- Name: el_get_workflow_context(integer); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION el_get_workflow_context(integer) RETURNS SETOF el_workflow_context
    LANGUAGE plpgsql
    AS $_$
DECLARE
    result EL_Workflow_Context;
BEGIN
   FOR result in
       SELECT context FROM workflow WHERE workflow=$1
   LOOP
       RETURN NEXT result;
   END LOOP;

   RETURN;

END;
$_$;


--
-- Name: getnodes(); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION getnodes() RETURNS SETOF node_list
    LANGUAGE plpgsql
    AS $$
DECLARE
   result node_list;
BEGIN
    FOR result in
        (SELECT
            node.obj, node.name as UNI, count(*) as camera_number
        FROM
            _objs as node,
            _objs as camera
        WHERE
            node.otype='D'
            AND node.subtype='N'
            AND node.deleted=0
            AND camera.otype='D'
            AND camera.subtype='C'
            AND camera.deleted=0
            AND node.name = camera.node_ip
        GROUP BY node.obj, node.name)
        UNION ALL
        (SELECT
            node.obj, node.name as UNI, 0
        FROM
            _objs as node
        WHERE
            otype='D'
            AND subtype='N'
            AND deleted=0
            AND name NOT IN (SELECT node_ip FROM _objs WHERE otype='D' AND subtype='C' AND deleted=0)
        GROUP BY node.obj, node.name)
        ORDER BY camera_number ASC
    LOOP
        RETURN NEXT result;
    END LOOP;

    RETURN;
END
$$;


--
-- Name: getobject(uuid, uuid, uuid); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION getobject(realm uuid, userobj uuid, objectobj uuid) RETURNS SETOF object_list_with_credentials
    LANGUAGE plpgsql
    AS $_$
BEGIN
    RETURN QUERY
        SELECT _objs.realm, _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.type, _objs.otype, _objs.subtype, _objs.protected, concatFields(set2role.permission)::varchar as permission, concatFields(permission_type.credentials || set2role.special_credentials)::varchar as credentials
            FROM _objs, permission_type, _links as set2role, _links as user2role, _links as object2set
            WHERE
                ($1 IS NULL OR _objs.realm = $1)
                AND set2role.link_type = 'set2role'
                AND user2role.link_type = 'user2role'
                AND object2set.link_type LIKE '%2set'
                AND _objs.obj = object2set.obj_res
                AND set2role.obj_cons = user2role.obj_cons
                AND object2set.obj_cons = set2role.obj_res
                AND set2role.permission = permission_type.permission
                AND user2role.obj_res = userobj
                AND object2set.obj_res = objectobj
            GROUP BY _objs.realm, _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.type, _objs.otype, _objs.subtype, _objs.protected
            ORDER BY _objs.name;
END
$_$;


--
-- Name: getobjects(uuid, uuid, uuid, character, character); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION getobjects(realm uuid, userobj uuid, roleobj uuid, objtype character, objsubtype character) RETURNS SETOF object_list_with_credentials
    LANGUAGE plpgsql
    AS $_$
BEGIN
    RETURN QUERY
        SELECT _objs.realm, _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.type, _objs.otype, _objs.subtype, _objs.protected, concatFields(set2role.permission)::varchar AS permission, concatFields(permission_type.credentials || set2role.special_credentials)::varchar as credentials
            FROM _objs, permission_type, _links as set2role, _links as user2role, _links as object2set
            WHERE
                ($1 IS NULL OR _objs.realm = $1)
                AND (objtype IS NULL OR _objs.otype = objtype)
                AND (objsubtype IS NULL OR _objs.subtype = objsubtype)
                AND _objs.deleted = 0
                AND set2role.link_type = 'set2role'
                AND user2role.link_type = 'user2role'
                AND object2set.link_type LIKE '%2set'
                AND _objs.obj = object2set.obj_res
                AND set2role.obj_cons = user2role.obj_cons
                AND object2set.obj_cons = set2role.obj_res
                AND set2role.permission = permission_type.permission
                AND user2role.obj_res = userobj
                AND (roleobj IS NULL OR user2role.obj_cons = roleobj)
            GROUP BY _objs.realm, _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.type, _objs.otype, _objs.subtype, _objs.protected
            ORDER BY _objs.name;
END
$_$;


--
-- Name: getobjects(uuid, uuid, uuid, uuid, character, character); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION getobjects(realm uuid, userobj uuid, roleobj uuid, setobj uuid, objtype character, objsubtype character) RETURNS SETOF object_list_with_credentials
    LANGUAGE plpgsql
    AS $_$
BEGIN
    RETURN QUERY
        SELECT object.realm, object.obj, object.udid, object.name, object.description, object.type, object.otype, object.subtype, object.protected, object.permission, object.credentials
            FROM (
                SELECT _objs.realm, _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.type, _objs.otype, _objs.subtype, _objs.protected, concatFields(set2role.permission)::varchar AS permission, concatFields(permission_type.credentials || set2role.special_credentials)::varchar as credentials
                    FROM _objs, permission_type, _links as set2role, _links as user2role, _links as object2set
                    WHERE
                        ($1 IS NULL OR _objs.realm = $1)
                        AND (objtype IS NULL OR _objs.otype = objtype)
                        AND (objsubtype IS NULL OR _objs.subtype = objsubtype)
                        AND _objs.deleted = 0
                        AND set2role.link_type = 'set2role'
                        AND user2role.link_type = 'user2role'
                        AND object2set.link_type LIKE '%2set'
                        AND _objs.obj = object2set.obj_res
                        AND set2role.obj_cons = user2role.obj_cons
                        AND object2set.obj_cons = set2role.obj_res
                        AND set2role.permission = permission_type.permission
                        AND user2role.obj_res = userobj
                        AND (roleobj IS NULL OR user2role.obj_cons = roleobj)
                    GROUP BY _objs.realm, _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.type, _objs.otype, _objs.subtype, _objs.protected
            ) AS object, _links as object2set
            WHERE
                object2set.link_type LIKE '%2set'
                AND object.obj = object2set.obj_res
                AND object2set.obj_cons = setobj
            ORDER BY object.name;
END
$_$;


--
-- Name: getobjectsfromrole(integer, integer, character, character); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION getobjectsfromrole(realm uuid, roleobj uuid, otype character, subtype character) RETURNS SETOF object_list_without_credentials
    LANGUAGE plpgsql
    AS $_$
BEGIN
    RETURN QUERY
        SELECT _objs.realm, set2role.obj_res as setobj, _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.type, _objs.otype, _objs.subtype, _objs.protected
            FROM _objs, _links as set2role, _links as object2set
            WHERE
                ($1 IS NULL OR _objs.realm = $1)
                AND ($2 IS NULL OR _objs.otype = $3)
                AND ($3 IS NULL OR _objs.subtype = $4)
                AND _objs.deleted = 0
                AND set2role.link_type = 'set2role'
                AND object2set.link_type LIKE '%2set'
                AND object2set.obj_res = _objs.obj
                AND set2role.obj_res = object2set.obj_cons
                AND set2role.obj_cons = $2
            ORDER BY _objs.name;
END
$_$;


--
-- Name: getobjectsfromset(uuid, uuid, character, character); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION getobjectsfromset(realm uuid, setobj uuid, otype character, subtype character) RETURNS SETOF object_list_with_credentials
    LANGUAGE plpgsql
    AS $_$
BEGIN
    RETURN QUERY
        SELECT _objs.realm, _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.type, _objs.otype, _objs.subtype, _objs.protected
            FROM _objs, _links as object2set
            WHERE
                ($1 IS NULL OR _objs.realm = $1)
                AND ($2 IS NULL OR _objs.otype = $3)
                AND ($3 IS NULL OR _objs.subtype = $4)
                AND _objs.deleted = 0
                AND object2set.link_type LIKE '%2set'
                AND object2set.obj_res = _objs.obj
                AND object2set.obj_cons = $2
            ORDER BY _objs.name;
END
$_$;


--
-- Name: getroles(uuid); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION getroles(realm uuid) RETURNS SETOF object_list
    LANGUAGE plpgsql
    AS $_$
BEGIN
    RETURN QUERY
        SELECT _objs.realm, _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.type, _objs.protected
            FROM _objs
            WHERE
                _objs.deleted = 0
                AND ($1 IS NULL OR _objs.realm = $1)
                AND _objs.otype = 'R'
            ORDER BY _objs.name;
END
$_$;


--
-- Name: getroles(integer, integer); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION getroles(realm uuid, userobj uuid) RETURNS SETOF object_list
    LANGUAGE plpgsql
    AS $_$
BEGIN
    RETURN QUERY
        SELECT _objs.realm, _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.type, _objs.protected
            FROM _objs, _links as user2role
            WHERE
                _objs.deleted = 0
                AND ($1 IS NULL OR _objs.realm = $1)
                AND user2role.link_type = 'user2role'
                AND _objs.obj = user2role.obj_cons
                AND _objs.otype = 'R'
                AND user2role.obj_res = $2
            ORDER BY _objs.name;
END
$_$;


--
-- Name: getrolesforset(uuid, uuid); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION getrolesforset(realm uuid, setobj uuid) RETURNS SETOF object_list
    LANGUAGE plpgsql
    AS $_$
BEGIN
    RETURN QUERY
        SELECT _objs.realm, _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.type, _objs.protected
            FROM _objs, _links as set2role
            WHERE
                _objs.deleted = 0
                AND ($1 IS NULL OR _objs.realm = $1)
                AND set2role.link_type = 'set2role'
                AND _objs.obj = set2role.obj_cons
                AND _objs.otype = 'R'
                AND set2role.obj_res = $2
            ORDER BY _objs.name;
END
$_$;


--
-- Name: getsets(uuid); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION getsets(realm uuid) RETURNS SETOF object_list
    LANGUAGE plpgsql
    AS $_$
BEGIN
    RETURN QUERY
        SELECT _objs.realm, _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.type, _objs.protected
            FROM _objs
            WHERE
                _objs.deleted = 0
                AND ($1 IS NULL OR _objs.realm = $1)
                AND _objs.otype = 'S'
            ORDER BY _objs.name;
END
$_$;


--
-- Name: getsets(uuid, uuid); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION getsets(realm uuid, objectobj uuid) RETURNS SETOF object_list
    LANGUAGE plpgsql
    AS $_$
BEGIN
    RETURN QUERY
        SELECT _objs.realm, _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.type, _objs.protected
            FROM _objs, _links as object2set
            WHERE
                ($1 IS NULL OR _objs.realm = $1)
                AND _objs.deleted = 0
                AND _objs.otype = 'S'
                AND object2set.obj_cons = _objs.obj
                AND object2set.obj_res = objectobj
                AND object2set.link_type LIKE '%2set'
            ORDER BY _objs.name;
END
$_$;


--
-- Name: getsets(uuid, uuid, uuid, uuid); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION getsets(realm uuid, userobj uuid, roleobj uuid, objectobj uuid) RETURNS SETOF object_list
    LANGUAGE plpgsql
    AS $_$
BEGIN
    RETURN QUERY
        SELECT DISTINCT _objs.realm, _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.type, _objs.protected
            FROM _objs, _links as user2role, _links as set2role, _links as object2set
            WHERE
                _objs.deleted = 0
                AND ($1 IS NULL OR _objs.realm = $1)
                AND user2role.link_type = 'user2role'
                AND set2role.link_type = 'set2role'
                AND _objs.obj = set2role.obj_res
                AND _objs.otype = 'S'
                AND user2role.obj_res = userobj
                AND (roleobj IS NULL OR user2role.obj_cons = roleobj)
                AND set2role.obj_cons = user2role.obj_cons
                AND (
                    objectobj IS NULL OR (
                        object2set.obj_res = objectobj
                        AND object2set.obj_cons = set2role.obj_res
                        AND object2set.link_type LIKE '%2set'
                    )
                )
            ORDER BY _objs.name;
END
$_$;


--
-- Name: getsetsfromrole(uuid, uuid); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION getsetsfromrole(realm uuid, roleobj uuid) RETURNS SETOF object_list_with_credentials
    LANGUAGE plpgsql
    AS $_$
BEGIN
    RETURN QUERY
        SELECT _objs.realm, _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.type, _objs.otype, _objs.subtype, _objs.protected, set2role.permission::varchar, (permission_type.credentials || set2role.special_credentials)::varchar
            FROM _objs, _links as set2role
            LEFT OUTER JOIN permission_type ON set2role.permission = permission_type.permission
            WHERE
                _objs.deleted = 0
                AND ($1 IS NULL OR _objs.realm = $1)
                AND set2role.link_type = 'set2role'
                AND set2role.obj_res = _objs.obj
                AND set2role.obj_cons = $2
            ORDER BY _objs.name;
END
$_$;


--
-- Name: getusers(uuid); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION getusers(realm uuid) RETURNS SETOF object_list
    LANGUAGE plpgsql
    AS $_$
BEGIN
    RETURN QUERY
        SELECT _objs.realm, _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.type, _objs.protected
            FROM _objs
            WHERE
                _objs.deleted = 0
                AND ($1 IS NULL OR _objs.realm = $1)
                AND _objs.otype = 'U'
            ORDER BY _objs.name;
END
$_$;


--
-- Name: getusers(uuid, uuid); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION getusers(realm uuid, roleobj uuid) RETURNS SETOF object_list
    LANGUAGE plpgsql
    AS $_$
BEGIN
    RETURN QUERY
        SELECT _objs.realm, _objs.obj, _objs.udid, _objs.name, _objs.description, _objs.type, _objs.protected
            FROM _objs, _links as user2role
            WHERE
                _objs.deleted = 0
                AND ($1 IS NULL OR _objs.realm = $1)
                AND user2role.link_type = 'user2role'
                AND _objs.obj = user2role.obj_res
                AND _objs.otype = 'U'
                AND user2role.obj_cons = roleobj
            ORDER BY _objs.name;
END
$_$;


--
-- Name: initcamnodeid(); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION initcamnodeid() RETURNS void
    LANGUAGE plpgsql
    AS $$
DECLARE
    objrec RECORD;
BEGIN
    FOR objrec IN SELECT * FROM _objs WHERE otype='D' AND subtype in ('C','A') AND deleted=0 LOOP
    BEGIN
        INSERT INTO  _obj_attr VALUES (objrec.obj, 'NODEID', objrec.node_id);
    EXCEPTION WHEN unique_violation THEN
        UPDATE _obj_attr SET val=objrec.node_id WHERE attr='NODEID';
    END;
    END LOOP;
END;
$$;


--
-- Name: merge_obj_attr(integer, character varying, character varying); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION merge_obj_attr(objid uuid, name character varying, value character varying) RETURNS void
    LANGUAGE plpgsql
    AS $$
BEGIN
    BEGIN
        INSERT INTO _obj_attr (obj, attr, val) VALUES (objid, name, value);
    EXCEPTION WHEN unique_violation THEN
        UPDATE _obj_attr SET val = value WHERE obj = objid AND attr = name;
    END;
END;
$$;


--
-- Name: purge_expired_objects(integer, integer); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION purge_expired_objects(integer, integer) RETURNS integer
    LANGUAGE plpgsql
    AS $_$
DECLARE
    num_objs ALIAS FOR $1;
    preserve_hours ALIAS FOR $2;
    result integer;
BEGIN
    CREATE TEMPORARY TABLE exp_objs (o integer) ON COMMIT DROP;

    INSERT INTO exp_objs
    SELECT obj FROM _objs WHERE ((otype='D' and subtype <> 'N') OR otype='X')
    AND deleted=1 AND protected=0 AND (now() at time zone 'UTC' - (preserve_hours || 'hour')::interval) > rtime
    AND obj > 99 AND NOT EXISTS (SELECT 1 FROM event WHERE objid=obj)
    ORDER BY rtime LIMIT num_objs;

    DELETE FROM confdb.obj_block WHERE obj IN (SELECT o FROM exp_objs);
    DELETE FROM confdb._obj_attr WHERE obj IN (SELECT o FROM exp_objs);
    DELETE FROM confdb._links WHERE obj_res IN (SELECT o FROM exp_objs) OR obj_cons IN (SELECT o FROM exp_objs);
    DELETE FROM confdb._objs WHERE obj IN (SELECT o FROM exp_objs);

    GET DIAGNOSTICS result = ROW_COUNT;

    RETURN result;
END;
$_$;

CREATE OR REPLACE FUNCTION identity_objid() RETURNS uuid
    LANGUAGE plpgsql
    AS $_$
BEGIN
    RETURN '00000000-0000-0000-0000-000000000053';
END;
$_$;
