package Overcast::Common;

use 5.008008;
use strict;
use warnings;
use Time::Local;

require Exporter;

our @ISA = qw(Exporter);

our %EXPORT_TAGS = ( 'all' => [ qw(
 UNIX2ReadableTime ReadableTime2UNIX DBTime2UNIX IsObjId IsDebug LogDebug LogInfo LogWarn LogError LogFatal AddTask TaskManager ArrayTable
) ] );

our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

our @EXPORT = qw(
 UNIX2ReadableTime ReadableTime2UNIX DBTime2UNIX IsObjId IsDebug LogDebug LogInfo LogWarn LogError LogFatal AddTask TaskManager ArrayTable
);

our $VERSION = '0.01';

my $APL      = $ENV{APL};
my %Tasks;

#--------------------------------
# Logging                       |
#--------------------------------

sub _log {
    my ($level, $msg) = @_;
    chomp $msg;
    warn "[" . UNIX2ReadableTime(time,2) . "] $level $msg\n";
}

sub LogWarn  { _log "WARNING", @_ }
sub LogDebug { _log "DEBUG",   @_ }
sub LogInfo  { _log "INFO",    @_ }
sub LogError { _log "ERROR",   @_ }
sub LogFatal { _log "FATAL",   @_; die @_ }
sub IsDebug  { return $ENV{DEBUG} ? 1 : 0 }

#--------------------------------
# Task Management               |
#--------------------------------

sub AddTask {
    my $task = shift;
    $task->{created} = time;
    $task->{started} = $task->{finished} = $task->{inprogress} = 0;
    $Tasks{$task->{id}} = $task;
}

sub _start_task {
    my $task = shift;
    $task->{started} = time;
    $task->{inprogress} = 1;
    
    $task->{fn}->($task);
    
    $task->{finished} = time;
    $task->{inprogress} = 0;
}

sub TaskManager {
    while (1) {
        foreach my $t (values %Tasks) {
            next if $t->{inprogress};
            _start_task $t if time - $t->{finished} >= $t->{interval};
        }
        sleep 1;
        system('touch /tmp/taskmanager');
    }
}

#--------------------------------
# Miscellaneous			|
#--------------------------------

# Prints pretty-formatted view of 2-dimensional array
sub ArrayTable {
    return if not $_[0] or not $_[1];
    my @names = @{$_[0]};
    my @table = @{$_[1]};
    my $fh = $_[2] || \*STDOUT;
    return if not @names or not @table;
    no warnings 'uninitialized';
    
    my $num_of_fields = @names;
    my $num_of_rows = @table;
    
    # Get maximum length required for each field
    my @maxlen;
    $maxlen[$_] = length($names[$_]) foreach (0..$#names);
    my $r;
    for(my $i = 0; $i < $num_of_rows; ++$i) {
	for (my $j = 0; $j < $num_of_fields; $j++) {
	    my $len = length($table[$i][$j]);
	    $maxlen[$j] = $len if $len > $maxlen[$j];
	}
    }
    
    my $pre = 2;
    my $post = 2;
    my $total_len = 1;
    $total_len += ($pre+$post+$_+1) foreach (@maxlen);
    
    my $line = '-' x $total_len;
    my $post_spaces = 0;
    my $header = "|";
    
    # Print header
    for(my $i = 0; $i < $num_of_fields; ++$i) {
	$post_spaces = $maxlen[$i] + $post - length($names[$i]);
	$header .= (' ' x $pre) . $names[$i] . (' ' x $post_spaces) . '|';
    }
    print $fh "$line\n$header\n$line\n";
    
    # Print rows
    for(my $i = 0; $i < $num_of_rows; ++$i) {
	my $row = '|';
	for(my $j = 0; $j < $num_of_fields; ++$j) {
	    my $max_len = $maxlen[$j];
	    $post_spaces = $max_len + $post - length($table[$i][$j]);
	    $row .=  ((' ' x $pre) . $table[$i][$j] . (' ' x $post_spaces) . '|');
	}
	print $fh "$row\n";	
    }
    
    print $fh "$line\n";
} # Sub ArrayTable

# Converts UNIX timestamp to human readable time string
sub UNIX2ReadableTime {
    my ($time, $format_code) = @_;
    $format_code = 2 if not defined $format_code;
    my $format;
    my ($tsec,$tmin,$thour,$tday,$tmon,$tyear)=(gmtime($time))[0..5];
    my $year = $tyear - 100;
    if ($format_code == 0) {
        $format = "%02u%02u%02u%02u%02u%02u";
    } elsif ($format_code == 1) {
        $format = "%02u%02u%02u_%02u%02u%02u";
    } elsif ($format_code == 2) {
        $year = $tyear + 1900;
        $format = "%04u-%02u-%02u %02u:%02u:%02u";
    }
    return sprintf($format,$year,$tmon+1,$tday,$thour,$tmin,$tsec);
}

# Converts readable time string to UNIX timestamp
sub ReadableTime2UNIX {
    my $ts = shift;
    if ($ts =~ /^(\d{2})\D*(\d{2})\D*(\d{2})\D*(\d{2})\D*(\d{2})\D*(\d{2})$/) {
        my ($year,$mon,$day,$hour,$min,$sec) = ($1,$2,$3,$4,$5,$6);
        return timegm($sec,$min,$hour,$day,$mon-1,$year+100);
    }
    return -1;
}

# Converts DB timestamp to UNIX timestamp
sub DBTime2UNIX {
    return undef if not defined $_[0];
    if ($_[0] =~ /^(\d{4})-(\d{2})-(\d{2}) (\d{2}):(\d{2}):(\d{2})/) {
        my ($year,$mon,$day,$hour,$min,$sec) = ($1,$2,$3,$4,$5,$6);
        return timegm($sec,$min,$hour,$day,$mon-1,$year-1900);
    }
    return undef;
}

# Checks OBJID for validity
sub IsObjId {
    my $objid = shift;
    return $objid =~ /^[0-9A-F]{8}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{12}$/i ? 1 : 0;
}


1;
__END__
