SET search_path TO apl;

--
-- Name: av_add_geo_positions(uuid, anyarray); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION av_add_geo_positions(uuid, anyarray) RETURNS void
    LANGUAGE plpgsql
    AS $_$
DECLARE
   obj_id         ALIAS FOR $1;
   geo_list       ALIAS FOR $2;
   geo double precision[];
   merged_geo_list text;
   ts integer;
   geo_json text;
BEGIN

IF array_length(geo_list, 1) IS NULL THEN
   RETURN;
END IF;

FOREACH geo SLICE 1 IN ARRAY geo_list
LOOP
     geo_json := '{"ts":'||geo[1]::integer||',"lat":'||geo[2]||',"lon":'||geo[3]||',"alt":'||geo[4]||'}';
     ts := geo[1]::integer - (geo[1]::integer % 60); -- align by minutes
     merged_geo_list := (SELECT GDATA FROM avatar_geo_history WHERE AVATARID=obj_id AND ts = extract('epoch' from GTIME) + extract(timezone from current_time));
     IF merged_geo_list IS NULL THEN
        INSERT INTO avatar_geo_history(AVATARID, GTIME, GDATA) VALUES (obj_id, TIMESTAMP 'epoch' + ts * INTERVAL '1 second', '['||geo_json||']');
     ELSE
        merged_geo_list := trim(trailing ']' from merged_geo_list);
        merged_geo_list :=  merged_geo_list || ',' || geo_json || ']';
        UPDATE avatar_geo_history SET GDATA=merged_geo_list WHERE AVATARID=obj_id AND ts = extract('epoch' from GTIME) + extract(timezone from current_time);
     END IF;
END LOOP;

--RAISE EXCEPTION '%', $2;

END;
$_$;




--
-- Name: el_get_event(integer); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION el_get_event(integer) RETURNS SETOF el_event
    LANGUAGE plpgsql
    AS $_$
DECLARE
   result EL_Event;
BEGIN

FOR result in
    SELECT
               event.objid,
               eventsource,
               extract('epoch' from utc_when),
               extract('epoch' from utc_from),
               extract('epoch' from utc_to),
               priority,
               state,
               message,
               note,
               eventtype,
               externalid,
               eventproperty.name,
               eventproperty.value,
               -- fields bellow used in next SELECT
               NULL as witness_objid,
               NULL as witness_name,
               NULL as witness_otype,
               NULL as witness_subtype,
               NULL as witness_timedst,
               NULL as witness_tz
    FROM event
    LEFT JOIN eventproperty ON eventproperty.eventid=event.eventid
    WHERE event.eventid=$1
    UNION
    SELECT NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
              eventwitness.objid,
              _objs.name,
              otype,
              subtype,
              _obj_attr.val,
              0
    FROM event, _objs, eventwitness, _obj_attr
    WHERE event.eventid=$1 AND eventwitness.eventid=event.eventid AND _objs.obj = eventwitness.objid
      AND _obj_attr.obj=eventwitness.objid AND _obj_attr.attr='TIME_ZONE'
    ORDER BY objid ASC
LOOP
RETURN NEXT result;
END LOOP;

RETURN;

END
$_$;


--
-- Name: el_get_event_for_merge(uuid, integer, character varying); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION el_get_event_for_merge(uuid, integer, character varying) RETURNS integer
    LANGUAGE plpgsql
    AS $_$
DECLARE
   obj_id  ALIAS FOR $1;
   event_id integer := NULL;
   external_id  ALIAS FOR $3;
BEGIN
   IF (external_id IS NULL OR length(external_id) = 0) THEN
      event_id := (SELECT MAX(eventid) FROM event WHERE objid=$1 AND eventsource=$2);
   ELSE
      event_id := (SELECT MAX(eventid) FROM event WHERE objid=$1 AND eventsource=$2 AND externalid=external_id);
   END IF;

   RETURN event_id;

END;
$_$;



--
-- Name: el_insert_event(uuid, integer, integer, integer, bigint, bigint, bigint, integer, integer, integer, character varying, character varying); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION el_insert_event(uuid, integer, integer, integer, bigint, bigint, bigint, integer, integer, character varying, character varying) RETURNS integer
    LANGUAGE plpgsql
    AS $_$
BEGIN

INSERT INTO event (objid, eventsource, workflow, state, utc_when, utc_from, utc_to, priority, eventtype, message, externalid)
VALUES ($1, $2, $3, $4,
        TIMESTAMP 'epoch' + $5 * INTERVAL '1 second',
        TIMESTAMP 'epoch' + $6 * INTERVAL '1 second',
        TIMESTAMP 'epoch' + $7 * INTERVAL '1 second',
        $8, $9, $10, $11);

RETURN (SELECT currval('transdb.seqeventid'));

END;
$_$;



--
-- Name: el_insert_property(integer, character varying, character varying); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION el_insert_property(integer, character varying, character varying) RETURNS void
    LANGUAGE plpgsql
    AS $_$
BEGIN

INSERT INTO eventproperty(eventid, name, value) VALUES($1, $2, $3);

END;
$_$;



--
-- Name: el_merge_witnesses(integer, uuid, uuid[]); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION el_merge_witnesses(integer, uuid, uuid[]) RETURNS SETOF el_object_info
    LANGUAGE plpgsql
    AS $_$
DECLARE
   event_id         ALIAS FOR $1;
   obj_id           ALIAS FOR $2;
   witnesses_       ALIAS FOR $3;
   witnesses        uuid[];
   merged_witnesses text[];
   result           EL_Object_Info;
BEGIN

-- postgres 8.1.x requires this
IF witnesses_ IS NULL THEN
   witnesses = ARRAY[obj_id];
ELSE
   witnesses = witnesses_;
END IF;

merged_witnesses := (SELECT COALESCE(
                              string_to_array(replace(val, ' ', ''), ','), -- just for case removing spaces between comas
                              CAST(ARRAY[obj_id] as text[]) -- if previous expression is NULL keep array of original obj_id.
                            ) || CAST(witnesses AS text[]) || CAST(ARRAY[obj_id] as text[]) -- adding objid of event to witnesses too
                    FROM _obj_attr
                    WHERE obj=obj_id AND attr='ASSOCIATE');

INSERT INTO eventwitness
(
   SELECT DISTINCT
      event_id,
      CAST (merged_witnesses[i] AS uuid)
   FROM generate_series(1, array_upper(merged_witnesses, 1)) as i
);

IF FOUND THEN

  FOR result in
      SELECT o.obj, o.name, o.type, o.otype, o.subtype, COALESCE(a1.val, 'UTC') as tz, 0, COALESCE(a2.val, '0') as vl_rate
        FROM eventwitness w
       INNER JOIN _objs o ON o.obj = w.objid
        LEFT OUTER JOIN _obj_attr a1 ON a1.obj=o.obj AND a1.attr='TIME_ZONE'
        LEFT OUTER JOIN _obj_attr a2 ON a2.obj=o.obj AND a2.attr='STAT_VL_RATE_LAST'
       WHERE w.eventid = event_id
  LOOP
     RETURN NEXT result;
  END LOOP;
END IF;

RETURN;

END
$_$;



--
-- Name: el_update_event(bigint, bigint, integer, integer, integer, character varying, character varying, integer); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION el_update_event(bigint, bigint, integer, integer, character varying, character varying, integer) RETURNS void
    LANGUAGE plpgsql
    AS $_$
BEGIN

UPDATE event SET
          utc_from = TIMESTAMP 'epoch' + $1 * INTERVAL '1 second',
          utc_to   = TIMESTAMP 'epoch' + $2 * INTERVAL '1 second',
          state    = $3,
          priority = $4,
          message  = $5,
          note     = $6
          WHERE eventid = $7;

END;
$_$;



--
-- Name: getcameradowntimesummary(GUID[], bigint, bigint, character varying); Type: FUNCTION; Schema: apl; Owner: postgres
--

CREATE OR REPLACE FUNCTION getcameradowntimesummary(objids varchar[], ts_from bigint, ts_to bigint, granularity character varying) RETURNS SETOF downtime_period
    LANGUAGE plpgsql
    AS $$
DECLARE
    result downtime_period;
    row camera_downtime%ROWTYPE;
    u_from bigint;
    u_to bigint;
    ol_from bigint;
    ol_to bigint;
    inter interval;
    ts_ol_from timestamp without time zone;
    ts_ol_to timestamp without time zone;
    ts_loop_from timestamp without time zone;
    ts_loop_to timestamp without time zone;
    ts_loop timestamp without time zone;
    ts_loop_ol_from timestamp without time zone;
    ts_loop_ol_to timestamp without time zone;
BEGIN
    result := (0, 0, 0, 0);
    inter := ('1 ' || granularity)::interval;
    FOR row IN
        SELECT * FROM camera_downtime
	WHERE objids @> array[objid] AND extract('epoch' from coalesce(utc_to,now() at time zone 'UTC')) > ts_from
	    AND extract('epoch' from utc_from) < ts_to
	ORDER BY utc_from ASC
    LOOP
	u_from := extract('epoch' from row.utc_from)::int;
	u_to := extract('epoch' from coalesce(row.utc_to,now() at time zone 'UTC'))::int;
	-- Find overlap of given interval and current one
	IF u_from < ts_from THEN ol_from := ts_from; ELSE ol_from := u_from; END IF;
	IF u_to > ts_to THEN ol_to := ts_to; ELSE ol_to := u_to; END IF;
	-- convert to timestamp
	ts_ol_from = timestamp 'epoch' + ol_from * interval '1 second';
	ts_ol_to = timestamp 'epoch' + ol_to * interval '1 second';
	ts_loop_from := date_trunc(granularity, timestamp 'epoch' + ol_from * interval '1 second');
	ts_loop_to := date_trunc(granularity, timestamp 'epoch' + ol_to * interval '1 second') + inter;

	-- inner loop
	ts_loop := ts_loop_from;
	WHILE ts_loop < ts_loop_to
	LOOP
	    IF (ts_loop, ts_loop + inter) overlaps (ts_ol_from, ts_ol_to)
	    THEN
		-- Find interval overlap and compute number of seconds in that overlap
		IF ts_loop < ts_ol_from
		THEN
		    ts_loop_ol_from := ts_ol_from;
		ELSE
		    ts_loop_ol_from := ts_loop;
		END IF;
		IF ts_loop + inter > ts_ol_to
		THEN
		    ts_loop_ol_to := ts_ol_to;
		ELSE
		    ts_loop_ol_to := ts_loop + inter;
		END IF;
		-- Found resulting interval, add it to function result
		result.objid := row.objid;
		result.ts_from := extract(epoch from ts_loop)::int;
		result.duration := extract(epoch from ts_loop_ol_to - ts_loop_ol_from)::int;
		result.status := row.status;
		RETURN NEXT result;
	    END IF;
	    ts_loop := ts_loop + inter;
	END LOOP;
	--RETURN NEXT result;
    END LOOP;
RETURN;
END
$$;



