SET search_path TO transdb;

-- 
-- Compute hour size (integer) as sum of chunk sizes (float)
--
CREATE OR REPLACE FUNCTION update_cm_stat_size() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
DECLARE
    hsize real;
    chunk_ts varchar;
    chunk_size real;
    selector text[];
BEGIN
    hsize := 0::real;
    FOR chunk_ts IN SELECT jsonb_object_keys(NEW.details)
    LOOP
        selector[1] := chunk_ts;
        selector[2] := 'size';
        chunk_size := (NEW.details #>> selector)::real;
        hsize = hsize + chunk_size;
    END LOOP;
    NEW.size := ceil(hsize);
    return NEW;
END;
$$;


CREATE TRIGGER trg_cm_stat_hour_upd_details BEFORE UPDATE ON cm_stat_hour FOR EACH ROW EXECUTE PROCEDURE update_cm_stat_size();

----------------------------------------------------------------------
-- CM usage
--
SET search_path TO apl;

CREATE OR REPLACE FUNCTION cm_usage_total() RETURNS jsonb
    LANGUAGE plpgsql
    AS $$
DECLARE
    total_usage integer := 0;
    pstat       record;
    pool_usage  jsonb   := '{}'::jsonb;
    result      jsonb   := '{}'::jsonb;
BEGIN
    FOR pstat IN SELECT sum(h.size) AS sz, a.val AS poolid 
                    FROM cm_stat_hour h INNER JOIN _obj_attr a ON h.objid=a.obj 
                    WHERE a.attr='STORAGE_POOL' GROUP BY poolid 
    LOOP
        total_usage := total_usage + pstat.sz;
        pool_usage := pool_usage || jsonb_build_object(pstat.poolid, pstat.sz);
    END LOOP;
    result := jsonb_build_object('totalUsage', total_usage, 'poolUsage', pool_usage);
    
    RETURN result;
END;
$$;


CREATE OR REPLACE FUNCTION cm_usage(objids uuid[], realmid uuid, stream integer, ts_from timestamp, ts_to timestamp) RETURNS jsonb
    LANGUAGE plpgsql
    AS $$
DECLARE
    total_usage integer := 0;
    pstat       record;
    objs        record;
    query       varchar;
    chunk_ts    varchar;
    cond1       varchar := '';
    cond2       varchar := '';
    cond3       varchar := '';
    selector    text[];
    result      jsonb   := '{}'::jsonb;
    pool_usage  jsonb   := '{}'::jsonb;
    hour_size   real;
    cur_size    int;
    chunk_len   int;
    ts_from_e   int := extract(epoch FROM ts_from)::int;
    ts_to_e     int := extract(epoch FROM ts_to)::int;
BEGIN
    
    IF objids IS NULL AND realmid IS NULL AND stream IS NULL AND ts_from IS NULL AND ts_to IS NULL THEN
        RAISE EXCEPTION 'All arguments cannot be empty';
    END IF;
    
    IF objids IS NOT NULL AND realmid IS NOT NULL THEN
        RAISE EXCEPTION 'Both realmid and objids cannot be provided';
    END IF;

    IF realmid IS NOT NULL THEN
        SELECT array(SELECT obj FROM _objs WHERE realm=realmid) AS objs INTO objs;
        objids := objs.objs;
    END IF;

    IF stream IS NOT NULL THEN
        cond1 := cond1 || ' AND stream=' || stream;
        cond2 := cond2 || ' AND stream=' || stream;
    END IF;
    IF objids IS NOT NULL AND array_length(objids, 1) > 0 THEN
        cond1 := cond1 || ' AND objid = ANY($1)';
        cond2 := cond2 || ' AND objid = ANY($1)';
    END IF;
    IF ts_from IS NOT NULL THEN
        IF ts_from = date_trunc('hour', ts_from) THEN
            cond1 := cond1 || ' AND hts >= $2';
        ELSE
            cond1 := cond1 || E' AND hts >= date_trunc(\'hour\',$2) + interval \'1 hour\'';
            cond3 := cond3 || E'hts = date_trunc(\'hour\',$2)';
        END IF;
    END IF;
    IF ts_to IS NOT NULL THEN
        IF ts_to = date_trunc('hour', ts_to) THEN
            cond1 := cond1 || ' AND hts < $3';
        ELSE
            cond1 := cond1 || E' AND hts < date_trunc(\'hour\',$3)';
            IF length(cond3) > 0 THEN
                cond3 := cond3 || ' OR ';
            END IF;
            cond3 := cond3 || E'hts = date_trunc(\'hour\',$3)';
        END IF;
    END IF;

    IF length(cond3) > 0 THEN
        cond3 := ' AND (' || cond3 || ')';
    END IF;

    query := 'SELECT sum(h.size) AS sz, a.val AS poolid ' || 
             'FROM cm_stat_hour h INNER JOIN _obj_attr a ON h.objid=a.obj ' ||
             E'WHERE a.attr=\'STORAGE_POOL\'' || cond1 || ' GROUP BY poolid';

    FOR pstat IN EXECUTE query USING objids, ts_from, ts_to LOOP
        total_usage := total_usage + pstat.sz;
        pool_usage := pool_usage || jsonb_build_object(pstat.poolid, pstat.sz);
    END LOOP;

    -- Do sencond query to count chunk sizes in border hours
    --
    IF length(cond3) > 0 THEN
        query := 'SELECT h.objid, h.stream, h.hts, h.details, a.val AS poolid ' || 
                 'FROM cm_stat_hour h INNER JOIN _obj_attr a ON h.objid=a.obj ' ||
                 E'WHERE a.attr=\'STORAGE_POOL\'' || cond2 || cond3;
        FOR pstat IN EXECUTE query USING objids, ts_from, ts_to LOOP
            hour_size := 0.0;

            FOR chunk_ts IN SELECT jsonb_object_keys(pstat.details)
            LOOP
                selector[1] := chunk_ts;
                selector[2] := 'len';
                chunk_len := (pstat.details #>> selector)::int;
                IF (chunk_ts::int >= ts_from_e) AND (chunk_ts::int + chunk_len <= ts_to_e) THEN
                    selector[1] := chunk_ts;
                    selector[2] := 'size';
                    hour_size := hour_size + (pstat.details #>> selector)::real;
                END IF;
            END LOOP;

            total_usage := total_usage + hour_size::int;
            IF pool_usage ? pstat.poolid THEN
                cur_size := (pool_usage ->> pstat.poolid)::int;
                cur_size := cur_size + hour_size::int;
                pool_usage := pool_usage || jsonb_build_object(pstat.poolid, cur_size);
            ELSE
                pool_usage := pool_usage || jsonb_build_object(pstat.poolid, hour_size::int);
            END IF;
        END LOOP;
    END IF;

    result := jsonb_build_object('totalUsage', total_usage, 'poolUsage', pool_usage);

    RETURN result;
END;
$$;
