SET search_path TO transdb;

-------------------------------------------------------------------
----- cm_jobs

CREATE TABLE cm_jobs (
    id            bigserial    NOT NULL PRIMARY KEY,
    objid         uuid  NOT NULL,
    chunk         varchar      NOT NULL,
    stream        smallint     NOT NULL,
    ts            timestamp without time zone NOT NULL,
    length        smallint     NOT NULL DEFAULT 30, 
    size          integer      NOT NULL,
    create_ts     timestamp without time zone default (now() at time zone 'UTC'),
    update_ts     timestamp without time zone,
    deferred_ts   timestamp without time zone,
    proc_details  jsonb
);


REVOKE ALL ON TABLE cm_jobs FROM PUBLIC;
GRANT SELECT,INSERT,UPDATE,DELETE ON TABLE cm_jobs TO apl;

-- Permissions on implicit sequence
GRANT SELECT,UPDATE ON SEQUENCE cm_jobs_id_seq TO apl;


-------------------------------------------------------------------
----- cm_stat_hour

CREATE TABLE cm_stat_hour (
    objid       uuid NOT NULL,
    stream      smallint NOT NULL,
    hts         timestamp without time zone NOT NULL,
    expires     timestamp without time zone,
    size        integer NOT NULL,
    details     jsonb NOT NULL
);


REVOKE ALL ON TABLE cm_stat_hour FROM PUBLIC;
GRANT SELECT,INSERT,UPDATE,DELETE ON TABLE cm_stat_hour TO apl;

-------------------------------------------------------------------
----- cm_bw_cache

CREATE TABLE cm_bw_cache (
    objid       uuid NOT NULL,
    ts          timestamp without time zone NOT NULL,
    bytes       bigint NOT NULL
);

REVOKE ALL ON TABLE cm_bw_cache FROM PUBLIC;
GRANT SELECT,INSERT,UPDATE,DELETE ON TABLE cm_bw_cache TO apl;


-------------------------------------------------------------------
----- cm_bw_stat

CREATE TABLE cm_bw_stat (
    objid       uuid NOT NULL,
    ts          timestamp without time zone NOT NULL,
    consumed    integer NOT NULL
);


REVOKE ALL ON TABLE cm_bw_stat FROM PUBLIC;
GRANT SELECT,INSERT,UPDATE,DELETE ON TABLE cm_bw_stat TO apl;

--------------------------------------------------------------------------
--  apl schema
--
SET search_path TO apl;

CREATE OR REPLACE FUNCTION date_trunc_min(ts timestamp, min integer) RETURNS timestamp AS $$
BEGIN
    RETURN date_trunc('hour',ts) + make_interval(mins => extract(minute from ts)::int / min * min);
END
$$ LANGUAGE 'plpgsql';

CREATE OR REPLACE FUNCTION cm_wipe_stat(obj uuid, ts_from timestamp, ts_to timestamp, chunk_len integer) RETURNS void AS $$
DECLARE
    hrec transdb.cm_stat_hour%ROWTYPE;
    ts_from_epoch integer;
    ts_to_epoch integer;
    chunk_ts varchar;
    details_filtered jsonb;
    hsize real;
    chunk_size real;
    selector text[];
BEGIN
    -- First delete records that fit into interval "completely" (without hour splitting)
    DELETE FROM transdb.cm_stat_hour WHERE objid=obj AND hts >= ts_from AND hts < date_trunc('hour',ts_to);
    -- Other ones should be filtered
    ts_from_epoch = extract(epoch FROM ts_from)::integer;
    ts_to_epoch = extract(epoch FROM ts_to)::integer;
    raise notice 'ts_from_epoch: %', ts_from_epoch;
    raise notice 'ts_to_epoch: %', ts_to_epoch;
    FOR hrec IN SELECT * FROM transdb.cm_stat_hour WHERE objid=obj AND hts >= date_trunc('hour',ts_from) AND hts <= ts_to
    LOOP
        raise notice 'Got hour record: obj=%; stream=%; size=%', hrec.objid, hrec.stream, hrec.size;
        details_filtered := hrec.details;
        hsize := hrec.size::real;
        FOR chunk_ts IN SELECT jsonb_object_keys(hrec.details)
        LOOP
            IF (chunk_ts::integer >= ts_from_epoch) AND (chunk_ts::integer + chunk_len <= ts_to_epoch) THEN
            -- drop these chunks
                raise notice 'Drop chunks %', chunk_ts;
                details_filtered := details_filtered - chunk_ts;
                selector[1] := chunk_ts;
                selector[2] := 'size';
                chunk_size := (hrec.details #>> selector)::real;
                hsize = hsize - chunk_size;
            END IF;
            UPDATE transdb.cm_stat_hour SET details=details_filtered, size=hsize::integer WHERE objid=hrec.objid AND hts=hrec.hts AND stream=hrec.stream;
        END LOOP;
    END LOOP;
END
$$ LANGUAGE 'plpgsql';


CREATE OR REPLACE VIEW cm_jobs AS SELECT * FROM transdb.cm_jobs;
REVOKE ALL ON TABLE cm_jobs FROM PUBLIC;
GRANT SELECT,INSERT,UPDATE,DELETE ON TABLE cm_jobs TO apl;


CREATE OR REPLACE VIEW cm_stat_hour AS SELECT * FROM transdb.cm_stat_hour;
REVOKE ALL ON TABLE cm_stat_hour FROM PUBLIC;
GRANT SELECT,INSERT,UPDATE,DELETE ON TABLE cm_stat_hour TO apl;


CREATE OR REPLACE VIEW cm_bw_cache AS SELECT * FROM transdb.cm_bw_cache;
REVOKE ALL ON TABLE cm_bw_cache FROM PUBLIC;
GRANT SELECT,INSERT,UPDATE,DELETE ON TABLE cm_bw_cache TO apl;


CREATE OR REPLACE VIEW cm_bw_stat AS SELECT * FROM transdb.cm_bw_stat;
REVOKE ALL ON TABLE cm_bw_stat FROM PUBLIC;
GRANT SELECT,INSERT,UPDATE,DELETE ON TABLE cm_bw_stat TO apl;
