package Overcast::Session;

use 5.008008;
use strict;
use warnings;
use JSON;
use FileHandle;
use IPC::Open2;
use Digest;
use LWP::UserAgent;

require Exporter;

our @ISA = qw(Exporter);

our %EXPORT_TAGS = ( 'all' => [ qw(
    CheckSession CheckSessionExt OpenSession CloseSession LastCode
) ] );

our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

our @EXPORT = qw(
    CheckSession CheckSessionExt OpenSession CloseSession LastCode
);

our $VERSION = '1.00';

our $SESSION_COOKIE_NAME = "PHPSESSID";
our $TOKEN_COOKIE_NAME = "token";

# Vars ---------------------------------------------------------------
our $Proto = 'https'; # http/https
our $SID = '';
our $Token = '';
our $Host = '';
our $Err = '';
our $LastRsp = '';
our $Redis;

# Init and tune UA
our $UA = LWP::UserAgent->new(ssl_opts => { verify_hostname => 0 });
$UA->cookie_jar({});
$UA->timeout(10);
$UA->agent("perl/Session.pm");

# Preloaded methods go here.

sub _sha512hex {
    my $src = shift;
    
    local $SIG{PIPE} = 'IGNORE';
    my $pid = open2(*Reader, *Writer, "/usr/bin/sha512sum --text");
    print Writer $src;
    close Writer;
    my $digest = <Reader>;
    chomp $digest if defined $digest;
    close Reader;
    if (defined $digest) {
	$digest = (split(/\s+/, $digest))[0];
    }
    return $digest;
}

sub _getCookies {
    my %cookies;
    $UA->cookie_jar()->scan( sub {
	$cookies{$_[1]} = $_[2];
    } );
    return \%cookies;
}

sub _getCookie {
    my $name = shift;
    my $cookies = _getCookies;
    return $cookies->{$name};
}

sub OpenSession {

    $Host = shift;
    my ($username, $password, $proto) = @_;
    $Proto = $proto if $proto and $proto =~ /^https?$/i;
    $Err = '';
    $SID = '';

    if (! $Host) {
	$Err = "Remote address must be specified!";
	return;
    }

    eval {
        my $encKey;
        my $resp = $UA->get("$Proto://$Host/api/call/getLoginInfo");
        my $json = _parseJSON($resp);
        $encKey = $json->{loginInfo}{encryptionKey};
        my $cookies = _getCookies;
        $Token = $cookies->{$TOKEN_COOKIE_NAME};

        # Compute SHA-512 digest
        my $tmp = $encKey._sha512hex($password).$encKey;
        my $digest = _sha512hex($tmp);

        # logging in..
        $resp = $UA->post(
        "$Proto://$Host/api/call/login",
        {
            credentials => $digest,
            name        => $username,
            token       => $Token
        }
        );
        $json = _parseJSON($resp);
        die("Authrorization error\n#",$resp->content) if $json->{error};
    };
    if ($@) {
        chomp $@;
        $Err = $@;
        return;
    }
    # Success
    my $cookies = _getCookies;
    $SID = $cookies->{$SESSION_COOKIE_NAME};
    $Token = $cookies->{$TOKEN_COOKIE_NAME};
    # Clear cookies
    #$ua->cookie_jar()->clear;
    
    if (! $SID) {
        $Err = "No SessionId in response";
        return;
    }

    return wantarray ? ($SID, $Token) : $SID;
}

sub _parseJSON {
    my $httpResp = shift;
    die("HTTP ERROR [".$httpResp->code."]") if $httpResp->is_error;

    my $json = eval { decode_json($httpResp->content) };
    $LastRsp = $json;
    die("Bad JSON returned\n#".$httpResp->content) if $@;
    die("Request failed\n#".$httpResp->content) if $json->{error};
    
    return $json;
}

sub _setCookie {
    my ($name, $val) = @_;
    return if not defined $name or not defined $val;
    $UA->default_headers(HTTP::Headers->new('Cookie'=>$name."=".$val));
}

sub _setSessionCookie {
    return unless $_[0];
    _setCookie($SESSION_COOKIE_NAME, $_[0]);
}

sub CloseSession {
    my $sid = shift;

    if (! $Host) {
	$Err = "Host should be specified!";
	return;
    }
    _setSessionCookie($sid || $SID);
    my $url = "$Proto://$Host/api/call/logout";
    $url .= "&token=$Token" if $Token;
    my $resp = $UA->get($url);
    eval {
	my $json = _parseJSON($resp);
    };
    if ($@) {
	chomp $@;
	$Err = $@;
	return;
    }
    $SID = '' if not $sid or ($sid and $sid eq $SID);
    return 1;
}

sub LastCode {
    return '' if not $LastRsp or ref $LastRsp ne 'HASH';
    return $LastRsp->{code};
}

sub CheckSession {
    my $session = shift;
    my $xsession = shift;
    
    if (not grep { $_ eq  'Overcast/Redis.pm' } keys %INC) {
        require Overcast::Redis;
        Overcast::Redis->import();
    }
    
    # Check if X-Session header provided
    if ($xsession and $xsession eq $ENV{OVERCAST_LOGIN_HEADER}) {
        return 1;
    }
    return 0 if not defined $session;
    
    $Redis = RedisConnect() if not $Redis or not $Redis->ping;
    my $key = "SESSION_${session}";
    my $sessData = $Redis->get($key);
    return 0 if not $sessData;
    my $sess = eval { decode_json($sessData) };
    return 0 if $@;
    return 0 if not $sess or ref($sess) ne 'HASH';
    my $status = $sess->{status};
    return 0 if not $status;
    if ($status eq 'LoggedIn') {
        return wantarray ? (1, $sess) : 1;
    }
}

sub CheckSessionExt {
    my $req = shift; # Mojo::Message::Request

    my $cookie = $req->headers->header('Cookie');
    my $xsession = $req->headers->header('X-session');
    return (0, "Unauthorized", 401) if not $cookie and not $xsession;
    
    my $sessid = $1 if defined $cookie and $cookie =~ /PHPSESSID=(\w+?)\b/;
    my ($ok, $sess) = CheckSession($sessid, $xsession);
    return(0, "Unauthorized", 401) if not $ok;
    if (not $sessid and $xsession) { # Internal call, e.g. health check
        my $xrealmid = $req->headers->header('X-realmid');
        my $xroleid  = $req->headers->header('X-roleid');
        $req->{realm_id} = lc $xrealmid if $xrealmid;
        $req->{role_id} = lc $xroleid if $xroleid;
        $req->{intcall} = 1;
    } else {
        return(0, "Cannot determine client's realm", 400) if not $sess->{realm};
        $req->{realm_id} = lc $sess->{realm};
        $req->{role_id} = lc $sess->{roleid} if defined $sess->{role_id};
    }
    
    return (1, "", 200);
}

1;
