define("motion", ["jquery", "jquery-ui", "jquery-ui-timepicker-addon", "log", "api", "jquery.mask", "jquery.databind", "jquery.tools"], function(){
"use strict";

/**
 * @class motion Incapsulates all logic related with editing motion configuration
 * This functional closely connected with motion schema (motion/etc/motion_schema.json)
 */
var motion = {

	loadAttempts: 0,
	defaultObjSize: 6553, // line size of newly created zone and line
	maxSize: 60000, // max value that x,y coordinates can have
	objid: null,  // id of camera being edited
	props: null,
	cameraAttributes: {}, // Attributes of current camera

	/**
	 *
	 * @param {Object} props array of init properties: objid and trackers on/off
	 */
	init: function(props) {

		this.objid = props['objid'];
		this.props = props;
		var self = this;

		var api = new API();
		$.when(
			api.getAttributes({obj: this.objid}),
			this.playerLoaded()
		)
			.done(function(res_attributes) {
				try {
					self.cameraAttributes = res_attributes[0].list;
				} catch (e) {
					self.retryInit(props);
				}

				self.initPlayer();
				self.initUI();
				self.addEventHandlers();
				self.checkEncoder();
			})
			.fail(function() {
				this.retryInit(props);
			});
	},

	checkEncoder: function() {
		if (this.cameraAttributes.ENCODER_SETTING_OVERRIDE === 'no') {
			alert('Currently, the option "encoder programming" is disabled, so all changes made to the configuration will not be applied. To apply changes after save, you must restart the encoder by clicking "Reset request" button in general tab');
		}
	},

	retryInit: function(props) {
		this.loadAttempts += 1;

		if (this.loadAttempts > 10) {
			Log.error('Failed to load motion configuration!');
			return;
		}
		Log.warning('Failed to load motion configuration, retry in 2 sec');
		setTimeout(function() {motion.init(props);}, 2000);
	},

	/**
	 * @return {Object} jQuery deferred object, it resolves when player plugin fully loaded
 	 */
	playerLoaded: function(d) {
		if (!d) {
			d = new $.Deferred();
		}

		var iframe_player = $('#motionPlayer');

		if (!iframe_player.size() || !$('#motionPlayer')[0].contentWindow || !$('#motionPlayer')[0].contentWindow.document.getElementById('_applet_player_livePlayer')) {
			setTimeout(function() {motion.playerLoaded(d)}, 100);
		} else {
			d.resolve();
		}

		return d.promise();
	},

	/**
	 * Wait until player loads, then init zone configurator and calibrators
	 */
	initPlayer: function() {
		this.player = null;
		var iframe_player = $('#motionPlayer');

		if (!iframe_player.size() || !$('#motionPlayer')[0].contentWindow || !$('#motionPlayer')[0].contentWindow.document.getElementById('_applet_player_livePlayer')) {
			var self = this;
			setTimeout(function() {self.initPlayer();}, 500);
			return;
		}

		this.player = $('#motionPlayer')[0].contentWindow.document.getElementById('_applet_player_livePlayer');
		this.playerJS = $('#motionPlayer')[0].contentWindow.window._mplayer_livePlayer;

		this.player.initMDCalibrator();
		this.player.mdcalibrator.horizontalCells = this.cameraAttributes.VAE_MOTION_CELLS_H;
		this.player.mdcalibrator.verticalCells = this.cameraAttributes.VAE_MOTION_CELLS_V;
		this.player.mdcalibrator.activeMap = this.cameraAttributes.VAE_MOTION_ZONEMAP;
		this.player.mdcalibrator.pixelNoiseLevel = this.cameraAttributes.VAE_MOTION_PIXELNOISE;
		this.player.mdcalibrator.cellNoiseLevel = this.cameraAttributes.VAE_MOTION_CELLNOISE;
		this.player.mdcalibrator.minimumObjectSize = this.cameraAttributes.VAE_MOTION_ZONENOISELEVEL;
		this.player.mdcalibrator.analysisFPS = this.cameraAttributes.VAE_MOTION_ANALYSIS_INTENSITY;
		this.player.setMode(this.player.mdcalibrator);
	},

	checkArcDate: function(attr) {
		if (typeof(motion.playerJS) == 'undefined' || motion.player == null || motion.playerJS == null) {
			setTimeout(function() {motion.checkArcDate();}, 500);
		}else if (motion.playerJS.getCurrentTS() == 0) {
			setTimeout(function() {motion.checkArcDate();}, 500);
		}else{
			var plaerStartTS = motion.playerJS.getCurrentTS();
			var datefrom = $('#calendar').datetimepicker('getDate').getTime();
			if(Math.abs(plaerStartTS-datefrom)>2000){
				motion.player.player.stop();
				// TODO: check if message is needed here
				motion.player.displayMessage(__("No archive"));
				motion.player.displaySign(2);
				Log.warning('Where is no data on selected time in archive. The player was stoped.');
			}
		}
	},

	/**
	 * Add callbacks to all needed UI events
	 */
	addEventHandlers: function() {

		$('#VAE_MOTION_PIXELNOISE').change(function() {
			motion.player.mdcalibrator.pixelNoiseLevel = $('#VAE_MOTION_PIXELNOISE').val();
		});
		$('#VAE_MOTION_CELLNOISE').change(function() {
			motion.player.mdcalibrator.cellNoiseLevel = $('#VAE_MOTION_CELLNOISE').val();
		});
		$('#VAE_MOTION_ZONENOISELEVEL').change(function() {
			motion.player.mdcalibrator.minimumObjectSize = $('#VAE_MOTION_ZONENOISELEVEL').val();
		});
		$('.buttons button.apply').click(function() {motion.save()});

		$('#button_archive').click(function() {
			//get UTC TS
			var datefrom = $('#calendar').datetimepicker('getDate');
			var dateto = new Date();
			function add_zero (val){
				if(val<10){
					val = "0" + val;
				}
				return val;
			}
			var str_datefrom = "&ts_from=" + datefrom.getUTCFullYear() + "-" + add_zero (datefrom.getUTCMonth() + 1) + "-" + add_zero (datefrom.getUTCDate()) + " " + add_zero (datefrom.getUTCHours()) + "-" + add_zero (datefrom.getUTCMinutes()) + "-" + add_zero (datefrom.getUTCSeconds());
			var str_dateto = "&ts_to=" + dateto.getUTCFullYear() + "-" + add_zero (dateto.getUTCMonth() + 1) + "-" + add_zero (dateto.getUTCDate()) + " " + add_zero (dateto.getUTCHours()) + "-" + add_zero (dateto.getUTCMinutes()) + "-" + add_zero (dateto.getUTCSeconds());

			var str_html = '<iframe width="700" height="500" scrolling="no" id="motionPlayer" src="/TibcoGI/JSXAPPS/mx/cell_player.php?name=livePlayer&camera='+motion.objid+'&width=700&height=500'+str_datefrom+str_dateto+'"></iframe><br />';
			$('#player').html(str_html);
			motion.initPlayer();
			motion.checkArcDate();
		});

		$('#button_live').click(function() {
			$('#player').html('<iframe width="700" height="500" scrolling="no" id="motionPlayer" src="/TibcoGI/JSXAPPS/mx/cell_player.php?name=livePlayer&camera='+motion.objid+'&width=700&height=500"></iframe><br />');
			motion.initPlayer();
		});

		$("#tabs_menu").tabs(".tabs_form > div", {
			onClick: function(event, tabIndex) {
				if (!motion.player)
					return;
			}
		});
	},

	/**
	 * Init UI accordingly to available trackers, features and motion schema
	 */
	initUI: function() {
		$('#VAE_MOTION_PIXELNOISE').val(this.cameraAttributes.VAE_MOTION_PIXELNOISE);
		$('#VAE_MOTION_CELLNOISE').val(this.cameraAttributes.VAE_MOTION_CELLNOISE);
		$('#VAE_MOTION_ZONENOISELEVEL').val(this.cameraAttributes.VAE_MOTION_ZONENOISELEVEL);

		$('#VAE_MOTION_PIXELNOISE').rangeinput({css: {input: 'range-input', slider: 'range-slider', progress: 'range-progress', handle: 'range-handle'}});
		$('#VAE_MOTION_CELLNOISE').rangeinput({css: {input: 'range-input', slider: 'range-slider', progress: 'range-progress', handle: 'range-handle'}});
		$('#VAE_MOTION_ZONENOISELEVEL').rangeinput({css: {input: 'range-input', slider: 'range-slider', progress: 'range-progress', handle: 'range-handle'}});
		$('#calendar').datetimepicker();
		var ts = new Date().getTime();
		ts = ts - (ts % 60000);
		$('#calendar').datetimepicker('setDate', (new Date(ts)) );
		motion.setDatepickerRange();
	},

	setDatepickerRange: function() {
		var objList = [];
		objList.push(this.objid);
		var objid = this.objid;
		$( "#calendar" ).datepicker( "option", "maxDate", new Date() );
		var api = new API();
		api.getStorageUsageSpread({
			objList: JSON.stringify(objList)
		})
			.done(function(response){
				if(typeof(response.list[objid][0]) != 'undefined'){
					$( "#calendar" ).datepicker( "option", "minDate", new Date(response.list[objid][0].timestamp * 1000) );
				}
			});
	},

	/**
	 * Fill zone editor fields with data of currently selected zone in configurator,
	 * provide datalinking between them
	 * @param objectIndex index of selected zone
	 */
	initDialog: function(objectIndex) {
		var dialog = $('#zone_editor');

		if (this.dataObj) {
			try {
				dialog.unlink(this.dataObj);
			} catch(e) {}
		}

		this.dataObj = this.player.configurator.objects[objectIndex];
		dialog.link(this.dataObj, {
			'jaoselect_trk_objs': null,  // exclude classificator selector checkboxes
			'jaoselect_postures': null
		});

		// Dialog view depending on object style: polygon, line, multiline
		$('#zone_rules, #zone_rules > li, #zone_object_filter').show();
		$('#detection_type').show();
		if (this.dataObj.style === 'polygon') {
			dialog.find('#rule_linecounter_a, #rule_linecounter_b, #rule_cross, #rule_cross2, #rule_order').hide();

			if ($('#rule_presence').size()) {
				var html = $('#rule_presence').html().replace('Cross (any direction)', 'Inside');
				$('#rule_presence').html(html);
			}
		}
		if (this.dataObj.style === 'line' || this.dataObj.style === 'multiline') {
			dialog.find('#detection_type').hide();
			if (!$('#pane_channel input:checkbox[name="enablecntline"]').prop("checked")) {
				$('#rule_linecounter_a, #rule_linecounter_b').hide();
			}
			$('#rule_enter, #rule_exit, #rule_appear, #rule_disappear, #rule_stop, #rule_dwell, #rule_loiters, #rule_abobj, #rule_rmobj').hide();

			if (this.dataObj.style === 'line') {
				$('#rule_cross2, #rule_order').hide();
			}

			if (this.dataObj.style === 'multiline') {
				$('#rule_cross').hide();
			}

			if ($('#rule_presence').size()) {
				// Change "inside rule" on "cross any direction" for VCA
				html = $('#rule_presence').html().replace('Inside', 'Cross (any direction)');
				$('#rule_presence').html(html);
			}
		}

		// Set correct values of form fields
		for (var i in this.dataObj) {

			if ((typeof this.dataObj[i] === 'function' || $.isPlainObject(this.dataObj[i]) || typeof this.dataObj[i] === 'undefined') && i != 'trk_objs')
				continue;

			$(this.dataObj).triggerHandler( "changeField!", [i, this.dataObj[i]] );
			var field = dialog.find('[name="' + i + '"]');
			if (field.hasClass('range-input')) {
				field.data("rangeinput").setValue(this.dataObj[i]);
			}
			field.change();
		}
	},

	/**
	 * Close motion iframe and return to admin GUI (works only if page was opened from motion tab in admin GUI)
	 */
	close: function() {
	},

	handleRestart: function() {
	},

	/**
	 * Serialize all configuration data into json and save it to server
	 */
	save: function() {
		var self = this;

		// Try..catch block to catch user input errors
		try {
			$('input.error').removeClass('error');
			var VAE_MOTION_CELLS_H_val = this.player.mdcalibrator.horizontalCells;
			var VAE_MOTION_CELLS_V_val = this.player.mdcalibrator.verticalCells;
			var VAE_MOTION_ZONEMAP_val = this.player.mdcalibrator.activeMap;
			var VAE_MOTION_PIXELNOISE_val = this.player.mdcalibrator.pixelNoiseLevel;
			var VAE_MOTION_CELLNOISE_val = this.player.mdcalibrator.cellNoiseLevel;
			var VAE_MOTION_ZONENOISELEVEL_val = this.player.mdcalibrator.minimumObjectSize;
			var VAE_MOTION_ANALYSIS_INTENSITY_val = this.player.mdcalibrator.analysisFPS;

			var attributes = {
				VAE_MOTION_CELLS_H:        VAE_MOTION_CELLS_H_val,
				VAE_MOTION_CELLS_V:        VAE_MOTION_CELLS_V_val,
				VAE_MOTION_ZONEMAP:        VAE_MOTION_ZONEMAP_val,
				VAE_MOTION_PIXELNOISE:     VAE_MOTION_PIXELNOISE_val,
				VAE_MOTION_CELLNOISE:      VAE_MOTION_CELLNOISE_val,
				VAE_MOTION_ZONENOISELEVEL: VAE_MOTION_ZONENOISELEVEL_val
			};

			var api = new API();
			api.setAttributes({
				obj: this.objid,
				attributes: JSON.stringify(attributes)
			})
				.done(function() {
					Log.info('Configuration saved succesfully!');
					self.handleRestart();
				});

		} catch(e) {
			if (e.name != 'inputError')
				throw e;

			if (typeof e.prepare === 'function') {
				e.prepare();
			}

			// Get index of pane which contains error element
			var pane = $(e.field).parents('#panes > div');
			var paneIndex = $('#panes > div').index(pane);

			// Open pane, focus index field and add error class to it
			$('#tabs_menu > div').eq(paneIndex).click();
			$(e.field).addClass('error').focus();
			Log.error(e.message);
		}
	}
};

return motion;

});
