#!/usr/bin/perl
#
#  Daemon program for Axis servers and cameras (bios after 2.10 only) relay and input control.
#  PLEASE UPGRADE AXIS SOFTWARE IF IT'S EARLIE THEN 2.10 !!!
#
#  Environment:
#	$APL       - /opt/sarch - software root directory
#	$APL_CONF  - $APL/var/conf
#
#  1. Reads devices' configs at $APL_CONF/<devicename>/conf for all AXIS virtual devices
#  2. Creates and regularly updates $APL_CONF/<devicename>/pid all virtual devices
#  3. Reads commands for relays from $APL/var/xio/axis
#  4. Writes virtual devices' logs at $APL/store/<devicename>/<YYMMDD>/<HH>.log
#  5. Updates virtual devices' alert files (by current state of port) at $APL_CONF/<devicename>/alert
#  6. Writes daemon log to $APL/var/log/xio/axis.log
#  7. Regularly check for connectivity and raise an alert flag $APL/var/alert/xio/axis
#  8. Support of notifications
#
#  Sample 'conf'-file:
#
#  DEVID=r1                     - device ID
#  HW_MODEL=AXIS                - device handling hardware, AXIS only for this daemon
#  USRNAME=root                 - username if AXIS server is username and password-protected
#  PASSWD=pass                  - password if AXIS server is password-protected
#  HW_PORT=1                    - number (1-4 for input, 1 for output)
#  HW_PORT_TYPE=OUT             - port type (IN/OUT)
#  RELAY_PULSE_TIME=1000        - pulse time in milliseconds for relays with auto-shutoff, 0 for steady signal
#  IP=1.2.3.4 or a1.company.net - AXIS server IP or URL
#  ACTIONRELAYS=r1,r9           - relays to reset to 0 in the case of action detected
#  ACTIONCMD=/opt/sarch/bin/ref - execute a command in the case of action detected
#  MOTIONEMAIL=email@site.net   - email for mailing
#  MOTIONEMAILST=e-mail once    - email srategy
#                every event
#                once in 15 min
#                once in 30 min
#                once in a hour

use strict;
use IO::Socket::INET;
use LWP::UserAgent;
use HTTP::Request::Common;
use Time::Local;
use FileHandle;
use File::Copy;
use IO::File;
use IO::Select;
use NextCAM::Init;
use Thread;

use lib "/opt/sarch/common/lib";
use lib "/opt/sarch/fw/bin/perl/FwIface";
use lib "/opt/sarch/fw/bin/perl";
use lib "/opt/sarch/elog/bin/perl";
use lib "/opt/sarch/elog/bin/perl/ELogIface";

use NextCAM::Servicing::Client;
use FwIface::Types;
use ELog;

my $DEBUG = 2;	# use 1 for errors output only and 2 for both debug and errors
my ($pname)=reverse(split(/\//,$0));
my ($usrpsw, $curstate, $port)=('', 0, 8510);

TermPrevious($pname);


my $service_client = NextCAM::Servicing::Client->new("s_master", 10000);
my $elog_client = ELogClient->new($service_client->getProtocol("skm.eventlog"));

my $pipe = new IO::File "+<$ENV{APL}/var/xio/axis";
die "Can not open $ENV{APL}/var/xio/axis\n$!\n" if not defined $pipe;
my $handles = new IO::Select();
$handles->add($pipe);

WriteDaemonLog('AXIS daemon started...');

my (%conf, %IP, @devs_polling);

ReadConfigs();
RecheckEachCameraNotifications();

sub ReleaseAndExit {
    debug(2, "Release and exit");
    WriteDaemonLog('AXIS daemon stoped...');
    Release();
    exit(1);
}
$SIG{TERM} = \&ReleaseAndExit;

# -------------------------- reset alert flags ---------------------------------
foreach my $dev (keys %conf) { CreateAlert($dev,0); }

# ------------------ Listen for notifications in the separate thread -----------
my $t = Thread->new( sub { listenSocket($port, %conf) } );



INFINITE: while(1) { # ---------------------- main cycle -----------------------
  my ($answ, $err);
  # -------------------------- update PID for all devices ----------------------
  foreach my $dev (keys %conf) {
    open(PID,"> $ENV{APL_CONF}/$dev/pid"); print PID $$; close PID;
    if($conf{$dev}{PULSE} and $conf{$dev}{PULSE} < time) {
      $conf{$dev}{PULSE}=0;
      CreateAlert($dev,0);
    }
  }
  # -------------------------- check sensors, polling --------------------------
  foreach my $dev (@devs_polling) {
    next if $conf{$dev}{HW_PORT_TYPE} ne 'IN';
	
    $answ = `$ENV{APL}/xio/bin/axis_sen_ctl.pl devid=$dev $conf{$dev}{HW_PORT}`;
    if($err = isErrorAnswer($answ)) {
	onLostCommunication ($dev, $conf{$dev}{IP});
	$curstate = 0;
    }
    else { # restored
      $curstate = $1 if $answ =~ /.+=([01])/;
      onRestoreCommunication($dev, $conf{$dev}{IP});
    }
    if($curstate != $conf{$dev}{LASTSTATE}) {
      CreateAlert($dev,$curstate);
      WriteLog($dev, $curstate, $conf{$dev}{LASTSTATE},"SENSOR");
      $conf{$dev}{LASTSTATE}=$curstate;
      $conf{$dev}{EVENT}=1;
      appendEventlog($conf{$dev}{ASSOCIATE}, "[$dev]: port #$conf{$dev}{HW_PORT} triggered to $curstate");
      `$conf{$dev}{ACTIONCMD}` if($conf{$dev}{ACTIONCMD});
      foreach (split(/,/,$conf{$dev}{ACTIONRELAYS})) { `echo $_=0 > $ENV{APL}/var/dev/$_/pipe`; }
    }
  }

  # ---------------------- check and handle pipe commands ----------------------
  my ($ready) = IO::Select->select($handles,undef,undef,2);
  if ($ready) {
    my $line = $pipe->getline;
    
    if($line =~ /^HUP\s/) {
        eval {
	    local $SIG{ALRM}=sub{die 'ALARM'};
	    alarm 10;
	    debug(2, "HUP signal received!");
	    Release();
	    ReadConfigs();
	    RecheckEachCameraNotifications();
	};
	alarm 0;
	next INFINITE;
    }
    
    $line =~ /(\w+)=(\d)\s+(.*)/;
    if($conf{$1} and (($2==0) or ($2==1))) {
      my $cmd= $conf{$1}{RELAY_PULSE_TIME} ?
                ( $2 ? "/$conf{$1}{RELAY_PULSE_TIME}\\" : '\\' )
              : ( $2 ? '/' : '\\' );
      $answ = `$ENV{APL}/xio/bin/axis_sen_ctl.pl devid=$1 $conf{$1}{HW_PORT}:$cmd`;
      if($err = isErrorAnswer($answ)) {
	onLostCommunication ($1, $conf{$1}{IP});
      }
      else { # restored
        onRestoreCommunication($1, $conf{$1}{IP});
      }
      CreateAlert($1,$2);
      $conf{$1}{PULSE}=time+int($conf{$1}{RELAY_PULSE_TIME}/1000) if $conf{$1}{RELAY_PULSE_TIME};
      WriteLog($1, $2, $conf{$1}{LASTSTATE},$3);
      $conf{$1}{LASTSTATE}=$conf{$1}{RELAY_PULSE_TIME} ? 0 : $2;
      $conf{$1}{EVENT}=1;
      `$conf{$1}{ACTIONCMD}` if($conf{$1}{ACTIONCMD});
      foreach (split(/,/,$conf{$1}{ACTIONRELAYS})) { `echo $_=0 > $ENV{APL}/var/dev/$_/pipe`; }
     }
  }
  
  # ------------------------ send email notifications --------------------------
  foreach my $dev (keys %conf) {
	SendMailNotify($dev);
  }
}
return 0;

# -------------------------------- CreateAlert ---------------------------------
sub CreateAlert : locked {
  my ($_dev,$_state) = @_;
  open(_ALERT,"> $ENV{APL_CONF}/$_dev/alert");
  print _ALERT '1' if $_state;
  close _ALERT;
} # sub CreateAlert

# ------------------------------ SendMailNotify --------------------------------
sub SendMailNotify {
    my ($dev) = @_;
    my $strategy;
    if($conf{$dev}{MOTIONEMAILST}=~/once\sin\sa\shour/) { $strategy=60; }
    elsif($conf{$dev}{MOTIONEMAILST}=~/once\sin\s(\d+)\smin/) { $strategy=$1; }
    elsif($conf{$dev}{MOTIONEMAILST}=~/e-mail\sonce/) { $strategy=1; }
    elsif($conf{$dev}{MOTIONEMAILST}=~/every\sevent/) { $strategy=0; }
    else { $strategy=-1; }
    #
    if($conf{$dev}{MOTIONEMAILST}=~/once\sin/) {
      if((time-$conf{$dev}{EVENTTIME})/60 > $strategy ) {
      return if not $conf{$dev}{EVENT};
      my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = gmtime($conf{$dev}{EVENTTIME}); $year %= 100; $mon++;
      SendMail($dev,$strategy,sprintf("%02d%02d%02d%02d%02d%02d",$year,$mon,$mday,$hour,$min,$sec));
      $conf{$dev}{EVENTTIME}=time;
      $conf{$dev}{EVENT}=0;
      }
    }
    elsif($conf{$dev}{MOTIONEMAILST}=~/e-mail\sonce/ or $conf{$dev}{MOTIONEMAILST}=~/every\sevent/) {
      return if not $conf{$dev}{EVENT};
      $conf{$dev}{MOTIONEMAILST}='' if $conf{$dev}{MOTIONEMAILST}=~/e-mail\sonce/;
      $conf{$dev}{EVENT}=0;
      my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = gmtime(time); $year %= 100; $mon++;
      SendMail($dev,$strategy,sprintf("%02d%02d%02d%02d%02d%02d",$year,$mon,$mday,$hour,$min,$sec));
    }
}

# ---------------------------------- SendMail ----------------------------------
sub SendMail
{
  my ($_devid,$_strategy,$_time)=@_;
  $_strategy=sprintf("%d",$_strategy);
  `perl $ENV{APL}/xio/bin/sendemail.pl $_devid $_strategy $_time`;
} # sub SendMail

# ------------------------------- WriteDaemonLog -------------------------------
sub WriteDaemonLog : locked {
  my ($_msg) = @_;
  my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = gmtime(time); $year %= 100; $mon++;
  open(LOG,">> $ENV{APL}/var/log/xio/axis.log");
  print LOG sprintf("%02d%02d%02d%02d%02d%02d %s\n",$year,$mon,$mday,$hour,$min,$sec,$_msg);
  close LOG;
} # sub WriteDaemonLog

# ---------------------------------- WriteLog ----------------------------------
sub WriteLog : locked {
  my ($_dev, $_curstate, $_laststate,$_userid)=@_;
  my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = gmtime(time); $year %= 100; $mon++;
  mkdir sprintf("$ENV{APL}/store/$_dev/%02d%02d%02d",$year,$mon,$mday);
  open(LOG,sprintf(">> $ENV{APL}/store/$_dev/%02d%02d%02d/%02d.log",$year,$mon,$mday,$hour));
  print LOG sprintf("%02d%02d%02d%02d%02d%02d  %d  %d  %s\n",$year,$mon,$mday,$hour,$min,$sec,$_curstate,$_laststate,$_userid);
  close LOG;
} # sub WriteLog

sub isErrorAnswer {
    my ($str) = @_;
    if(($str =~ /Error:(.+)/) or
       ($str =~ /\b(Bad\s+Request)/) or
       ($str =~ /(^HTTP ERROR.+)/) or
       ($str =~ /(Request failed:.+)/)) {
	    return $1;
    }

    return '';
}

#sub parseArray {
#    my ($regexp, @arr) = @_;
#    foreach (@arr) {
#        return ($1, $2, $3, $4, $5, $6, $7, $8, $9) if (/$regexp/);
#    }
#    return ();
#}

sub TermPrevious {
  my ($prog) = @_;
  local $SIG{TERM}='IGNORE';
  my @pids = `ps ax|grep $prog|cut -c 1-6`;
  chomp(@pids);
  debug(2, "Processes to terminate: @pids");
  `killall $prog 2>/dev/null`;
  # wait while previous program terminated
  while(@pids != 1) {
    my $id = shift(@pids);
    if  (kill(0,$id)) {
	push(@pids, $id);
	debug(2, "Task $id is alive");
    } else {
	debug(2, "Task $id is dead");
    }
    sleep(0.5);
  }
} # sub TermPrevious


sub onLostCommunication : locked {
    my ($dev, $ip) = @_;
    return if $conf{$dev}{LOSTCOMM};
    error(1, "Daemon lost communication with [$dev], $ip");
    open(ALERT,"> $ENV{APL}/var/alert/xio/axis");
    print ALERT "Daemon lost communication with [$dev], IP:$ip";
    close ALERT;
    WriteDaemonLog("Daemon lost communication with [$dev], IP:$ip");
    $conf{$dev}{LOSTCOMM}=1;
}

sub onRestoreCommunication : locked {
    my ($dev, $ip) = @_;
    return if not $conf{$dev}{LOSTCOMM};
    debug(1, "Daemon restored communication with [$dev], $ip");
    WriteDaemonLog("Daemon restored communication with [$dev], IP:$ip") if ($conf{$dev}{LOSTCOMM});
    $conf{$dev}{LOSTCOMM}=0;
}

sub get_cam_devid {
    my $ip = shift;
    my %cf = GetCfgs( ('DEVICETYPE'=>'CAMERA', 'DEVIP'=>$ip ) );
    foreach my $dev (keys %cf) {
	return $dev if($cf{$dev}{CAMERAMODEL} eq 'axis');
    }
    return '';
}

sub createMask {
    my ($num, $sign) = @_;
    my $mask = '';
    for (my $i=1; $i<5; $i++) {
	if($i == $num) {
    	    $mask .= "$sign";
	}
	else {
	    $mask .= 'x';
	}
    }
    return $mask;
}

#sub appendEventlog {
#    my ($objid, $message) = @_;
#    my $ua = LWP::UserAgent->new;
#    my $url_elog = "http://s_master/api/cgi-bin/eventlog.cgi?modify=append&utc_event=NOW&objid=$objid&message=$message";
#    my $req = GET $url_elog;
#    my $rsp = $ua->request($req);
#    html_print($rsp->content);
#}

sub appendEventlog {
   my ($objid, $message) = @_;
   my $action = Action->new(
              {
               name       => "create",
               parameters => {"objid"     => $objid,
                              "source"    => 3,
                              "msg"       => $message,
                              "eventtype" => 1,
                              "lifespan"  => 1,
                              "when"      =>time
                             }
             });

   my $resp;
   eval
   {
       $resp = $elog_client->submitAction($action);
   };
   if ($@ && $@->UNIVERSAL::isa("FwIface::FwException"))
   {
       $resp = "FwIface::FwException: ". $@->{errorId}.": ".$@->{what}. "\n";
   }
   else
   {
       $resp = "New event created: " . $resp->{parameters}{eventid} . "\n";
   }

   WriteDaemonLog($resp);

   # FIXME: what is need for?
   html_print($resp);
}

sub html_print {
    print "Cache-Control: no-store, no-cache, must-revalidate\n"
	 ."Cache-Control: post-check=0, pre-check=0\n"
	 ."Pragma: no-cache\n"
	 ."Content-Type: text/xml\n\n";
    print "@_";
}

sub disableEvents {
    my ($dev) = @_;
    my ($cam_devid, $answ, $err) = get_cam_devid($conf{$dev}{IP});
    
    if(defined($conf{$dev}{ON_EVENT_ACTION})) {
    	$answ = `$ENV{APL}/cam/bin/cam_axis_ctl.pl devid=$cam_devid -$conf{$dev}{ON_EVENT_ACTION}`;
	error (1, "Can't remove Action group $conf{$dev}{ON_EVENT_ACTION}: $err") if($err = isErrorAnswer($answ));
    }

    if(defined($conf{$dev}{ON_EVENT})) {
	$answ = `$ENV{APL}/cam/bin/cam_axis_ctl.pl devid=$cam_devid Event.$conf{$dev}{ON_EVENT}.Enabled=no`;
	error (1, "Can't disable Event $conf{$dev}{ON_EVENT}: $err") if($err = isErrorAnswer($answ));
    }

    if(defined($conf{$dev}{OFF_EVENT})) {
	$answ = `$ENV{APL}/cam/bin/cam_axis_ctl.pl devid=$cam_devid Event.$conf{$dev}{OFF_EVENT}.Enabled=no`;
	error (1, "Can't disable Event $conf{$dev}{OFF_EVENT}: $err") if($err = isErrorAnswer($answ));
    }
}


sub Release {
    foreach my $dev (keys %conf) {
	disableEvents($dev);
	delete $conf{$dev};
    }

    # release %IP hash
    foreach my $ip (sort keys %IP) {
        my $tmp = $IP{$ip};
	my @arr = @$tmp;
        foreach (@arr) {
	    pop(@arr);
        }

	delete $IP{$ip};
    }
    
    foreach(@devs_polling) {
	pop(@devs_polling);
    }
}

sub ReadConfigs {
    # ------------------- now read all configs into hash -----------------------
    %conf =  GetCfgs( ('DEVICETYPE'=>'RELAY') );
    %conf = (%conf, GetCfgs( ('DEVICETYPE'=>'SENSOR') ) );
    foreach my $dev (keys %conf) {
	$conf{$dev}{ACTIONRELAYS}='';
	$conf{$dev}{ACTIONCMD}='';
	$conf{$dev}{PULSE}=0;
	$conf{$dev}{LASTSTATE}=0;
	$conf{$dev}{EVENT}=0;
	$conf{$dev}{EVENTTIME}=time;
	$conf{$dev}{LOSTCOMM}=0;
	$conf{$dev}{MOTIONEMAILST}='none' if not defined $conf{$dev}{MOTIONEMAILST};
	
	if ( not $conf{$dev}{HW_MODEL} =~ /AXIS/
	     or !defined($conf{$dev}{DEVID})
	     or !defined($conf{$dev}{HW_PORT})
	     or !defined($conf{$dev}{HW_PORT_TYPE})
	     or (defined $conf{$dev}{ACTIVE} and $conf{$dev}{ACTIVE} =~ /NO/)) {
	    delete $conf{$dev}
	}
    } # foreach $dev (keys %conf)

    debug(2,"Configs read as:");

    my ($key);
    foreach my $dev (sort keys %conf) {
	debug(2,"[$dev]");
	foreach $key (sort keys %{$conf{$dev}}) {
	    debug(2,"\t$key=$conf{$dev}{$key}");
	}
    } # my $dev (sort keys %conf)
} # sub ReadConfigs

sub RecheckEachCameraNotifications {
    # ---------- create new hash, where keys contain IP addresses and values ---
    # ---------- contain references to an array of devices ---------------------

    my ($ip, $dev);
    foreach $dev (keys %conf) {
	$ip = $conf{$dev}{IP};
	my ($tmp, @arr) = ($IP{$ip});
	@arr = @$tmp if defined($tmp);
	push(@arr, $dev);
	$IP{$ip} = \@arr;
    }


    # ------------------------------ print hash --------------------------------
    debug(2, "Our IP hash:");
    foreach $ip (sort keys %IP) {
	debug (2, "IP: $ip");
        my $tmp = $IP{$ip};
	  my @arr = @$tmp;
        foreach (@arr) {
	    debug (2, "\tDevice: $_");
        }
    }


    # ------------------ Getting out own hostname and IP address ---------------
    my ($hostname, $addr) = `hostname`;
    chomp($hostname);
    $addr = join ".", unpack('C4',inet_aton($hostname));
    if($addr eq '127.0.0.1') {
	$addr = $1 if `/sbin/ifconfig eth0` =~ /inet (\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})/;
    }

    my ($ip, $err, $answ, $add);
    foreach $ip (keys %IP) {
        my $tmp = $IP{$ip};
	my @arr = @$tmp;

        # ---------- check out HTTP API version for every IP address -----------
	debug(2, "Check HTTP API version at $ip ...");
    
	my $dev = $IP{$ip}[0];
	$answ = `$ENV{APL}/xio/bin/axis_sen_ctl.pl devid=$dev Properties.API.HTTP.Version`;
	if($err = isErrorAnswer($answ)) {
	    error(1, $err);
	    foreach (@arr) {
		push(@devs_polling, $_);
	    }
	    next;
	}
		    
        debug(2, $answ);

	# -- Add all devices with HTTP API version < 2 to an array for polling -
	my $ver = ($answ =~ /[\w\.]+=(\d+)/) ? $1 : undef;
        if(!defined($ver) or $ver < 2) {
	    error(1, "HTTP API version is undefined") if not defined $ver;
	    error(1, "HTTP API version is $ver, smaller then 2") if($ver < 2);
	    foreach (@arr) {
		push(@devs_polling, $_);
	    }
	    next;
	}

        debug(2, "HTTP API version at $ip is $ver\n");
    
        # ------- Check list of the TCP Event Servers for each IP address ------

	my $count = 0;
        foreach $dev (@arr) {
	    next if ($conf{$dev}{DEVICETYPE} ne 'SENSOR' and $conf{$dev}{DEVICETYPE} ne 'RELAY');
	    debug(2, "Check list of the Event Servers at $ip ...");
	    $answ = `$ENV{APL}/xio/bin/axis_sen_ctl.pl devid=$dev EventServers.TCP`;
	    $err = isErrorAnswer($answ);
	    if($err = isErrorAnswer($answ)) {
		error(1, $err);
		push(@devs_polling, $dev);
		next;
	    }
	    $count++;
	    last;
	}
	next if($err or !$count);

	debug(2, "TCP Event Servers at $ip:\n$answ");

	my $tcp_serv = '';
	$count = 0;
	# ------ Possibly add current server to the list of Event Servers ------
	debug(2, "Find current \"$addr\" server in the list of the Event Servers at $ip ...");

	while($answ =~ /EventServers\.TCP\.(T\d+)\.Address=(.+)\s*\w+\.EventServers\.TCP\.\1\.Port=(\d+)/g) {
	    if($2 eq $addr or $2 eq $hostname) {
		$tcp_serv = $1;
		$port = $3;
    		debug (2, "TCP Event Server: $tcp_serv, port=$port\n");
		$count++;
		last;
	    }
	    else { debug(2, "EVENT_ADDR:$2,ADDR:$addr,HOST:$hostname"); }
	}
	if(!$count) {
	    debug (2, "The current \"$addr\" server is not found in the list of the Event Servers at $ip. We add one.");
	    my $add = "group:=EventServers.TCP "
		 ."template:=tcp_config "
		 ."EventServers.TCP.T.Name:=$hostname "
		 ."EventServers.TCP.T.Address:=$addr "
		 ."EventServers.TCP.T.Port:=$port";
	    $answ = `$ENV{APL}/xio/bin/axis_sen_ctl.pl devid=$dev $add`;
	    if($err = isErrorAnswer($answ)) {
		error(1, $err);
		#onLostCommunication($dev, $ip);
		foreach (@arr) {
		    push(@devs_polling, $_);
		}
		next;
	    }
	    elsif($answ =~ /(T\d+)\s(OK)/) {
		$tcp_serv = $1;
		debug(2, "$1 $2\n");
	    }
	}

	# ------------ Check list of the Events for each IP address ------------
	my $Events = `$ENV{APL}/xio/bin/axis_sen_ctl.pl devid=$dev Event`;
	if($err = isErrorAnswer($Events)) {
	    error(1, $err);
	    #onLostCommunication($dev, $ip);
	    foreach (@arr) {
		push(@devs_polling, $_);
	    }
	    next;
	}

        my $event;
	foreach $dev (@arr) {
	    next if $conf{$dev}{DEVICETYPE} ne 'SENSOR' or $conf{$dev}{HW_PORT_TYPE} ne 'IN';
	    
	    my $hw_port = $conf{$dev}{HW_PORT};
	    my $inputs_mask = createMask($hw_port, '1');
	
	    # -- Find an Event triggered when digital input connector is high --
	    if(!($Events =~ /Event\.(E\d+)\.Type=T\s*(.+\s)*\w+\.Event\.\1\.Enabled=(yes|no)\s*(.+\s)*\w+\.Event\.\1\.HWInputs=$inputs_mask\s*(.+\s)*\w+\.Event\.\1\.Weekdays=1111111\s*(.+\s)*\w+\.Event\.\1\.Starttime=00:00\s*(.+\s)*\w+\.Event\.\1\.Duration=24:00/)) {
    		# Create an Event triggered when digital input connector is high
    		debug (2, "Create an Event at $ip for device [$dev] triggered when digital input connector is high");
		$add = "group:=Event "
		      ."template:=event "
		      ."Event.E.Name:=Input${hw_port}_on_$tcp_serv "
		      ."Event.E.Type:=T "
		      ."Event.E.HWInputs:=$inputs_mask "
		      ."Event.E.Duration:=24:00";
		$answ = `$ENV{APL}/xio/bin/axis_sen_ctl.pl devid=$dev $add`;
		if($err = isErrorAnswer($answ)) {
		    error(1, $err);
		    #onLostCommunication($dev, $ip);
		    disableEvents($dev);
		    push(@devs_polling, $dev);
		    next;
		}
		elsif($answ =~ /(E\d+)\s(OK)/) {
		    $event = $1;
		    debug(2, "$1 $2\n");
		}
	    }
	    else {
		$event = $1;
		if($3 eq 'no') {
		    $answ = `$ENV{APL}/xio/bin/axis_sen_ctl.pl devid=$dev Event.$1.Enabled=yes`;
		    if($err = isErrorAnswer($answ)) {
			error(1, $err);
			#onLostCommunication($dev, $ip);
			disableEvents($dev);
			push(@devs_polling, $dev);
			next;
		    }
		    debug (2, "Event $1 enabled at $ip for device [$dev] triggered when digital input connector is high");
		} else {
		    debug (2, "We already have needed event for [$dev] at $ip triggered when digital input connector is high");
		}
	    }
	    $conf{$dev}{ON_EVENT} = $event;

	    # Find an Action for Event triggered when digital input connector is high
	    if(!($Events =~ /Event\.$event\.Actions\.(A\d+)\.Type=N\s*(.+\s)*\w+\.Event\.$event\.Actions\.\1\.Protocol=TCP\s*(.+\s)*\w+\.Event\.$event\.Actions\.\1\.Server=$tcp_serv\s*(.+\s)*\w+\.Event\.$event\.Actions\.\1\.Message=$ip:$dev:input$hw_port=1/)) {
		# Create an Action to the last Event for TCP notification sending
		debug (2, "Create an Action to $event at $ip for device [$dev]");
		$add = "group:=Event.$event.Actions "
		      ."template:=tcpaction "
		      ."Event.$event.Actions.A.Server:=$tcp_serv "
		      ."Event.$event.Actions.A.Message:=$ip:$dev:input$hw_port=1";
		$answ = `$ENV{APL}/xio/bin/axis_sen_ctl.pl devid=$dev $add`;
		if($err = isErrorAnswer($answ)) {
		    error(1, $err);
		    #onLostCommunication($dev, $ip);
		    disableEvents($dev);
		    push(@devs_polling, $dev);
		    next;
		}
		elsif($answ =~ /(A\d+)\s(OK)/) {
		    debug(2, "$1 $2\n");
		}
	    }
	    else { debug (2, "We already have needed Action to event $event for [$dev] at $ip triggered when digital input connector is high"); }
	
	    $inputs_mask = createMask($hw_port, '0');
	
	    # -- Find an Event triggered when digital input connector is low ---
	    if(!($Events =~ /Event\.(E\d+)\.Type=T\s*(.+\s)*\w+\.Event\.\1\.Enabled=(yes|no)\s*(.+\s)*\w+\.Event\.\1\.HWInputs=$inputs_mask\s*(.+\s)*\w+\.Event\.\1\.Weekdays=1111111\s*(.+\s)*\w+\.Event\.\1\.Starttime=00:00\s*(.+\s)*\w+\.Event\.\1\.Duration=24:00/)) {
		# Create an Event triggered when digital input connector is low
		debug (2, "Create an Event at $ip for device [$dev] triggered when digital input connector is low");
		$add = "group:=Event "
		      ."template:=event "
		      ."Event.E.Name:=Input${hw_port}_off_$tcp_serv "
		      ."Event.E.Type:=T "
		      ."Event.E.HWInputs:=$inputs_mask "
		      ."Event.E.Duration:=24:00";
		$answ = `$ENV{APL}/xio/bin/axis_sen_ctl.pl devid=$dev $add`;
		if($err = isErrorAnswer($answ)) {
		    error(1, $err);
		    #onLostCommunication($dev, $ip);
		    disableEvents($dev);
		    push(@devs_polling, $dev);
		    next;
		}
		elsif($answ =~ /(E\d+)\s(OK)/) {
		    $event = $1;
		    debug(2, "$1 $2\n");
		}
	    }
	    else {
		$event = $1;
		if($3 eq 'no') {
		    $answ = `$ENV{APL}/xio/bin/axis_sen_ctl.pl devid=$dev Event.$1.Enabled=yes`;
		    if($err = isErrorAnswer($answ)) {
			error(1, $err);
			#onLostCommunication($dev, $ip);
			disableEvents($dev);
			push(@devs_polling, $dev);
			next;
		    }
		    debug (2, "Event $1 enabled at $ip for device [$dev] triggered when digital input connector is low");
		} else {
		    debug (2, "We already have needed event for [$dev] at $ip triggered when digital input connector is low");
		}
	    }
	    $conf{$dev}{OFF_EVENT} = $event;
	    
	    # Find an Action for Event triggered when digital input connector is low
	    if(!($Events =~ /Event\.$event\.Actions\.(A\d+)\.Type=N\s*(.+\s)*\w+\.Event\.$event\.Actions\.\1\.Protocol=TCP\s*(.+\s)*\w+\.Event\.$event\.Actions\.\1\.Server=$tcp_serv\s*(.+\s)*\w+\.Event\.$event\.Actions\.\1\.Message=$ip:$dev:input$hw_port=0/)) {
		# Create an Action to the last Event for TCP notification sending
		debug (2, "Create an Action to $event at $ip for device [$dev]");
		$add = "group:=Event.$event.Actions "
		      ."template:=tcpaction "
		      ."Event.$event.Actions.A.Server:=$tcp_serv "
		      ."Event.$event.Actions.A.Message:=$ip:$dev:input$hw_port=0";
		$answ = `$ENV{APL}/xio/bin/axis_sen_ctl.pl devid=$dev $add`;
		if($err = isErrorAnswer($answ)) {
		    error(1, $err);
		    #onLostCommunication($dev, $ip);
		    disableEvents($dev);
		    push(@devs_polling, $dev);
		    next;
		}
		elsif($answ =~ /(A\d+)\s(OK)/) {
		    debug(2, "$1 $2\n");
		}
	    }
	    else { debug (2, "We already have needed Action to event $event for [$dev] at $ip triggered when digital input connector is low"); }

	    # ------------ Check current state on a sensor control -------------
	    my $state = 0;
	    $answ = `$ENV{APL}/xio/bin/axis_sen_ctl.pl devid=$dev $conf{$dev}{HW_PORT}`;
	    if(!($err = isErrorAnswer($answ))) {
    		$state = $1 if $answ =~ /.+=([01])/;
		if($state == 1) {
		    CreateAlert($dev,$state);
		    WriteLog($dev, $state, $conf{$dev}{LASTSTATE},"SENSOR");
		    $conf{$dev}{LASTSTATE}=$state;
		    $conf{$dev}{EVENT}=1;
		}
	    }
	}
	
	# ------------ Check list of the Events for each IP address ------------
	$Events = `$ENV{APL}/xio/bin/axis_sen_ctl.pl devid=$dev Event`;
	if($err = isErrorAnswer($Events)) {
	    error(1, $err);
	    #onLostCommunication($dev, $ip);
	    foreach (@arr) {
		disableEvents($_);
		push(@devs_polling, $_);
	    }
	    next;
	}
	
	foreach $dev (@arr) {
	    next if $conf{$dev}{DEVICETYPE} ne 'RELAY' or $conf{$dev}{HW_PORT_TYPE} ne 'OUT';
	    
	    my $hw_port = $conf{$dev}{HW_PORT};
	    my $inputs_mask = createMask($hw_port, '1');

	    # -- Find an Event triggered when digital input connector is high --
	    if($Events =~ /Event\.(E\d+)\.Type=T\s*(.+\s)*\w+\.Event\.\1\.Enabled=yes\s*(.+\s)*\w+\.Event\.\1\.HWInputs=$inputs_mask\s*(.+\s)*\w+\.Event\.\1\.Weekdays=1111111\s*(.+\s)*\w+\.Event\.\1\.Starttime=00:00\s*(.+\s)*\w+\.Event\.\1\.Duration=24:00/) {
		$event = $1;
		my $duration = (defined($conf{$dev}{RELAY_PULSE_TIME})) ? ($conf{$dev}{RELAY_PULSE_TIME})/1000 : 0;
		# Find an activate output port Action for Event triggered when digital input connector is high
		if(!($Events =~ /Event\.$event\.Actions\.(A\d+)\.Type=N\s*(.+\s)*\w+\.Event\.$event\.Actions\.\1\.Protocol=HW\s*(.+\s)*\w+\.Event\.$event\.Actions\.\1\.Output=$hw_port\s*(.+\s)*\w+\.Event\.$event\.Actions\.\1\.Duration=(.+)/)) {
		    # Create an Action to the last Event for activate output port
		    debug (2, "Create an Action to $event at $ip for device [$dev]");
		    $add = "group:=Event.$event.Actions "
			  ."template:=hwaction "
			  ."Event.$event.Actions.A.Output:=$hw_port "
			  ."Event.$event.Actions.A.Duration:=$duration";
		    $answ = `$ENV{APL}/xio/bin/axis_sen_ctl.pl devid=$dev $add`;
		    if($err = isErrorAnswer($answ)) {
			error(1, $err);
			#onLostCommunication($dev, $ip);
			disableEvents($dev);
			push(@devs_polling, $dev);
			next;
		    }
		    elsif($answ =~ /(A\d+)\s(OK)/) {
			debug(2, "$1 $2\n");
		    }
		}
		elsif($5 ne $duration) {
		    debug(2, "Try to change duration in Action $1 of Event $event");
		    $answ = `$ENV{APL}/xio/bin/axis_sen_ctl.pl devid=$dev Event.$event.Actions.$1.Duration=$duration`;
		    debug(2, "$answ");
		}
		else { debug (2, "We already have needed Action to event $event for [$dev] at $ip triggered when digital input connector is high"); }
		$conf{$dev}{ON_EVENT_ACTION} = "Event.$event.Actions.$1";
	    }
	}
    }
}
		   
sub debug : locked { print STDERR "DEBUG @_\n" if $DEBUG>=shift }
sub error : locked { print STDERR "ERROR @_\n" if $DEBUG>=shift }

sub listenSocket {
    my ($port, %conf) = @_;
    my ($str, $server, $client) = ('');
    $server = IO::Socket::INET->new(LocalPort => $port,
    				    Type      => SOCK_STREAM,
				    Reuse     => 1,
				    Listen    => 10 )
    or die "Can't start server tcp on port $port: $!\n";
    
    debug(2, "TCP server started at PORT#: $port\n");

    while($client = $server->accept()) {
	my $message = <$client>;
	my $peer = $client->peerhost();
	if($message =~ /(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}):(\w\d+):input([1-4])=([0-1])/) {
	    my ($ip, $dev, $input, $state) = ($1, $2, $3, $4);
	    debug(2, "Parsed notification message: IP=$1, dev=$2, input=$3, state=$4");
	    if($ip ne $peer) {
		debug(2, "Peer host IP $peer distinguish from device IP $ip");
	    }
	    else {
		CreateAlert($dev,$state);
		WriteLog($dev, $state, !$state,"SENSOR");
		$conf{$dev}{EVENT}=1;
		`$conf{$dev}{ACTIONCMD}` if($conf{$dev}{ACTIONCMD});
		foreach (split(/,/,$conf{$dev}{ACTIONRELAYS})) { `echo $_=0 > $ENV{APL}/var/dev/$_/pipe`; }
		appendEventlog($conf{$dev}{ASSOCIATE}, "[$dev]: port #$input triggered to $state");
	    }
	}
	else {
	    error(1, "$message: Illegal notification type, peer host: $peer");
	}
	
	close($client);
    }

    close ($server);
}
