#!/usr/bin/perl -w
#  $Id: cam_axis_ctl.pl 8851 2007-10-15 16:46:11Z ryabovol $
# -----------------------------------------------------------------------------
#  The part of 'IP cameras retrival' project
#  get/set AXIS parameters for sensor controls and relays
#  require admin username and password in sensor control parameters
#  based on AXIS API v2
# -----------------------------------------------------------------------------
#  Author: teetov, 
#  Edited by:
#  QA by:
#  Copyright: videoNEXT LLC
# -----------------------------------------------------------------------------
#  usage:
#         axis_sen_ctl.pl devid=s1                                       - show axis model and version
#         axis_sen_ctl.pl objid=123                                      - show axis model and version
#         axis_sen_ctl.pl devid=s1  Image.I0.RateControl.MaxBitrate      - show axis parameters
#         axis_sen_ctl.pl objid=123 Image.I0.RateControl.MaxBitrate      - show axis parameter
#         axis_sen_ctl.pl devid=s1  Image.I0.RateControl.MaxBitrate=1000 - set parameter
#         axis_sen_ctl.pl objid=123 Image.I0.RateControl.MaxBitrate=1000 - set parameter
#         axis_sen_ctl.pl devid=s1  group:=Event template:=event         - add an Event with all parameters set to their default value
#         axis_sen_ctl.pl objid=123 group:=Event template:=event         - add an Event with all parameters set to their default value
#         axis_sen_ctl.pl devid=s1  -Event.E0.Actions.A0                 - remove an action A0 from event E0
#         axis_sen_ctl.pl objid=123 -Event.E0.Actions.A0                 - remove an action A0 from event E0
#         axis_sen_ctl.pl devid=s1  1                                    - check current state of axis input 1
#         axis_sen_ctl.pl objid=123 1                                    - check current state of axis input 1
#         axis_sen_ctl.pl devid=s1  1:/1000\                             - turn axis output 1 on for 1 second and after that turn it off
#         axis_sen_ctl.pl objid=123 1:/1000\                             - turn axis output 1 on for 1 second and after that turn it off


use NextCAM::Init;
use strict;
use LWP::UserAgent;
use HTTP::Request::Common;

# VARS -------------------------------------------------------------------------
my ($P) = (split(/\//, $0))[-1]; #name of the program
my $device = shift @ARGV;

# PROC -------------------------------------------------------------------------
sub usage {
   my $err=shift;
   print STDERR "\n$err\n" if $err;
   print STDERR "\nAXIS provisioning helper\nUsage:\n"
	."  $P devid=s1                                       - show model and version\n"
	."  $P objid=123                                      - show model and version\n"
	."  $P devid=s1  Image.I0.RateControl.MaxBitrate      - show parameter\n"
	."  $P objid=123 Image.I0.RateControl.MaxBitrate      - show parameter\n"
	."  $P devid=s1  Image.I0.RateControl.MaxBitrate=1000 - set parameter\n"
	."  $P objid=123 Image.I0.RateControl.MaxBitrate=1000 - set parameter\n"
	."  $P devid=s1  group:=Event template:=event         - add an Event with all parameters set to their default value\n"
	."  $P objid=123 group:=Event template:=event         - add an Event with all parameters set to their default value\n"
	."  $P devid=s1  -Event.E0.Actions.A0                 - remove an action A0 from event E0\n"
	."  $P objid=123 -Event.E0.Actions.A0                 - remove an action A0 from event E0\n"
	."  $P devid=s1  1                                    - check current state of axis input 1\n"
	."  $P objid=123 1                                    - check current state of axis input 1\n"
	."  $P devid=s1  1:/1000\\                             - turn axis output 1 on for 1 second and after that turn it off\n"
	."  $P objid=123 1:/1000\\                             - turn axis output 1 on for 1 second and after that turn it off\n\n";
}

sub send_request {
 my ($url, $username, $password) = @_;
 my $ua = LWP::UserAgent->new;
 $ua->timeout(5);
 my $req = GET $url;
 $req->authorization_basic($username, $password);
 my $rsp=$ua->request($req);
 return ($rsp->is_success) ? $rsp->content."\n" : "HTTP ERROR [".$rsp->code."]";
}

sub get_axis_param {
 my $cf = shift;
 my $key = join ',',@_;
 my $res = send_request("http://$cf->{IP}:$cf->{HTTP_PORT}/axis-cgi/admin/param.cgi?action=list&group=$key",
				$cf->{USRNAME}, $cf->{PASSWD});
 return ($res =~ /HTTP ERROR/) ? $res.": Cannot get $key\n" : $res;
}

sub add_axis_param {
 my $cf = shift;
 my $val = join '&',@_;
 my $res = send_request("http://$cf->{IP}:$cf->{HTTP_PORT}/axis-cgi/admin/param.cgi?action=add&$val",
				$cf->{USRNAME}, $cf->{PASSWD});
 return ($res =~ /HTTP ERROR/) ? $res.": Cannot add $val\n" : $res;
}

sub del_axis_param {
 my $cf = shift;
 my $key = join ',',@_;
 my $res = send_request("http://$cf->{IP}:$cf->{HTTP_PORT}/axis-cgi/admin/param.cgi?action=remove&group=$key",
				$cf->{USRNAME}, $cf->{PASSWD});
 return ($res =~ /HTTP ERROR/) ? $res.": Cannot get $key\n" : $res;
} 

sub set_axis_param {
 my $cf = shift;
 my $val = join '&',@_;
 my $res = send_request("http://$cf->{IP}:$cf->{HTTP_PORT}/axis-cgi/admin/param.cgi?action=update&$val",
				$cf->{USRNAME}, $cf->{PASSWD});
 return ($res =~ /HTTP ERROR/) ? $res.": Cannot set $val\n" : $res;
}

sub set_and_check {
 my $cf = shift;
 my $val = join "\n",@_;
 my $key = join ',', map {/^(.+)=.+/} @_;
 my $old = get_axis_param($cf, $key);
 my $set = set_axis_param($cf, @_);
 my $new = get_axis_param($cf, $key);
 return " Old value:\n$old Set value:\n$val\n $set New value:\n$new";
}

sub check_axis_inputs {
 my $cf = shift;
 my $key = join ',',@_;
 my $res = send_request("http://$cf->{IP}:$cf->{HTTP_PORT}/axis-cgi/io/input.cgi?check=$key",
				$cf->{USRNAME}, $cf->{PASSWD});
 return ($res =~ /HTTP ERROR/) ? $res.": Cannot get $key\n" : $res;
}

sub setup_axis_outputs {
 my $cf = shift;
 my $key = join ',',@_;
 my $res = send_request("http://$cf->{IP}:$cf->{HTTP_PORT}/axis-cgi/io/output.cgi?action=$key",
				$cf->{USRNAME}, $cf->{PASSWD});
 return ($res =~ /HTTP ERROR/) ? $res.": Cannot get $key\n" : $res;
}


# MAIN -------------------------------------------------------------------------
my $cfg;
usage(), exit if not $device;
if($device =~ /devid=([sr]+\d+)/) {	#devid parameter
   my %conf = GetCfgs('DEVID' => $1);
   print(STDERR "CFG ERROR: Device $device is not defined in system\n"), exit 1	if not %conf;
   $cfg = $conf{$1};
}elsif($device =~ /objid=(\d+)/) {	#objid parameter
   my %conf = GetCfgs('OBJID' => $1);
   print(STDERR "CFG ERROR: Device $device is not defined in system\n"), exit 1	if not %conf;
   my $devid = (keys %conf)[0];
   $cfg = $conf{$devid};
}else {
  usage("CFG ERROR: Wrong Device '$device' specification");
  exit 1;
}
print(STDERR "CFG ERROR: This is not AXIS sensor control or relay! HW_MODEL=$cfg->{HW_MODEL}\n"),exit 1    if $cfg->{HW_MODEL} ne 'AXIS';

if(defined($cfg->{ASSOCIATE})) {
    my %conf = GetCfgs('OBJID' => $cfg->{ASSOCIATE});
    $cfg->{HTTP_PORT} = (%conf) ? $conf{(keys %conf)[0]}{HTTP_PORT} : 80;
}
else { $cfg->{HTTP_PORT} = 80; }


if(not @ARGV) {	# only show model & version
  print get_axis_param($cfg, 'Properties.Firmware.Version', 'brand.ProdShortName');
  exit 0;
}

my (@get, @set, @add, @del, @chk_inputs, @set_outputs);
foreach(@ARGV) {
 push(@get, $1)     if /^(\w[\w\.]+)$/;
 push(@set, $1)     if /^(\w[\w\.]+=.+)$/;
 push(@add, $1.$2)  if /^(\w[\w\.]+):(=.+)$/;
 push(@del, $1)     if /^-(\w[\w\.]+)$/;

 if($_ =~ /^(\d),?(\d)?,?(\d)?,?(\d)?[,\d]*$/) {
    push(@chk_inputs, $1);
    push(@chk_inputs, $2) if defined $2;
    push(@chk_inputs, $3) if defined $3;
    push(@chk_inputs, $4) if defined $4;
 }
 
 push(@set_outputs,$1) if /^(\d:[\/\\0-9]+)$/;
}

print set_and_check($cfg, @set)  if @set;
print get_axis_param($cfg, @get) if @get;
print add_axis_param($cfg, @add) if @add;
print del_axis_param($cfg, @del) if @del;
print check_axis_inputs($cfg, @chk_inputs) if @chk_inputs;
print setup_axis_outputs($cfg,@set_outputs) if @set_outputs;
