#!/usr/bin/perl

#
#  Daemon program for APC PowerSwitch series
#
#  Environment:
#	$APL       - /opt/sarch - software root directory
#	$APL_CONF  - $APL/store for now and $APL/var/dev later
#
#  Sample 'conf'-file:
#
#  DEVID=r1                     - device ID
#  HW_MODEL=APC_MS              - device handling hardware, APC_MS only for this daemon
#  HW_PORT=1                    - number (1-8 for AP9211)
#  RELAY_PULSE_TIME=1000        - pulse time in milliseconds for relays with auto-shutoff, 0 for steady signal
#  IP=1.2.3.4 or a1.company.net - APC_MS server IP or URL
#  ACTIONRELAYS=r1,r9           - relays to reset to 0 in the case of action detected
#  ACTIONCMD=/opt/sarch/bin/ref - execute a command in the case of action detected
#  MOTIONEMAIL=email@site.net   - email for mailing
#  MOTIONEMAILST=e-mail once    - email srategy
#                every event
#                once in 15 min
#                once in 30 min
#                once in a hour


use strict;
use Net::SNMP;
use IO::File;
use IO::Select;
use NextCAM::Init;

my $DEBUG=0;

my ($pname)=reverse(split(/\//,$0));
TermPrevious($pname);
my %conf;

ReadConfigs();

debug(5,"APC_MS daemon started.");


# ---------------------------- reset alert flags ------------------------------
foreach my $dev (keys %conf) { CreateAlert($dev,0); }

# ---------------------------- open control inputs ----------------------------
my ($usrpsw,$curstate)=('',0);
my $pipe = new IO::File "+<$ENV{APL}/var/xio/apc_ms";
die "Can not open $ENV{APL}/var/xio/apc_ms\n$!\n" if not defined $pipe;
my $handles = new IO::Select();
$handles->add($pipe);
$handles->add('STDIN') if $DEBUG;

WriteDaemonLog('APC_MS daemon started...');

# =============================================================================


while(1) { # ------------------------- main cycle -----------------------------
  sleep 1;
  foreach my $dev (keys %conf) {
    open(PID,"> $ENV{APL_CONF}/$dev/pid"); print PID $$; close PID;
  }
  # ------------------- check and handle pipe commands ------------------------
 my $nextcmd;
 my ($ready) = IO::Select->select($handles,undef,undef,.2);

  if ($ready) {
    foreach my $in (@$ready) {
      next if not $nextcmd=<$in>;
      if($nextcmd=~/HUP/) {
        ReadConfigs();
        next;
      }
      $nextcmd =~/(\w+)=(\d)\s+(.*)/;
      debug(5,"command: $1=$2");
      next if not ($conf{$1} and (($2==0) or ($2==1)));
      # --------------------- process command ---------------
      $curstate=ProcessCmd($1,$2);
      if($curstate == -2) { # lost communication
        if(not $conf{$1}{LOSTCOMM}) {
          open(ALERT,"> $ENV{APL}/var/alert/xio/apc_ms");
          print ALERT "Daemon lost communication with [$1], IP:$conf{$1}{IP}";
          close ALERT;
          WriteDaemonLog("Daemon lost communication with [$1], IP:$conf{$1}{IP}");
          $conf{$1}{LOSTCOMM}=1;
        }
      } # lost communication
      else {
        WriteDaemonLog("Daemon restored communication with [$1], IP:$conf{$1}{IP}") if ($conf{$1}{LOSTCOMM});
        $conf{$1}{LOSTCOMM}=0;
      } # lost communication
      # ---- write log, create alert, because of relay event ------
      WriteLog($1, $2, $conf{$1}{LASTSTATE},$3);
      CreateAlert($1,$2);
      #
      `$conf{$1}{ACTIONCMD}` if($conf{$1}{ACTIONCMD});
      foreach (split(/,/,$conf{$1}{ACTIONRELAYS})) { `echo $_=0 > $ENV{APL}/var/dev/$_/pipe`; }
      $conf{$1}{PULSE}=time+int($conf{$1}{RELAY_PULSE_TIME}/1000) if $conf{$1}{RELAY_PULSE_TIME};
      $conf{$1}{LASTSTATE}=$conf{$1}{RELAY_PULSE_TIME} ? 0 : $2;
      $conf{$1}{EVENT}=1;
      } # foreach handle
  } #  if $ready @ check and handle pipe commands
  # --------------- for pulsed output switch back to OFF --------------
  foreach my $dev (keys %conf) {
    next if not $conf{$dev}{PULSE};
    if( $conf{$dev}{PULSE} <= time ) {
      `echo $_=0 > $ENV{APL}/var/dev/$dev/pipe`;
      $conf{$dev}{PULSE} = 0;
    }
  }
  # ----------------- send email notifications ------------------------
  foreach my $dev (keys %conf) {
    my $strategy;
    if($conf{$dev}{MOTIONEMAILST}=~/once\sin\sa\shour/) { $strategy=60; }
    elsif($conf{$dev}{MOTIONEMAILST}=~/once\sin\s(\d+)\smin/) { $strategy=$1; }
    elsif($conf{$dev}{MOTIONEMAILST}=~/e-mail\sonce/) { $strategy=1; }
    elsif($conf{$dev}{MOTIONEMAILST}=~/every\sevent/) { $strategy=0; }
    else { $strategy=-1; }
    #
    if($conf{$dev}{MOTIONEMAILST}=~/once\sin/) {
      if((time-$conf{$dev}{EVENTTIME})/60 > $strategy ) {
      next if not $conf{$dev}{EVENT};
      my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = gmtime($conf{$dev}{EVENTTIME}); $year %= 100; $mon++;
      SendMail($dev,$strategy,sprintf("%02d%02d%02d%02d%02d%02d",$year,$mon,$mday,$hour,$min,$sec));
      $conf{$dev}{EVENTTIME}=time;
      $conf{$dev}{EVENT}=0;
      }
    }
    elsif($conf{$dev}{MOTIONEMAILST}=~/e-mail\sonce/ or $conf{$dev}{MOTIONEMAILST}=~/every\sevent/) {
      next if not $conf{$dev}{EVENT};
      $conf{$dev}{MOTIONEMAILST}='' if $conf{$dev}{MOTIONEMAILST}=~/e-mail\sonce/;
      $conf{$dev}{EVENT}=0;
      my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = gmtime(time); $year %= 100; $mon++;
      SendMail($dev,$strategy,sprintf("%02d%02d%02d%02d%02d%02d",$year,$mon,$mday,$hour,$min,$sec));
    }
  } # send email notifications
} # --------------------------- main cycle -------------------------------
return 0; # ===================================================================


# ------------------------------------------------------------ ProcessCmd -----
sub ProcessCmd {
  my($dev,$state)=@_; debug(5,"ProcessCmd($dev,$state)");

  my ($apc_port,$apc_state)=($conf{$dev}{HW_PORT},2-$state); debug(5,"APC PORT:$apc_port  APC STATE: $apc_state");
  my $IP=$conf{$dev}{IP};

  my ($session, $error) = Net::SNMP->session(
     -hostname  => $conf{$dev}{IP}, # '207.207.189.91',
     -community => 'private',
     -port      => 161,
     -timeout   => 3
  );

  return 0 if !defined($session);
  my $sPDUOutletCtl     = "1.3.6.1.4.1.318.1.1.4.4.2.1.3.$apc_port"; debug(5,"PDU: $sPDUOutletCtl");
  my $result = $session->set_request( -varbindlist => [$sPDUOutletCtl,INTEGER,$apc_state] );
  debug(5,$session->error) if (!defined($result));
  debug(5,$result->{$sPDUOutletCtl});
  $session->close;
  return 1;
} # sub ProcessCmd

# ----------------------------------------------------------- ReadConfigs -----
sub ReadConfigs
{
foreach my $dev (keys %conf) {
  $conf{$dev}=undef;
}
# --- now read all configs into hash ---
%conf = GetCfgs( ('DEVICETYPE'=>'RELAY') );

foreach my $dev (keys %conf) {
  $conf{$dev}{ACTIONRELAYS}='';
  $conf{$dev}{ACTIONCMD}='';
  $conf{$dev}{PULSE}=0;
  $conf{$dev}{LASTSTATE}=0;
  $conf{$dev}{EVENT}=0;
  $conf{$dev}{EVENTTIME}=time;
  $conf{$dev}{LOSTCOMM}=0;
  $conf{$dev}{MOTIONEMAILST}='none' if not defined $conf{$dev}{MOTIONEMAILST};
  delete $conf{$dev} if not $conf{$dev}{'HW_MODEL'}=~/APC_MS/i or (defined $conf{$dev}{'ACTIVE'} and $conf{$dev}{'ACTIVE'}=~/NO/);
} # foreach $dev (keys %conf)

debug(5,"Using configs:");

my ($key,$line);
foreach my $dev (sort keys %conf) {
  debug(5,"[$dev]");
  foreach $key (sort keys %{$conf{$dev}}) {
    debug(5,"$key=$conf{$dev}{$key}");
  }
  debug(5,'--------------------------------------------');
}
} # sub ReadConfigs
# ----------------------------------------------------------------- debug -----
sub debug
{
  print STDERR "DEBUG @_\n" if $DEBUG>=shift
} # sub debug
# ----------------------------------------------------------- CreateAlert -----
sub CreateAlert {
  my ($_dev,$_state) = @_;
  open(_ALERT,"> $ENV{APL_CONF}/$_dev/alert");
  print _ALERT '1' if $_state;
  close _ALERT;
} # sub CreateAlert

# -------------------------------------------------------------- SendMail -----
sub SendMail
{
  my ($_devid,$_strategy,$_time)=@_;
  $_strategy=sprintf("%d",$_strategy);
  `perl $ENV{APL}/xio/bin/sendemail.pl $_devid $_strategy $_time`;
} # sub SendMail

# -------------------------------------------------------- WriteDaemonLog -----
sub WriteDaemonLog {
  my ($_msg) = @_;
  my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = gmtime(time); $year %= 100; $mon++;
  open(LOG,">> $ENV{APL}/var/log/xio/axis.log");
  print LOG sprintf("%02d%02d%02d%02d%02d%02d %s\n",$year,$mon,$mday,$hour,$min,$sec,$_msg);
  close LOG;
} # sub WriteDaemonLog

# -------------------------------------------------------------- WriteLog -----
sub WriteLog {
  my ($_dev, $_curstate, $_laststate,$_userid)=@_;
  my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = gmtime(time); $year %= 100; $mon++;
  mkdir sprintf("$ENV{APL}/store/$_dev/%02d%02d%02d",$year,$mon,$mday);
  open(LOG,sprintf(">> $ENV{APL}/store/$_dev/%02d%02d%02d/%02d.log",$year,$mon,$mday,$hour));
  print LOG sprintf("%02d%02d%02d%02d%02d%02d  %d  %d  %s\n",$year,$mon,$mday,$hour,$min,$sec,$_curstate,$_laststate,$_userid);
  close LOG;
} # sub WriteLog

# ---------------------------------------------------------- TermPrevious -----
sub TermPrevious {
  my ($prog) = @_;
  local $SIG{TERM}='IGNORE';
  `killall $prog 2>/dev/null`;
} # sub TermPrevious

