#!/usr/bin/perl
#
#  Daemon program for ACTi servers and cameras relay and input control.
#
#  Environment:
#	$APL       - /opt/sarch - software root directory
#	$APL_CONF  - $APL/var/conf
#
#  1. Reads devices' configs at $APL_CONF/<devicename>/conf for all ACTi virtual devices
#  2. Creates and regularly updates $APL_CONF/<devicename>/pid all virtual devices
#  3. Reads commands for relays from $APL/var/xio/acti
#  4. Writes virtual devices' logs at $APL/store/<devicename>/<YYMMDD>/<HH>.log
#  5. Updates virtual devices' alert files (by current state of port) at $APL_CONF/<devicename>/alert
#  6. Writes daemon log to $APL/var/log/xio/acti.log
#  7. Regularly check for connectivity and raise an alert flag $APL/var/alert/xio/acti
#
#  Sample 'conf'-file:
#
#  DEVID=r1                     - device ID
#  HW_MODEL=ACTi                - device handling hardware, ACTi only for this daemon
#  USRNAME=root                 - username if ACTi server is username and password-protected
#  PASSWD=pass                  - password if ACTi server is password-protected
#  HW_PORT=1                    - number (1-4 for input, 1 for output)
#  HW_PORT_TYPE=OUT             - port type (IN/OUT)
#  IP=1.2.3.4 or a1.company.net - ACTi server IP or URL

use strict;
use LWP::UserAgent;
use HTTP::Request::Common;
use Time::Local;
use FileHandle;
use File::Copy;
use IO::File;
use IO::Select;
use NextCAM::Init;
use lib "/opt/sarch/common/lib"; # service API
use lib "/opt/sarch/fw/bin/perl/FwIface";
use lib "/opt/sarch/elog/bin/perl";
use lib "/opt/sarch/elog/bin/perl/ELogIface";


$SIG{INT} = sub { die WriteDaemonLog('ACTi daemon terminated by Ctrl-C'); };
$SIG{TERM}= sub { die WriteDaemonLog('ACTi daemon killed'); };

my $DEBUG = 2;	# use 1 for errors output only and 2 for both debug and errors
my ($pname)=reverse(split(/\//,$0));

TermPrevious($pname);


my $service_client = NextCAM::Servicing::Client->new("s_master", 10000);
my $elog_client = ELogClient->new($service_client->getProtocol("skm.eventlog"));

my $pipe = new IO::File "+<$ENV{APL}/var/xio/acti";
die "Can not open $ENV{APL}/var/xio/acti\n$!\n" if not defined $pipe;
my $handles = new IO::Select();
$handles->add($pipe);

WriteDaemonLog('ACTi daemon started...');

my ($curstate, %conf) = (0);

debug(2, "ACTi daemon started with configs:");
ReadConfigs();

# -------------------------- reset alert flags ---------------------------------
foreach my $dev (keys %conf) { CreateAlert($dev,0); }

while(1) { # ---------------------------- main cycle ---------------------------
  my ($answ, $err);
  # -------------------------- update PID for all devices ----------------------
  foreach my $dev (keys %conf) {
    open(PID,"> $ENV{APL_CONF}/$dev/pid"); print PID $$; close PID;
    if($conf{$dev}{PULSE} and $conf{$dev}{PULSE} < time) {
      $conf{$dev}{PULSE}=0;
      CreateAlert($dev,0);
    }
    # -------------------------- check sensors, polling --------------------------
    next if $conf{$dev}{HW_PORT_TYPE} ne 'IN';
	
    $answ = `$ENV{APL}/xio/bin/acti_sen_ctl.pl devid=$dev group=mpeg4 DIO_STATUS`;
    if($err = isErrorAnswer($answ)) {
	onLostCommunication ($dev, $conf{$dev}{IP});
	$curstate = 0;
    }
    else { # restored
      $answ =~ /.+=([0-9A-F]x[0-9A-F]{2})/i;
      if('0x0'.$conf{$dev}{HW_PORT} eq ($1 & '0x0'.$conf{$dev}{HW_PORT})) {
	$curstate = 1;
      } else { $curstate = 0; }
      onRestoreCommunication($dev, $conf{$dev}{IP});
    }
    if($curstate != $conf{$dev}{LASTSTATE}) {
      CreateAlert($dev,$curstate);
      WriteLog($dev, $curstate, $conf{$dev}{LASTSTATE},"SENSOR");
      $conf{$dev}{LASTSTATE}=$curstate;
      $conf{$dev}{EVENT}=1;
      debug(2, "[$dev]: port#$conf{$dev}{HW_PORT} triggered to $curstate");
      appendEventlog($conf{$dev}{ASSOCIATE} ? $conf{$dev}{ASSOCIATE} : $conf{$dev}{OBJID}, "[$dev]: port #$conf{$dev}{HW_PORT} triggered to $curstate");
      `$conf{$dev}{ACTIONCMD}` if($conf{$dev}{ACTIONCMD});
      foreach (split(/,/,$conf{$dev}{ACTIONRELAYS})) { `echo $_=0 > $ENV{APL}/var/dev/$_/pipe`; }
    }
  }

  # ---------------------- check and handle pipe commands ----------------------
  my ($ready) = IO::Select->select($handles,undef,undef,.2);
  if ($ready) {
    my $line = $pipe->getline;
    
    if($line =~ /^HUP\s/) {
        eval {
	    local $SIG{ALRM}=sub{die 'ALARM'};
	    alarm 2;
	    debug(2, "HUP signal received!");
	    ReadConfigs();
	};
	alarm 0;
	next;
    }

  }
  
  # ------------------------ send email notifications --------------------------
#  foreach my $dev (keys %conf) {
#	SendMailNotify($dev);
#  }
}
return 0;

# -------------------------------- CreateAlert ---------------------------------
sub CreateAlert {
  my ($_dev,$_state) = @_;
  open(_ALERT,"> $ENV{APL_CONF}/$_dev/alert");
  print _ALERT '1' if $_state;
  close _ALERT;
} # sub CreateAlert

# ------------------------------ SendMailNotify --------------------------------
sub SendMailNotify {
    my ($dev) = @_;
    my $strategy;
    if($conf{$dev}{MOTIONEMAILST}=~/once\sin\sa\shour/) { $strategy=60; }
    elsif($conf{$dev}{MOTIONEMAILST}=~/once\sin\s(\d+)\smin/) { $strategy=$1; }
    elsif($conf{$dev}{MOTIONEMAILST}=~/e-mail\sonce/) { $strategy=1; }
    elsif($conf{$dev}{MOTIONEMAILST}=~/every\sevent/) { $strategy=0; }
    else { $strategy=-1; }
    #
    if($conf{$dev}{MOTIONEMAILST}=~/once\sin/) {
      if((time-$conf{$dev}{EVENTTIME})/60 > $strategy ) {
      return if not $conf{$dev}{EVENT};
      my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = gmtime($conf{$dev}{EVENTTIME}); $year %= 100; $mon++;
      SendMail($dev,$strategy,sprintf("%02d%02d%02d%02d%02d%02d",$year,$mon,$mday,$hour,$min,$sec));
      $conf{$dev}{EVENTTIME}=time;
      $conf{$dev}{EVENT}=0;
      }
    }
    elsif($conf{$dev}{MOTIONEMAILST}=~/e-mail\sonce/ or $conf{$dev}{MOTIONEMAILST}=~/every\sevent/) {
      return if not $conf{$dev}{EVENT};
      $conf{$dev}{MOTIONEMAILST}='' if $conf{$dev}{MOTIONEMAILST}=~/e-mail\sonce/;
      $conf{$dev}{EVENT}=0;
      my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = gmtime(time); $year %= 100; $mon++;
      SendMail($dev,$strategy,sprintf("%02d%02d%02d%02d%02d%02d",$year,$mon,$mday,$hour,$min,$sec));
    }
}

# ---------------------------------- SendMail ----------------------------------
sub SendMail
{
  my ($_devid,$_strategy,$_time)=@_;
  $_strategy=sprintf("%d",$_strategy);
  `perl $ENV{APL}/xio/bin/sendemail.pl $_devid $_strategy $_time`;
} # sub SendMail

# ------------------------------- WriteDaemonLog -------------------------------
sub WriteDaemonLog {
  my ($_msg) = @_;
  my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = gmtime(time); $year %= 100; $mon++;
  open(LOG,">> $ENV{APL}/var/log/xio/acti.log");
  print LOG sprintf("%02d%02d%02d%02d%02d%02d %s\n",$year,$mon,$mday,$hour,$min,$sec,$_msg);
  close LOG;
} # sub WriteDaemonLog

# ---------------------------------- WriteLog ----------------------------------
sub WriteLog {
  my ($_dev, $_curstate, $_laststate,$_userid)=@_;
  my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = gmtime(time); $year %= 100; $mon++;
  mkdir sprintf("$ENV{APL}/store/$_dev/%02d%02d%02d",$year,$mon,$mday);
  open(LOG,sprintf(">> $ENV{APL}/store/$_dev/%02d%02d%02d/%02d.log",$year,$mon,$mday,$hour));
  print LOG sprintf("%02d%02d%02d%02d%02d%02d  %d  %d  %s\n",$year,$mon,$mday,$hour,$min,$sec,$_curstate,$_laststate,$_userid);
  close LOG;
} # sub WriteLog

sub isErrorAnswer {
    my ($str) = @_;
    if(($str =~ /Error:(.+)/i) or
       ($str =~ /(^HTTP ERROR.+)/)) {
	    return $1;
    }

    return '';
}

sub TermPrevious {
  my ($prog) = @_;
  local $SIG{TERM}='IGNORE';
  `killall $prog 2>/dev/null`;
} # sub TermPrevious


sub onLostCommunication {
    my ($dev) = @_;
    return if $conf{$dev}{LOSTCOMM};
    error(1, "Daemon lost communication with [$dev], IP:$conf{$dev}{IP}");
    open(ALERT,"> $ENV{APL}/var/alert/xio/acti");
    print ALERT "Daemon lost communication with [$dev], IP:$conf{$dev}{IP}";
    close ALERT;
    WriteDaemonLog("Daemon lost communication with [$dev], IP:$conf{$dev}{IP}");
    $conf{$dev}{LOSTCOMM}=1;
}

sub onRestoreCommunication {
    my ($dev) = @_;
    return if not $conf{$dev}{LOSTCOMM};
    debug(1, "Daemon restored communication with [$dev], IP:$conf{$dev}{IP}");
    WriteDaemonLog("Daemon restored communication with [$dev], IP:$conf{$dev}{IP}") if ($conf{$dev}{LOSTCOMM});
    $conf{$dev}{LOSTCOMM}=0;
}

#sub appendEventlog {
#    my ($objid, $message) = @_;
#    my $ua = LWP::UserAgent->new;
#    my $url_elog = "http://s_master/api/cgi-bin/eventlog.cgi?modify=append&utc_event=NOW&objid=$objid&message=$message";
#    my $req = GET $url_elog;
#    my $rsp = $ua->request($req);
#    html_print($rsp->content);
#}

sub appendEventlog {
   my ($objid, $message) = @_;
   my $action = FwIface::Action->new(
              {
               name       => "create",
               parameters => {"objid"     => $objid,
                              "source"    => 3,
                              "msg"       => $message,
                              "eventtype" => 1,
                              "lifespan"  => 1,
                              "when"      =>time
                             }
             });

   my $rsp;
   eval
   {
       $resp = $elog_client->submitAction($action);
   };
   if ($@ && $@->UNIVERSAL::isa("FwIface::FwException"))
   {
       $rsp = "FwIface::FwException: ". $@->{errorId}.": ".$@->{what}. "\n";
   }
   else
   {
       $rsp = "New event created: " . $resp->{parameters}{eventid} . "\n";
   }

   WriteDaemonLog($rsp);

   # FIXME: what is need for?
   html_print($rsp);
}

sub html_print {
    print "Cache-Control: no-store, no-cache, must-revalidate\n"
	 ."Cache-Control: post-check=0, pre-check=0\n"
	 ."Pragma: no-cache\n"
	 ."Content-Type: text/xml\n\n";
    print "@_\n\n";
}

sub ReadConfigs {
    # ------------------- now read all configs into hash -----------------------
    %conf =  GetCfgs( ('DEVICETYPE'=>'RELAY') );
    %conf = (%conf, GetCfgs( ('DEVICETYPE'=>'SENSOR') ) );
    foreach my $dev (keys %conf) {
	$conf{$dev}{ACTIONRELAYS}='';
	$conf{$dev}{ACTIONCMD}='';
	$conf{$dev}{PULSE}=0;
	$conf{$dev}{LASTSTATE}=0;
	$conf{$dev}{EVENT}=0;
	$conf{$dev}{EVENTTIME}=time;
	$conf{$dev}{LOSTCOMM}=0;
	$conf{$dev}{MOTIONEMAILST}='none' if not defined $conf{$dev}{MOTIONEMAILST};
	
	if ( not $conf{$dev}{HW_MODEL} =~ /ACTi/
	     or !defined($conf{$dev}{DEVID})
	     or !defined($conf{$dev}{HW_PORT})
	     or !defined($conf{$dev}{HW_PORT_TYPE})
	     or (defined $conf{$dev}{ACTIVE} and $conf{$dev}{ACTIVE} =~ /NO/)) {
	    delete $conf{$dev}
	}
    } # foreach $dev (keys %conf)

    debug(2,"Configs read as:");

    my ($key);
    foreach my $dev (sort keys %conf) {
	debug(2,"[$dev]");
	foreach $key (sort keys %{$conf{$dev}}) {
	    debug(2,"\t$key=$conf{$dev}{$key}");
	}
    } # my $dev (sort keys %conf)
} # sub ReadConfigs

sub debug : locked { print STDERR "DEBUG @_\n" if $DEBUG>=shift }
sub error : locked { print STDERR "ERROR @_\n" if $DEBUG>=shift }
