#!/usr/bin/perl -w
# -----------------------------------------------------------------------------
#  The part of 'IP cameras retrival' project
#  get/set ACTi parameters for sensor controls and relays
#  require admin username and password in sensor control parameters
#  based on ACTi URL SDK
# -----------------------------------------------------------------------------
#  Author: ryabovol, 
#  Edited by:
#  QA by:
#  Copyright: videoNEXT LLC
# -----------------------------------------------------------------------------
#  usage:
#         acti_sen_ctl.pl devid=s1                              - show ACTi system information
#         acti_sen_ctl.pl objid=123                             - show ACTi system information
#         acti_sen_ctl.pl devid=s1 group=mpeg4 DIO_STATUS       - show parameter
#         acti_sen_ctl.pl objid=123 group=mpeg4 DIO_STATUS      - show parameter
#         acti_sen_ctl.pl devid=s1 group=mpeg4 DIO_OUTPUT=0x01  - set parameter
#         acti_sen_ctl.pl objid=123 group=mpeg4 DIO_OUTPUT=0x01 - set parameter

use NextCAM::Init;
use strict;
use LWP::UserAgent;
use HTTP::Request::Common;

# VARS -------------------------------------------------------------------------
my ($P) = (split(/\//, $0))[-1]; #name of the program
my $device = shift @ARGV;

# PROC -------------------------------------------------------------------------
sub usage {
   my $err=shift;
   print STDERR "\n$err\n" if $err;
   print STDERR "\nACTi provisioning helper\nUsage:\n"
	."  $P devid=s1                              - show ACTi system information\n"
	."  $P objid=123                             - show ACTi system information\n"
	."  $P devid=s1 group=mpeg4 DIO_STATUS       - show parameter\n"
	."  $P objid=123 group=mpeg4 DIO_STATUS      - show parameter\n"
	."  $P devid=s1 group=mpeg4 DIO_OUTPUT=0x01  - set parameter\n"
	."  $P objid=123 group=mpeg4 DIO_OUTPUT=0x01 - set parameter\n\n";
}

sub show_system_info {
  my $cf = shift;
  print "\n".get_set_acti_param('system', $cf, 'SYSTEM_INFO');
}

sub send_request {
 my ($url, $username, $password) = @_;
 my $ua = LWP::UserAgent->new;
 $ua->timeout(3);
 my $req = GET $url;
 my $rsp=$ua->request($req);
 return ($rsp->is_success) ? $rsp->content."\n" : "HTTP ERROR [".$rsp->code."]";
}

sub get_set_acti_param {
 my $group = shift;
 my $cf = shift;
 my $key = join '&',@_;
 my $res = send_request("http://$cf->{IP}:$cf->{HTTP_PORT}/cgi-bin/$group?USER=$cf->{USRNAME}&PWD=$cf->{PASSWD}&$key");
 return ($res =~ /HTTP ERROR/) ? $res.": Cannot get $key\n" : $res;
}

sub set_and_check {
 my $group = shift;
 my $cf = shift;
 my $val = join "\n",@_;
 my $key = join '&', map {/^(.+)=.+/} @_;
 my $old = get_set_acti_param($group, $cf, $key);
 my $set = get_set_acti_param($group, $cf, @_);
 my $new = get_set_acti_param($group, $cf, $key);
 return " Old value:\n$old Set value:\n$val\n$set New value:\n$new";
}

# MAIN -------------------------------------------------------------------------
my $cfg;
usage(), exit if not $device;
if($device =~ /devid=([sr]\d+)/) {	#devid parameter
   my %conf = GetCfgs('DEVID' => $1);
   print(STDERR "CFG ERROR: Device $device is not defined in system\n"), exit 1	if not %conf;
   $cfg = $conf{$1};
}elsif($device =~ /objid=(\d+)/) {	#objid parameter
   my %conf = GetCfgs('OBJID' => $1);
   print(STDERR "CFG ERROR: Device $device is not defined in system\n"), exit 1	if not %conf;
   my $devid = (keys %conf)[0];
   $cfg = $conf{$devid};
}else {
  usage("CFG ERROR: Wrong Device '$device' specification");
  exit 1;
}
print(STDERR "CFG ERROR: This is not ACTi sensor control or relay! HW_MODEL=$cfg->{HW_MODEL}\n"),exit 1    if $cfg->{HW_MODEL} ne 'ACTi';

if(defined($cfg->{ASSOCIATE}) and $cfg->{ASSOCIATE}) {
    my %conf = GetCfgs('OBJID' => $cfg->{ASSOCIATE});
    $cfg->{HTTP_PORT} = (%conf) ? $conf{(keys %conf)[0]}{HTTP_PORT} : 80;
}
else { $cfg->{HTTP_PORT} = 80; }
        

if(not @ARGV) {	# only show system information
  show_system_info($cfg);
  exit 0;
}

my $group=shift @ARGV;
if($group =~ /group=(.+)/) {
    $group = $1;
}
else { $group = ''; }

if(not @ARGV or !$group) {	# only show system information
  show_system_info($cfg);
  exit 0;
}

my (@get, @set);
foreach(@ARGV) {
 push(@get, $1)     if /^(\w[\w\.]+)$/;
 push(@set, $1)     if /^(\w[\w\.]+=.+)$/;
}

print set_and_check($group, $cfg, @set)  if @set;
print get_set_acti_param($group, $cfg, @get) if @get;
