/*******************************************************************************************
 *                                                                                         *
 *       vMX.WallcellView class - visual representation of Wall Cell object                *
 *                                                                                         *
 ******************************************************************************************/


jsx3.lang.Class.defineClass(
	"mx.vMX.WallcellView", jsx3.lang.Object, null, function(WallcellView, WallcellView_prototype) {

	var vMX = mx.vMX;

	WallcellView.IMG_PLACEHOLDER = "images/design1/vMX/magnifier_placeholder.png";

	WallcellView.IMG_VNC         = "images/design1/vnc/vnc_image.jpg";

	WallcellView_prototype.init = function(wallcell, block) {
		this._wc = wallcell;
		this._block = block;
		//this._dirty = true;
		this.wallLayout = null;
	};

	WallcellView_prototype.getCellImg = function(cellid) { return this._block.getDescendantOfName(cellid); };

	WallcellView_prototype.setContainer = function(container) { this.wallLayout = container; };

	WallcellView_prototype.getWallcell = function() { return this._wc; };

	WallcellView_prototype.getBlock = function() { return this._block; };

	WallcellView_prototype.getWallLayout = function() { return this.wallLayout; };


	WallcellView_prototype.create = function() {
		vMX._log("WallcellView_prototype.create");

		this._block.setName("Wallcell " + String(this._wc.row) + "." + String(this._wc.col));

		// Set cell border css
		this._block.setClassName("vmx_wallcell");

		if(this._wc.monitor) { // subscribe to monitor events and draw cell images

			this._block.setEvent("1;", jsx3.gui.Interactive.JSXDOUBLECLICK);
			this._block.subscribe(jsx3.gui.Interactive.JSXDOUBLECLICK, this, "onDoubleClick");
			this._block.subscribe(jsx3.gui.Interactive.DESTROY, this, "onDestroy");

			/** WallcellView must subscribe to notifications of monitor and cell changes,
			 * such as STATE_CHANGED, CELL_RESIZE, CELL_REPLACE, NEW_LAYOUT
			 * As far as updateView() is rather quick, it will hanle all events except
			 * MonitorCell.STATE_CHANGED. This one will be handled by separate handler
			 */
			this._wc.monitor.subscribe(vMX.Monitor.NEW_LAYOUT, this, "updateView");
			this._wc.monitor.subscribe(vMX.Monitor.CELL_REPLACE, this, "updateView");
			this._wc.monitor.subscribe(vMX.Monitor.CELL_RESIZE, this, "updateView");
			this._wc.monitor.subscribe(vMX.Monitor.NAME_DESC_CHANGED, this, "onNameDescChanged");

			this._wc.subscribe(vMX.WallCell.BEFORE_REMOVE_MONITOR, this, "onRemoveMonitor");

			this.updateView();
		}
		else { // Draw placeholder image
			var cellImg = new jsx3.gui.Image("placeholder");
			cellImg.setRelativePosition(jsx3.gui.Block.ABSOLUTE);
			cellImg.setWidth("100%").setHeight("100%");
			cellImg.setSrc(WallcellView.IMG_PLACEHOLDER);

			this._block.setChild(cellImg);
			cellImg.repaint();
		}
	};

	WallcellView_prototype.onNameDescChanged = function(event) {
		var mon = this._wc.monitor;
		if(!mon) return;

		var strTip = "Monitor '" + unescape(mon.name) + "' [id: " + mon.objid + "]";
		this._block.setTip(strTip);
	};


	WallcellView_prototype.updateView = function() {
		vMX._log("vMX.WallcellView_prototype.updateView");
		var objMonitor = this._wc.monitor;
		if(!objMonitor) return;


		if(objMonitor.state != "connected" ||
			!objMonitor.initialized ||
			objMonitor.getCellsArray().length == 0)
		{
			var strMessage;
			if(objMonitor.state == "disconnected")
				strMessage = "DISCONNECTED";
			else if (objMonitor.state == "not_licensed")
				strMessage = "Monitor isn't licensed";
			else if(!objMonitor.initialized || objMonitor.getCellsArray().length == 0)
				strMessage = "Display does not have cells.<br>Use layout template for initializing display";

			var strText = "<table width='100%' height='100%'><tr><td><center><font size='3'>" +
					  strMessage + "</font></center></td></tr></table>";
			var blkMessage = new jsx3.gui.Block("message_block", 0, 0, "100%", "100%", strText);
			blkMessage.setRelativePosition(jsx3.gui.Block.ABSOLUTE);
			blkMessage.setCanDrop(jsx3.Boolean.TRUE);
			//blkMessage.setClassName('drop');

			if(!objMonitor.initialized || objMonitor.getCellsArray().length == 0) {
				blkMessage.setEvent("1;", jsx3.gui.Interactive.DROP);
				blkMessage.subscribe(jsx3.gui.Interactive.DROP, this, "onDropCell");
				//registerDropHandler(blkMessage.getId(), this, this.onDropCell);
			} else {
				blkMessage.setEvent("mx.MATRIX2.getServer().getJSXByName('controlPane').hideMask();mx.vMX.showHideMask(false);",
							  jsx3.gui.Interactive.DROP);
				/*registerDropHandler(blkMessage.getId(), this, function() {
					mx.MATRIX2.getServer().getJSXByName('controlPane').hideMask();
					mx.vMX.showHideMask(false);
				});*/
			}
			this._block.removeChildren();
			this._block.setChild(blkMessage);

			this._block.setCanDrag(jsx3.Boolean.FALSE);
		}
		else {

			//Clear old cell images
			this._block.removeChildren();

			// Set block text (for dragging)
			this._block.setText("ID: " + objMonitor.objid);
			this._block.setFontSize(0);
			this._block.setCanDrag(jsx3.Boolean.TRUE);

			var pos = this._block.getAbsolutePosition(vMX.getWallLayout(this._wc.wall).getContentBlock().getRendered());

			// Create new inner block for "16:9" monitor view
			// Compute position of inner block
			var innerPos = { T:0, L:0, W:0, H:0 };
			if (pos.W / pos.H < 1.778) { // < 16:9 (4:3)
				innerPos.W = pos.W;
				innerPos.H = Math.floor(pos.W * 9/16);
				innerPos.T = Math.floor( (pos.H - innerPos.H) / 2);
			} else {
				innerPos.H = pos.H;
				innerPos.W = Math.floor(pos.H * 16/9);
				innerPos.L = Math.floor( (pos.W - innerPos.W) / 2);
			}
			var imageBlock = new jsx3.gui.Block("image_block", innerPos.L, innerPos.T, innerPos.W, innerPos.H, strText);
			imageBlock.setRelativePosition(jsx3.gui.Block.ABSOLUTE);
			this._block.setChild(imageBlock);

			for(var i = objMonitor.iterator(); i.hasNext();) {
				var img = new jsx3.gui.Image();
				var c = i.next();

				var coeff_x = (pos.W * (innerPos.W / pos.W))/objMonitor.hres;
				var coeff_y = (pos.H * (innerPos.H / pos.H))/objMonitor.vres;

				if(coeff_x != 0 && coeff_y != 0) {
					img.setHeight(c.h * coeff_y).
					setWidth(c.w * coeff_x).
					setTop(c.y * coeff_y).
					setLeft(c.x * coeff_x);
				} else {
					img.setHeight(parseFloat(100*c.h/objMonitor.vres+1) + "%").
					setWidth(parseFloat(100*c.w/objMonitor.hres+1) + "%").
					setTop(100*c.y/objMonitor.vres + "%").
					setLeft(100*c.x/objMonitor.hres + "%");
				}
				img.setName(c.id).
					setRelativePosition(jsx3.gui.Block.ABSOLUTE).
					setOverflow(jsx3.gui.Block.OVERFLOWEXPAND);

				if (c.isVNC()) { // Set pre-defined picture for VNC cell
					//img.setSrc(WallcellView.IMG_VNC);
					img.setClassName("loadingIndication").setSrc(vMX.getCellSnapshot(objMonitor.objid, c.id, "vnc"));
				}
				else if (c.isWeb()) {
					//img.setSrc(vMX.MagnifierCell.IMG_WEB);
					img.setClassName("loadingIndication").setSrc(vMX.getCellSnapshot(objMonitor.objid, c.id, "web"));
				}
				else {
					img.setClassName("loadingIndication").setSrc(vMX.getCamImage(c));
				}

				img.setCanDrop(jsx3.Boolean.TRUE);
				img.setEvent("1;", jsx3.gui.Interactive.DROP);
				img.subscribe(jsx3.gui.Interactive.DROP, this, "onDropCell");
				img.subscribe(jsx3.gui.Interactive.DESTROY, this, "onDestroyImage");
				if (parseInt(c.groupid)) {
					img.setCSSOverride("border: 1px solid red");
					img.setEvent("1;", jsx3.gui.Interactive.JSXCLICK);
					img.unsubscribe(jsx3.gui.Interactive.JSXCLICK, this);
					img.subscribe(jsx3.gui.Interactive.JSXCLICK, this, "onPopupAlert");
				}
				//this._block.setChild(img);
				imageBlock.setChild(img);
				if (!jsx3.GO('magnifier')) {
					registerDropHandler(img.getId(), this, this.onDropCell);
					img.setClassName('drop');
				}
				// Attach reference to MonitorCell object
				img.moncell = c;

				c.subscribe(vMX.MonitorCell.STATE_CHANGED, this, "onCellChanged");
				c.subscribe(vMX.MonitorCell.TYPE_CHANGED, this, "onCellChanged");
				c.subscribe(vMX.WallLayout.UPDATE_SNAPSHOTS, this, "onCellChanged");
			}
		}

		this.onNameDescChanged();
		this._block.repaint();
	};

	WallcellView_prototype.onPopupAlert = function(event) {

		// Ctrl-click or shift-click on APC cell with event
		if (!(event.context.objEVENT.ctrlKey() || event.context.objEVENT.shiftKey()) ||
			!parseInt(event.target.moncell.groupid) ||
			!parseInt(event.target.moncell.eventid))
			return;

		vMX.getEventInfo(event.target.moncell.eventid);
	};

	WallcellView_prototype.onRemoveMonitor = function(event) {
		if(!this._wc.monitor) {
			vMX._log(2, "WallcellView.onRemoveMonitor called for empty cell");
			return;
		}
		this._wc.monitor.unsubscribe(vMX.Monitor.NEW_LAYOUT, this);
		this._wc.monitor.unsubscribe(vMX.Monitor.CELL_REPLACE, this);
		this._wc.monitor.unsubscribe(vMX.Monitor.CELL_RESIZE, this);
		this._wc.monitor.unsubscribe(vMX.Monitor.NAME_DESC_CHANGED, this);
	};

	/**
	 * Handler for jsx3.Interactive.DESTROY
	 */
	WallcellView_prototype.onDestroy = function(event) {
		if(this._wc.monitor) {
			this._wc.monitor.unsubscribe(vMX.Monitor.NEW_LAYOUT, this);
			this._wc.monitor.unsubscribe(vMX.Monitor.CELL_REPLACE, this);
			this._wc.monitor.unsubscribe(vMX.Monitor.CELL_RESIZE, this);
			this._wc.monitor.unsubscribe(vMX.Monitor.NAME_DESC_CHANGED, this);
		}

		this._wc.unsubscribe(vMX.WallCell.BEFORE_REMOVE_MONITOR, this);
	};

	/**
	 * Here we have to unsubscribe from MonitorCell events
	 */
	WallcellView_prototype.onDestroyImage = function(event) {
		var moncell = event.target.moncell;
		if(moncell) {
			moncell.unsubscribe(vMX.MonitorCell.STATE_CHANGED, this);
			moncell.unsubscribe(vMX.MonitorCell.TYPE_CHANGED, this);
			moncell.unsubscribe(vMX.WallLayout.UPDATE_SNAPSHOTS, this);
			delete event.target.moncell;
		}
	};


	WallcellView_prototype.onCellChanged = function(event) {

		if(event && event.target && event.target.id) {
			var cellimg = this.getCellImg(event.target.id);
			if(!cellimg) { vMX._log(2, "Error in WallcellView.onCellStateChanged() : no image found!"); return; }

			switch(event.subject) {
				case vMX.MonitorCell.TYPE_CHANGED:
					var src;
					if (event.target.isVNC())
						src = WallcellView.IMG_VNC;
					else if (event.target.isWeb())
						src = vMX.MagnifierCell.IMG_WEB;
					else
						src = vMX.getCamImage(event.target);

					cellimg.setSrc(src).repaint();
				break;
				case vMX.MonitorCell.STATE_CHANGED:
					if (event.target.isVNC()) break;
					if (event.changed_items.search(/srcObjid|tour_selected_item|all/) > -1)
						cellimg.setClassName("loadingIndication").setSrc(vMX.getCamImage(event.target)).repaint();
					if (event.changed_items.search(/groupid/) > -1) {
						cellimg.setCSSOverride(parseInt(event.target.groupid) ? "border: 1px solid red" : 'border:none;').repaint();
						cellimg.setEvent("1;", jsx3.gui.Interactive.JSXCLICK);
						cellimg.unsubscribe(jsx3.gui.Interactive.JSXCLICK, this);
						cellimg.subscribe(jsx3.gui.Interactive.JSXCLICK, this, "onPopupAlert");
					}
				break;
				case vMX.WallLayout.UPDATE_SNAPSHOTS:
					if(event.layout != this.wallLayout) break;
					cellimg.setClassName("loadingIndication").setSrc(event.url).repaint();
				break;
			}
		}
	};


	WallcellView_prototype.onDropCell = function(event, target) {
		var objid = this._wc.monitor.objid;
		var cellid;

		if (event.type) {
			var objType = event.type;
			if (objType != 'camera' && objType != 'view')
				return;
			cellid = jsx3.GO(target.id).moncell.id;
		} else {
			objType = event.context.objSOURCE.getName();
			var strId = event.context.strDRAGID;
			if(event.target && event.target.moncell) {
				cellid = event.target.moncell.id;
			}
		}


		if (cellid && parseInt(this._wc.monitor.getCellById(cellid).groupid) && (
			objType == 'camera' ||
			objType == 'cell_dev_type' ||
			objType == 'matrixWitnesses' ||
			objType == 'screenshot')) {

			alert(__("Cannot drop device on APC cell"));
			return;
		}

		switch(objType) {
			case 'image_spl_h':
				vMX.splitMonitor(objid, {rows:2, cols:1});
				break;
			case 'image_spl_v':
				vMX.splitMonitor(objid, {rows:1, cols:2});
				break;
			case 'image_spl_c':
				vMX.splitMonitor(objid, {rows:2, cols:2});
				break;
			case 'image_spl_3x2':
				vMX.splitMonitor(objid, {rows:2, cols:3});
				break;
			case 'image_spl_4x3':
				vMX.splitMonitor(objid, {rows:3, cols:4, exclusive:true});
				break;
			case 'image_popup':
				if (vMX.checkMonitorInSeveralWalls(objid)) {
					alert(__("Monitor can not be assigned for alerts because it is shared between multiple Video Walls"));
					break;
				}
				vMX.splitMonitor(objid, {rows:2, cols:3, apc:true});
				break;
			case 'image_vnc':
				if (parseInt(this._wc.monitor.getCellById(cellid).groupid)) {
					alert("Cannot drop VNC item on APC cell");
					break;
				}
				vMX.assignURL(objid, cellid, "vnc://");
				break;
			case 'url_control':
				vMX.assignURL(objid, cellid, vMX.MagnifierCell.PrevWebURL, "Play", 0);
				break;
			case 'camera':
				if(!event.cred('L')) {
					vMX._log(4, "Live credential is missing: cell" + cellid + ", objid: " + event.obj);
					alert("Not sufficient credentials to play the video stream.\n Please contact your system administrator.");
				} else {
					vMX.assignStream(event.obj, objid, cellid);
				}
				break;
			case 'matrixWitnesses':
			case 'screenshot':

				var eLog2 = jsx3.GO("eLog2Root");
				var dialog = eLog2.getDescendantOfName('eLog2dialog');
				var eventId = dialog.getAttribute("eventId");
				var utc_from = dialog.getAttribute("utctime_from");
				var utc_to = dialog.getAttribute("utctime_to");
				var t_zone = dialog.getAttribute("t_zone");
				var t_dst = dialog.getAttribute("t_dst");
				var objDev;

				if(objType == 'screenshot') {
					objDev = resourceTree.getObject(dialog.getAttribute("objId"));

					// Check archive credentials
					if(!objDev.cred('A')) {
						vMX._log(4, "Archive credential is missing: cell" + cellid + ", srcObjid: " + objDev.obj);
						alert("Not sufficient credentials to play the video stream");
						break;
					}

					if(objDev.type == "camera") {
						vMX.assignStream(objDev.obj, objid, cellid,
									{ 'from':utc_from, 'to':utc_to, 'pos':utc_from }
						);
					}
					else{
						alert(__("Cannot drop non-camera object into cell!"));
					}
				}
				else {
					var witnessesDoc = dialog.getDescendantOfName('matrixWitnesses').getXML();
					var objId = witnessesDoc.selectSingleNode("//record[@jsxid='"+strId+"']").getAttribute("objid");
					objDev = resourceTree.getObject(objId);
					if(objDev.type == "camera"){

						// Check archive credentials
						if(!objDev.cred('A')) {
							vMX._log(4, "Archive credential is missing: cell" + cellid + ", srcObjid: " + objDev.obj);
							alert("Not sufficient credentials to play the video stream");
							break;
						}

						vMX.assignStream(objDev.obj, objid, cellid,
								{ 'from':utc_from, 'to':utc_to, 'pos':utc_from }
						);
					}
					else{
						alert(__("Cannot drop non-camera object to Magnifier cell!"));
					}
				}
				break;
			case 'image_tour':
				if(!cellid) break;
				if (parseInt(this._wc.monitor.getCellById(cellid).groupid)) {
					alert(__("Cannot drop touring on APC cell"));
					break;
				}
				vMX.setTouring(objid, cellid, 15, new jsx3.util.List());
				break;
			case 'image_elog2':
			case 'image_elog2_detailed':
				vMX.assignURL(objid, cellid, vMX.MonitorCell.ALERTLOG_URL);
				break;
            case 'view':
                var data;
                $.ajax({
                    data: {'function': 'getAttribute', 'obj': event.obj, 'attribute': 'DATA'},
					async: false,
                    success: function(json) {
                        data = $.parseJSON(json.value);
                    }
                });

                if (data == null || data.content == null || data.content.contents == null) {
            	    appLogger.warn("Null value in view. Cannot load");
            	    break;
            	}

                try {
                    var arrCells = this._wc.monitor.deserializeFromJson(data.content.contents);
                } catch(e) {
                    var msg = e.toSource ? e.toSource() : e.message;
                    appLogger.error("Error deserializing view: " + msg);
                    break;
                }
                vMX.splitMonitor(this._wc.monitor.objid, arrCells);
                break;
		}

		this.getWallLayout().onDropWall(event);
	};


	WallcellView_prototype.onDoubleClick = function(event) {

		if(!this._wc.monitor)
			vMX._log(2, "handleOnDoubleClick called for empty cell");
		else if(this._wc.monitor.state == "disconnected")
			alert("Connection with Display Server was lost");
		else if(this._wc.monitor.state == "not_licensed")
			alert("Monitor isn't licensed");
		else if(this._wc.monitor.getCellsArray().length == 0)
			alert("Display does not have cells.\nUse layout template for initializing display");
		else {
			// If no problems detected, show Magnifier

			this.wallLayout.raiseMagnifier(this);
		}
	};

	}
);
