/*******************************************************************************************
 *
 *       vMX.WallLayout class - visual representation of vMX Wall object
 *
 ******************************************************************************************/


jsx3.lang.Class.defineClass(
	"mx.vMX.WallLayout", jsx3.lang.Object, [mx.vMX.CellContainer], function(WallLayout, WallLayout_prototype) {

	var vMX = mx.vMX;


	WallLayout.QUERY_SNAPSHOTS_TIMEOUT = 15000;

	WallLayout.SELECTOR_HEIGHT = 30;

	WallLayout.SNAPS_PER_TURN = 5;

	// event identifier
	WallLayout.UPDATE_SNAPSHOTS = "vmxupdatesnapshots";


	/**
	 * Constuction
	 */
	WallLayout_prototype.init = function(wall, tab) {
		this.wall = wall;
		this.tab = tab;
		this._cells = new jsx3.util.List;

		// Set dirty state
		this._dirty = true;

		// Was layout painted at least once
		this._painted = false;

		// Manage snapshots refreshing
		this.querySnapshotsEnabled = false;
		this.querySnapshotsId = 0;

		// Create wall content block as a child of the tab block
		var contentBlock = new jsx3.gui.Block("content_block_"+wall.objid, 0, WallLayout.SELECTOR_HEIGHT, "100%", "100%");
		contentBlock.setRelativePosition(jsx3.gui.Block.ABSOLUTE);
		contentBlock.setCanDrop(jsx3.Boolean.TRUE);
		contentBlock.setEvent("1", jsx3.gui.Interactive.DROP);
		this.tab.getContentChild().setEvent("mx.vMX.showHideMask(false);jsx3.gui.Interactive.hideSpy();", jsx3.gui.Interactive.DROP);
		this.tab.getContentChild().setChild(contentBlock);
		this.contentBlock = contentBlock;

		// Modify tab item appearance
		tab.setAttribute("wallid", wall.objid);
		tab.setAttribute("show_cell_headers", "0");

		// Create instance of vMX.SpecialPainting
		this.objPainting = new vMX.SpecialPainting(this.contentBlock);

		// Subscribe to events published by corresponding vMX.Wall object
		this.wall.subscribe(vMX.Wall.NEW_LAYOUT, this, "onNewLayout");
		this.wall.subscribe(vMX.Wall.LAYOUT_CHANGED, this, "onLayoutChanged");
		this.wall.subscribe(vMX.Wall.SHOW_HEADERS_CHANGED, this, "onHeadersChanged");
		this.wall.subscribe(vMX.Wall.CRED_CHANGED, this, "onCredChanged");

		// Destructor
		this.contentBlock.subscribe(jsx3.gui.Interactive.DESTROY, this, "onDestroy");
		// Top-level 'onDrop' processing
		this.contentBlock.subscribe(jsx3.gui.Interactive.DROP, this, "onDropWall");
	};

	WallLayout_prototype.getContentBlock = function() {
		return this.contentBlock;
	};

	WallLayout_prototype.getTab = function() {
		return this.tab;
	};

	WallLayout_prototype.getWall = function() {
		return this.wall;
	};

	WallLayout_prototype.getMagnifier = function() {
		return this.magnifier;
	};

	WallLayout_prototype.setMagnifier = function(m) {
		this.magnifier = m;
	};


	WallLayout_prototype.getCellAt = function(i, j) {
		for (var iter = this.iterator(); iter.hasNext();) {
			var wcview = iter.next();
			if (wcview.getBlock().getName() == "Wallcell " + i + "." + j)
				return wcview;
		}
		return null;
	};


	WallLayout_prototype.isVisible = function() {
		return this.contentBlock && this.contentBlock.getVisibility() == jsx3.gui.Block.VISIBILITYVISIBLE;
	};

	WallLayout_prototype.setDirty = function() {
		this._dirty = true;
	};

	WallLayout_prototype.getDirty = function() {
		return this._dirty;
	};


	WallLayout_prototype.recalcLayout = function(width, height, padding) {

		var arrCellPos = new Array(this.wall.rows);

		// cell content width and height + 2 * border_width
		var cellW = parseInt((width - (this.wall.cols + 1) * padding) / this.wall.cols);
		var cellH = parseInt((height - (this.wall.rows + 1) * padding) / this.wall.rows);

		for (var i = 0; i < this.wall.rows; i++)
			for (var j = 0; j < this.wall.cols; j++) {
				if (!arrCellPos[i]) arrCellPos[i] = new Array(this.wall.cols);

				arrCellPos[i][j] = {
					L : padding + j * (cellW + padding),
					T : padding + i * (cellH + padding),
					W : cellW - 2,
					H : cellH - 2
				};
			}

		return arrCellPos;
	};


	/**
	 * Clears layout view if already exists and paints it from scratch
	 */
	WallLayout_prototype.paintLayout = function() {

		if (this.magnifier) this.magnifier.doClose();
		this.clearCells();
		this.contentBlock.removeChildren();
		this.contentBlock.setEvent("mx.vMX.getServer().getJSXByName('controlPane').hideMask();mx.vMX.showHideMask(false);",
			jsx3.gui.Interactive.DROP);

		this.contentBlock.setDynamicProperty("jsxbgcolor", "@vmx_bg");

		// Set border css properties
		this.contentBlock.setClassName("vmx_layout");

		// Compute wallcell geometry
		var dims = this.contentBlock.getParent().getAbsolutePosition();
		var padding = 3;
		var cellpos = this.recalcLayout(dims.W, dims.H, padding);

		for (var i = 0; i < this.wall.rows; i++)
			for (var j = 0; j < this.wall.cols; j++) {
				var wallcell = this.wall.getCellAt(i, j);
				var blkWC = new jsx3.gui.Block("wallcell", cellpos[i][j].L, cellpos[i][j].T, cellpos[i][j].W, cellpos[i][j].H);

				blkWC.setBackgroundColor(vMX.getServer().getJSXByName('block_main').getBackgroundColor());
				blkWC.setRelativePosition(jsx3.gui.Block.ABSOLUTE);
				blkWC.setOverflow(jsx3.gui.Block.OVERFLOWHIDDEN);
				this.contentBlock.setChild(blkWC);

				var wcview = new vMX.WallcellView(wallcell, blkWC);
				wcview.create();
				this.addCell(wcview);
			}

		this.tab.repaint();
		this.contentBlock.getParent().repaint();

		// Apply credentials only if layout is visible!
		if (this.isVisible()) {
			this.applyCredentials();
		}

		// Unset 'dirty' flag
		this._dirty = false;

		// Set 'painted' flag
		this._painted = true;
	};


	/**
	 * Recalculates position of wallcell views and magnifier and repaints them.
	 * Do not touch anything else
	 */
	WallLayout_prototype.updateLayout = function() {

		var dims = this.contentBlock.getAbsolutePosition();
		var padding = 3;
		var cellpos = this.recalcLayout(dims.W, dims.H, padding);

		for (var i = 0; i < this.wall.rows; i++)
			for (var j = 0; j < this.wall.cols; j++) {
				var wcview = this.getCellAt(i, j);
				if (!wcview) {
					vMX._log(2, "failed to get wallcell view at " + i + "." + j);
					continue;
				}
				var block = wcview.getBlock();
				block.setLeft(cellpos[i][j].L);
				block.setTop(cellpos[i][j].T);
				block.setWidth(cellpos[i][j].W);
				block.setHeight(cellpos[i][j].H, true);

				// update wallcell view content
				if (wcview.getWallcell().monitor) wcview.updateView();
			}

		var magnifier = this.getMagnifier();
		if (magnifier) {
			this.objPainting.setBaseRect(magnifier.getWCView().getBlock().
				getAbsolutePosition(this.contentBlock.getRendered()));
			magnifier.onMoveResize();
		}

		this._dirty = false;
	};

	/**
	 * Creates vMX.Magnifier object and displays it along with surrounding vector shapes
	 */
	WallLayout_prototype.raiseMagnifier = function(wcview) {
		var objP = this.objPainting;
		objP.setBaseRect(wcview.getBlock().getAbsolutePosition(this.contentBlock.getRendered()));
		objP.lightOn();

		var magnifier = new vMX.Magnifier(wcview);
		this.setMagnifier(magnifier);

		// Set handler for 'onmouseover'
		var darkblock = this.contentBlock.getChild("darkblock");
		darkblock.setCanSpy(jsx3.Boolean.TRUE);
		darkblock.doSpyOver = function(objEvent) {
			this.doEvent(jsx3.gui.Interactive.JSXMOUSEOVER, {objEVENT:objEvent});
		};
		darkblock.subscribe(jsx3.gui.Interactive.JSXMOUSEOVER, magnifier, "checkPreviewSnapshots");

		window.setTimeout(function() {
			magnifier.repaint();
			//objP.paintTraceLines(magnifier.getDialog());
		}, 100);
	};

	/**
	 * Shows/hides transparent 'mask' block
	 */
	WallLayout_prototype.showHideMask = function(bShow) {

		if (this.contentBlock.getDescendantOfName(this.contentBlock.getId() + "_mask")) {
			// Mask is already shown and cannot be removed
			return;
		}

		// show/hide mask for darkblock if visible
		var darkblock = this.objPainting.darkblock;
		if (darkblock && darkblock.getVisibility() == jsx3.gui.Block.VISIBILITYVISIBLE)
			bShow ? darkblock.showMask() : darkblock.hideMask();
		else {
			for (var iter = this.iterator(); iter.hasNext();) {
				var wcview = iter.next();
				if (!wcview.getWallcell().monitor) {
					if (bShow)
						wcview.getBlock().showMask();
					else
						wcview.getBlock().hideMask();
				}
			}
		}
	};


	/**
	 * Updates layout view acording to the wall creadentials
	 */
	WallLayout_prototype.applyCredentials = function() {

		var blkConfMenu = vMX.getServer().getJSXByName("menuconfblock");

		// Show blocking mask if user isn't allowed to manipulate monitors
		if (this.wall.cred.indexOf('M') < 0) {
			if (this.magnifier) this.magnifier.doClose();
			this.contentBlock.showMask("");
			var blkMask = this.contentBlock.getDescendantOfName(this.contentBlock.getId() + "_mask");
			blkMask.setDynamicProperty("jsxcursor", "@No").repaint();
			hideLayoutControls();
		} else {
			this.contentBlock.hideMask();
			showLayoutControls();
		}

		// Enable/disable Configuration Menu 'Save', 'Manage' and 'Load' items
		blkConfMenu.getDescendantOfName("conf_menu_save").setEnabled(this.wall.cred.indexOf('m') < 0 ? 0 : 1, true);
		blkConfMenu.getDescendantOfName("conf_menu_manage").setEnabled(this.wall.cred.indexOf('m') < 0 ? 0 : 1, true);
		blkConfMenu.getDescendantOfName("conf_menu_load").setEnabled(this.wall.cred.indexOf('r') < 0 ? 0 : 1, true);
		blkConfMenu.getDescendantOfName("conf_menu_new").setEnabled(this.wall.cred.indexOf('M') < 0 ? 0 : 1, true);
	};


	/**
	 * Enables/disables updating cell images for all monitors in the wall.
	 * Only WallLayout.SNAPS_PER_TURN snapshots are loaded by one call to
	 * 'updateSnapshots'.
	 */
	WallLayout_prototype.enableSnapshots = function(bEnable) {
		vMX._log("enableSnapshots: " + bEnable);

		if (bEnable && !this.querySnapshotsEnabled) {
			var self = this;
			this.querySnapshotsId = window.setTimeout(function() {
				self.updateSnapshots();
			},
				WallLayout.QUERY_SNAPSHOTS_TIMEOUT);
		}
		else if (!bEnable && this.querySnapshotsEnabled && this.querySnapshotsId != 0) {
			window.clearTimeout(this.querySnapshotsId);
		}

		this.querySnapshotsEnabled = bEnable;
	};


	/**
	 * Is called for timeout. Loops around the cells of the corresponding wall object
	 * and refreshes camera snapshots for them. When snaphots are downloaded and cached in
	 * temporary native 'Image' objects, and event is published, so both wallcell views
	 * and magnifier cell which are subscribed to it, can handle it and update their
	 * images.
	 * Warning! Found a caching issue in IE: it doesn't free memory from cached images
	 * so snapshots refreshing should be disabled for IE.
	 */
	WallLayout_prototype.updateSnapshots = function(bRestarted) {

		// Run method after a period of time if snapshots refreshing enabled
		if (this.querySnapshotsEnabled && !bRestarted) {
			var self = this;
			this.querySnapshotsId = window.setTimeout(function() {
				self.updateSnapshots();
			},
				WallLayout.QUERY_SNAPSHOTS_TIMEOUT);
		}

		var arrSnapData = []; // Array of cells and their image urls
		var cnt = 0;
		var bHaveCellsToUpdate = false;
		WallLayout._CNT = 0;

		O: for (var wci = this.wall.iterator(); wci.hasNext();) {
			var wcell = wci.next();
			if (!wcell.monitor) continue;

			for (var ci = wcell.monitor.iterator(); ci.hasNext();) {
				if (cnt >= WallLayout.SNAPS_PER_TURN) break O;
				var cell = ci.next();
				if (cell.empty())
					continue;
				else
					bHaveCellsToUpdate = true;
				if (cell.snapshot_updated) continue;


				cell.snapshot_updated = 1;
				cnt++;
				var url;
				if (cell.isWeb() || cell.isVNC()) {
					url = vMX.getCellSnapshot(
						    wcell.monitor.objid,
						    cell.id,
						    (cell.isWeb() ? "web" : "vnc"));
				}
				else {
					url = vMX.getCamImage(cell.srcObjid, cell.isArchive()?cell.arc_pos:0);;
				}

				var self = this;
				var img = new Image();

				arrSnapData.push([cell, url])

				img.onload = img.onerror = function() {
					try {

						if (++WallLayout._CNT == cnt) {
							for (i = 0; i < arrSnapData.length; i++) {
								arrSnapData[i][0].publish({'subject':WallLayout.UPDATE_SNAPSHOTS, 'layout':self,
									'target':arrSnapData[i][0], 'url':arrSnapData[i][1]});
							}
						}
					} catch(e) {
						vMX._log(3, "Exception in onload: " + e.description);
					}
				};

				img.src = url;
			}
		}

		// If all available live snapshots already updated, clear cell marks and start from the very beginning
		if (cnt < WallLayout.SNAPS_PER_TURN) {
			//vMX._log("Updated: " + cnt + ". " + "Need to clear 'updated' cell marks");
			for (var wci = this.wall.iterator(); wci.hasNext();) {
				var monitor = wci.next().monitor;
				if (!monitor) continue;
				for (var ci = monitor.iterator(); ci.hasNext();)
					delete ci.next().snapshot_updated;
			}
			// do recursive call if no snapshots were updated
			if (cnt == 0 && bHaveCellsToUpdate) this.updateSnapshots(true);
		}
	};


	WallLayout_prototype.newLayout = function() {
		var delay = 0;
		for (var iter = this.iterator(); iter.hasNext();) {
			var mon = iter.next().getWallcell().monitor;
			if (mon && mon.state != "not_licensed") {
				window.setTimeout("mx.vMX.splitMonitor(" + mon.objid +
					", {'rows':1, 'cols':1})", delay);
				delay += 300;
			}
		}
	};

	WallLayout_prototype.hide = function() {
		this.contentBlock.setVisibility(jsx3.gui.Block.VISIBILITYHIDDEN, true);
	};


	WallLayout_prototype.show = function() {
		this.contentBlock.setVisibility(jsx3.gui.Block.VISIBILITYVISIBLE, true);
	};


	/************************************************************************
	 *                           Event Handlers                             *
	 * **********************************************************************/

	/**
	 * Is called manually from WallCellView and MagnifierCell 'onDrop'
	 * handlers for top-level event processing
	 */
	WallLayout_prototype.onDropWall = function(event) {
		vMX.getServer().getJSXByName('controlPane').hideMask();
		vMX.showHideMask(false);
		jsx3.gui.Interactive.hideSpy(); // Hide speglass for camera snapshots in Resource tree
	};

	WallLayout_prototype.onMagnifierClose = function(event) {
		this.magnifier = null;
	};

	/**
	 * Destructor
	 */
	WallLayout_prototype.onDestroy = function(event) {
		vMX._log("LAYOUT " + this.wall.objid + ": ONDESTROY");
		this.wall.unsubscribe(vMX.Wall.NEW_LAYOUT, this);
		this.wall.unsubscribe(vMX.Wall.LAYOUT_CHANGED, this);
		this.wall.unsubscribe(vMX.Wall.SHOW_HEADERS_CHANGED, this);
		this.wall.unsubscribe(vMX.Wall.CRED_CHANGED, this);

		// Remove WallLayout-specific attributes from tab object
		this.tab.removeAttribute("show_cell_headers").removeAttribute("wallid");
	};

	/**
	 * If layout is changed, it must be repainted
	 */
	WallLayout_prototype.onNewLayout = function(event) {
		this.paintLayout();
	};

	WallLayout_prototype.onLayoutChanged = function(event) {
		var strChanged = event.strChanged;
		vMX._log("LAYOUT CHANGED: " + strChanged);
		if (strChanged.search(/name/) > -1) {
			var newText = this.wall.name + " [" + this.wall.objid + "]";
			this.tab.setAttribute("initText", newText);
			if (this.isVisible()) {
				mx.CT.setTabDesignSelected(this.tab);
			}
			else {
				mx.CT.setTabDesignDeselected(this.tab);
			}
		}
		if (strChanged.search(/desc/) > -1) {
			//TODO: Reflect in some way
		}
		if (strChanged.search(/layout/) > -1) {
			this.paintLayout();
		}
	};

	WallLayout_prototype.onCredChanged = function(event) {
		vMX._log("applying new credentials to wall layout " + this.wall.objid);
		if (this.isVisible()) this.applyCredentials();
	};

	/**
	 * Is called when wall content block is resized. Here we need to repaint all wall cell views
	 * and magnifier trace region
	 */
	WallLayout_prototype.onLayoutResize = function(event) {

		if (this.isVisible())
			this.updateLayout();
		else
			this._dirty = true;
	};

	/**
	 * Update view of 'toggle headers' button
	 */
	WallLayout_prototype.onHeadersChanged = function(event) {
		var bShow = event.target.showHeaders;
		var curValue = this.tab.getAttribute("show_cell_headers") == "1";
		if (curValue != bShow) {
			this.tab.setAttribute("show_cell_headers", bShow ? "1" : "0");
			//this.repaintHeadersButton();
		}
	};

	WallLayout_prototype.repaintHeadersButton = function() {
		var bShow = Number(this.tab.getAttribute("show_cell_headers"));
		var imgSrc = bShow ? "images/design1/vMX/hide_headers.png" : "images/design1/vMX/show_headers.png";
		this.tab.setText(this.tab.getText().replace(/src="images\/design1\/vMX\/(show|hide)_headers.png"/i,
			function(m, g) {
				return 'src="' + imgSrc + '"'
			}), true);
	};


	/**
	 * Fires when user press 'show/hide headers' imagebutton located on the tab item
	 * It iterates through all monitor cells that belongs to current wall and
	 * toggles their text headers
	 */
	WallLayout_prototype.onShowCellHeaders = function() {

		var bShow = this.tab.getAttribute("show_cell_headers") == "1";

		var assignData = [];
		for (var wci = this.wall.iterator(); wci.hasNext();) {
			var wcell = wci.next();
			var mon = wcell.monitor;
			if (!mon) continue;
			for (var ci = mon.iterator(); ci.hasNext();) {
				var c = ci.next();

				// Do not touch tour and vnc cells
				if (!c.isVideo()) continue;
				// Also exclude cells restricted by license
				if (c.text.search(/License limit/i) > -1) continue;

				var objC = resourceTree.getObject(c.srcObjid);
				var assignStruct = {
					jsxid : jsx3.xml.CDF.getKey(), cellid : c.id, monid : mon.objid, state : c.state,
					scale : c.scale, show_text : (bShow ? "" : (objC ? objC.name : "No device")),
					arc_from : c.arc_from, arc_to : c.arc_to, arc_pos : c.arc_pos, srcObjid : c.srcObjid
				};
				if (assignStruct.state == "Invalid") assignStruct.state = "Play";

				assignData.push(assignStruct);
			}
		}

		if (assignData.length > 0) vMX.assignStream(assignData);//vMX.controlCells(ctrlData);
	};

	/**
	 * Fires when this wall layout is activated by pressing tab button ot in any othe way
	 */
	WallLayout_prototype.onActivate = function(objPrevLayout) {

		vMX._log("[LAYOUT " + this.wall.objid + "] - Activate");
		// Retrieve list of layouts for the current wall
		vMX.getLayoutList(this.wall.objid);

		if (!objPrevLayout) {

			//hideElementsWithName("url_control");
			hideElementsWithName("image_ge");
			showElementsWithName("image_popup");
			showElementsWithName("image_vnc");

			// Replace default code for configuration management routines, save old values in static variables
			WallLayout.CONFSAVEDIALOG = LSC.saveDialog;
			WallLayout.CONFLOADDIALOG = LSC.loadDialog;
			WallLayout.CONFMANAGEDIALOG = LSC.manageDialog;
			LSC.saveDialog = vMX.confSaveDialog;
			LSC.loadDialog = vMX.confLoadDialog;
			LSC.manageDialog = vMX.confManageDialog;

			// Replace handler for 'File->New'
			var confMenuNew = vMX.getServer().getJSXByName("menuconfblock").getChild("conf_menu_new");
			WallLayout.CONFMENUNEW = confMenuNew.getEvent(jsx3.gui.Interactive.EXECUTE);
			confMenuNew.setEvent("LSC.hideConfMenu();mx.vMX.clearWallLayout();", jsx3.gui.Interactive.EXECUTE);

			// Close save-load-manage dialogs
			mx.Dialog.close("save");
			mx.Dialog.close("load");
			mx.Dialog.close("manage");

			// hide'File' menu
			LSC.hideConfMenu();

            // Enable DROP from vMX tab to Resource Tree
            mx.MATRIX2.getServer().getJSXByName("resource_wrapper").setEvent("mx.vMX.onWallcellDrop(objSOURCE);", jsx3.gui.Interactive.DROP);
		}

		if (this._dirty) {
			if (!this._painted) {
				this.paintLayout();
			} else {
				this.updateLayout();
			}
		}

		this.applyCredentials();

		this.sync({'activate':(objPrevLayout ? 0 : 1), 'group':1, 'archive':1, 'timeline':1});

		// Turn on updating camera snapshots for entire wall
		this.enableSnapshots(true);

		vMX.refreshState();
	};


	/**
	 * Fires when another wall layout is deactivated and focus is passsed to
	 * another layout or Matrix2 tab
	 */
	WallLayout_prototype.onDeactivate = function(objNewLayout) {
		vMX._log("[LAYOUT " + this.wall.objid + "] - Deactivate");

		if (objNewLayout == null) { // Focus passed to a Matrix2 tab
			var blkConfMenu = vMX.getServer().getJSXByName("menuconfblock");

			// Restore default values for configuration management routines
			if (WallLayout.CONFSAVEDIALOG) {
				LSC.saveDialog = WallLayout.CONFSAVEDIALOG;
				delete WallLayout.CONFSAVEDIALOG;
			}
			if (WallLayout.CONFLOADDIALOG) {
				LSC.loadDialog = WallLayout.CONFLOADDIALOG;
				delete WallLayout.CONFLOADDIALOG;
			}
			if (WallLayout.CONFMANAGEDIALOG) {
				LSC.manageDialog = WallLayout.CONFMANAGEDIALOG;
				delete vMX.CONFMANAGEDIALOG;
			}

			// Unhide cell menu items
			//showElementsWithName("url_control");

			if (!jsx3.GO('ge_iframe')) {
				showElementsWithName("image_ge");
			}
			hideElementsWithName("image_popup");
			hideElementsWithName("image_vnc");

			// Now we have to enable conf menu items if they were disabled because of credentials lack
			blkConfMenu.getChild("conf_menu_save").setEnabled(1, true);
			blkConfMenu.getChild("conf_menu_manage").setEnabled(1, true);
			blkConfMenu.getChild("conf_menu_load").setEnabled(1, true);

			// Restore handler for 'File->New'
			var confMenuNew = vMX.getServer().getJSXByName("menuconfblock").getChild("conf_menu_new");
			confMenuNew.setEvent(WallLayout.CONFMENUNEW, jsx3.gui.Interactive.EXECUTE);
			confMenuNew.setEnabled(1, true);

			// show layout controls pane if it was hidden according to credentials
			showLayoutControls();

			// Close save-load-manage dialogs
			mx.Dialog.close("save");
			mx.Dialog.close("load");
			mx.Dialog.close("manage");

			// hide'File' menu
			LSC.hideConfMenu();

            // Disable DROP to Resource Tree
            mx.MATRIX2.getServer().getJSXByName("resource_wrapper").removeEvent(jsx3.gui.Interactive.DROP);

			// Bring Group control and timeline to valid state
			this.sync({'deactivate' : 1});
		}

		this.enableSnapshots(false);
		this.showHideMask(false);
	};


	/**
	 * Set up state of matrix2 GUI elements which are reused by vMX: menus, configuration dialogs,
	 * group control, timeline. Every time wall layout is activated it chanhes behavior of these elemtns,
	 * and when matrix2 tab is activated, layout must synchronize them with current matrix2 state.
	 */
	WallLayout_prototype.sync = function(target) {
		vMX._log("Sync: " + vMX.dump_obj(target));

		var objControl = vMX.getControlInstance();

		// Can be called only when wall layout is visible or when it is deactivated
		if (!this.isVisible() && !target['deactivate']) return;

		// Define vars for 'activate', 'deactivate' and 'group' targets
		if (target['activate'] || target['deactivate'] || target['group']) {
			var btnPlay = objControl.getDescendantOfName("play_pause");
			var btnForward = objControl.getDescendantOfName("catchup_stepforward");
			var btnBackward = objControl.getDescendantOfName("revert_stepback");
		}

		// Matrix2 tab is active
		if (target['deactivate']) {
			// Restore 'onNewImage' callback
			if (WallLayout.GC_ONNEWIMAGE != null) {
				objControl.controls.onNewImage = WallLayout.GC_ONNEWIMAGE;
				delete WallLayout.GC_ONNEWIMAGE;
			}

			// Restore default playbutton handler
			if (WallLayout.GC_ONGROUPPLAY != null) {
				btnPlay.setEvent(WallLayout.GC_ONGROUPPLAY, jsx3.gui.Interactive.EXECUTE);
				btnPlay.unsubscribeAll(jsx3.gui.Interactive.EXECUTE);
				delete WallLayout.GC_ONGROUPPLAY;
			}

			// Restore default handlers for 'forward' and 'backward' buttons
			if (WallLayout.GC_ONFORWARD != null) {
				btnForward.setEvent(WallLayout.GC_ONFORWARD, jsx3.gui.Interactive.EXECUTE);
				btnForward.unsubscribeAll(jsx3.gui.Interactive.EXECUTE);
				delete WallLayout.GC_ONFORWARD;
			}
			if (WallLayout.GC_ONBACKWARD != null) {
				btnBackward.setEvent(WallLayout.GC_ONBACKWARD, jsx3.gui.Interactive.EXECUTE);
				btnBackward.unsubscribeAll(jsx3.gui.Interactive.EXECUTE);
				delete WallLayout.GC_ONBACKWARD;
			}

			// Restore Timeline callback
			// Assign own callback to the Timeline object
			var timeline = mx.TIMELINE.timeline;
			if (timeline && WallLayout.TL_SELECTEDRANGE != null) {
				timeline.onSelectionChange = WallLayout.TL_SELECTEDRANGE;
				delete WallLayout.TL_SELECTEDRANGE;
			}
			if (timeline && WallLayout.TL_REQUESTXML != null) {
				timeline.onGetData = WallLayout.TL_REQUESTXML;
				delete WallLayout.TL_REQUESTXML;
			}

			if (timeline && WallLayout.TL_ONTIMECHANGE != null) {
				timeline.onTimeChange = WallLayout.TL_ONTIMECHANGE;
				delete WallLayout.TL_ONTIMECHANGE;
			}


			// force 'GroupControl' object to restore its state according to the current players group
			objControl.controls.setAccess(true,	objControl.controls.isLiveModeEnabled, objControl.controls.isArchiveModeEnabled);
			objControl.controls.checkTimeline();
		}


		// Wall tab is active, do sync with magnifier cells state
		if (target['activate']) {

			objControl.getDescendantOfName("record").setEnabled(0, true);
			objControl.getDescendantOfName("saveImage").setEnabled(0, true);

			btnPlay.setDisabledImage("images/design1/player_controls/play_disabled.png");
			btnPlay.setOverImage(null);
			btnForward.setImage("images/design1/player_controls/forward.png");
			btnBackward.setImage("images/design1/player_controls/back.png");
			btnForward.setDisabledImage("images/design1/player_controls/forward_disabled.png");
			btnBackward.setDisabledImage("images/design1/player_controls/back_disabled.png");
			btnForward.setOverImage("images/design1/player_controls/forward_hover.png");
			btnBackward.setOverImage("images/design1/player_controls/back_hover.png");
			btnPlay.setEnabled(0).repaint();
			btnBackward.setEnabled(0).repaint();
			btnForward.setEnabled(0).repaint();

			objControl.getDescendantOfName("date").setText("", true);
			objControl.getDescendantOfName("time").setText("", true);

			// Assign own callback to the Timeline object
			var timeline = mx.TIMELINE.timeline;
			if (timeline) {
				WallLayout.TL_SELECTEDRANGE = timeline.onSelectionChange;
				WallLayout.TL_REQUESTXML = timeline.onGetData;
				WallLayout.TL_ONTIMECHANGE = timeline.onTimeChange;
				timeline.onSelectionChange = vMX.onSetTimelineRange;
				timeline.onGetData = vMX.onRequestXML;
				timeline.onTimeChange = vMX.onTimeChange;
			}

			// Reassign 'onNewImage' callback to disable updating of 'date' and 'time' fields
			if (!WallLayout.GC_ONNEWIMAGE) {
				WallLayout.GC_ONNEWIMAGE = objControl.controls.onNewImage;
				objControl.controls.onNewImage = function() {
				};
			}

			// Backup handler for click on 'play_pause'
			if (!WallLayout.GC_ONGROUPPLAY) {
				WallLayout.GC_ONGROUPPLAY = btnPlay.getEvent(jsx3.gui.Interactive.EXECUTE);
				btnPlay.setEvent("1", jsx3.gui.Interactive.EXECUTE);
			}

			// Backup handlers for 'forward' and 'backward' buttons
			if (!WallLayout.GC_ONFORWARD) {
				WallLayout.GC_ONFORWARD = btnForward.getEvent(jsx3.gui.Interactive.EXECUTE);
				btnForward.setEvent("1", jsx3.gui.Interactive.EXECUTE);
			}
			if (!WallLayout.GC_ONBACKWARD) {
				WallLayout.GC_ONBACKWARD = btnBackward.getEvent(jsx3.gui.Interactive.EXECUTE);
				btnBackward.setEvent("1", jsx3.gui.Interactive.EXECUTE);
			}
		}


		// synchronize 'play_pause' and 'archive' buttons state with the state of magnifier group
		if (target['group']) {
			var prevState = btnPlay.getEnabled();
			var newPlayState, newForwardState, newBackwardState;
			var imgForwardDisabled = "images/design1/player_controls/forward_disabled.png";
			var imgBackwordDisabled = "images/design1/player_controls/back_disabled.png";


			btnPlay.unsubscribeAll(jsx3.gui.Interactive.EXECUTE);
			btnForward.unsubscribeAll(jsx3.gui.Interactive.EXECUTE);
			btnBackward.unsubscribeAll(jsx3.gui.Interactive.EXECUTE);

			if (!this.magnifier) {
				newPlayState = /*newArchiveState = */newForwardState =
					newBackwardState = jsx3.gui.Form.STATEDISABLED;
			}
			else {

				btnPlay.subscribe(jsx3.gui.Interactive.EXECUTE, this.magnifier, "groupPlay");
				btnForward.subscribe(jsx3.gui.Interactive.EXECUTE, this.magnifier, "groupJumpForward");
				btnBackward.subscribe(jsx3.gui.Interactive.EXECUTE, this.magnifier, "groupJumpBackward");

				var group = this.magnifier.getGroup();
				if (group.size() == 0) {
					newPlayState = /*newArchiveState = */newForwardState =
						newBackwardState = jsx3.gui.Form.STATEDISABLED;
				}
				else {
					var leader = group.get(0).getMonCell();
					var bLive = leader.arc_from <= 0 && leader.arc_to <= 0;

					if (leader.state == "Play") {
						btnPlay.setTip(__("Pause"));
						btnPlay.setImage('images/design1/player_controls/pause.png');
						newPlayState = jsx3.gui.Form.STATEENABLED;
						newBackwardState = jsx3.gui.Form.STATEENABLED;
						if (bLive) {
							newForwardState = jsx3.gui.Form.STATEDISABLED;
							btnBackward.setTip(__("Quick switch to archive"));
							btnForward.setTip(null);
						}
						else {
							newForwardState = jsx3.gui.Form.STATEENABLED;
							btnBackward.setTip(__("Jump backward on 30 sec"));
							btnForward.setTip(__("Jump forward on 30 sec"));
						}

					}
					else if (leader.state == "Pause") {
						btnPlay.setTip(__("Play"));
						btnPlay.setImage('images/design1/player_controls/play.png');
						newPlayState = jsx3.gui.Form.STATEENABLED;
						if (bLive) {
							newForwardState = jsx3.gui.Form.STATEDISABLED;
							newBackwardState = jsx3.gui.Form.STATEDISABLED;
						}
						else {
							newBackwardState = jsx3.gui.Form.STATEENABLED;
							newForwardState = jsx3.gui.Form.STATEENABLED
						}
					}
					else {
						newPlayState = jsx3.gui.Form.STATEDISABLED;
					}

					// got archive stream, must check 'switchToArchive' button
					//if (!bLive) bLiveChecked = false;
				}
			}

			btnPlay.setEnabled(newPlayState).repaint();
			btnForward.setEnabled(newForwardState).repaint();
			btnBackward.setEnabled(newBackwardState).repaint();
		}

		// update state and position of GroupControl slider
		if (target['archive']) {
			var group = this.magnifier ? this.magnifier.getGroup() : null;
			var leader = group && group.size() ? group.get(0).getMonCell() : null;

			if (leader && leader.arc_from > 0 && leader.arc_pos > 0 && leader.arc_to > 0) {

				objControl.getDescendantOfName('date').setText(mx.TIME.timestamp2date(leader.arc_pos * 1000, 'DD/MM/YYYY'), true);
				objControl.getDescendantOfName('time').setText(mx.TIME.timestamp2date(leader.arc_pos * 1000, 'HH:II:SS'), true);
				mx.TIMELINE.timeline.setTime(leader.arc_pos * 1000, false);
			}
			else {
				objControl.getDescendantOfName('date').setText('', true);
				objControl.getDescendantOfName('time').setText('', true);
			}
		}


		if (target['timeline']) {

			this.checkTimeline();

		}

	};


	/**
	 * Checks whether timeline should be activated or hidden
	 */
	WallLayout_prototype.checkTimeline = function() {

		var show = function(bShow, from, to) {
			if (bShow)
				mx.TIMELINE.showTimeLine('gn');
				if (from && to) {

					mx.TIMELINE.setSelectedInterval(from, to, false, false, 'gn');
					mx.TIMELINE.setTime(from, false, 'gn');
				}
			else
				mx.TIMELINE.hideTimeLine('gn');
		};

		if (!this.magnifier || this.magnifier.getGroup().size() == 0) {
			show(false);
		} else {
			var leader = this.magnifier.getGroup().get(0).getMonCell();
			if (leader.arc_from > 0 && leader.arc_to > 0)
				show(true, leader.arc_from * 1000, leader.arc_to * 1000);
			else {
				show(false);
			}
		}
	};

}
);
