//------------------------------------------------------------------------------------------
// Wall Class. Incapsulates system object called 'Wall' and provides methods for manipulating
// its data, most of them dealing with wall cells.
//------------------------------------------------------------------------------------------

jsx3.lang.Class.defineClass(
	"mx.vMX.Wall",
	jsx3.lang.Object,
	[jsx3.util.EventDispatcher, mx.vMX.CellContainer],
	function(Wall, Wall_prototype) {

		var vMX = mx.vMX;

		Wall.NEW_LAYOUT = "vmxnewwalllayout";
		Wall.LAYOUT_CHANGED = "vmxwalllayoutchanged";
		Wall.DESTROY = "vmxwalldestroy";
		Wall.SHOW_HEADERS_CHANGED = "vmxshowheaderschanged";
		Wall.CRED_CHANGED = "vmxcredchanged";


		Wall_prototype.init = function(nodeWall, cred) {

			this.initialized = false;
			this.state = "";
			this._cells = new jsx3.util.List;

			// Current state of headers
			//var showHeaders = false;

			if (nodeWall instanceof jsx3.xml.Entity) {
				// First convert to resource format
				var wallRes = this.fromXML(nodeWall, cred);
				this.update(wallRes);
				this.initialized = true;
			}
			else if (nodeWall instanceof Object) {
				this.update(nodeWall);  // Already in resource format
			}
		};

		Wall_prototype.update = function(wallRes) {
			try {
				var bNewLayout = false;
				var bLayoutChanged = false;
				var bCredChanged = false;
				var strChanged = "";
				var lstProcessed = new jsx3.util.List;

				var objid = wallRes.obj;
				var name  = wallRes.name;
				var desc  = wallRes.location;
				var rows  = Number(wallRes.attributes["WALLH"]);
				var cols  = Number(wallRes.attributes["WALLW"]);

				// Credentials
				var cred = wallRes.credentials;
				if (this.cred != cred) bCredChanged = true;
				this.cred = cred;

				this.objid = objid;
				if (name != this.name)
					strChanged += "name;";
				if (desc != this.desc)
					strChanged += "desc;";
				if (rows != this.rows || cols != this.cols)
					bNewLayout = true;

				this.name = name;
				this.desc = desc;
				this.rows = rows;
				this.cols = cols;

				// Remove existing cells and create new empty cell matrix
				if (bNewLayout) this.resetLayout();

				// Parse wall layout
				var xmlConf = wallRes.attributes["XML_CONF"];
				var cellMap = xmlConf.split(";");
				var self = this;
				$.each(cellMap, function() {
					if (this == undefined || this == "")
						return; // Empty element

					var arrDims = this.split(",");
					var row = arrDims[0];
					var col = arrDims[1];
					var monid = arrDims[2];

					if (row == undefined || col == undefined)
						return;

					var cell = self.getCellAt(row, col);
					if (!cell) throw new jsx3.lang.Exception("Failed to obtain WallCell object for " +
						"row=" + row + ", col=" + col);
					var monitor = vMX.getMonitorById(monid);

					if (self.initialized && !bNewLayout) {

						if ((cell.monitor == null && monitor != null) ||
							(cell.monitor != null && monitor == null) ||
							(cell.monitor != null && cell.monitor.objid != monid))
							bLayoutChanged = true;
					}

					cell.monitor = monitor;
					lstProcessed.add(cell);
				});

				// search for removed cells
				for (var iter = this.iterator(); this.initialized && !bNewLayout && iter.hasNext();) {
					var cell = iter.next();
					if (cell.monitor && !lstProcessed.contains(cell)) {
						bLayoutChanged = true;
						cell.removeMonitor();
					}
				}

				if (bLayoutChanged) strChanged += "layout";

				if (this.initialized) {
					if (bNewLayout)
						this.publish({'subject':vMX.Wall.NEW_LAYOUT, 'target':this});
					else if (strChanged || bLayoutChanged)
						this.publish({'subject':vMX.Wall.LAYOUT_CHANGED, 'target':this,
							'strChanged':strChanged});
					else if (bCredChanged)
						this.publish({'subject':vMX.Wall.CRED_CHANGED, 'target':this});
				}

			}
			catch(e) {
				var ex = jsx3.lang.NativeError.wrap(e);
				vMX._log(2, "Exception caught in Wall.update: " + ex.getMessage());
			}
		};

		Wall_prototype.fromXML = function(nodeWall, cred) {
			var objResult = {};
			objResult.attributes = [];

			objResult.obj = nodeWall.getAttribute("wallid");
			objResult.name = nodeWall.getAttribute("name");
			objResult.description = nodeWall.getAttribute("desc");
			objResult.attributes["WALLH"] = Number(nodeWall.getAttribute("rows"));
			objResult.attributes["WALLW"] = Number(nodeWall.getAttribute("cols"));

			// Credentials
			objResult.credentials = cred;

			// Construct XML_CONF
			var xmlConf = "";
			var nodeWallCells = nodeWall.selectNodes(
				"/data/record[@wallid=" + objResult.obj + "]/record[@monid]");
			for (var iter = nodeWallCells.iterator(); iter.hasNext();) {
				var node = iter.next();

				var monid = node.getAttribute("monid");
				var row = node.getAttribute("row");
				var col = node.getAttribute("col");

				xmlConf += row + "," + col + "," + monid + ";";
			}

			objResult.attributes["XML_CONF"] = xmlConf;

			return objResult;
		};

		Wall_prototype.resetLayout = function() {
			this.state = "";
			this.clearCells();
			for (var i = 0; i < this.rows; i++)
				for (var j = 0; j < this.cols; j++)
					this.addCell(new vMX.WallCell(i, j));
		};

		/**
		 * Look through cell headers to guess the state of per-wall
		 *'toggle-cell-headers' button
		 */
		Wall_prototype.checkHeaders = function() {

			var bShow = false;

			L: for (var wci = this.iterator(); wci.hasNext();) {
				var wcell = wci.next();
				var mon = wcell.monitor;
				if (!mon) continue;
				for (var i = mon.iterator(); i.hasNext();) {
					var c = i.next();
					if (!c.touring && c.text && c.text.search(/License limit/i) == -1) {
						bShow = true;
						break L;
					}
				}
			}

			if (this.showHeaders != bShow) {
				this.showHeaders = bShow;
				this.publish({'subject' : vMX.Wall.SHOW_HEADERS_CHANGED, 'target':this});
			}
		};

		Wall_prototype.getMonitorAt = function(row, col) {
			var cell = this.getCellAt(row, col);
			return cell ? cell.monitor : null;
		};

		Wall_prototype.setMonitorAt = function(row, col, monitor) {
			var cell = this.getCellAt(row, col);
			if (cell && monitor) cell.monitor = monitor;
		};

		Wall_prototype.getCellAt = function(row, col) {
			return this._cells.filter(function(c) {
				return c.row == row && c.col == col;
			}).get(0);
		};

		Wall_prototype.getMonitorById = function(objid) {
			var cell = this._cells.filter(function(c) {
				return this.monitor.objid == objid;
			});
			return cell ? cell.monitor : null;
		};

		Wall_prototype.toString = function() {
			return "[Wall " + "id=" + this.objid + ", name=" + this.name + ", rows=" + this.rows +
				", cols=" + this.cols + ", Cells: " + this._cells.size() + "]";
		};

	}
	);
