//------------------------------------------------------------------------------------------
// Monitor Cell class. Incapsulates a 'Display Cell' object (in terms of Display Server spec)
//------------------------------------------------------------------------------------------
jsx3.lang.Class.defineClass(
	"mx.vMX.MonitorCell",
	jsx3.lang.Object,
	[jsx3.util.EventDispatcher],
	function(MonitorCell, MonitorCell_prototype) {

		var vMX = mx.vMX;

		MonitorCell.STATE_CHANGED = "vmxstatechanged";
		MonitorCell.TYPE_CHANGED = "vmxcelltypechanged";

		/*
		 * Analytics modes
		 * -1 if stream has no analytics and icon should be hidden
		 */
		MonitorCell.ANL_MODES = ['disable', 'alerts', 'tracks', 'rules'];
		MonitorCell.NO_ANALYTICS = -1;

		/*
		 * Stream quality (resolution) modes
		 * Corresponds to 'streamnum' URL parameter
		 */
		MonitorCell.STREAM_QUALITY_LOW = 0;
		MonitorCell.STREAM_QUALITY_NORMAL = 1;
		MonitorCell.STREAM_QUALITY_DEFAULT = -1;

		/*
		 * Sound volume values
		 */
		MonitorCell.SOUND_MUTE = 0;
		MonitorCell.SOUND_MAX = 100;

		// Dummy URL for Alert Log
		MonitorCell.ALERTLOG_URL = "http://alertlog";

		MonitorCell_prototype.getProperties = function() {
			var props = {};
			for (var item in this) {
				if (typeof(this[item]) == "function") continue;
				if (item.charAt(0) == '_') continue;
				if (item == "initialized") continue;
				if (item == "assocImages") continue;

				props[item] = this[item];
			}
			props["cellid"] = this.id;
			props["monid"] = this.monitor.objid;

			return props;
		};

		MonitorCell_prototype.init = function(obj) {
			if (obj instanceof jsx3.xml.Entity)
				this.fromXML(obj);
			else
				vMX._log(2, "Not a jsx3.xml.Entity passed to MonitorCell.init()");

			this.open_timeout = this.play_timeout = 10; // Required for web service calls
			this.initialized = true;
			this.assocImages = [];
		};

		MonitorCell_prototype.fromXML = function(objNode) {
			var strChanged = "";
			var bCellTypeChanged = false; // Basic cell was replaced by tour one or vice versa
			this.id = objNode.getAttribute("cellid");
			//objNode.setAttribute("vncurl","vnc://127.0.0.1");
			var item;
			for (item in {
				'srcObjid'  : 0,
				'state'     : 1,
				'scale'     : 2,
				'text'      : 3,
				'arc_from'  : 4,
				'arc_to'    : 5,
				'arc_pos'   : 6,
				'groupid'   : 7,
				'eventid'   : 8,
				'vncurl'    : 9,
				'analytics' : 10,
				'streamnum' : 11,
				'audioObjid': 12,
				'volume'    : 13,
				'weburl'    : 14,
				'refresh'   : 15}) {
				var val = item == "srcObjid" ? parseInt(objNode.getAttribute(item)) : objNode.getAttribute(item);
				if (this.initialized && this[item] != val) strChanged += (item + ";");
				if (item == "vncurl" || item == "weburl") { // Special case
					if ( (!val && this[item]) || (val && !this[item]) ) {
						bCellTypeChanged = true;
					}
				}

				this[item] = val;
			}

			/** Normalize APC-specific attributes */
			if (!this.groupid) this.groupid = 0;
			if (!this.eventid) this.eventid = 0;
			/** Default values for URLs*/
			if (!this.vncurl)  this.vncurl  = "";
			if (!this.weburl)  this.weburl  = "";

			// Analytics mode should be set to 'none' by default
			if (this.analytics == null) this.analytics = MonitorCell.NO_ANALYTICS;

			if (this.streamnum == null) this.streamnum = MonitorCell.STREAM_QUALITY_DEFAULT;

			if (this.audioObjid == null) this.audioObjid = 0;
			if (this.volume == null) this.volume = MonitorCell.SOUND_MUTE;

			// Refresh interval is 0 by default
			if (!this.refresh) this.refresh = 0;
			this.refresh = Number(this.refresh);

			/** Normally, cell geometry shouldn't be changed inside 'fromXML', so do not check.
			 */
			for (item in {'x':0, 'y':1, 'w':2, 'h':3})
				this[item] = Number(objNode.getAttribute(item));

			/** Parse tour info if present **/
			var nodeTouring = objNode.getFirstChild();

			if (nodeTouring) {
				if (!this.touring) {
					bCellTypeChanged = true;
					this.touring = {};
					this.touring.source = new jsx3.util.List;
				}
				// If srcObjid was changed, for tour cell this means that another record was selected
				if (this.initialized && strChanged.indexOf("srcObjid") > -1)
					strChanged += "tour_selected_item;";

				var delay = nodeTouring.getAttribute("delay");
				var pos   = nodeTouring.getAttribute("pos");
				if (this.initialized && this.touring.delay != delay) strChanged += "tour_delay;";
				if (this.initialized && this.touring.pos != pos) {
					if (strChanged.indexOf("tour_selected_item") == -1)
						strChanged += "tour_selected_item;";
				}
				this.touring.delay = delay;
				this.touring.pos   = pos;
				var tmpList = new jsx3.util.List;
				var nodelistSrc = nodeTouring.getChildNodes();
				for (var iter = nodelistSrc.iterator(); iter.hasNext();) {
					var n = iter.next();
					var id = n.getAttribute("id");
					var options = n.getAttribute("options");
					var presetid = n.getAttribute("presetid");
					if (!presetid) presetid = 0;
					tmpList.add({
						"id"       : id,
						"options"  : options,
						"presetid" : presetid
					});
				}

				if (this.initialized) {
					// compare two lists
					var size = tmpList.size();
					if (size != this.touring.source.size()) {
						strChanged += "tour_source;";
						if (this.touring.source.size() == 0 && size > 0) strChanged += "scale;"; // Workaround for DE1913
					}
					else {
						for (var j = 0; j < size; j++) {
							var o1 = tmpList.get(j), o2 = this.touring.source.get(j);
							if (
								   o1.id != o2.id
								|| o1.options != o2.options
								|| o1.presetid != o2.presetid
								)
							{
								strChanged += "tour_source;";
								break;
							}
						}
					}
				}

				this.touring.source = tmpList;
			}
			else if (this.touring) {
				bCellTypeChanged = true;
				this.touring = null;
			}

			if (this.initialized) {
				if (bCellTypeChanged)
					this.publish({'subject':vMX.MonitorCell.TYPE_CHANGED, 'target':this});
				if (strChanged.length)
					this.publish({'subject':vMX.MonitorCell.STATE_CHANGED, 'target':this, 'changed_items':strChanged});
			}
		};

		MonitorCell_prototype.isTour = function() {
			return this.touring ? true : false;
		};

		MonitorCell_prototype.isVNC = function() {
			return this.vncurl ? true : false;
		};

		MonitorCell_prototype.isWeb = function() {
			return this.weburl ? true : false;
		};

		MonitorCell_prototype.isAlertLog = function() {
			if (!this.isWeb)
				return false;
			return this.weburl == MonitorCell.ALERTLOG_URL;
		};

		MonitorCell_prototype.isVideo = function() {
			return ! (this.vncurl || this.touring || this.weburl);
		};

		MonitorCell_prototype.isArchive = function() {
			if (!this.isVideo()) return false;
			return this.arc_from > 0;
		};

		MonitorCell_prototype.setContainer = function(objMonitor) {
			this.monitor = objMonitor;
		};

		MonitorCell_prototype.empty = function() {
			return this.srcObjid == 0 && ! this.vncurl && ! this.weburl && ! this.touring;
		};

		MonitorCell_prototype.isFullScreen = function() {
			return this.scale === "Proportional" || this.scale === "MaxSize" || this.scale === "Original";
		};

		MonitorCell_prototype.toString = function() {
			return "[Cell " + this.id + " x=" + this.x + ", y=" + this.y + ", w=" + this.w + ", h=" + this.h +
				", srcObjid=" + this.srcObjid + ", arc_from=" + this.arc_from + ", arc_to=" + this.arc_to +
				", arc_pos=" + this.arc_pos + ", state=" + this.state + ", scale=" + this.scale + "]";
		};


		MonitorCell_prototype.equals = function(objCell) {
			return (this.id == objCell.id && this.srcObjid == objCell.srcObjid &&
				this.arc_from == objCell.arc_from &&
				this.arc_to == objCell.arc_to);
		};
	}
	);
