/****************************************************************************************************
*                                                                                                   *
* Wall Monitor Class. Incapsulates system object called "Wall monitor", which is a representation   *
* of a physical monitor connected to a box running Display Server. Class in                         *
*                                                                                                   *
****************************************************************************************************/

jsx3.lang.Class.defineClass(
  "mx.vMX.Monitor",
  jsx3.lang.Object,
  [jsx3.util.EventDispatcher, mx.vMX.CellContainer],
  function(Monitor, Monitor_prototype) {

    var vMX = mx.vMX;

    Monitor.CELL_RESIZE = "vmxcellresize";
    Monitor.CELL_REPLACE = "vmxcellreplace";
    Monitor.NEW_LAYOUT = "vmxnewlayout";
    Monitor.NAME_DESC_CHANGED = "vmxnamedescchanged";

    Monitor.prototype.init = function(objid, name, desc) {
      this.initialized = false;
      this.objid = objid;
      this.name = name;
      this.desc = desc;
      this.state = "disconnected";

      // monitor cells
      this._cells = new jsx3.util.List;
    };

    Monitor.prototype.getCellById = function(cellid) {
        for (var i = 0; i < this._cells.size(); i++)
            if (this._cells.get(i).id == cellid) return this._cells.get(i);
        return null;
    };

    /**
     * Main construction and checks. Called when new monitor layout is received
     * from the vMX daemon. Here layout is analyzed and events are published if
     * necessary
     */
    Monitor_prototype.fromXML = function(nodeMonitor, nodeCells) {
      var statenum = nodeMonitor.getAttribute("statenum");
      var hres = nodeMonitor.getAttribute("hres");
      var vres = nodeMonitor.getAttribute("vres");
      var cellscount = nodeMonitor.getAttribute("cellscount");
      var mon_state = nodeMonitor.getAttribute("state") || "connected";

      var verdict = vMX.Monitor.NEW_LAYOUT;

      /** If basic monitor parameters have been changed complete layout should be updated **/
      if(!this.initialized || this.hres != hres ||
         this.vres != vres || this.state != mon_state) {
          verdict = vMX.Monitor.NEW_LAYOUT;
      }

      this.initialized = true;
      this.statenum = statenum;
      this.hres = hres;
      this.vres = vres;
      this.cellscount = cellscount;
      this.state = mon_state;

      var tmpCells = new jsx3.util.List;
      for(var i = 0; i < nodeCells.size(); i++) {
          var node = nodeCells.get(i);
          tmpCells.add({'x':Number(node.getAttribute('x')), 'y':Number(node.getAttribute('y')),
              'w':Number(node.getAttribute('w')), 'h': Number(node.getAttribute('h')),
              'id':node.getAttribute('cellid')});
      }

      var objScanResult = {'verdict' : vMX.Monitor.NEW_LAYOUT};
      if(this._cells.size() > 0) {
        objScanResult = this.smartScan(tmpCells);
        verdict = objScanResult.verdict;
      }

      switch(verdict) {
        case vMX.Monitor.NEW_LAYOUT:
          this.clearCells();
          for(var iter = nodeCells.iterator(); iter.hasNext();) {
            var cell = new vMX.MonitorCell(iter.next());
            this.addCell(cell);
          }
          this.publish({'subject' : verdict});
          break;
        case vMX.Monitor.CELL_REPLACE:
          objScanResult.remove_cell.unsubscribeAllFromAll();
          this.removeCell(objScanResult.remove_cell);
          var replaceCells = [];
          for(var iter = nodeCells.iterator(); iter.hasNext();) {
            var node = iter.next();
            var cell = this.getCellById(node.getAttribute("cellid"));
            if(cell) {
              cell.fromXML(node);
            }
            else {
              var newCell = new vMX.MonitorCell(node);
              this.addCell(newCell);
              replaceCells.push(newCell);
            }
          }
          this.publish({'subject' : verdict, 'remove_cell' : objScanResult.remove_cell,
                        'replace_cells' : replaceCells});
          break;
        case vMX.Monitor.CELL_RESIZE:
          for(var iter = nodeCells.iterator(); iter.hasNext();) {
            var node = iter.next();
            this.getCellById(node.getAttribute("cellid")).fromXML(node);
          }
          this.publish({'subject' : verdict, 'diff' : objScanResult.diff});
          break;
        default:
          for(var iter = nodeCells.iterator(); iter.hasNext();) {
            var node = iter.next();
            this.getCellById(node.getAttribute("cellid")).fromXML(node);
          }
      }
    };


    /**
     * Sets and traces changes of basic attributes of system object: 'name' and 'description'.
     * They are returnes by 'get_resources' web service method which reads values from _objs DB table
     */
    Monitor_prototype.setNameDesc = function(name, desc) {
      if(name != this.name || desc != this.desc) var bChanged = true;
      this.name = name; this.desc = desc;
      if(bChanged) this.publish({'subject' : vMX.Monitor.NAME_DESC_CHANGED});
    };

    Monitor_prototype.smartScan = function(lstCells) {

      var NO_CHANGE = 3;    // no difference found between layouts
      var FULL_CHANGE = 0;  // two different layouts
      var RGN_DIFF = 1;     // layouts differ in particular region(s)
      var CELL_REPLACE = 4; // One cell should be replaced by several new ones

      var objResult = {'verdict' : vMX.Monitor.NEW_LAYOUT};

      var lstCmp = function(l1, l2) {
        if(l1.size() != l2.size()) return false;
        var bFound;
        for(var i = 0; i < l1.size(); i++) {
          var el = l1.get(i);
          bFound = false;
          for(var j = 0; j < l2.size(); j++) {
            if(l2.get(j).id == el.id) { bFound = true; break; }
          }
          if(!bFound) return false;
        }
        return true;
      };

      var objCmp = function(o1, o2, arrDiff) {
        if(o1.orientation == "none" || o2.orientation == "none") {
          return FULL_CHANGE;
        }
        else if(o1.orientation != o2.orientation || !lstCmp(o1.cells, o2.cells)) {
          arrDiff.splice(0, arrDiff.length);
          return FULL_CHANGE;
        } else if(o1.orientation != "cell" && o1.subcontainer1pct != o2.subcontainer1pct) {
           arrDiff.push({x:o1.x, y:o1.y, w:o1.w, h:o1.h, oldpct:o1.subcontainer1pct, newpct:o2.subcontainer1pct});
           return RGN_DIFF & objCmp(o1.region1, o2.region1, arrDiff) & objCmp(o1.region2, o2.region2, arrDiff);
        } else if(o1.orientation != "cell") {
           return objCmp(o1.region1, o2.region1, arrDiff) & objCmp(o1.region2, o2.region2, arrDiff);
        } else {
           return NO_CHANGE;
        }
      };

      // First check for cell replacing
      if(this._cells.size() < lstCells.size()) {
        for(var iter = this.iterator(); iter.hasNext();) {
          var cell = iter.next();
          var replaceCells = lstCells.filter(function(c){return c.id.indexOf(cell.id + " | Cell") == 0});
          if(replaceCells.size()) {
            objResult.verdict = vMX.Monitor.CELL_REPLACE;
            objResult.remove_cell = cell;
            objResult.replace_cells = replaceCells.toArray(1);
            for(var iter2 = lstCells.iterator(); iter2.hasNext();) {
              var c_new = iter2.next();
              if(replaceCells.contains(c_new)) continue;
              var c_old = this.getCellById(c_new.id);
              if(c_old == null || c_new.x != c_old.x || c_new.y != c_old.y || c_new.w != c_old.w || c_new.h != c_old.h)
                { objResult.verdict = vMX.Monitor.NEW_LAYOUT; break; }
            }
            break;
          }
        }
      }
      else {
        var obj1 = {}, obj2 = {};
        vMX.recursiveScan(0, 0, this.hres, this.vres, this._cells, obj1);
        vMX.recursiveScan(0, 0, this.hres, this.vres, lstCells, obj2);

        var arrDiff = [];
        var v =  objCmp(obj1, obj2, arrDiff);
        objResult.verdict = (v == FULL_CHANGE ? vMX.Monitor.NEW_LAYOUT : (v==RGN_DIFF ? vMX.Monitor.CELL_RESIZE : "nochange"));
        objResult.diff = arrDiff;
      }

      return objResult;
    };

    Monitor_prototype.simpleScan = function(lstCells) {
      var objResult = { verdict : "nochange" };
      if(this.getCellsArray().length != lstCells.size()) {
        objResult.verdict = vMX.Monitor.NEW_LAYOUT;
      }
      else {
        for(var iter = lstCells.iterator(); iter.hasNext();) {
          var c1 = iter.next();
          var c2 = this.getCellById(c1.id);
          if(c2 == null) { objResult.verdict = vMX.Monitor.NEW_LAYOUT; break; }
          if(c1.x != c2.x || c1.y != c2.y || c1.w != c2.w || c1.h != c2.h)
            { objResult.verdict = vMX.Monitor.NEW_LAYOUT; break; }
        }
      }

      return objResult;
    };

    Monitor.prototype.toString = function() {
      return "[Monitor " + this.objid + ", name=" + this.name + ", desc=" + this.desc +
              ", hres=" + this.hres + ", vres=" +
              this.vres + ", cellscount=" + this._cells.size() + "]";
    };

    Monitor_prototype.serializeToJson = function() {
        var scanRes = {};

        var process = function(obj) {
            if (obj.orientation != "cell") {
                return {
                    'type':         'splitter',
                    'orientation':  obj.orientation,
                    'subcont_size': String(Math.round(obj.subcontainer1pct) + "%"),
                    'children':     [
                        process(obj.region1),
                        process(obj.region2)
                    ]
                };
            } else {
                var cell = obj.cells.get(0);

                if (cell.empty()) {
                    return {'type': 'empty_block'};
                }
                else if (cell.isVideo()) {
                    return {
                        'type': 'camera',
                        'obj' : cell.srcObjid
                    };
                }
                else if (cell.isTour()) {
                    var nodes = [];
                    var presets = [];

                    for (var i = cell.touring.source.iterator(); i.hasNext();) {
                        var src = i.next();
                        nodes.push(src.id);
                        presets.push(src.presetid > 0 ? src.presetid : "");
                    }
                    return {
                        'type':    'touring',
                        'objects': nodes,
                        'presets': presets,
                        'delay':   cell.touring.delay,
                        'state':   true  // list open/close
                    }
                }
                else if (cell.isVNC()) {
                    return {
                        'type':         'vnc',
                        'url':          cell.vncurl
                    }
                }
                else if (cell.isWeb()) {
                    return {
                        'type':         'url',
                        'url':          cell.weburl,
                        'refresh':      cell.refresh
                    };
                }
                else { // default
                    return {'type': 'empty_block'};
                }
            }
        };

        vMX.recursiveScan(0, 0, this.hres, this.vres, this.getCellsList(), scanRes);
        return process(scanRes);
    };

    Monitor_prototype._cellFromRect = function(rect, obj) {
        var cellid = rect.cellid != null ? rect.cellid : jsx3.xml.CDF.getKey();
        var cell = {
				jsxid:jsx3.xml.CDF.getKey(), id:cellid, cellid:cellid,
                x:rect.x, y:rect.y, w:rect.w, h:rect.h,
				srcObjid:0, state:"Play", scale:"Cell",
				arc_from:0, arc_to:0, vncurl:"",
				arc_pos:0, groupid: 0, open_timeout:10, play_timeout:10,
				weburl:"", refresh:0, streamnum:vMX.MonitorCell.STREAM_QUALITY_DEFAULT,
				audioObjid:0, volume: vMX.MonitorCell.SOUND_MUTE, 
				analytics: vMX.MonitorCell.NO_ANALYTICS
		};
        switch (obj.type) {
            case "empty_block" :
                cell.state = "Empty";
                break;
            case "camera" :
                cell.srcObjid = obj.obj;
                cell.streamnum = vMX.getDefaultStreamNumber(cell.srcObjid);
                cell.audioObjid = vMX.getAudioObjid(cell.srcObjid);
                cell.analytics = vMX.getDefaultAnlState(cell.srcObjid);
                break;
            case "url" :
                cell.weburl = obj.url;
                cell.refresh = obj.refresh;
                break;
            case "vnc" :
                cell.vncurl = obj.url;
                break;
            case "touring" :
                var touring = {};
			    touring.source = new jsx3.util.List;
                touring.delay = obj.delay;
                touring.pos = 0;
                for (var i = 0; i < obj.objects.length; i++) {
                    touring.source.add({
                        "id" : obj.objects[i],
                        "options" : "",
                        "presetid" : obj.presets[i]
                    });
                }
                cell.touring = touring;
                if (obj.objects.length == 0)
                    cell.state = "Empty";
            default:
                cell.state = "Empty";
        }

        return cell;
    };

    Monitor_prototype.deserializeFromJson = function(objJson) {
        var arrCells = [];
        var me = this;
        var scan = function(obj, rect) {
            if (obj.type == "splitter") {
                var horiz = obj.orientation == jsx3.gui.Splitter.ORIENTATIONV;
                var pct = parseFloat(obj.subcont_size);
                var rect1 = {
                    x : rect.x,
                    y : rect.y,
                    w : horiz ? rect.w : parseInt(rect.w * pct/100),
                    h : horiz ? parseInt(rect.h * pct/100) : rect.h
                };
                var rect2 = {
                    x : horiz ? rect.x : rect.x + rect1.w + 1,
                    y : horiz ? rect.y + rect1.h + 1 : rect.y,
                    w : horiz ? rect1.w : rect.w - rect1.w - 1,
                    h : horiz ? rect.h - rect1.h - 1 : rect1.h
                };

                scan(obj.children[0], rect1);
                scan(obj.children[1], rect2);
            }
            else {
                arrCells.push(me._cellFromRect(rect, obj));
            }
        };

        var baseRect = { x : 0, y : 0, w : this.hres, h : this.vres };

        if (objJson != null) {
    	    scan(objJson, baseRect);
        }

        return arrCells;
    };
  }
);
