/******************************************************************************
 *                                                                             *
 *    vMX.MagnifierCell class - visual representation of monitor cell object   *
 *                                                                             *
 ******************************************************************************/
jsx3.lang.Class.defineClass(
	"mx.vMX.MagnifierCell", jsx3.lang.Object, null, function(MagnifierCell, MagnifierCell_prototype) {

	var vMX = mx.vMX;

	MagnifierCell.COMPONENT_GUI_URL = "components/gui_cell.xml";

	MagnifierCell.COMPONENT_TOUR_URL = "components/tour_cell_vmx.xml";

	MagnifierCell.COMPONENT_VNC_URL  = "components/vnc_cell.xml";

	MagnifierCell.COMPONENT_WEB_URL  = "components/url_cell.xml";

	MagnifierCell.IMG_LIVE = "images/design1/gui_cell/cell_live.png";

	MagnifierCell.IMG_ARCHIVE = "images/design1/gui_cell/cell_archive.png";

	MagnifierCell.IMG_ZOOM = "images/design1/vMX/zoom.png";

	MagnifierCell.IMG_ZOOM_PRESSED = "images/design1/vMX/zoom_pressed.png";

	MagnifierCell.IMG_VNC_START = "images/design1/vnc/start_sharing.png";
	MagnifierCell.IMG_VNC_STOP = "images/design1/vnc/stop_sharing.png";

	MagnifierCell.IMG_SETTINGS = "images/design1/resources/settings.png";

	MagnifierCell.IMG_WEB = "images/design1/vMX/webcell.jpg";

	MagnifierCell.ANL_TIPS = [
		__("Analytics off"),
		__("Alerts only"),
		__("Tracks/Alerts"),
		__("Tracks/Alerts/Rules")
	];

	/*
	 * Vars and constants dealing with Web Cell
	 */
	MagnifierCell.DEFAULT_REFRESH_INTERVAL = 60;  // 1 minute
	MagnifierCell.DEFAULT_WEBURL = mx.MATRIX2.getIdentityValue('URL_HOME_PAGE');
	MagnifierCell.PrevRefreshInterval = MagnifierCell.DEFAULT_REFRESH_INTERVAL;
	MagnifierCell.PrevWebURL = MagnifierCell.DEFAULT_WEBURL;


	MagnifierCell_prototype.init = function(objMonCell, blkDest, magnifier) {
		this._magnifier = magnifier;
		this._moncell = objMonCell;
		this._mainblock = blkDest;

		// Snapshot update enabled by default
		this.snapshotUpdateEnabled = true;

		if (magnifier) magnifier.addCell(this);
		this._mainblock.origName = this._mainblock.getName();
	};

	MagnifierCell_prototype.getMagnifier = function() {
		return this._magnifier;
	};

	MagnifierCell_prototype.getMainBlock = function() {
		return this._mainblock;
	};

	MagnifierCell_prototype.setContainer = function(objMagnifier) {
		this._magnifier = objMagnifier;
	};

	MagnifierCell_prototype.getMonCell = function() {
		return this._moncell;
	};

	MagnifierCell_prototype.inGroup = function() {
		return this._magnifier.getGroup().contains(this) ? true : false;
	};

	MagnifierCell_prototype.toString = function() {
		return "MagnifierCell";
	};

	MagnifierCell_prototype.getRefresh = function() {
		if (!this.isWeb())
			return 0;

		var refresh = this._mainblock.getDescendantOfName("intervalSec").getValue();
		var objCheckbox = this._mainblock.getDescendantOfName("checkboxRefresh");
		if (objCheckbox.getRendered() != null && !objCheckbox.getRendered().children[0].checked)
			refresh = 0;

		return refresh;
	};

	MagnifierCell_prototype.isVideo = function() {
		return this._moncell.isVideo();
	};

	MagnifierCell_prototype.isTour = function() {
		return this._moncell.isTour();
	};

	MagnifierCell_prototype.isVNC = function() {
		return this._moncell.isVNC();
	};

	MagnifierCell_prototype.isWeb = function() {
		return this._moncell.isWeb();
	};

	/**
	 * Actually creates magnifier cell view. vMX reuses matrix2 'gui_cell' and
	 * 'tour_cell' components and dynamically modifies their DOM to add vMX-specific
	 * GUI elements such as 'zoom-in/zoom-out' button and 'play/pause' youtube-like
	 * button. create' method loads those components and performs all the necessary
	 * modifications, but if only renders basic cell skeleton. Other painting,
	 * specific for each cell is done inside 'updateView'
	 */
	MagnifierCell_prototype.create = function() {

		if (this.isTour() && this._magnifier) {
			// Remove cell from group if it gonna be replaced with tour cell
			var group = this._magnifier.getGroup();
			if (group.contains(this)) group.remove(this);
		}

		this._moncell.unsubscribe(vMX.MonitorCell.STATE_CHANGED, this);
		this._moncell.unsubscribe(vMX.MonitorCell.TYPE_CHANGED, this);

		this._mainblock.removeChildren();
		this._mainblock.setOverflow(jsx3.gui.Block.OVERFLOWEXPAND);

		this._mainblock.setName(this._moncell.id);
		this._mainblock.magn_cell = this;


		if (this.isTour()) { // Create tour cell

			mx.language.load_translate(this._mainblock, vMX.MagnifierCell.COMPONENT_TOUR_URL);

			// Setup 'afterRepaint' hook
			var me = this;
			this._mainblock.getChild("cell_blk").onAfterPaint = function() {
				me._magnifier.onAfterCellRepaint(me._moncell.id);
			};

			// Change component's load type so as it should be painted asynchronously
			this._mainblock.getChild("cell_blk").setLoadType(jsx3.app.Model.LT_SLEEP_PAINT);

			// Add image block for displaying static camera image
			var layoutTour = this._mainblock.getDescendantOfName("layout_tour");
			layoutTour.setCols("*,50%", true);
			var imageCell = new jsx3.gui.Image("image_cell", 0, 0, "100%", "100%");
			layoutTour.getChild("cell_tour_player").setChild(imageCell);

			// Set css class for cell header text
			this._mainblock.getDescendantOfName("cell_dev_name").setClassName("cell_header_text_grey");

			// Modify tour cell header buttons
			var header = this._mainblock.getDescendantOfName("cell_btn_tour").getParent();
			header.getParent().setCols("*,95", true);
			var btnZoom = new jsx3.gui.ImageButton("cell_zoom", 0, 0, 20, 20);
			btnZoom.setImage(vMX.MagnifierCell.IMG_ZOOM);
			header.insertBefore(btnZoom, header.getChild("cell_btn_tour"));
			var btnSettings = new jsx3.gui.ImageButton("cell_settings", 0, 0, 20, 20);
			btnSettings.setImage(vMX.MagnifierCell.IMG_SETTINGS);
			btnSettings.setEvent("1", jsx3.gui.Interactive.EXECUTE);
			header.insertBefore(btnSettings, header.getChild("cell_btn_tour"));

			// Hide PTZ block
			var ptzBlock = this._mainblock.getDescendantOfName("block_ptzpreset");
			if (ptzBlock) ptzBlock.setVisibility(jsx3.gui.Block.VISIBILITYHIDDEN);

			// Finally, set/remove event handlers
			var btnTour = header.getChild("cell_btn_tour");
			var btnClose = header.getChild("cell_btn_close");
			var listTour = this._mainblock.getDescendantOfName("list_tour");
			var imageButtonMask = this._mainblock.getDescendantOfName("imageButtonMask");
			var slider = this._mainblock.getDescendantOfName("delay_slider_tour");
			var tourPane = this._mainblock.getDescendantOfName("cell_tour_player");

			btnTour.removeEvent(jsx3.gui.Interactive.EXECUTE);
			btnClose.removeEvent(jsx3.gui.Interactive.EXECUTE);
			listTour.setEvent("false;", jsx3.gui.Interactive.DROP);
			listTour.setEvent("false;", jsx3.gui.Interactive.EXECUTE);
			imageButtonMask.removeEvent(jsx3.gui.Interactive.EXECUTE);
			slider.removeEvent(jsx3.gui.Interactive.CHANGE).removeEvent(jsx3.gui.Interactive.INCR_CHANGE);

			tourPane.setCanDrop(jsx3.Boolean.TRUE).subscribe(jsx3.gui.Interactive.DROP, this, "onDrop");

			btnTour.subscribe(jsx3.gui.Interactive.EXECUTE, this, "listOpenClose");
			btnSettings.subscribe(jsx3.gui.Interactive.EXECUTE, this, "btnTourSettings");
			btnClose.subscribe(jsx3.gui.Interactive.EXECUTE, this, "btnTourClose");
			listTour.subscribe(jsx3.gui.Interactive.DROP, this, "onListTourDrop");
			listTour.subscribe(jsx3.gui.Interactive.EXECUTE, this, "editPTZPreset");
			listTour.setClassName('drop');
			registerDropHandler(listTour.getId(), this, this.onListTourDrop);
			listTour.subscribe(jsx3.gui.Interactive.SELECT, this, "onTourSelectRecord");
			listTour.setSortPath("jsxid");
			listTour.repaint();
			btnZoom.subscribe(jsx3.gui.Interactive.EXECUTE, this, "onZoom");
			imageButtonMask.subscribe(jsx3.gui.Interactive.EXECUTE, this, "listTourObjectDelete");
			slider.subscribe(jsx3.gui.Interactive.CHANGE, this, "delaySliderChange");
			slider.setEvent("this.getParent().getParent().getDescendantOfName('delay_txt_tour').setText(Math.floor(fpVALUE+5)).repaint();", jsx3.gui.Interactive.INCR_CHANGE);

			// Subscribe to 'onmouseover' event
			this._mainblock.setCanSpy(jsx3.Boolean.TRUE);
			this._mainblock.doSpyOver = function(objEvent) {
				this.doEvent(jsx3.gui.Interactive.JSXMOUSEOVER, {objEVENT:objEvent});
			};
			this._mainblock.subscribe(jsx3.gui.Interactive.JSXMOUSEOVER, this, "onMouseOver");

			if (parseInt(this._moncell.groupid)) {
				header.getParent().setCols("*,75", true);
				imageButtonMask.setDisplay(jsx3.gui.Block.DISPLAYNONE, true);
				btnSettings.setDisplay(jsx3.gui.Block.DISPLAYNONE, true);
			}

		}
		else if (this.isVideo()) { // Ordinary magnifier cell
			vMX._log("create standard cell view for cell " + this._moncell.id);
			mx.language.load_translate(this._mainblock, vMX.MagnifierCell.COMPONENT_GUI_URL);

			var cellBlk = this._mainblock.getChild("cell_blk");

			// Setup 'afterRepaint' hook
			cellBlk.setLoadType(jsx3.app.Model.LT_SLEEP_PAINT);
			var me = this;
			cellBlk.onAfterPaint = function() {
				me._magnifier.onAfterCellRepaint(me._moncell.id);
			};

			this._mainblock.getDescendantOfName("image_player").setName("image_cell").setVisibility(jsx3.gui.Block.VISIBILITYVISIBLE);
			this._mainblock.getDescendantOfName("cell_player").removeChild(this._mainblock.getDescendantOfName("iframe_player"));
			this._mainblock.getDescendantOfName("message_player").setVisibility(jsx3.gui.Block.VISIBILITYHIDDEN);

			// Set css class for cell header text
			//this._mainblock.getDescendantOfName("cell_dev_name").setClassName("cell_header_text_grey");

			this._mainblock.getDescendantOfName("cell_dev_state").setEvent("true", jsx3.gui.Interactive.JSXCLICK);
			this._mainblock.getDescendantOfName("cell_dev_state").subscribe(jsx3.gui.Interactive.JSXCLICK, this, "onToggleStreamType");

			// Change event for clicking on cell header
			this._mainblock.getDescendantOfName("cell_select").setEvent("1", jsx3.gui.Interactive.JSXCLICK);
			this._mainblock.getDescendantOfName("cell_select").subscribe(jsx3.gui.Interactive.JSXCLICK, this, "onSelect");

			// Replace default ptz button handler
			this._mainblock.getDescendantOfName("cell_joystick").removeEvent(jsx3.gui.Interactive.EXECUTE);
			this._mainblock.getDescendantOfName("cell_joystick").subscribe(jsx3.gui.Interactive.EXECUTE, this, "onCallPTZ");

			// Replace default analytics button handler
			this._mainblock.getDescendantOfName("cell_anl").setEvent("true", jsx3.gui.Interactive.EXECUTE);
			this._mainblock.getDescendantOfName("cell_anl").subscribe(jsx3.gui.Interactive.EXECUTE, this, "onSetAnalytics");

			// Replace default resolution button handler
			this._mainblock.getDescendantOfName("cell_stream_quality").setEvent("true", jsx3.gui.Interactive.EXECUTE);
			this._mainblock.getDescendantOfName("cell_stream_quality").subscribe(jsx3.gui.Interactive.EXECUTE, this, "onChangeStream");

            // Replace default sount button handler
            this._mainblock.getDescendantOfName("cell_sound").setEvent("true", jsx3.gui.Interactive.EXECUTE);
			this._mainblock.getDescendantOfName("cell_sound").subscribe(jsx3.gui.Interactive.EXECUTE, this, "onSetSound");

			//Create additional pane to where state button will be placed and the button itself

			var cellPlayer = this._mainblock.getDescendantOfName("cell_player");
			var blkState = new jsx3.gui.Block("cell_state_pane", "45%", "44%", "10%", "12%");
			blkState.setOverflow(jsx3.gui.Block.OVERFLOWEXPAND);
			blkState.setZIndex(4);
			blkState.setRelativePosition(jsx3.gui.Block.ABSOLUTE);
			cellPlayer.setChild(blkState);

			var btnStateToggle = new jsx3.gui.ImageButton("state_image", 0, 0, 30, 30);
			btnStateToggle.setRelativePosition(jsx3.gui.Block.ABSOLUTE);
			blkState.setChild(btnStateToggle);
			btnStateToggle.subscribe(jsx3.gui.Interactive.EXECUTE, this, "onToggleStreamState");

			var pane = this._mainblock.getDescendantOfName("cell_pin").getParent();
			pane.getChild("cell_pin").setDisplay(jsx3.gui.Block.DISPLAYNONE);
			pane.getChild("cell_jump").setDisplay(jsx3.gui.Block.DISPLAYNONE);
			//pane.getChild("cell_sound").setDisplay(jsx3.gui.Block.DISPLAYNONE);
			pane.getChild("cell_timezone").setDisplay(jsx3.gui.Block.DISPLAYNONE);
            pane.getChild("set_target").setDisplay(jsx3.gui.Block.DISPLAYNONE);
			this._mainblock.getDescendantOfName("cell_close").
				removeEvent(jsx3.gui.Interactive.EXECUTE).
				subscribe(jsx3.gui.Interactive.EXECUTE, this, "onClose");
			cellBlk.setCanDrop(jsx3.Boolean.TRUE).subscribe(jsx3.gui.Interactive.DROP, this, "onDrop");
			cellBlk.setClassName('drop');
			registerDropHandler(cellBlk.getId(), this, this.onDrop);

			/** Insert 'cell zoom' button **/
			var btnZoom = new jsx3.gui.ImageButton("cell_zoom");
			btnZoom.setImage(vMX.MagnifierCell.IMG_ZOOM);
			btnZoom.setWidth(20).setHeight(20);
			btnZoom.subscribe(jsx3.gui.Interactive.EXECUTE, this, "onZoom");
			pane.insertBefore(btnZoom, pane.getChild("cell_stream_quality"));

			// Do a small hack to enable MOUSEOVER and MOUSEOUT event processing
			// for a jsx3.gui.Block object
			var blkImage = this._mainblock.getDescendantOfName("cell_player");
			blkImage.setCanSpy(jsx3.Boolean.TRUE);
			blkImage.doSpyOver = function(objEvent) {
				this.doEvent(jsx3.gui.Interactive.JSXMOUSEOVER, {objEVENT:objEvent});
			};
			blkImage.doSpyOut = function(objEvent) {
				this.doEvent(jsx3.gui.Interactive.JSXMOUSEOUT, {objEVENT:objEvent});
			};
			blkImage.subscribe(jsx3.gui.Interactive.JSXMOUSEOVER, this, "onMouseOver");
			blkImage.subscribe(jsx3.gui.Interactive.JSXMOUSEOUT, this, "onMouseOut");
		}
		else if (this.isVNC()) { // VNC cell
			vMX._log("create VNC cell view for cell " + this._moncell.id);
			//mx.language.load_translate(this._mainblock, vMX.MagnifierCell.COMPONENT_VNC_URL);
			this._mainblock.loadAndCache(vMX.MagnifierCell.COMPONENT_VNC_URL);

			var cellBlk = this._mainblock.getChild("cell_blk");

			// Setup 'afterRepaint' hook
			cellBlk.setLoadType(jsx3.app.Model.LT_SLEEP_PAINT);
			var me = this;
			cellBlk.onAfterPaint = function() {
				me._magnifier.onAfterCellRepaint(me._moncell.id);
			};

			// Subscribe to events
			cellBlk.setCanDrop(jsx3.Boolean.TRUE).subscribe(jsx3.gui.Interactive.DROP, this, "onDrop");
			cellBlk.setClassName('drop');
			registerDropHandler(cellBlk.getId(), this, this.onDrop);

			cellBlk.getDescendantOfName("vnc_url").
				subscribe(jsx3.gui.Interactive.EXECUTE, this, "onStartVNCSharing");
			cellBlk.getDescendantOfName("buttonShare").
				subscribe(jsx3.gui.Interactive.EXECUTE, this, "onStartStopVNCSharing");
			cellBlk.getDescendantOfName("input_vnc_login").
				subscribe(jsx3.gui.Interactive.EXECUTE, this, "onStartVNCSharing");
			cellBlk.getDescendantOfName("input_vnc_password").
				subscribe(jsx3.gui.Interactive.EXECUTE, this, "onStartVNCSharing");
			cellBlk.getDescendantOfName("cell_close").removeEvent(jsx3.gui.Interactive.EXECUTE).
				subscribe(jsx3.gui.Interactive.EXECUTE, this, "onClose");
			cellBlk.getDescendantOfName("buttonConfig").
				subscribe(jsx3.gui.Interactive.EXECUTE, this, "onShowHideVNCConfig");
		}
		else if (this.isWeb()) { // Web cell

			mx.language.load_translate(this._mainblock, vMX.MagnifierCell.COMPONENT_WEB_URL);

			vMX._log("create Web cell view for cell " + this._moncell.id);
			this._mainblock.loadAndCache(vMX.MagnifierCell.COMPONENT_WEB_URL);

			var cellBlk = this._mainblock.getChild("main_url_block");
			cellBlk.setName("cell_blk").repaint();

			// Setup 'afterRepaint' hook
			cellBlk.setLoadType(jsx3.app.Model.LT_SLEEP_PAINT);
			var me = this;
			cellBlk.onAfterPaint = function() {
				// Checkbox is created dynamically, so must repaint it explicitly
				var objCheckbox = me._mainblock.getDescendantOfName("checkboxRefresh");
				if (objCheckbox.getRendered() != null) {
					objCheckbox.getRendered().children[0].checked = me._moncell.refresh ? 1 : 0;
				}
				me._magnifier.onAfterCellRepaint(me._moncell.id);
			};

			/* Slightly change url_cell component
			1. Remove Iframe
			2. Replace it with static image
			 */
			var blkFrame = this._mainblock.getDescendantOfName("frame");
			blkFrame.removeChild(this._mainblock.getDescendantOfName("url_iframe"));
			var blkImage = new jsx3.gui.Image;
			blkImage.setName("image_cell");
			blkImage.setClassName("loadingIndication").setSrc(vMX.getCellSnapshot(
					    this.getMagnifier().getMonitor().objid,
					    this._moncell.id, "web"));
			blkImage.setWidth("100%");
			blkImage.setHeight("100%");
			blkFrame.setChild(blkImage);
			blkImage.repaint();

			// Subscribe to events
			cellBlk.setCanDrop(jsx3.Boolean.TRUE).subscribe(jsx3.gui.Interactive.DROP, this, "onDrop");
			cellBlk.setClassName('drop');
			registerDropHandler(cellBlk.getId(), this, this.onDrop);

			/* Reassing event handlers */
			cellBlk.getDescendantOfName("adressUrl").setEvent("1", jsx3.gui.Interactive.EXECUTE).
				subscribe(jsx3.gui.Interactive.EXECUTE, this, "onOpenWebURL");
			this._mainblock.getDescendantOfName("buttonClose").
				setEvent("1", jsx3.gui.Interactive.EXECUTE).
				subscribe(jsx3.gui.Interactive.EXECUTE, this, "onClose");
			this._mainblock.getDescendantOfName("checkboxRefresh").
				setEvent("1", jsx3.gui.Interactive.JSXCLICK).
				subscribe(jsx3.gui.Interactive.JSXCLICK, this, "onCheckRefresh");

			/* Set initial values for url and refresh
			 * Restore saved values if any
			 */
			cellBlk.getDescendantOfName("adressUrl").setValue(MagnifierCell.PrevWebURL);
			cellBlk.getDescendantOfName("intervalSec").setValue(MagnifierCell.PrevRefreshInterval);
		}

		// if popup cell
		if (parseInt(this._moncell.groupid)) {
			this._mainblock.getDescendantOfName("cell_popup").setDisplay(jsx3.gui.Block.DISPLAYBLOCK).repaint();
			this._mainblock.getDescendantOfName("cell_popup").removeEvent(jsx3.gui.Interactive.EXECUTE).
				subscribe(jsx3.gui.Interactive.EXECUTE, this, "onPopupAlert");
			btnZoom.setDisplay(jsx3.gui.Block.DISPLAYNONE, true);
		}

		// Subscribe to MonitorCell events here
		this._moncell.subscribe(vMX.MonitorCell.STATE_CHANGED, this, "updateView");
		this._moncell.subscribe(vMX.MonitorCell.TYPE_CHANGED, this, "onCellTypeChanged");
		this._moncell.subscribe(vMX.WallLayout.UPDATE_SNAPSHOTS, this, "onUpdateSnapshot");


		// DESTRUCTORS
		this._mainblock.subscribe(jsx3.gui.Interactive.DESTROY, this, "onDestroyMain");
		this._mainblock.getChild('cell_blk').subscribe(jsx3.gui.Interactive.DESTROY, this, "onDestroyContent");
	};

	MagnifierCell_prototype.clearRefs = function() {
		// Unsubscribe
		this._moncell.unsubscribe(vMX.MonitorCell.STATE_CHANGED, this);
		this._moncell.unsubscribe(vMX.MonitorCell.TYPE_CHANGED, this);
		this._moncell.unsubscribe(vMX.WallLayout.UPDATE_SNAPSHOTS, this);

		// remove references
		this._moncell = this._mainblock = this._magnifier = null;
	};

	MagnifierCell_prototype.destroy = function() {
		this._mainblock.setName(this._mainblock.origName);
		this._mainblock.removeChildren();
		this._magnifier.removeCell(this);
		this.clearRefs();
	};


	MagnifierCell_prototype.setSelected = function(bSelected) {
		var group = this._magnifier.getGroup();
		if (bSelected) {

			// First remove another cell from the group
			for (var iter = group.iterator(); iter.hasNext();) {
				var cell = iter.next();
				cell.getMainBlock().getDescendantOfName('cell_header').setBackground(
					'background-image:url(./images/design1/gui_cell/cell_header_bg.gif);background-repeat:repeat-x;').
					repaint();
			}
			group.clear();
			// Now add selected cell to the group
			group.add(this);
			this._mainblock.getDescendantOfName('cell_header').setBackground(
				'background-image:url(./images/design1/gui_cell/cell_header_bg_blue.gif);background-repeat:repeat-x;').
				repaint();
		}
		else {
			group.remove(this);
			this._mainblock.getDescendantOfName('cell_header').setBackground(
				'background-image:url(./images/design1/gui_cell/cell_header_bg.gif);background-repeat:repeat-x;').
				repaint();
		}

		this._magnifier.getWCView().getWallLayout().sync({'group' : 1, 'archive' : 1, 'timeline' : 1});
	};

	/**
	 * Displays warning message inside the cell and hides it on timeout
	 */
	MagnifierCell_prototype.raiseMessage = function() {
		var objDest = this._mainblock;
		if (jsx3.CLASS_LOADER.IE) {
			var strI = __("Not sufficient credentials to play the video stream.<br><i>Please contact your system administrator.</i>");
			var strM = "<table width='100%' height='100%'><tr><td align='center' valign='middle' style='font-size:12px;'>" + strI + "</td></tr></table>";
			var blkMessage = objDest.getDescendantOfName('message_player');
			var blkImage = objDest.getDescendantOfName('image_cell');
			blkMessage.setText(strM, true).setVisibility(jsx3.gui.Block.VISIBILITYVISIBLE, true).setDisplay(jsx3.gui.Block.DISPLAYBLOCK, true);
			vMX.setOpacity(blkImage, 1.0);
			blkImage.setDisplay(jsx3.gui.Block.DISPLAYNONE).repaint();
			window.setTimeout(function() {
				if (blkMessage && blkImage) {
					blkMessage.setVisibility(jsx3.gui.Block.VISIBILITYHIDDEN, true);
					blkImage.setDisplay(jsx3.gui.Block.DISPLAYBLOCK, true);
				}
			}, 3000);
		} else if (jsx3.CLASS_LOADER.FX) {
			alert(__("Not sufficient credentials to play the video stream.\nPlease contact your system administrator."));
		} else {
			alert(__("Not supported browser type."));
		}
	};

	/*********************************************
	 * Event handlers
	 ********************************************/


	MagnifierCell_prototype.onMouseOver = function(event) {

		// First we have to look through rest of cell to find out
		// whether some of them are still in 'preview' state
		for (var iter = this._magnifier.iterator(); iter.hasNext();) {
			var mcell = iter.next();
			if (mcell != this && mcell._mouseover)
				mcell.onMouseOut();
		}

		if (dragged && !this._mouseover && this.isVideo()) {

			this._mouseover = true;

			var img = this._mainblock.getDescendantOfName('image_cell');

			this._tmpurl = img.getSrc();

			event.context.objEVENT.cancelAll();

			var oW = this._mainblock.getAbsolutePosition().W;
			var oH = this._mainblock.getAbsolutePosition().H;

			this.snapshotUpdateEnabled = false;

			// Hide cell state imagebutton when preview image is displayed
			this._mainblock.getDescendantOfName("cell_state_pane").setDisplay(jsx3.gui.Block.DISPLAYNONE, true);


			if (dragged_splitter) {
				if (dragged_splitter_type == '1x2') {
					if (img.getSrc() != "images/design1/layout/1x2.png") {
						img.setSrc("images/design1/layout/1x2.png");
						img.setHeight(10, false);
						img.setWidth(10, false);
						img.setTop((img.getAbsolutePosition().H / 2 - 5), false);
						img.setLeft((img.getAbsolutePosition().W / 2 - 5), false);
						img.repaint();
					}
				}
				else if (dragged_splitter_type == '2x1') {
					if (img.getSrc() != "images/design1/layout/2x1.png") {
						img.setSrc("images/design1/layout/2x1.png");
						img.setHeight(10, false);
						img.setWidth(10, false);
						img.setTop((img.getAbsolutePosition().H / 2 - 5), false);
						img.setLeft((img.getAbsolutePosition().W / 2 - 5), false);
						img.repaint();
					}
				}
				else if (dragged_splitter_type == '2x2') {
					if (img.getSrc() != "images/design1/layout/2x2.png") {
						img.setSrc("images/design1/layout/2x2.png");
						img.setWidth(10, false);
						img.setHeight(10, false);
						img.setTop((img.getAbsolutePosition().H / 2 - 5), false);
						img.setLeft((img.getAbsolutePosition().W / 2 - 5), false);
						img.repaint();
					}
				}
				else if (dragged_splitter_type == 4) {
					if (img.getSrc() != "images/design1/layout/map3d.png") {
						img.setSrc("images/design1/layout/map3d.png");
						img.setWidth(10, false);
						img.setHeight(10, false);
						img.setTop((img.getAbsolutePosition().H / 2 - 5), false);
						img.setLeft((img.getAbsolutePosition().W / 2 - 5), false);
						img.repaint();
					}
				}
			}
			else {
				if (glbLegacyNodename != "eventlog" && glbLegacyDevId) { // if preview live
					var src = window.jsServerProto + "://" + window.jsServerName + ":" + window.jsServerPort + "/domain/" + glbLegacyNodename +
						"/conf/live/" + glbLegacyDevId + ".jpg";
					vMX.setOpacity(img, 0.4);
					img.setClassName("loadingIndication").setSrc(src);
					img.repaint();
				}
			}

		}
	};

	MagnifierCell_prototype.onMouseOut = function(event) {

		if (this._mouseover) {

			var imageCell = this._mainblock.getDescendantOfName('image_cell');

			this.snapshotUpdateEnabled = true;

			// Display cell state imagebutton
			this._mainblock.getDescendantOfName("cell_state_pane").setDisplay(jsx3.gui.Block.DISPLAYBLOCK, true);


			imageCell.setLeft(0).setTop(0).setWidth("100%").setHeight("100%");

			if (this._moncell.scale != "Hidden")
				vMX.setOpacity(imageCell, 1.0);

			imageCell.setClassName("loadingIndication").setSrc(
			    this._tmpurl ?
				this._tmpurl :
				vMX.getCamImage(this._moncell)
			);
			imageCell.repaint();

			this._tmpurl = "";
			this._mouseover = false;
		}
	};


	MagnifierCell_prototype.onUpdateSnapshot = function(event) {
		//Skip tour cells
		if (this.isTour()) return;

		if (this.snapshotUpdateEnabled && this._magnifier.getWCView().getWallLayout() == event.layout) {
			//appLogger.warn("update snapshot for cell: " + this._moncell.id + "; url="+event.url);
			var imageCell =
				this.isVNC() ?
				this._mainblock.getDescendantOfName("vnc_image") :
				this._mainblock.getDescendantOfName("image_cell");
			imageCell.setClassName("loadingIndication").setSrc(event.url).repaint();
		}
	};

	MagnifierCell_prototype.onDestroyMain = function(event) {
		if (this._magnifier) this._magnifier.removeCell(this);
		// Remove references
		this._moncell = this._mainblock = this._magnifier = null;
	};

	MagnifierCell_prototype.onDestroyContent = function(event) {
		// Restore original mainblock name
		this._mainblock.setName(this._mainblock.origName);
		// Unsubscribe from MonitorCell events
		this._moncell.unsubscribe(vMX.MonitorCell.STATE_CHANGED, this);
		this._moncell.unsubscribe(vMX.MonitorCell.TYPE_CHANGED, this);
		this._moncell.unsubscribe(vMX.WallLayout.UPDATE_SNAPSHOTS, this);
	};

	MagnifierCell_prototype.onCellTypeChanged = function(event) {
		this.create();

		if (this.isVideo() || this.isVNC() || this.isWeb()) {
			this.updateView(null, false);
			this._mainblock.repaint();
		}
	};

	/**
	 * Is called when user clicks on cell header. Cell is added or removed from the group
	 * NOTE: Group includes only one cell, due to DS functional limitations
	 */
	MagnifierCell_prototype.onSelect = function(event) {
		if (this._moncell.srcObjid != 0) {
			var group = this._magnifier.getGroup();
			if (group.contains(this))
				this.setSelected(false);
			else
				this.setSelected(true);
		}
	};


	/**
	 * Only devices, touring and splitters can be dropped on the cell block
	 */
	MagnifierCell_prototype.onDrop = function(event) {

		// Return cell pane to normal state
		this.onMouseOut();

		if (event.type) {
			var objType = event.type;
			if (objType != 'camera')
				return;
		} else {
			objType = event.context.objSOURCE.getName();
			var strId = event.context.strDRAGID;
		}

		var objid = this._moncell.monitor.objid;
		var cellid = this._moncell.id;

		if (parseInt(this._moncell.groupid) && (
			objType == 'camera' ||
			objType == 'cell_dev_type' ||
			objType == 'matrixWitnesses' ||
			objType == 'screenshot')) {

			alert("Cannot drop device on APC cell");
			return;
		}

		switch (objType) {
			case 'image_spl_h':
				vMX.splitMonitor(objid, {rows:2, cols:1}, cellid);
				break;
			case 'image_spl_v':
				vMX.splitMonitor(objid, {rows:1, cols:2}, cellid);
				break;
			case 'image_spl_c':
				vMX.splitMonitor(objid, {rows:2, cols:2}, cellid);
				break;
			case 'image_spl_3x2':
				vMX.splitMonitor(objid, {rows:2, cols:3}, cellid);
				break;
			case 'image_spl_4x3':
				vMX.splitMonitor(objid, {rows:3, cols:4, exclusive:true}, cellid);
				break;
			case 'camera':
				var objC = event;
				if (!objC.cred('L')) {
					vMX._log(4, "Live credential is missing: cell" + cellid + ", srcObjid: " + objC.obj);
					if (!objC.cred('A')) {
						this.raiseMessage(); // Cannot show video at any case
					}
					else { // show archive instead of live
						var now = math.round(mx.MATRIX2.getServerTime() / 1000);
						vMX.assignStream(objC.obj, objid, cellid, {'from':now - 300, 'to': now, 'pos': now - 300});
					}
				}
				else {
					vMX.assignStream(objC.obj, objid, cellid);
				}
				break;
			case 'matrixWitnesses':
			case 'screenshot':

				var eLog2 = jsx3.GO("eLog2Root");
				var dialog = eLog2.getDescendantOfName('eLog2dialog');
				var eventId = dialog.getAttribute("eventId");
				var utc_from = dialog.getAttribute("utctime_from");
				var utc_to = dialog.getAttribute("utctime_to");
				var t_zone = dialog.getAttribute("t_zone");
				var objDev;

				if (objType == 'screenshot') {
					objDev = resourceTree.getObject(dialog.getAttribute("objId"));

					// Check archive credentials
					if (!objDev.cred('A')) {
						vMX._log(4, "Archive credential is missing: cell" + cellid + ", srcObjid: " + objDev.getAttribute("obj_id"));
						this.raiseMessage();
						break;
					}

					if (objDev.type == "camera") {
						vMX.assignStream(objDev.obj, objid, cellid, { 'from':utc_from, 'to':utc_to, 'pos':utc_from });
					}
					else {
						alert(__("Cannot drop non-camera object to Magnifier cell!"));
					}
				}
				else {
					var witnessesDoc = dialog.getDescendantOfName('matrixWitnesses').getXML();
					var objId = witnessesDoc.selectSingleNode("//record[@jsxid='" + strId + "']").getAttribute("objid");
					objDev = resourceTree.getObject(objId);
					if (objDev.type == "camera") {

						// Check archive credentials
						if (!objDev.cred('A')) {
							vMX._log(4, "Archive credential is missing: cell" + cellid + ", srcObjid: " + objDev.getAttribute("obj_id"));
							this.raiseMessage();
							break;
						}

						vMX.assignStream(objDev.obj, objid, cellid, { 'from':utc_from, 'to':utc_to, 'pos':utc_from });
					}
					else {
						alert(__("Cannot drop non-camera object to Magnifier cell!"));
					}
				}
				break;
			case 'cell_dev_type':
				var srcCellid = event.context.objSOURCE.getAncestorOfName("cell_blk").getParent().getName();
				if (cellid == srcCellid) break;
				var srcCell = this._moncell.monitor.getCellById(srcCellid);
				if (!srcCell) break;
				vMX.assignStream([
					{ 'cellid':srcCellid, 'monid':objid, 'state':"Empty", 'srcObjid':srcCell.srcObjid },
					{
						'cellid':cellid, 'monid':objid, 'state':"Play", 'scale':srcCell.scale,
						'srcObjid' : srcCell.srcObjid, 'text' : srcCell.text, 'arc_from' : srcCell.arc_from,
						'arc_to' : srcCell.arc_to, 'arc_pos' : srcCell.arc_pos, 'analytics' : srcCell.analytics,
						'streamnum' : srcCell.streamnum, 'audioObjid' : srcCell.audioObjid, 'volume' : srcCell.volume
					}
				]);
				break;
			case 'image_tour':
				if (parseInt(this._moncell.groupid)) {
					alert("Cannot drop touring on APC cell");
					break;
				}
				vMX.setTouring(objid, cellid, 15, new jsx3.util.List());
				break;
			case 'image_popup':
				if (parseInt(this._moncell.groupid)) {
					// Cell already is APC
					break;
				}
				if (vMX.checkMonitorInSeveralWalls(objid)) {
					alert("Monitor can not be assigned for Alerts because it is shared between multiple Video Walls");
					break;
				}
				mx.vMX.setApc(true, {cellid: cellid, monid: objid, groupid: vMX.getCurWall().objid});
				break;
			case 'image_vnc':
				if (parseInt(this._moncell.groupid)) {
					alert("Cannot drop VNC item on APC cell");
					break;
				}
				if (this._moncell.isFullScreen()) {
					alert("Cannot drop VNC item on fullscreen cell");
					break;
				}
				this.assignURL("vnc://");
				break;
			case 'url_control':
				if (parseInt(this._moncell.groupid)) {
					alert("Cannot drop URL control item on APC cell");
					break;
				}
				if (this._moncell.isFullScreen()) {
					alert("Cannot drop URL control item on fullscreen cell");
					break;
				}

				this.assignURL(MagnifierCell.PrevWebURL, "Play", 0);
				break;
			case 'image_elog2':
			case 'image_elog2_detailed':
				this.assignURL(vMX.MonitorCell.ALERTLOG_URL);
				break;

		}

		this._magnifier.getWCView().getWallLayout().onDropWall(event);
	};

	// Joystick button show/hide logic
	MagnifierCell_prototype.getJoystickDisplay = function() {
		var objCell = this._moncell;
		var objC = resourceTree.getObject(objCell.srcObjid);
		var show = jsx3.gui.Block.DISPLAYBLOCK;
		var hide = jsx3.gui.Block.DISPLAYNONE;
		var newDisplay = hide;
		if (!objCell.srcObjid) return hide; // hide if cell has no stream
		if (objCell.groupid && objCell.groupid > 0) return hide; // hide if APC cell
		var posCtl = objC ? objC.attr("POSITIONCTL") : "";
		if (objC && objC.cred('p'))
			newDisplay = (posCtl == "none") || (posCtl == "") ? hide : show;
		else
			newDisplay = hide;

		if (newDisplay == show && objCell.arc_from > 0) return hide;
		return newDisplay;
	};


	MagnifierCell_prototype.getAnlDisplay = function() {
		var objCell = this._moncell;
		var objC = resourceTree.getObject(objCell.srcObjid);
		if (!objCell.srcObjid || !objC) return jsx3.gui.Block.DISPLAYNONE; // hide if cell has no stream
		if (objCell.groupid && objCell.groupid > 0) return jsx3.gui.Block.DISPLAYNONE; // hide if APC cell
		if (objC.attr("VAE_ACTIVE") != "yes") return jsx3.gui.Block.DISPLAYNONE; // analytics is disabled for camera
		return jsx3.gui.Block.DISPLAYBLOCK;
	};

	MagnifierCell_prototype.getStreamDisplay = function() {
		var objCell = this._moncell;
		var objC = resourceTree.getObject(objCell.srcObjid);
		if (!objCell.srcObjid || !objC) return jsx3.gui.Block.DISPLAYNONE; // hide if cell has no stream
		if (objCell.groupid && objCell.groupid > 0) return jsx3.gui.Block.DISPLAYNONE; // hide if APC cell
		return jsx3.gui.Block.DISPLAYBLOCK;
	};

	MagnifierCell_prototype.getSoundDisplay = function() {
		var objCell = this._moncell;
		var objC = resourceTree.getObject(objCell.srcObjid);
		if (!objCell.srcObjid || !objC) return jsx3.gui.Block.DISPLAYNONE; // hide if cell has no stream
		if (objCell.groupid && objCell.groupid > 0) return jsx3.gui.Block.DISPLAYNONE; // hide if APC cell
		if (!objCell.isVideo()) return jsx3.gui.Block.DISPLAYNONE;
		if (objC.attr("AUDIO") != "on") return jsx3.gui.Block.DISPLAYNONE;
		return jsx3.gui.Block.DISPLAYBLOCK;
	};

	MagnifierCell_prototype.updateView = function(event, bRepaint) {

		if (this.isVNC()) {
			// Special case for VNC cell
			this.updateVNCView(event, bRepaint);
			return;
		}

		if (this.isWeb()) {
			// Special case for Web cell
			this.updateWebView(event, bRepaint);
			return;
		}

		var objCell = this._moncell;
		var blkDest = this._mainblock;

		//vMX._log("Painting cell: " + objCell.id + " srcObjid: " + objCell.srcObjid + " touring: " + (objCell.touring?true:false));
		var changed_items;
		if (event) {
			changed_items = event.changed_items;
			bRepaint = true;
		} else {
			changed_items = "all";
		}

		var lstRepaint = new jsx3.util.List(); // List of cell view items which need repaint

		var objSyncTargets = {}; // Argument for WallLayout.sync call, if needed

		var objC = resourceTree.getObject(objCell.srcObjid);

		vMX._log("changed_items: " + changed_items);
		// Let's enumerate blocks that may need repaint
		var imageCell; // Cell image
		var cellDevType; // Device type: camera, sensor, etc.
		var cellDevName; // cell header text
		var cellDevState; // Indicates Archive/Live type of video playing in the cell
		var stateImage; // Image in the center of the cell which indicates current state of the stream (play/pause/stopped)
		var blkMessage; // Block where warning messasges are displayed. Hidden by default
		var statePane; // Block where state imagebutton is placed
		var cellZoom; // zoom-in/zoom-out button
		var cellJoystick; // Joystick button
		var cellAnl; // Analytics button
		var cellStream; // Stream quality button
		var cellSound; // Sound on/off (currently doesn't work)
		var cellPopup; // Button for Alert/Popup cell

		//----------Tour cell-------------------------------------
		var cellTour; // pane where tour cell text is displayed
		var mxListTour; //Matrix containing list of tour cameras
		var delayText; //Text box displaying current delay value
		var delaySlider; // Slider control for selecting delay value

		// srcObjid changed
		if (changed_items.search(/srcObjid|all/) > -1 && this.isVideo()) {
			imageCell = blkDest.getDescendantOfName("image_cell");

			cellDevType = blkDest.getDescendantOfName("cell_dev_type");
			cellDevName = blkDest.getDescendantOfName("cell_dev_name");
			cellDevState = blkDest.getDescendantOfName("cell_dev_state");
			cellJoystick = blkDest.getDescendantOfName("cell_joystick");
			cellAnl = blkDest.getDescendantOfName("cell_anl");
			cellSound = blkDest.getDescendantOfName("cell_sound");

			var newDisplay = objCell.srcObjid ? jsx3.gui.Block.DISPLAYBLOCK : jsx3.gui.Block.DISPLAYNONE;
			if (objCell.groupid && objCell.groupid > 0)
				newDisplay = jsx3.gui.Block.DISPLAYNONE;

			if (cellDevState.getDisplay() != newDisplay) {
				cellDevType.setDisplay(objCell.srcObjid ? jsx3.gui.Block.DISPLAYBLOCK : jsx3.gui.Block.DISPLAYNONE);
				cellDevState.setDisplay(objCell.srcObjid ? jsx3.gui.Block.DISPLAYBLOCK : jsx3.gui.Block.DISPLAYNONE);
				lstRepaint.add(cellDevType);
				lstRepaint.add(cellDevState);
			}

			var newName = objCell.srcObjid ?
				(objC ? objC.name :
					objCell.text ? objCell.text : __("unknown device")) :
				__("No device");
			if (cellDevName.getText() != newName) {
				cellDevName.setText(newName);
				lstRepaint.add(cellDevName);
			}

			var newJoystickDisplay = this.getJoystickDisplay();
			if (cellJoystick.getDisplay() != newJoystickDisplay) {
				cellJoystick.setDisplay(newJoystickDisplay);
				lstRepaint.add(cellJoystick);
			}

			var newAnlDisplay = this.getAnlDisplay();
			if (cellAnl.getDisplay() != newAnlDisplay) {
				cellAnl.setDisplay(newAnlDisplay);
				lstRepaint.add(cellAnl);
			}

			var newSoundDisplay = this.getSoundDisplay();
			if (cellSound.getDisplay() != newSoundDisplay) {
				cellSound.setDisplay(newSoundDisplay);
				lstRepaint.add(cellSound);
			}

			imageCell.setSrc(vMX.getCamImage(objCell));
			lstRepaint.add(imageCell);
		}

		// Stream state changed (for standard cells only)
		if (changed_items.search(/state|all/) > -1 && this.isVideo()) {

			stateImage = blkDest.getDescendantOfName("state_image");
			blkMessage = blkDest.getDescendantOfName('message_player');
			cellZoom = blkDest.getDescendantOfName('cell_zoom');

			var cellTimezone = blkDest.getDescendantOfName("cell_timezone");
			if (!imageCell) imageCell = blkDest.getDescendantOfName("image_cell");
			if (!cellAnl) cellAnl = blkDest.getDescendantOfName('cell_anl');
			if (!cellStream) cellStream = blkDest.getDescendantOfName('cell_stream_quality');
			if (!cellSound) cellSound = blkDest.getDescendantOfName("cell_sound");

			if (blkMessage.getDisplay() == jsx3.gui.Block.DISPLAYBLOCK) {
				blkMessage.setDisplay(jsx3.gui.Block.DISPLAYNONE);
				imageCell.setDisplay(jsx3.gui.Block.DISPLAYBLOCK);
				stateImage.setDisplay(jsx3.gui.Block.DISPLAYBLOCK);
				lstRepaint.add(blkMessage);
				lstRepaint.add(imageCell);
			}

			if (objCell.state != "Play" && objCell.state != "Pause")
				stateImage.setCursor("default");
			else
				stateImage.setCursor("pointer");
			if (objCell.state == "Play")
				stateImage.setImage(vMX.STATEIMG["Pause"]);
			else if (objCell.state == "Pause")
				stateImage.setImage(vMX.STATEIMG["Play"]);
			else
				stateImage.setImage(vMX.STATEIMG[objCell.state]);

			stateImage.setDisplay(objCell.state == "Play" || objCell.state == "Pause" ? jsx3.gui.Block.DISPLAYBLOCK : jsx3.gui.Block.DISPLAYNONE);

			var newDisplay = (objCell.state == "Play" || objCell.state == "Pause") ? jsx3.gui.Block.DISPLAYBLOCK : jsx3.gui.Block.DISPLAYNONE;
			if (objCell.groupid && objCell.groupid > 0) { // Hide if cell is Alert/Popup one
				newDisplay = jsx3.gui.Block.DISPLAYNONE;
			}
			// Hide if no such camera in cache (probably current user has no credentials for this camera)
			if (!objC) {
				newDisplay = jsx3.gui.Block.DISPLAYNONE;
			}

			if (cellZoom.getDisplay() != newDisplay) {
				cellZoom.setDisplay(newDisplay);
				lstRepaint.add(cellZoom);
			}

			// Timezone button

			if (objCell.srcObjid && objC) {
				cellTimezone.setDisplay(jsx3.gui.Block.DISPLAYBLOCK);
				cellTimezone.setEvent('mx.TIME.applyCameraTimezone("' + objC.attr('TIME_ZONE') + '");', jsx3.gui.Interactive.EXECUTE);
			}
			else {
				cellTimezone.setDisplay(jsx3.gui.Block.DISPLAYNONE);
			}
			lstRepaint.add(cellTimezone);


			// Analytics button
			var newAnlDisplay = this.getAnlDisplay();
			if (cellAnl.getDisplay() != newAnlDisplay) {
				cellAnl.setDisplay(newAnlDisplay);
				lstRepaint.add(cellAnl);
			}

			var newStreamDisplay = this.getStreamDisplay();
			if (cellStream.getDisplay() != newStreamDisplay) {
				cellStream.setDisplay(newStreamDisplay);
				lstRepaint.add(cellStream);
			}

			var newSoundDisplay = this.getSoundDisplay();
			if (cellSound.getDisplay() != newSoundDisplay) {
				cellSound.setDisplay(newSoundDisplay);
				lstRepaint.add(cellSound);
			}

			objSyncTargets['group'] = objSyncTargets['archive'] = objSyncTargets['timeline'] = 1;

			lstRepaint.add(stateImage);
		}


		// Cell text changed. The only way to check license limitation
		if (changed_items.search(/text|all/) > -1 && this.isVideo()) {
			blkMessage = blkDest.getDescendantOfName('message_player');
			if (!imageCell) imageCell = blkDest.getDescendantOfName("image_cell");

			if (objCell.text.search(/License limit/i) > -1) {
				var strI = __("License limit exceeded.<br><i>Please contact your system administrator.</i>");
				var strM = "<table width='100%' height='100%'><tr><td align='center' valign='middle' style='font-size:12px;'>" + strI + "</td></tr></table>";
				blkMessage.setText(strM).setVisibility(jsx3.gui.Block.VISIBILITYVISIBLE).setDisplay(jsx3.gui.Block.DISPLAYBLOCK);
				imageCell.setDisplay(jsx3.gui.Block.DISPLAYNONE);
				lstRepaint.add(blkMessage);
				lstRepaint.add(imageCell);
			}
			else if (blkMessage.getDisplay() == jsx3.gui.Block.DISPLAYBLOCK) {
				blkMessage.setDisplay(jsx3.gui.Block.DISPLAYNONE);
				imageCell.setDisplay(jsx3.gui.Block.DISPLAYBLOCK);
				lstRepaint.add(blkMessage);
				lstRepaint.add(imageCell);
			}
		}


		// Cell stream switched to/from archive state. Non-tour cells only
		if (changed_items.search(/arc_from|arc_to|all/) > -1 && this.isVideo()) {
			if (!imageCell) imageCell = blkDest.getDescendantOfName("image_cell");
			if (!cellDevState) cellDevState = blkDest.getDescendantOfName("cell_dev_state");
			if (!cellJoystick) cellJoystick = blkDest.getDescendantOfName("cell_joystick");

			var newSrc = (objCell.arc_from > 0) ? vMX.MagnifierCell.IMG_ARCHIVE : vMX.MagnifierCell.IMG_LIVE;
			var newDisplay = this.getJoystickDisplay();
			if (cellDevState.getSrc() != newSrc) {
				cellDevState.setSrc(newSrc);
				lstRepaint.add(cellDevState);
			}
			if (cellJoystick.getDisplay() != newDisplay) {
				cellJoystick.setDisplay(newDisplay);
				lstRepaint.add(cellJoystick);
			}

			imageCell.setSrc(vMX.getCamImage(objCell));
			lstRepaint.add(imageCell);

			objSyncTargets['archive'] = objSyncTargets['timeline'] = objSyncTargets["group"] = 1;
		}


		// Archive position changed
		if (changed_items.search(/arc_pos|all/) > -1 && this.isVideo()) {
			objSyncTargets['archive'] = 1;
		}

		// Cell scale changed
		if (changed_items.search(/scale|all/) > -1 && ! this.isWeb()) {
			//vMX._log("scale changed. New scale: " + objCell.scale);
			if (!cellZoom) cellZoom = blkDest.getDescendantOfName('cell_zoom');
			if (!imageCell) imageCell = blkDest.getDescendantOfName('image_cell');

			if (objCell.state == "Play" || objCell.state == "Pause") {
				var newImage = (objCell.scale == "Proportional") ? vMX.MagnifierCell.IMG_ZOOM_PRESSED : vMX.MagnifierCell.IMG_ZOOM;
				if (cellZoom.getImage() != newImage) {
					cellZoom.setImage(newImage);
					lstRepaint.add(cellZoom);
				}
			}

			var newOpacity = objCell.scale == "Hidden" ? 0.2 : 1.0;
			if (vMX.getOpacity(imageCell) != newOpacity) {
				vMX.setOpacity(imageCell, newOpacity);
				lstRepaint.add(imageCell);
			}
		}

		// Analytics mode changed
		if (changed_items.search(/analytics|all/) > -1 && this.isVideo()) {
			if (!cellAnl) cellAnl = blkDest.getDescendantOfName('cell_anl');

			var newAnlDisplay = this.getAnlDisplay();
			if (newAnlDisplay == jsx3.gui.Block.DISPLAYBLOCK) {
				var anl = objCell.analytics == vMX.MonitorCell.NO_ANALYTICS ? 0 : objCell.analytics;
				var newMode = vMX.MonitorCell.ANL_MODES[anl];
				cellAnl.setImage("images/design1/gui_cell/cell_video_analytics_" + newMode + ".png");
				cellAnl.setTip(MagnifierCell.ANL_TIPS[anl]);
				cellAnl.setDisplay(jsx3.gui.Block.DISPLAYBLOCK);
			}
			else {
				cellAnl.setDisplay(jsx3.gui.Block.DISPLAYNONE);
			}
			lstRepaint.add(cellAnl);
		}

		// Stream number changed
		if (changed_items.search(/streamnum|all/) > -1 && this.isVideo()) {
			if (!cellStream) cellStream = blkDest.getDescendantOfName('cell_stream_quality');

			var newStreamDisplay = this.getStreamDisplay();
			if (newStreamDisplay == jsx3.gui.Block.DISPLAYBLOCK) {
				var streamnum = objCell.streamnum;
				if (streamnum == 1) {
					cellStream.setImage('images/design1/gui_cell/stream_normal.png')
					.setTip(__('Normal resolution'));
				} else {
					cellStream.setImage('images/design1/gui_cell/stream_low.png')
					.setTip(__('Low resolution'));
				}
				cellStream.setValue(streamnum);
			}
			else {
				cellStream.setDisplay(jsx3.gui.Block.DISPLAYNONE);
			}
			lstRepaint.add(cellStream);
		}

		// Volume changed
		if (changed_items.search(/volume|all/) > -1 && this.isVideo()) {
			if (!cellSound) cellSound = blkDest.getDescendantOfName('cell_sound');

			var newSoundDisplay = this.getSoundDisplay();
			if (newSoundDisplay == jsx3.gui.Block.DISPLAYBLOCK) {
				var volume = parseInt(objCell.volume);
				if (volume > 0) {
					cellSound.setImage("jsxuser:///images/design1/gui_cell/cell_audio_on.png");
				} else {
					cellSound.setImage("jsxuser:///images/design1/gui_cell/cell_audio_off.png");
				}
			}
			else {
				cellSound.setDisplay(jsx3.gui.Block.DISPLAYNONE);
			}
			lstRepaint.add(cellSound);
		}

		// Selected item changed in tour device list
		if (changed_items.search(/tour_selected_item|all/) > -1 && this.isTour()) {
			//vMX._log(4, "tours_selected_item changed");
			if (!mxListTour) mxListTour = blkDest.getDescendantOfName("list_tour");
			if (!cellDevName) cellDevName = blkDest.getDescendantOfName("cell_dev_name");
			if (!cellZoom) cellZoom = blkDest.getDescendantOfName('cell_zoom');
			if (!imageCell) imageCell = blkDest.getDescendantOfName("image_cell");

			var newVis = (objCell.state == "Play" || objCell.state == "Pause") ? jsx3.gui.Block.VISIBILITYVISIBLE : jsx3.gui.Block.VISIBILITYHIDDEN;
			if (cellZoom.getVisibility() != newVis) {
				cellZoom.setVisibility(newVis);
				lstRepaint.add(cellZoom);
			}

			var newName = objC ? objC.name : (objCell.text ? objCell.text : __("Touring Control"));
			cellDevName.setText(newName);
			lstRepaint.add(cellDevName);

			imageCell.setSrc(vMX.getCamImage(objCell.srcObjid));
			lstRepaint.add(imageCell);

			var lstRecs = mxListTour.getXML().selectNodes("//record[@obj_id]");
			//var objSelRec = mxListTour.getXML().selectSingleNode("//record[@obj_id='" + objCell.srcObjid + "']");
			var objSelRec = lstRecs.get(objCell.touring.pos);
			if (objSelRec) {
				mxListTour.deselectAllRecords();
				mxListTour.selectRecord(objSelRec.getAttribute("jsxid"));
			}
		}

		// Tour device list changed
		if (changed_items.search(/tour_source|all/) > -1 && this.isTour()) {
			//vMX._log(4, "tour_source changed");

			if (!mxListTour) mxListTour = blkDest.getDescendantOfName("list_tour");
			if (!cellDevName) cellDevName = blkDest.getDescendantOfName("cell_dev_name");
			if (!cellZoom) cellZoom = blkDest.getDescendantOfName('cell_zoom');
			if (!cellTour) cellTour = blkDest.getDescendantOfName("cell_tour_player");

			// Hide ptz-preset box
			this._mainblock.getDescendantOfName("block_ptzpreset").
				setVisibility(jsx3.gui.Block.VISIBILITYHIDDEN, true);

			var newVis = (objCell.touring.source.size()) ? jsx3.gui.Block.VISIBILITYVISIBLE : jsx3.gui.Block.VISIBILITYHIDDEN;
			if (cellZoom.getVisibility() != newVis) {
				cellZoom.setVisibility(newVis);
				lstRepaint.add(cellZoom);
			}

			var newText = (objCell.touring.source.size()) ? "" : "<table width='100%' height='100%'><tr><td align='center' valign='middle' style='font-size:12px;'>" + __("Drag and drop Device, Set of devices, or Role to the list on the right, to activate the touring.") + "</td></tr></table>";
			if (cellTour.getText().length != newText.length) {
				cellTour.setText(newText);
				lstRepaint.add(cellTour);
			}

			mxListTour.clearXmlData();
			var idSel, idFirst;
			for (i = 0; i < objCell.touring.source.size(); i++) {
				var dev = objCell.touring.source.get(i);
				var treeObject = resourceTree.getObject(dev.id);
				//if (!resourceTree.getObject(dev.id))
				//	continue;
				var text = treeObject ? treeObject.name : __("<unknown device>");
				var name = text;
				var ptz = treeObject ? treeObject.attr("POSITIONCTL") : "none";
				if (dev.presetid > 0) {
					name = text = Gettext.strargs(
						__('%1/preset %2'), [name, dev.presetid]
						);
				}
				var nodeRecord = {
					'jsxid'  : jsx3.xml.CDF.getKey(),
					'jsxtext': name,
					'obj_id' : dev.id,
					'name'   : name,
					'ptz'    : ptz,
					'preset' : dev.presetid,
					'pos'    : i
				};

				mxListTour.insertRecord(nodeRecord, null, false);
				if (dev.id == objCell.srcObjid) {
					idSel = nodeRecord.jsxid;
				}
			}
			if (objCell.touring.source.size() == 0) {
				cellDevName.setText(__("Touring control"));
			}

			mxListTour.deselectAllRecords();
			mxListTour.selectRecord(idSel);
			lstRepaint.add(mxListTour);
		}

		// Tour delay value changed
		if (changed_items.search(/tour_delay|all/) > -1 && this.isTour()) {
			//vMX._log("tour_delay changed");
			blkDest.getDescendantOfName("delay_txt_tour").setText(objCell.touring.delay).repaint();
			blkDest.getDescendantOfName("delay_slider_tour").setValue(objCell.touring.delay - 5).repaint();
		}

		// Cell APC status changed
		if (changed_items.search(/groupid|all/) > -1 && (this.isVideo() || this.isTour()) ) {
			vMX._log("Groupid changed. New groupid: " + objCell.groupid);
			if (!cellPopup) cellPopup = blkDest.getDescendantOfName('cell_popup');
			if (!cellDevState) cellDevState = blkDest.getDescendantOfName("cell_dev_state");
			if (!cellDevType) cellDevType = blkDest.getDescendantOfName("cell_dev_type");

			var newDisplay;

			if (objCell.groupid && objCell.groupid > 0) {
				cellPopup.setDisplay(jsx3.gui.Block.DISPLAYBLOCK);
				newDisplay = jsx3.gui.Block.DISPLAYNONE;
			}
			else {
				cellPopup.setDisplay(jsx3.gui.Block.DISPLAYNONE);
				newDisplay = objCell.srcObjid ? jsx3.gui.Block.DISPLAYBLOCK : jsx3.gui.Block.DISPLAYNONE;
			}

			if (cellDevType) {
				cellDevType.setDisplay(newDisplay);
					lstRepaint.add(cellDevType);
			}
			if (cellDevState) {
				cellDevState.setDisplay(newDisplay);
				lstRepaint.add(cellDevState);
			}

			lstRepaint.add(cellPopup);
		}

		// At last we have to repaint updated blocks
		if (bRepaint) {
			var lstReady = new jsx3.util.List();
			for (var i = 0; i < lstRepaint.size(); i++) {
				var item = lstRepaint.get(i);
				if (!lstReady.contains(item)) {
					item.repaint();
					lstReady.add(item);
				}
			}

			if (this.inGroup()) {
				this._magnifier.getWCView().getWallLayout().sync(objSyncTargets);
				// If cell is in group, check whether it should be removed from it
				if (objCell.srcObjid == 0) this.setSelected(false);
			}
		}
	};

	MagnifierCell_prototype.updateVNCView = function(event, bRepaint) {
		if (!this.isVNC()) return;

		var objCell = this._moncell;
		var blkDest = this._mainblock;

		var changed_items;
		if (event) {
			changed_items = event.changed_items;
			bRepaint = true;
		} else {
			changed_items = "all";
		}

		var lstRepaint = new jsx3.util.List(); // List of cell view items which need repaint
		vMX._log("changed_items: " + changed_items);

		if (changed_items.search(/vncurl|state|all/) > -1) {
			var blkUrl = blkDest.getDescendantOfName("vnc_url");
			var blkLogin = blkDest.getDescendantOfName("input_vnc_login");
			var blkPwd = blkDest.getDescendantOfName("input_vnc_password");
			var vncImage = blkDest.getDescendantOfName("vnc_image");

			// Parse new VNC URL
			var url="", login="", password="";
			var result = vMX.parseURL(objCell.vncurl);
			if (! result["part"]) {
				url = objCell.vncurl;
				login = password = "";
			}
			else {
				url = "vnc://" + result["part"];
				if (result["login"] != null) {
					login = result["login"];
				}
				if (result["password"] != null) {
					password = result["password"];
				}
			}
			// Update controls
			blkUrl.setValue(url);
			blkLogin.setValue(login);
			blkPwd.setValue(password);

			// Update image
			vncImage.setClassName("loadingIndication").setSrc(vMX.getCellSnapshot(
					    this.getMagnifier().getMonitor().objid,
					    objCell.id, "vnc"));

			if (bRepaint) {
				blkUrl.repaint();
				blkLogin.repaint();
				blkPwd.repaint();
				vncImage.repaint();
			}
		}
		if (changed_items.search(/state|all/) > -1) {
			var btnShare = blkDest.getDescendantOfName("buttonShare");
			// Update start-stop-sharing button view
			if (objCell.state == "Play") {
				btnShare.setImage(vMX.MagnifierCell.IMG_VNC_STOP);
				btnShare.setTip(__("Stop sharing"));
			}
			else {
				btnShare.setImage(vMX.MagnifierCell.IMG_VNC_START);
				btnShare.setTip(__("Start sharing"));
			}

			if (bRepaint) {
				btnShare.repaint();
			}
		}
	};

	MagnifierCell_prototype.updateWebView = function(event, bRepaint) {
		if (!this.isWeb()) return;

		var objCell = this._moncell;
		var blkDest = this._mainblock;

		var changed_items;
		if (event) {
			changed_items = event.changed_items;
			bRepaint = true;
		} else {
			changed_items = "all";
		}

		var lstRepaint = new jsx3.util.List(); // List of cell view items which need repaint
		vMX._log("changed_items: " + changed_items);

		if (changed_items.search(/weburl|state|all/) > -1) {
			var blkUrl = blkDest.getDescendantOfName("adressUrl");
			var imageCell = blkDest.getDescendantOfName("image_cell");
			var urlLabel = blkDest.getDescendantOfName("url_label");
			var buttonConfig = blkDest.getDescendantOfName("buttonConfig");
			var layoutCols = blkDest.getDescendantOfName("layout ( | )");

			// Update controls
			blkUrl.setValue(objCell.weburl);
			imageCell.setSrc(vMX.getCellSnapshot(
					    this.getMagnifier().getMonitor().objid,
					    objCell.id, "web"));

			// Hide URL pane for AlertLog cell
			if (objCell.isAlertLog()) {
				blkUrl.setVisibility(jsx3.gui.Block.VISIBILITYHIDDEN);
				urlLabel.setText("Alert Log");
				buttonConfig.setVisibility(jsx3.gui.Block.VISIBILITYHIDDEN);
				layoutCols.setCols("100,*,50", true);
			}
			else {
				blkUrl.setVisibility(jsx3.gui.Block.VISIBILITYVISIBLE);
				urlLabel.setText("URL");
				buttonConfig.setVisibility(jsx3.gui.Block.VISIBILITYVISIBLE);
				layoutCols.setCols("40,*,50", true);
			}

			if (bRepaint) {
				blkUrl.repaint();
				imageCell.repaint();
				urlLabel.repaint();
				buttonConfig.repaint();
			}
		}
		if (changed_items.search(/refresh|all/) > -1) {
			var newRefresh = objCell.refresh ? objCell.refresh : vMX.MagnifierCell.PrevRefreshInterval;
			this._mainblock.getDescendantOfName("intervalSec").setValue(newRefresh);
			if (bRepaint)
				this._mainblock.getDescendantOfName("intervalSec").repaint();
		}
	};

	MagnifierCell_prototype.onZoom = function(event) {
		if (this._moncell.state != "Play" && this._moncell.state != "Pause") {
			alert("Can't resize cell in '" + this._moncell.state + "' state");
		}
		else {
			// #773. VNC cell interferes with "Full Screen" mode of Video Cell
			// Output alert if any vnc cell is on screen
			var objMonitor = this.getMagnifier().getMonitor();
			var lstCells = objMonitor.getCellsList();

			if ( this._moncell.scale == "Cell" &&
				 lstCells.filter(
					function(c){ return c.isVNC()&&c.state!="Empty" }
				 ).size() )
			{
				alert(__("Zoom-In is not supported when VNC is active"));
			}
			else
			{
				var ctrlData = [];
				var ctrlStruct = {};
				ctrlStruct.jsxid = "0";
				ctrlStruct.cellid = this._moncell.id;
				ctrlStruct.monid = this._moncell.monitor.objid;
				ctrlStruct.text = this._moncell.text;
				ctrlStruct.state = this._moncell.state;
				ctrlStruct.volume = this._moncell.volume;
				if (this._moncell.scale == "Proportional")
					ctrlStruct.scale = "Cell";
				else
					ctrlStruct.scale = "Proportional";

				ctrlData[0] = ctrlStruct;
				vMX.controlCells(ctrlData);
			}
		}
	};

	MagnifierCell_prototype.onToggleStreamState = function(event) {
		vMX._log("onToggleStreamState");
		if (this._moncell.state != "Play" && this._moncell.state != "Pause") {
			vMX._log(3, "onToggleStreamState() called for cell with " + this._moncell.state + " stream!");
		}
		else {
			var ctrlData = [];
			var ctrlStruct = {};
			ctrlStruct.jsxid = "0";
			ctrlStruct.cellid = this._moncell.id;
			ctrlStruct.monid = this._moncell.monitor.objid;
			ctrlStruct.state = this._moncell.state == "Play" ? "Pause" : "Play";
			ctrlStruct.scale = this._moncell.scale;
			ctrlStruct.volume = this._moncell.volume;
			ctrlData.push(ctrlStruct);
			vMX.controlCells(ctrlData);
		}
	};


	/**
	 * Handles click on cell header imagebutton that switches stream
	 * to/from archive state
	 */
	MagnifierCell_prototype.onToggleStreamType = function(event) {
		if (!this._magnifier.getGroup().contains(this)) return; // selected cells only

		var c = this._moncell;
		if (c.arc_from <= 0 && c.arc_to <= 0)
			this._magnifier.groupSwitchToArchive(null, true);
		else
			this._magnifier.groupSwitchToLive();
	};

	/**
	 * Show PTZ dialog. Here we need to perform some suspicious
	 * and unclear manipulations in order to make mx.PTZ 'load' method
	 * to work correctly with our magnifier cell
	 */
	MagnifierCell_prototype.onCallPTZ = function(event) {
		if (!this._moncell.srcObjid) {
			vMX._log(2, "Tried to show PTZ for cell with empty stream: " + this._moncell.id);
			alert(__("Can't show PTZ dialog for cell with empty stream!"));
		}
		var cellBlk = this._mainblock.getChild("cell_blk");
		cellBlk.setAttribute("objectId", String(this._moncell.srcObjid));
		cellBlk.setAttribute("cellId", jsx3.xml.CDF.getKey());
		mx.PTZ.load(event.target);
	};

	MagnifierCell_prototype.onPopupAlert = function(event) {
		if (!parseInt(this._moncell.eventid))
			return;
		vMX.getEventInfo(this._moncell.eventid);
	};

	/*
	 * Analytics button handler
	 * Cycles over 4 analytics modes
	 */
	MagnifierCell_prototype.onSetAnalytics = function(event) {
		var c = this._moncell;
		var anl = parseInt(c.analytics);
		if (anl == null || anl == vMX.MonitorCell.NO_ANALYTICS)
			anl = 0;

		if (anl == vMX.MonitorCell.ANL_MODES.length - 1)
			anl = 0;
		else
			anl = anl + 1;

		var cellProps = c.getProperties();
		cellProps.analytics = anl;
		vMX.assignStream( [cellProps] );
		return true;
	}


	/*
	 * Stream switch button handler
	 * Cycles over 2 stream quality modes: low and normal
	 */
	MagnifierCell_prototype.onChangeStream = function(event) {
		var c = this._moncell;
		var streamnum = parseInt(c.streamnum);
		if (streamnum == null || streamnum < 0)
			streamnum = vMX.getDefaultStreamNumber(c.srcObjid);
		else
			streamnum = streamnum === vMX.MonitorCell.STREAM_QUALITY_LOW  ?
						vMX.MonitorCell.STREAM_QUALITY_NORMAL :
						vMX.MonitorCell.STREAM_QUALITY_LOW;

		var cellProps = c.getProperties();
		cellProps.streamnum = streamnum;
		vMX.assignStream( [cellProps] );
		return true;
	}

	MagnifierCell_prototype.onSetSound = function(event) {
		var c = this._moncell;
		var vol = 0;
		if (c.volume != vMX.MonitorCell.SOUND_MUTE) {
			vol = vMX.MonitorCell.SOUND_MUTE;
		} else {
			vol = vMX.MonitorCell.SOUND_MAX;
		}

		var ctrlData = [
    		    {
                	'jsxid':'0', 'cellid':c.id, 'state':c.state, 'scale':c.scale, 'text':c.text,
			'monid':this._moncell.monitor.objid, 'volume' : String(vol)
		    }
		];
		vMX.controlCells(ctrlData);
	}


	MagnifierCell_prototype.onClose = function(event) {

		//if (this._moncell.srcObjid) {
			var ctrlData = [
				{ 'jsxid':'0', 'cellid':this._moncell.id, 'state':'Empty', 'scale':'Cell', 'text':"", 'monid':this._moncell.monitor.objid }
			];
			vMX.controlCells(ctrlData);
		//}
	};

	/********************************************************************************************
	 *                               VNC cell operations                                         *
	 *******************************************************************************************/

	MagnifierCell_prototype.onStartVNCSharing = function(event) {
		if (! this.isVNC()) return;
		this.startVNCSharing();
	};

	MagnifierCell_prototype.onStartStopVNCSharing = function(event) {
		if (! this.isVNC()) return;

		if (this._moncell.state == "Play")
			this.stopVNCSharing();
		else
			this.startVNCSharing();
	};

	MagnifierCell_prototype.startVNCSharing = function() {
		if (! this.isVNC()) return;

		// Check for full-screen cells
		// VNC sharing is prohibited when any other cell is in full-screen mode
		var objMonitor = this.getMagnifier().getMonitor();
		var lstCells = objMonitor.getCellsList();
		if ( ! this._moncell.isFullScreen() &&
				 lstCells.filter(
					function(c){ return c.isFullScreen() }
				 ).size() )
		{
			alert(__("Zoom-In is not supported when VNC is active"));
			return;
		}

		// parse given URL, Login and Password and
		// try to constuct resulting URL for vMX
		var url = this._mainblock.getDescendantOfName("vnc_url").getValue();
		var result = vMX.parseURL(url);
		if (! result["part"]) {
			// Wrong URL format!
			alert(__("Bad URL format"));
			return;
		}
		var login = "", password = "";
		if (
			 (result["login"] != null && result["login"].length > 0) ||
			 (result["password"] != null && result["password"].length > 0)
			)
		{
			login = result["login"];
			password = result["password"];
		}
		else {
			login = this._mainblock.getDescendantOfName("input_vnc_login").getValue();
			password = this._mainblock.getDescendantOfName("input_vnc_password").getValue();
		}
		if (login == null) login = "";
		if (password == null) password = "";

		var urlSend = "vnc://";
		if (login.length > 0 || password.length > 0) {
			urlSend += login + ":" + password + "@";
		}
		urlSend += result["part"];
		var urlShow = "vnc://" + result["part"];

		// Update View
		this._mainblock.getDescendantOfName("vnc_url").
			setValue(urlShow).repaint();
		this._mainblock.getDescendantOfName("input_vnc_login").
			setValue(login).repaint();
		this._mainblock.getDescendantOfName("input_vnc_password").
			setValue(password).repaint();

		this.assignURL(urlSend);
	};

	MagnifierCell_prototype.stopVNCSharing = function() {
		if (! this.isVNC()) return;

		var ctrlData = [
			{ 'jsxid':'0', 'cellid':this._moncell.id, 'state':'Stop', 'scale':'Cell', 'text':"", 'monid':this._moncell.monitor.objid }
		];
		vMX.controlCells(ctrlData);
	};

	MagnifierCell_prototype.onShowHideVNCConfig = function(event) {
		var layout = this._mainblock.getDescendantOfName("layoutVNC");
		var btn = event.target;
		var rows = layout.getRows();
		var isShown = rows.split(',')[1] > 0 ? true : false;

		if(!isShown){
			layout.setRows("25,25,*",true);
			btn.setImage("jsxuser:///images/design1/url/url_collapse.png");
			btn.setTip(__("Hide VNC config"));
		} else {
			layout.setRows("25,0,*",true);
			btn.setImage("jsxuser:///images/design1/url/url_expand.png");
			btn.setTip(__("Show VNC config"));
		}
		btn.repaint();
	};


	/********************************************************************************************
	 *                               Web cell operations                                        *
	 *******************************************************************************************/

	 MagnifierCell_prototype.onOpenWebURL = function(event) {
		 if (! this.isWeb())
			return;
		 this.openWebURL();
	 };

	 MagnifierCell_prototype.openWebURL = function() {
		 if (! this.isWeb()) return;

		 var urlBox = this._mainblock.getDescendantOfName("adressUrl");
		 var url = urlBox.getValue();

		 var refresh = this.getRefresh();

		 var protocol = "http:";
		 if(url.search(/:\/\//g)!=-1) {
			 protocol = url.slice(0, url.search(/:\/\//g)+1).toLowerCase();
			 url = url.slice(url.search(/:\/\//g)+3, url.length).toLowerCase();
		 }

		 url = protocol+"//"+url;
		 urlBox.setValue(url);

		 this.assignURL(url, "Play", refresh);
	 };

	 MagnifierCell_prototype.onCheckRefresh = function(event) {
		 this.openWebURL();
	 };


	 MagnifierCell_prototype.assignURL = function(url, state, refresh) {
		vMX.assignURL(
			 this._moncell.monitor.objid,
			 this._moncell.id,
			 url,
			 state,
			 refresh
		);

		// Schedule snapshot update
		var self = this;
		setTimeout(function() {
			if (!self.isWeb() && !self.isVNC())
				return;
			var url = mx.vMX.getCellSnapshot(
				    self._moncell.monitor.objid,
				    self._moncell.id,
				    (self.isWeb() ? "web" : "vnc"));
			var img = new Image();

			var moncell = self._moncell;
			img.onload = img.onerror = function() {
				moncell.publish( {'subject':mx.vMX.WallLayout.UPDATE_SNAPSHOTS,
					          'target':moncell, 'url':url,
					          'layout':self._magnifier.getWCView().getWallLayout()} );
			};
			img.src = url;
		}, 5000);
	 };

	/********************************************************************************************
	 *                               Touring operations                                         *
	 *******************************************************************************************/


	/**
	 * Fires when user drag&drops a camera or a set of cameras on the tour list area
	 * Result: devices are added to the list
	 */
	MagnifierCell_prototype.onListTourDrop = function(obj) {

		vMX._log("onListTourDrop:");

		if (!this.isTour()) {
			vMX._log(2, "onListTourDrop() called for non-tour cell!");
			return false;
		}

		if (obj.type != 'camera' && obj.type != 'set' && obj.type != 'role') {
			alert(__("Only Camera, Role or Set could be added to touring component. Please select appropriate type."));
			return;
		}
		if (parseInt(this._moncell.groupid)) {
			alert("Cannot drop device on APC cell");
			return;
		}

		var objIds = [];
		switch (obj.type) {
			case 'camera': objIds = resourceTree.getObjects({'obj': [obj.obj], 'cred': 'L'});
			break;
			case 'set': objIds = resourceTree.getObjects({'obj': obj.objects, 'type': 'camera', 'cred': 'L'});
			break;
			case 'role': objIds = resourceTree.getObjects({'type': 'camera', 'cred': 'L'});
		}

		// Copy all touring cams to temporary list
		var listSrc = new jsx3.util.List(this._moncell.touring.source);
		var delay = this._moncell.touring.delay;

		$.each(objIds, function(i, v) {
			/*if (listSrc.filter(
				function(o) {
					return Boolean(o.id == v);
				}).size()) {
				return;
			}*/
			// Now add camera to the list
			listSrc.add({id:v, options:"", presetid:"0"});
		});

		// logging
		var str = "";
		for (var i = 0; i < listSrc.size(); i++) {
			str += listSrc.get(i).id + " ";
		}
		vMX._log("tourOnDrop: " + str);

		if (!this._moncell.touring.source.equals(listSrc))
			vMX.setTouring(this._moncell.monitor.objid, this._moncell.id, delay, listSrc);

		return false;
	};

	/**
	 * Fires when user clicks on a small red cross placed left to the camera name on the tour device list
	 * Result: camera is removed from the list
	 */
	MagnifierCell_prototype.listTourObjectDelete = function(event) {

		if (!this.isTour()) {
			vMX._log(2, "listTourObjectDelete called for a non-tour cell");
			return;
		}

		var objThis = event.target;
		var id = objThis.getParent().getParent().getRecord(objThis.emGetSession().recordId).obj_id;
		var pos = objThis.getParent().getParent().getRecord(objThis.emGetSession().recordId).pos;

		var delay = this._mainblock.getDescendantOfName("delay_txt_tour").getText() || 15;
		var srcList = this._moncell.touring.source.clone();
		srcList.removeAt(pos);
		vMX.setTouring(this._moncell.monitor.objid, this._moncell.id, delay, srcList);
	};

	/**
	 * Handles click on 'open/close' button, which is located on the cell header near the 'close' one
	 * Result: tour device list is opened/collapsed
	 */
	MagnifierCell_prototype.listOpenClose = function(event) {
		var objLayout = this._mainblock.getDescendantOfName("layout_tour");
		var strCols = objLayout.getCols();
		if (strCols.endsWith(",0"))
			objLayout.setCols("*,50%", true);
		else
			objLayout.setCols("*,0", true);
	};

	/**
	 * Handler for 'close' button. Removes the tour from the cell.
	 * Result: stream is closed, if cell has tour, it will be removed
	 */
	MagnifierCell_prototype.btnTourClose = function(event) {
		if (!this.isTour()) {
			vMX._log(2, "btnTourClose() called for a non-tour cell!");
		} else {
			vMX.assignStream(this._moncell.srcObjid, this._moncell.monitor.objid, this._moncell.id, null, "Empty");
		}
	};

	/**
	 * Fires when user drags the tour slider
	 */
	MagnifierCell_prototype.delaySliderChange = function(event) {
		if (!this.isTour()) {
			vMX._log(2, "btnTourClose() called for a non-tour cell!");
		} else {
			var fpValue = event.context.fpVALUE;
			this._mainblock.getDescendantOfName('delay_txt_tour').setText(Math.floor(fpValue) + 5).repaint();
			vMX.setTouring(this._moncell.monitor.objid, this._moncell.id, Math.floor(fpValue) + 5, this._moncell.touring.source.toArray(1));
		}
	};

	MagnifierCell_prototype.onTourSelectRecord = function(event) {
		var strRecordId = event.context.strRECORDID;
		var objid = this._mainblock.getDescendantOfName("list_tour").getRecord(strRecordId).obj_id;
		var imageCell = this._mainblock.getDescendantOfName("image_cell");
		imageCell.setClassName("loadingIndication").setSrc(vMX.getCamImage(objid)).repaint();

		var objC = resourceTree.getObject(objid);
		var cellDevName = this._mainblock.getDescendantOfName("cell_dev_name");
		var newName = "";
		if (objid) {
			if (objC) {
				newName = objC.name;
			}
			//else if (this._moncell.text) {
			//	newName = this._moncell.text;
			//}
			else {
				newName = __("unknown device");
			}
		}
		else {
			newName = __("No device");
		}
		cellDevName.setText(newName, true);
	};

	MagnifierCell_prototype.btnTourSettings = function(event) {
		var objLayout = this._mainblock.getDescendantOfName("layout_settings");
		var strRows = objLayout.getRows();
		if (strRows.endsWith(",0"))
			objLayout.setRows("*,45", true);
		else
			objLayout.setRows("*,0", true);
	};

	MagnifierCell_prototype.editPTZPreset = function(event) {
		var objMatrix = event.target;
		var strId = event.context.strRECORDID;
		var objEvent = event.context.objEVENT;

		var cell = objMatrix.getAncestorOfName("cell_blk");
				var doc = objMatrix.getXML();
				var node = doc.selectSingleNode("//record[@jsxid='" + strId + "']");

				if(node.getAttribute("ptz") == "none"){
						return false;
				}

				var inputPreset = cell.getDescendantOfName("textboxPtzPreset");
				if(node && node.getAttribute("preset") != ""){
						inputPreset.setValue(node.getAttribute("preset"), true);
				}

		var btnPreset = cell.getDescendantOfName("buttonPreset");
		btnPreset.setEvent("1;", jsx3.gui.Interactive.EXECUTE);
		btnPreset.subscribe(jsx3.gui.Interactive.EXECUTE, this, "setPTZPreset");

		var top = objEvent.clientY() - objMatrix.getAbsolutePosition().T + 5;
				cell.getDescendantOfName("block_ptzpreset").
			setTop(top, true).
			setHeight(26, true).
			setAttribute("edit_id",strId).
			setAttribute("start_value",node.getAttribute("preset")).
			setVisibility(jsx3.gui.Block.VISIBILITYVISIBLE, true);


		return true;
	};

	MagnifierCell_prototype.setPTZPreset = function(event) {
		var cell = event.target.getAncestorOfName("cell_blk");
		var preset = cell.getDescendantOfName("textboxPtzPreset").getValue();
		var blkPreset = cell.getDescendantOfName("block_ptzpreset");

		if (preset != null && preset != "" && preset != parseInt(preset)) {
						blkPreset.setHeight(50, true);
						cell.getDescendantOfName("textMsgPreset").
				setText(__("Wrong preset id!"), true).
				setColor("#FF0000", true).
				setDisplay(jsx3.gui.Block.DISPLAYBLOCK, true);
						return;
				}
		else {
			if (preset == null || preset == "")
				preset = 0;
			var matrix = cell.getDescendantOfName("list_tour");
			var strId = blkPreset.getAttribute("edit_id");
			var startVal = blkPreset.getAttribute("start_value");
			var record = matrix.getXML().selectSingleNode("//record[@jsxid='" + strId + "']");
					record.setAttribute("preset", preset);

	                var text = "";
	                var curText = record.getAttribute("name");
	                if(preset != "" && preset != "0"){
	                        text = "/preset " + preset;
	                        curText = curText.replace(/\/preset \d+$/, "");
	                }
	                record.setAttribute("jsxtext", curText + text);
	                matrix.repaint();

			if(parseInt(preset) != parseInt(startVal)) {
				var srcList = this._moncell.touring.source.clone();
				var elem = srcList.get(record.getAttribute("pos"));
				if (elem) {
					elem.presetid = preset;
				}
				else {
					// No such element in touring
				}
				vMX.setTouring(
					this._moncell.monitor.objid,
					this._moncell.id,
					this._moncell.touring.delay,
					srcList
				);
			}
		}

		blkPreset.setVisibility(jsx3.gui.Block.VISIBILITYHIDDEN, true);
	};
}
	);
