//------------------------------------------------------------------------------------------
// vMX.Magnifier class
//------------------------------------------------------------------------------------------


jsx3.lang.Class.defineClass("mx.vMX.Magnifier", jsx3.lang.Object, [mx.vMX.CellContainer], function(Magnifier, Magnifier_prototype) {

	var vMX = mx.vMX;

	/**
	 * Construction
	 */
	Magnifier_prototype.init = function(wcview) {
		if (!wcview.getWallcell().monitor)
			throw new jsx3.lang.Exception(__("Can't create magnifier for empty wall cell!"));

		var objMonitor = wcview.getWallcell().monitor;
		var blkParent = wcview.getWallLayout().getContentBlock();

		$('#' + blkParent.getId() + ' .drop').each(function() {
			mx.MATRIX2.getServer().getJSXById(this.id).setClassName('drop-disable').repaint();
		});

		var pos = blkParent.getAbsolutePosition();
		var coeff = 3 * (pos.W / objMonitor.hres + pos.H / objMonitor.vres) / 8;
		var dlg_x = (pos.W - objMonitor.hres * coeff) / 2;
		var dlg_y = (pos.H - objMonitor.vres * coeff) / 2;
		var dlg_w = Math.floor(objMonitor.hres * coeff) + 8 + 2;
		var dlg_h = Math.floor(objMonitor.vres * coeff) + 36 + 2;
		var spl_wmin = (dlg_w - 4) / objMonitor.hres;
		var spl_hmin = (dlg_h - 42) / objMonitor.vres;

		if (dlg_x < 0) dlg_x = 0;
		if (dlg_y < 0) dlg_y = 0;
		if (dlg_w > pos.W) dlg_w = pos.W;
		if (dlg_h > pos.H) dlg_h = pos.H;

		var maindlg = new jsx3.gui.Dialog("magnifier");
		var captionBar = maindlg.getCaptionBar();
		//captionBar.setText("Magnifier [monitor id:" + objMonitor.objid + "]");
		captionBar.setText("<div style='position: absolute; top: 0px; left: 0px; " +
			"background-image: url(images/design1/vMX/header.png); " +
			"width: 49px; height: 11px;'></div>");
		captionBar.setHeight(22);
		captionBar.setBackground("background-image: none;");
		captionBar.setBackgroundColor("transparent");
		captionBar.setBorder("0px solid #E4E4E4");

		maindlg.setBackgroundColor("#E4E4E4");
		maindlg.setCSSOverride("border-color: white #949B9E #949B9E white;" +
			"border-style: solid; border-width: 1px; overflow: hidden;");

		maindlg.setLeft(dlg_x);
		maindlg.setTop(dlg_y);
		maindlg.setHeight(dlg_h);
		maindlg.setWidth(dlg_w);

		//AFTER_MOVE and BEFORE_RESIZE events are handled
		maindlg.subscribe(jsx3.gui.Interactive.AFTER_MOVE, this, "onMoveResize");
		maindlg.subscribe(jsx3.gui.Interactive.AFTER_RESIZE, this, "onMoveResize");

		blkParent.setChild(maindlg);
		maindlg.getCaptionBar().removeChild(0);
		maindlg.getCaptionBar().removeChild(0);

		// Add handler for 'close' action
		var btnClose = captionBar.getChild(0);
		btnClose.removeEvent(jsx3.gui.Interactive.EXECUTE);
		btnClose.subscribe(jsx3.gui.Interactive.EXECUTE, this, "onClose");
		btnClose.setDynamicProperty("jsximage", null);
		btnClose.setImage("images/design1/gui_cell/cell_close.png");

		// Add 'maximize' button to magnifier caption bar
		var jButton = new jsx3.gui.ToolbarButton(maindlg.getName() + "_btn_max", null);
		jButton.subscribe(jsx3.gui.Interactive.EXECUTE, this, "onToggleState");
		jButton.setImage("images/design1/gui_cell/cell_full_screen.png");
		jButton.setTip(__("Maximize/restore"));
		captionBar.insertBefore(jButton, btnClose);

		// Add 'clear' button to magnifier caption bar
		var btnClear = new jsx3.gui.ToolbarButton(maindlg.getName() + "_btn_clear", null);
		btnClear.subscribe(jsx3.gui.Interactive.EXECUTE, this, "onClearCells");
		btnClear.setImage("images/design1/vMX/trash.png");
		btnClear.setTip(__("Clear cells"));
		captionBar.insertBefore(btnClear, jButton);


		var blkCellPane = new jsx3.gui.Block('cellpane', 0, 0, '100%', '100%');
		blkCellPane.setBackgroundColor("#5F696D");
		blkCellPane.setCSSOverride('overflow: hidden');
		maindlg.setChild(blkCellPane);

		this._wcview = wcview;
		this._group = new jsx3.util.List;
		this._cells = new jsx3.util.List;
		this._maindlg = maindlg;
		this._coeff = coeff;
		this._wmin = spl_wmin * vMX.MIN_CELL_WIDTH;
		this._hmin = spl_hmin * vMX.MIN_CELL_HEIGHT;
		this.state = "window";

		maindlg.subscribe(jsx3.gui.Interactive.AFTER_RESIZE, this, "onMagnifierResize");
		objMonitor.subscribe(vMX.Monitor.NEW_LAYOUT, this, "repaint");
		wcview.getWallcell().subscribe(vMX.WallCell.BEFORE_REMOVE_MONITOR, this, "onClose");

		mx.MATRIX2.getServer().getJSXByName('eventInfoDialog').setVisibility(jsx3.gui.Block.VISIBILITYHIDDEN).repaint();
	};


	/**
	 * Clear current magnifier layout and paints a new one according to the state of the
	 * associated vMX.Monitor object
	 */
	Magnifier_prototype.repaint = function() {
		vMX._log(4, "repaint dialog starts");
		var objMonitor = this.getMonitor();
		if (objMonitor.state == "connected") {
			vMX.stopRefresh();
			vMX.Magnifier._INIT = 1;
			vMX.Magnifier._PAINTED_CELLS = {};
			if (vMX.Magnifier._STARTELOG_TIMEOUT) {
				window.clearTimeout(vMX.Magnifier._STARTELOG_TIMEOUT);
				vMX.Magnifier._STARTELOG_TIMEOUT = 0;
			}

			try {
				mx.ELog2.stopELog();
			} catch(e) {
				vMX._log(3, "Got exception when stopping ELog")
			}

		}

		var blkCellPane = this._maindlg.getChild("cellpane") || this._maindlg.getChild(1);

		// First unsubscribe and then look whether we can resubscribe
		objMonitor.unsubscribe(vMX.Monitor.CELL_RESIZE, this);
		objMonitor.unsubscribe(vMX.Monitor.CELL_REPLACE, this);

		if (objMonitor.state != "connected") {
			// Display warning message and return
			var strMessage = objMonitor.state == "disconnected" ? "DISCONNECTED" : "Monitor isn't licensed";
			var strText = "<table width='100%' height='100%'><tr><td><center><font size='3'>" +
				strMessage + "</font></center></td></tr></table>";
			var blkMessage = new jsx3.gui.Block("message_block", 0, 0, "100%", "100%", strText);
			blkCellPane.removeChildren();
			blkCellPane.setBackgroundColor(vMX.getServer().getJSXByName('block_main').getBackgroundColor());
			blkMessage.setRelativePosition(jsx3.gui.Block.ABSOLUTE);
			blkCellPane.setChild(blkMessage);

			this._group.clear();
		}
		else {

			// do cleanup before constructing new layout
			this.clearCells();
			blkCellPane.removeChildren();

			// If monitor in 'ok' state
			if (!this.doSplit(blkCellPane)) {
				// Failed to paint matrix2-like layout, gonna paint simple one
				vMX._log(3, "Failed to create matrix2-like layout in magnifier!");
				this._maindlg.setResize(jsx3.gui.Dialog.FIXED);
				this._maindlg.getDescendantOfName(this._maindlg.getName() + "_btn_max").
					setDisplay(jsx3.gui.Block.DISPLAYNONE);

				if (this.matrix !== false) {
					alert(__("Failed to create Matrix2-like layout. You will not be able to resize Magnifier and change its layout"));
				}
				for (var i = objMonitor.iterator(), j = 0; i.hasNext(); j++) {
					var c = i.next();
					var blkCell = new jsx3.gui.Block('blkCell' + j, parseInt(c.x * this._coeff),
						parseInt(c.y * this._coeff),
						parseInt(c.w * this._coeff),
						parseInt(c.h * this._coeff));
					blkCell.setRelativePosition(jsx3.gui.Block.ABSOLUTE);
					var magnCell = new vMX.MagnifierCell(c, blkCell, this);
					blkCellPane.setChild(blkCell);
					magnCell.create();
					magnCell.updateView();
				}
				this.matrix = false;
			} else {
				this.matrix = true;
				this._maindlg.setResize(jsx3.gui.Dialog.RESIZABLE);
				this._maindlg.getDescendantOfName(this._maindlg.getName() + "_btn_max").
					setDisplay(jsx3.gui.Block.DISPLAYBLOCK);

				objMonitor.subscribe(vMX.Monitor.CELL_RESIZE, this, "resizeSplitters");
				objMonitor.subscribe(vMX.Monitor.CELL_REPLACE, this, "onCellReplace");
			}
		}

		this._maindlg.getParent().repaint();

		this._wcview.getWallLayout().sync({'group' : 1, 'archive' : 1, 'timeline' : 1});
	};

	/**
	 * Recurses the magnifier child blocks and does the splitting into cells
	 */
	Magnifier_prototype._recursiveSplit = function(obj, block, bRepaintCell) {
		block.setAttribute("x", String(obj.x)).setAttribute("y", String(obj.y));
		block.setAttribute("w", String(obj.w)).setAttribute("h", String(obj.h));

		if (obj.orientation != "cell") {
			var spl = vMX.addSplitter(block, obj.orientation, obj.subcontainer1pct);
			spl.subscribe(jsx3.gui.Interactive.AFTER_RESIZE, this, "onSplitterResize");
			spl.setSubcontainer1Min(obj.orientation == jsx3.gui.Splitter.ORIENTATIONH ? this._wmin : this._hmin);
			spl.setSubcontainer2Min(obj.orientation == jsx3.gui.Splitter.ORIENTATIONH ? this._wmin : this._hmin);
			this._recursiveSplit(obj.region1, spl.getFirstChild());
			this._recursiveSplit(obj.region2, spl.getLastChild());
		} else {
			var magnCell = new vMX.MagnifierCell(obj.cells.get(0), block, this);
			magnCell.create();
			magnCell.updateView(null, this.getMonitor().getCellsList().size() > 1 ? bRepaintCell : true);
		}
	};

	/**
	 * Performs splitting of magnifier into cells
	 */
	Magnifier_prototype.doSplit = function(blkCellPane) {
		var objRes = {};
		var objMonitor = this.getMonitor();

		if (!vMX.recursiveScan(0, 0, objMonitor.hres, objMonitor.vres, objMonitor.getCellsList(), objRes))
			return false;
		this._recursiveSplit(objRes, blkCellPane);

		return true;
	};


	/**
	 * Destroys magnifier window
	 */
	Magnifier_prototype.doClose = function() {
		this.onClose();
	};


	Magnifier_prototype.getWallcell = function() {
		return this._wcview.getWallcell();
	};

	Magnifier_prototype.getWCView = function() {
		return this._wcview;
	};

	Magnifier_prototype.getDialog = function() {
		return this._maindlg;
	};

	Magnifier_prototype.getMonitor = function() {
		return this._wcview.getWallcell().monitor;
	};

	Magnifier_prototype.getGroup = function() {
		return this._group;
	};

	Magnifier_prototype.getCellById = function(cellid) {
		return this._cells.filter(function(c) {
			return c.getMainBlock().getName() == cellid ? true : false
		}).get(0);
	};

	Magnifier_prototype.toString = function() {
		return "Magnifier: " + this._maindlg;
	};


	/**
	 * Overrides 'removeCell' from base interface
	 */
	Magnifier_prototype.removeCell = function(cell) {
		cell.setContainer(null);
		this._cells.remove(cell);
		this._group.remove(cell);
	};

	Magnifier_prototype.recalcMinCellSize = function() {
		var cellpane = this._maindlg.getChild("cellpane");
		if (cellpane == null)
			return;

		var pos = cellpane.getAbsolutePosition();
		var minW = pos.W / this.getMonitor().hres * vMX.MIN_CELL_WIDTH;
		var minH = pos.H / this.getMonitor().vres * vMX.MIN_CELL_HEIGHT;
		var arrSplitters = this._maindlg.getDescendantsOfType("jsx3.gui.Splitter");
		for (var i = 0; i < arrSplitters.length; i++) {
			var spl = arrSplitters[i];
			if (spl.getOrientation() == jsx3.gui.Splitter.ORIENTATIONH) {
				spl.setSubcontainer1Min(minW);
				spl.setSubcontainer2Min(minW);
			} else {
				spl.setSubcontainer1Min(minH);
				spl.setSubcontainer2Min(minH);
			}
		}
	};


	/****************************************************
	 * Event handlers                                   *
	 ***************************************************/

	/**
	 * Method is called after repaint of a single magnifier cell is completed.
	 */
	Magnifier_prototype.onAfterCellRepaint = function(cellid) {

		if (!vMX.Magnifier._INIT) return;

		if (!vMX.Magnifier._PAINTED_CELLS) vMX.Magnifier._PAINTED_CELLS = {};
		var bPainted = vMX.Magnifier._PAINTED_CELLS[cellid];
		if (bPainted)
			vMX.Magnifier._PAINTED_CELLS[cellid]++;
		else
			vMX.Magnifier._PAINTED_CELLS[cellid] = 1;

		var arrCells = this.getCellsArray();
		var bReady = true;
		for (var i = 0; i < arrCells.length; i++) {
			var c = arrCells[i].getMonCell();
			if (vMX.Magnifier._PAINTED_CELLS[c.id] == null ||
				vMX.Magnifier._PAINTED_CELLS[c.id] < 2) {
				bReady = false;
				break;
			}
		}

		if (bReady) {
			vMX._log(3, "Magnifier repaint ready!!!");
			vMX.Magnifier._PAINTED_CELLS = {};
			// Restart wall state refreshing
			if (!vMX.QUERY_STATE_ENABLED) vMX.startRefresh();
			// Restart eventlog
			if (vMX.Magnifier._STARTELOG_TIMEOUT)
				window.clearTimeout(vMX.Magnifier._STARTELOG_TIMEOUT);
			vMX.Magnifier._STARTELOG_TIMEOUT =
				window.setTimeout("mx.vMX.Magnifier._STARTELOG_TIMEOUT = 0;" +
					"if(!mx.ELog2.hidden) mx.ELog2.startELog()", 5000);
		}

	};

	/**
	 * Fires when magnifier is dragged to a new position with a mouse or
	 * when it is resized by pressing 'Maximize' button
	 */
	Magnifier_prototype.onMoveResize = function(event) {

		// toggle magnifier to "window" state
		if (this.state == "max") {
			this.state = "window";
			//this._maindlg.getCaptionBar().getChild(0).setDynamicProperty("jsximage","@Max Icon").repaint();
			this.pos = null;
		}

		// normalize dialog position if it was moved out of the active pane
		var x = this._maindlg.getLeft(), y = this._maindlg.getTop(),
			w = this._maindlg.getWidth(), h = this._maindlg.getHeight();
		var ppos = this._maindlg.getParent().getAbsolutePosition();

		if (x < 0 || y < 0 || (x + w > ppos.W) || (y + h > ppos.H)) {

			if (x < 0) this._maindlg.setLeft(0, true);
			if (y < 0) this._maindlg.setTop(0, true);

			if (w > ppos.W)
				this._maindlg.setWidth(ppos.W, true);
			else if (x + w > ppos.W)
				this._maindlg.setLeft(ppos.W - w, true);
			if (h > ppos.H)
				this._maindlg.setHeight(ppos.H, true);
			else if (y + h > ppos.H)
				this._maindlg.setTop(ppos.H - h, true);

		}

		//this._wcview.getWallLayout().objPainting.paintTraceLines(this._maindlg);
	};

	/**
	 * Handler for AFTER_RESIZE event for magnifier dialog
	 */
	Magnifier_prototype.onMagnifierResize = function(event) {
		this.recalcMinCellSize();
	};


	/**
	 * Handles vMX.Monitor.CELL_RESIZE event. Fires when vMX.Monitor.smartScan
	 * detects the situation of cell resizing without any other significant changes
	 * This method does the actual resizing of magnifier splitters
	 */
	Magnifier_prototype.resizeSplitters = function(event) {
		vMX._log("Magnifier.resizeSplitters");

		var arrDiff = event.diff;
		if (!arrDiff || arrDiff.length == 0)
			throw new jsx3.lang.Exception("resizeSplitters: diff is NULL or empty!");

		var s = this._maindlg.getDescendantsOfType("jsx3.gui.Splitter");
		for (var i = 0; i < s.length; i++) {
			var blk = s[i].getParent();
			var x = blk.getAttribute("x"), y = blk.getAttribute("y");
			var w = blk.getAttribute("w"), h = blk.getAttribute("h");

			for (var j = 0; j < arrDiff.length; j++) {
				var el = arrDiff[j];
				if (x == el.x && y == el.y && w == el.w && h == el.h) {
					s[i].setSubcontainer1Pct(el.newpct + "%", true);
					break;
				}
			}
		}
	};

	/**
	 * Fires when a splitter is dropped into a magnifier cell. This causes
	 * a layout change when the target cell is destroyed and several new cells
	 * appear on that cell's place
	 */
	Magnifier_prototype.onCellReplace = function(event) {
		var objRes = {};
		var c = event.remove_cell;

		if (vMX.recursiveScan(c.x, c.y, c.w, c.h, new jsx3.util.List(event.replace_cells), objRes)) {
			var magnCell = this.getCellById(c.id);
			var blkCell = magnCell.getMainBlock();

			//magnCell.destroy();
			magnCell._mainblock.removeChildren();
			this.removeCell(magnCell);

			this._recursiveSplit(objRes, blkCell, true);
			blkCell.repaint();

			this._wcview.getWallLayout().sync({'group' : 1, 'archive' : 1, 'timeline' : 1});
		}
		else {
			vMX._log(2, "Error in Magnifier.onCellReplace: recursiveScan returned false");
		}
	};

	/**
	 * Fires after any of the splitters inside magnifier is resized.
	 * Performs conversion from the browser objects coords to Display Server
	 * screen coords and calls 'splitMonitor' to adjust DS screen layout
	 */
	Magnifier_prototype.onSplitterResize = function(event) {
		vMX._log("Magnifier.onSplitterResize");

		var objMonitor = this.getMonitor();
		var arCells = new Array(objMonitor.getCellsArray().length);
		for (var iter = objMonitor.iterator(), i = 0; iter.hasNext(); i++) {
			var c = iter.next();
			arCells[i] = {
				jsxid:jsx3.xml.CDF.getKey(), id:c.id, x:c.x, y:c.y, w:c.w, h:c.h,
				srcObjid:c.srcObjid, state:c.state, scale:"Cell", touring : c.touring,
				arc_from:c.arc_from, arc_to:c.arc_to, vncurl:c.vncurl,
				arc_pos:c.arc_pos, groupid: c.groupid, open_timeout:10, play_timeout:10,
				analytics:c.analytics, streamnum:c.streamnum, audioObjid:c.audioObjid,
				volume:c.volume, weburl:c.weburl, refresh:c.refresh
			};
		}

		var doRecalc = function(block) {
			var x = Number(block.getAttribute("x")), y = Number(block.getAttribute("y"));
			var w = Number(block.getAttribute("w")), h = Number(block.getAttribute("h"));
			var spl = block.getFirstChildOfType("jsx3.gui.Splitter");
			if (!spl) {
				var cellid = block.magn_cell.getMonCell().id;
				for (var i = 0; i < arCells.length; i++) {
					if (arCells[i].id == cellid) {
						arCells[i].x = x;
						arCells[i].y = y;
						arCells[i].w = w;
						arCells[i].h = h;
					}
				}
				return;
			}
			else {

				if (spl.getOrientation() == jsx3.gui.Splitter.ORIENTATIONH) {
					var x1 = x, y1 = y, w1 = Math.floor(parseFloat(spl.getSubcontainer1Pct()) * w / 100), h1 = h;
					var x2 = x + w1 + 1, y2 = y, w2 = w - w1 - 1, h2 = h;
				} else {
					var x1 = x, y1 = y, w1 = w, h1 = Math.floor(parseFloat(spl.getSubcontainer1Pct()) * h / 100);
					var x2 = x, y2 = y + h1 + 1, w2 = w, h2 = h - h1 - 1;
				}

				var first = spl.getFirstChild(), last = spl.getLastChild();
				first.setAttribute("x", String(x1)).setAttribute("y", String(y1)).setAttribute("w", String(w1)).setAttribute("h", String(h1));
				last.setAttribute("x", String(x2)).setAttribute("y", String(y2)).setAttribute("w", String(w2)).setAttribute("h", String(h2));
				doRecalc(first);
				doRecalc(last);
			}
		};

		doRecalc(event.target.getParent());

		// Validate cell sizes and reject splitting if some cells are too small
		/*var bResizeOK = true;
		 for(var i = 0; i < arCells.length; i++) {
		 var c = arCells[i];
		 if(c.w < vMX.MIN_CELL_WIDTH || c.h < vMX.MIN_CELL_HEIGHT) { bResizeOK = false; break; }
		 }
		 if(!bResizeOK) {
		 alert("Can't resize - cell is too small");
		 this.repaint();
		 }(*/
		if (!vMX.recursiveScan(0, 0, objMonitor.hres, objMonitor.vres, new jsx3.util.List(arCells), {})) {
			alert(__("Cannot resize.\nPlease try again."));
			this.repaint();
		}
		else {
			vMX.splitMonitor(objMonitor.objid, arCells);
		}
	};


	/**
	 * Handles click on dialog 'close' button
	 */
	Magnifier_prototype.onClose = function(event) {
		vMX._log("Magnifier.onClose");

		var objMonitor = this.getMonitor();
		objMonitor.unsubscribe(vMX.Monitor.NEW_LAYOUT, this);
		objMonitor.unsubscribe(vMX.Monitor.CELL_RESIZE, this);
		objMonitor.unsubscribe(vMX.Monitor.CELL_REPLACE, this);
		this.getWallcell().unsubscribe(vMX.WallCell.BEFORE_REMOVE_MONITOR, this);

		this._maindlg.doClose();
		var wallLayout = this._wcview.getWallLayout();
		wallLayout.objPainting.lightOff();
		wallLayout.setMagnifier(null);
		$('#' + wallLayout.getContentBlock().getId() + ' .drop-disable').each(function() {
			mx.MATRIX2.getServer().getJSXById(this.id).setClassName('drop').repaint();
		});
		wallLayout.updateLayout();
		wallLayout.sync({'group' : 1, 'archive' : 1, 'timeline' : 1});

		// Re-enable querying of current wall state
		if (!vMX.QUERY_STATE_ENABLED) vMX.startRefresh();
		vMX.Magnifier._PAINTED_CELLS = {};
		if (vMX.Magnifier._STARTELOG_TIMEOUT) {
			window.clearTimeout(vMX.Magnifier._STARTELOG_TIMEOUT);
			vMX.Magnifier._STARTELOG_TIMEOUT = 0;
		}
		try {
			if (!mx.ELog2.hidden) mx.ELog2.startELog();
		} catch(e) {
			vMX._log("Got exception when starting ELog")
		}
	};

	/**
	 * Handler for 'go fullscreen/windowed" button
	 */
	Magnifier_prototype.onToggleState = function(event) {
		if (this.state != "max") {
			this.state = "max";
			var pos = this._maindlg.getParent().getAbsolutePosition();
			this.pos = {
				L : this._maindlg.getLeft(), T : this._maindlg.getTop(),
				W : this._maindlg.getWidth(), H : this._maindlg.getHeight()
			};
			this._maindlg.setLeft(0).setTop(0).setWidth(pos.W).setHeight(pos.H, true);
		} else {
			this.state = "window";
			var pos = this.pos;
			if (!pos) {
				vMX._log(2, "Unable to minimize magnifier: pos is null");
				return;
			}
			this._maindlg.setLeft(pos.L).setTop(pos.T).setWidth(pos.W).setHeight(pos.H, true);
			this.pos = null;
		}

		this.recalcMinCellSize();
	};

	/**
	 * Handler for 'Clear cells' button
	 */
	Magnifier_prototype.onClearCells = function(event) {
		vMX.splitMonitor(this.getMonitor().objid, {'rows':1, 'cols':1});
	};

	/**
	 * 'handler for onmouseover' event. Here we must clear
	 * all preview snapshots that may persist in magnifier cells
	 */
	Magnifier_prototype.checkPreviewSnapshots = function(event) {
		for (var iter = this.iterator(); iter.hasNext();) {
			var mcell = iter.next();
			if (mcell._mouseover) mcell.onMouseOut();
		}
	};


	Magnifier_prototype.groupPlay = function(event) {
		if (this._group.size() == 0) return;

		var first = this._group.get(0).getMonCell();
		var state = first.state == "Play" ? "Pause" : "Play";

		var ctrlData = [];
		for (var i = 0; i < this._group.size(); i++) {

			var moncell = this._group.get(i).getMonCell();

			if (moncell.state != "Play" && moncell.state != "Pause")
				continue;

			var ctrlStruct = {};
			ctrlStruct.jsxid = i;
			ctrlStruct.cellid = this._group.get(i).getMonCell().id;
			ctrlStruct.monid = this.getMonitor().objid;
			ctrlStruct.state = state;
			ctrlStruct.text = this._group.get(i).getMonCell().text;
			ctrlStruct.scale = this._group.get(i).getMonCell().scale;
			ctrlData.push(ctrlStruct);
		}

		vMX.controlCells(ctrlData);
	};

	/**
	 * Sets archive parameters for current cell group
	 */
	Magnifier_prototype.groupSetStreamPos = function(event) {
		if (this._group.size() == 0) return;

		if (event.from != null && event.to != null) {
			var from = event.from, to = event.to, pos = event.pos || event.from;
		} else {
			vMX._log("groupSetStreamPos: bad arguments");
			return;
		}

		var arrCells = [];
		for (var iter = this._group.iterator(); iter.hasNext();) {
			var mcell = iter.next();
			var cell = mcell.getMonCell();

			// Do credentials check
			var objC = resourceTree.getObject(cell.srcObjid);
			if ((from == 0 && to == 0 && !objC.cred('L')) ||
				(from > 0 && to > 0 && !objC.cred('A'))) {
				mcell.raiseMessage();
				continue;
			}

			arrCells.push(
			{
				'cellid' : cell.id, 'monid' : this.getMonitor().objid,
				'state' : "Play", 'scale' : cell.scale, 'text' : cell.text,
				'arc_from' : from, 'arc_to' : to, 'srcObjid' : cell.srcObjid, 'arc_pos' : pos,
				'analytics' : cell.analytics, 'streamnum' : cell.streamnum,
				'audioObjid' : cell.audioObjid, 'volume' : cell.volume
			}
			);
		}

		vMX.assignStream(arrCells);
	};


	/**
	 * 'onclick' handler for 'switchToArchive' button on player control
	 * Can be called directly, 'bQuick' parameter stands for 'quick'
	 * switch, when start of archive interval is set to (now() - 60 sec)
	 */
	Magnifier_prototype.groupSwitchToArchive = function(event, bQuick) {
		if (this._group.size() == 0) return;

		var jumpInterval = bQuick ? 60 : 300;
		var now = Math.round(mx.MATRIX2.getServerTime() / 1000);

		this.groupSetStreamPos({'from': now - jumpInterval, 'to': now, 'pos': now - jumpInterval});
	};


	Magnifier_prototype.groupJumpBackward = function(event) {
		if (this._group.size() == 0) return;

		var leader = this._group.get(0).getMonCell();
		if (leader.arc_from <= 0 && leader.arc_to <= 0) { // quick switch to archive
			this.groupSwitchToArchive(event, true);
		}
		else { // jump backward on 30 sec
			this.groupSetStreamPos({'from':leader.arc_from, 'to':leader.arc_to,
				'pos':leader.arc_pos - 30});
		}
	};

	Magnifier_prototype.groupJumpForward = function(event) {
		if (this._group.size() == 0) return;

		var leader = this._group.get(0).getMonCell();
		if (leader.arc_from <= 0 && leader.arc_to <= 0) {
			vMX._log(3, "groupJumpForward() called for live video stream")
		}
		else { // jump forward on 30 sec
			this.groupSetStreamPos({'from':leader.arc_from, 'to':leader.arc_to,
				'pos':Number(leader.arc_pos) + 30});
		}
	};

	/**
	 * 'onclick' handler for 'switchToLive' button on player control
	 */
	Magnifier_prototype.groupSwitchToLive = function(event) {
		if (this._group.size() == 0) return;
		this.groupSetStreamPos({'from':0, 'to':0, 'pos':0});
	};

});
