//------------------------------------------------------------------------------------------
// vMX Package
//------------------------------------------------------------------------------------------
jsx3.lang.Package.definePackage("mx.vMX", function(vMX) {

	/**
	 * Interface for cell containers. Implemented by Wall, Monitor, Magnifier and WallLayout classes
	 */
	jsx3.Class.defineInterface("mx.vMX.CellContainer", null, function(CellContainer, CellContainer_prototype) {
		CellContainer_prototype.iterator = function() {
		return new jsx3.util.List.Iterator(this._cells);
	};

	CellContainer_prototype.getCellsArray = function() {
		return this._cells.toArray(true);
	};

	CellContainer_prototype.getCellsList = function() {
		return this._cells;
	};

	CellContainer_prototype.addCell = function(cell) {
		if(!this._cells.contains(cell)) this._cells.add(cell);
		cell.setContainer(this);
	};

	CellContainer_prototype.removeCell = function(cell) {
		cell.setContainer(null);
		this._cells.remove(cell);
	};
	CellContainer_prototype.clearCells = function() {
		this._cells.clear();
	};
	});

	vMX.WALLS    = new jsx3.util.List();
	vMX.MONITORS = [];
	vMX.DSPS     = new jsx3.util.List();
	vMX.LAYOUTS  = "";
	vMX.LAYOUT_LIST = new Array;
	vMX.WALL_LAYOUTS = {};
	vMX.WALL_SELECTOR = null;
	vMX.STATEIMG = {
		Play      : "images/design1/vMX/magnifier_play.png",
		Pause     : "images/design1/vMX/magnifier_pause.png",
		Stop      : "images/state_stop.jpg",
		Empty     : "pixs/no.gif",
		Invalid   : "pixs/no.gif",
		Opening   : "images/design1/vMX/magnifier_play.png",
		Closing   : "images/state_stop.jpg",
		Destroyed : "pixs/no.gif",
		None      : "pixs/no.gif"
	};

	// Manage timeouts
	vMX.QUERY_STATE_ENABLED = false;
	vMX.QUERY_STATE_TIMEOUT = 3000;
	vMX.QUERY_STATE_TID = 0;
	vMX.QUERY_OBJS_ENABLED = false;
	vMX.QUERY_OBJS_TIMEOUT = 20000;
	vMX.QUERY_OBJS_TID = 0;
	vMX.QUERY_SNAPSHOTS_ENABLED = false;
	vMX.QUERY_SNAPSHOTS_TIMEOUT = 4000;
	vMX.QUERY_SNAPSHOTS_TID = 0;
	vMX.GUARD_TIMEOUT = 60000;
	vMX.GUARD_TID = 0;

	// Cell size limits
	vMX.MIN_CELL_WIDTH = 200;
	vMX.MIN_CELL_HEIGHT = 180;

	//Session
	vMX.SID = "";
	// Role
	vMX.ROLEID = 0;

	// Snapshots to update per turn
	vMX.SNAPS_PER_TURN = 5;

	// Image cache
	vMX.IMAGE_CACHE = {};

	// Event info for APC cell
	vMX.CURR_WITNESSES = {};

	// Reusable jsx3.net.Service objects
	vMX.GetWallsSvc = null;
	vMX.GetMonitorStateSvc = null;
	vMX.SplitMonitorSvc = null;
	vMX.AssignStreamSvc = null;
	vMX.ControlCellsSvc = null;
	vMX.GetWallChangesSvc = null;
	vMX.GetAudioDevicesSvc = null;
	vMX.AssignAudioSvc = null;
	vMX.SetTouringSvc = null;
	vMX.GetLayoutListSvc = null;
	vMX.RecallLayoutSvc = null;
	vMX.DeleteLayoutSvc = null;
	vMX.SaveLayoutSvc = null;
	vMX.GetResourcesSvc = null;
	vMX.SetApcSvc = null;

	// Misc
	vMX.AWAIT_CFG_LIST = null;
	vMX.LAST_REFRESH_TIME = 0;

	// Logging and Debugging
	vMX._log = function(intLevel, strMessage, strArgs) {
		if(typeof intLevel == "string") {
			strMessage = intLevel;
			intLevel = 4;
		}
		if(intLevel <= vMX._MAX_LOG_LEVEL && vMX._LOG) vMX._LOG.log(intLevel, strMessage, strArgs);
	};

	vMX._appendHandlers = function () {
		var fn_append = function (fn, fn_name, fn_class) {
			var arrCode = fn.toString().replace(/\n/g, "").match(/function\((.*?)\)\s*\{(.*)\}/);
			var strCodeMod = "try {" + arrCode[2] + "}catch(e){" + "var ex = jsx3.lang.NativeError.wrap(e);" +
							 "vMX._log(2, 'Exception caught in " + (fn_class?fn_class:"mx.vMX") + "." + fn_name +
							 "() :' + ex.printStackTrace());}"
							  ;
			eval((fn_class?(fn_class + ".prototype."):"mx.vMX.") + fn_name + "=function(" + arrCode[1] + ") {" + strCodeMod + "};");
		};

		var pkg = jsx3.lang.Package.forName("mx.vMX");
		var arrStatic = pkg.getStaticMethods();
		for(var i = 0; i < arrStatic.length; i++) {
			var method = arrStatic[i];
			if(method.getName() != "_log" && method.getName() != "_appendHandlers") {
				fn_append(method.getFunction(), method.getName());
			}
		}
		var arrClasses = pkg.getClasses();
		for(var i = 0; i < arrClasses.length; i++) {
			var arrInstance = arrClasses[i].getInstanceMethods();
			for(var j = 0; j < arrInstance.length; j++) {
				var method = arrInstance[j];
				fn_append(method.getFunction(), method.getName(), arrClasses[i].getName());
			}
		}
	};

	vMX._LOG = null;
	vMX._MAX_LOG_LEVEL = 3; // disable logging by default

	vMX._INIT = true;

	//------------------------------------------------------------------------------------------
	// vMX package general-use methods
	//------------------------------------------------------------------------------------------

	vMX.getServer = function() {
		return mx.MATRIX2.getServer();
	};

	vMX.getMonitorById = function(objid) {
		for(var i = 0; i < vMX.MONITORS.length; i++) {
			if(vMX.MONITORS[i].objid == objid)
				return vMX.MONITORS[i];
		}
		return null;
	};

	vMX.getWallById = function(objid) {
		return vMX.WALLS.filter(function(item){return item.objid==objid?true:false}).get(0);
	};

	vMX.addMonitor = function(objid) {
		for(var i = 0; i < vMX.MONITORS.length; i++) {
			if(vMX.MONITORS[i].objid == objid)
				return vMX.MONITORS[i];
		}
		var mon = new vMX.Monitor(objid);
		vMX.MONITORS.push(mon);
		return mon;
	};

	vMX.getCurWall = function() {
		var curTab = mx.CT.getActiveTab();
		if (curTab != mx.CT.wallTab) {
			return null;
		}
		var wallid = vMX.WALL_SELECTOR.getValue();
		return wallid ? vMX.getWallById(wallid) : null;
	};

	vMX.getWallLayout = function(selector) {
		if(typeof(selector) == "string" || typeof(selector) == "number")
			return vMX.WALL_LAYOUTS[selector];

		if(!selector)
			var wall = vMX.getCurWall();
		else if(selector instanceof vMX.Wall)
			var wall = selector;

		return wall != null ? vMX.WALL_LAYOUTS[wall.objid] : null;
	};

	vMX.getCamImage = function(mixed, ts) {
		var objid;
		if (mixed instanceof vMX.MonitorCell) {
		    objid = mixed.srcObjid;
		    ts = mixed.arc_from > 0 ? (mixed.arc_pos > 0 ? mixed.arc_pos : mixed.arc_from) : 0;
		} else {
		    objid = mixed;
		}
		if (ts != null && ts > 0) {
		    return "/storage/snapshot?objid=" + objid + "&ts=" + ts + "&downscale" + "&_=" + Math.random();
		} else {
		    return resourceTree.getSnapshotHref(objid) || "images/design1/vMX/magnifier_placeholder.png";
		}
	};

	vMX.getCellSnapshot = function(monid, cellid, type) {
		return "vmxsnapshot.php?monid=" + monid +
			"&cellid=" + cellid + "&type=" + type +
			"&rand=" + jsx3.xml.CDF.getKey();
	}

	vMX.getControlInstance = function() {
		return vMX.getServer().getJSXByName('layoutControlComponent').controls.controlInstance;
	};

	vMX.getMagnifier = function(objWall) {
		var wallLayout = vMX.getWallLayout(objWall);
		return wallLayout == null ? null : wallLayout.getMagnifier();
	};

	/**
	 * vMX-specfic tab management, replacement for mx.CT method
	 */
	vMX.removeWallTabs = function(lstWalls) {
		var tabpane = mx.MATRIX2.getServer().getJSXByName('tabbedpaneMain');
		var tabs = tabpane.getChildren();
		var curLayout = vMX.getWallLayout();

		if(curLayout != null && lstWalls.contains(curLayout.getWall())) {
			var newTab = tabs[0];
			for(var objid in vMX.WALL_LAYOUTS) {
				if(!lstWalls.contains(vMX.WALL_LAYOUTS[objid].getWall()))
				newTab = vMX.WALL_LAYOUTS[objid].getTab();
			}
			// Activate new tab
			for (var i = 0; i < tabs.length; i++) {
				if(tabs[i] == newTab) {
					mx.CT.onTabSelect(newTab);
					break;
				}
			}
		}

		for(var iter = lstWalls.iterator(); iter.hasNext();) {
			var objTab = vMX.WALL_LAYOUTS[iter.next().objid].getTab();
			objTab.removeAttribute("wallid");
			objTab.removeAttribute("curType");
			objTab.getFirstChild().removeEvent(jsx3.gui.Interactive.DROP);
			objTab.getFirstChild().setCanDrop(jsx3.Boolean.TRUE);
			objTab.getFirstChild().setEvent("mx.MATRIX2.handleOnDrop(objSOURCE, this, strDRAGID);", jsx3.gui.Interactive.DROP);
			objTab.setDisplay(jsx3.gui.Block.DISPLAYNONE,true);
			objTab.getFirstChild().removeChildren();
		}
	};

	/**
	 * Handles comunication with web services. Automatically generates handlers for
	 * jsx3.net.Service events except ON_SUCCESS which should be implemented manually
	 */
	vMX.doServiceCall = function(strOp, strSvcName) {
		if(strSvcName == null) strSvcName = "Wall_svc";
		var f = function(mask, group) { return group.toUpperCase() };
		var base = strOp.replace(/_(\w)/g, f).replace(/^(\w)/, f);
		var svc = vMX[base + "Svc"];
		if(!svc) {
			vMX[base + "Svc"] = vMX.getServer().loadResource(strOp + "_xml");
			svc = vMX[base + "Svc"];

			var strOnSuccess = "on" + base + "Success"
			var strOnError = "on" + base + "Error";
			var strOnInvalid = "on" + base + "Invalid";
			if(!vMX[strOnError]) vMX[strOnError] = function(objEvent)
			{ vMX._log(2, "'" + strOp + "' call failed: " + objEvent.target.getRequest().getResponseText()); }
			if(!vMX[strOnInvalid]) vMX[strOnInvalid] = function(objEvent)
			{ vMX._log(2, "invalid call to " + strOp); }

			svc.subscribe(jsx3.net.Service.ON_SUCCESS, vMX[strOnSuccess]);
			svc.subscribe(jsx3.net.Service.ON_ERROR, vMX[strOnError]);
			svc.subscribe(jsx3.net.Service.ON_INVALID, vMX[strOnInvalid]);
		}

		var req = svc.getRequest();
		if(req) req.abort();

		svc.setOperationName(strOp);
		svc.setEndpointURL(mx.MATRIX2.getServerURL() + "/axis2/services/" + strSvcName);
		try {
			svc.doCall();
		} catch(e) {
			var ex = jsx3.lang.NativeError.wrap(e);
			vMX._log(2, "Exception in doCall() for '" + strOp + "': " + ex.printStackTrace());
		}
	};


	/**
	 * package initialization
	 */
	vMX.vmx_init = function(bPostInit) {

		if(!bPostInit) { // 1st stage stage - get session, set role.
			// Load all vMX modules
			jsx3.require("mx.vMX.Wall", "mx.vMX.WallCell", "mx.vMX.Monitor", "mx.vMX.MonitorCell", "mx.vMX.SpecialPainting",
						 "mx.vMX.Magnifier", "mx.vMX.MagnifierCell", "mx.vMX.WallcellView", "mx.vMX.WallLayout", "mx.vMX.DSP");
			vMX._INIT = true;
			vMX._LOG = jsx3.util.Logger.getLogger('global');
			vMX._log(4, "vMX application started");

			vMX.SID = window.jsSessionId;

			/*try {
				vMX._appendHandlers();
			} catch(e) {vMX._log(2, "Exception caught when appending handlers: " + e.description);}
			*/

			//vMX.getSessionId();
		}
		else { // 2nd stage init - actually run vMX interface
			//vMX.QUERY_OBJS_ENABLED = true;

			//vMX.getSystemObjects();
			//vMX.getAudioDevices();
		}
	};

	vMX.getWalls = function() {
		vMX.doServiceCall("get_walls");
	};

	vMX.getAudioDevices = function() {
		vMX.doServiceCall("get_audio_devices");
	};

	vMX.setRole = function(roleid) {
		vMX._log("set role: " + roleid);
		vMX.setSessionRole(roleid);
	};


	vMX.addSplitter = function(blkDest, splOrientation, pct, bDisableResize){
		var bgColor = vMX.getServer().getJSXByName('block_main').getBackgroundColor();
		blkDest.removeEvent(jsx3.gui.Interactive.JSXCLICK);
		ifr_count++;

		var objSplitter = new jsx3.gui.Splitter(('spl'+ifr_count), splOrientation);
		objSplitter.setSubcontainer1Pct(pct+"%");

		if(bDisableResize) objSplitter.setEvent("false;", jsx3.gui.Interactive.BEFORE_RESIZE);

		var blocks = objSplitter.getChildren();
		for(var i=0; i < blocks.length; i++) {
			blocks[i].setBackgroundColor(bgColor);
			blocks[i].setOverflow(jsx3.gui.Block.OVERFLOWEXPAND);
		}

		blkDest.setChild(objSplitter);
		return objSplitter;
	};

	vMX.showHideMask = function(bShow) {
		if (jsx3.GO('magnifier'))
			return;
		var curLayout = vMX.getWallLayout();
		if(curLayout) curLayout.showHideMask(bShow);
	};

	/*
	 * Drop event handler for resource tree. Need to handle d'n'd from Wallcell to resource tree
	 */
	vMX.onWallcellDrop = function(block) {
		if (!block)
			return;

		var result = block.getName().match(/^Wallcell (\d+)\.(\d+)/);
		if (result != null) {
			var row = result[1];
			var col = result[2];

			var layout = vMX.getWallLayout();
			if (layout != null) {
				var wcview = layout.getCellAt(row,col);
				if (wcview != null) {
					var monitor = wcview.getWallcell().monitor;
					var objJson = monitor.serializeToJson();
					VIEWS.save(monitor.name, false, true, monitor.objid);
				}
			}
		}
	};

	/**
	 * Have to provide package method for handling this event, because it should be called
	 * after the event script defined in component file, and unfortunately, methods defined
	 * via 'subscribe/publish' technology are called before this script
	 */
	vMX.onLayoutResize = function() {
		for(var iter = vMX.WALLS.iterator(); iter.hasNext();) {
			var wall = iter.next();
			var l = vMX.getWallLayout(wall);
			if(l)
				l.onLayoutResize();
			else
				vMX._log(2, "Failed to get layout for wall: " + wall)
		}
	};

	/**
	 * Handles click on per-tab 'show cell headers' button
	 */
	vMX.onShowCellHeaders = function(event) {
		var target = event.srcElement || event.target;

		// Event shouldn't bubble to ancestors
		event.cancelBubble = true;
		// W3C standard:
		if (event.stopPropagation) {
			event.stopPropagation()
		}

		var wallid = target.parentNode.parentNode.parentNode.getAttribute("wallid");
		var wallLayout = vMX.WALL_LAYOUTS[wallid];
		if(wallLayout) {
			wallLayout.onShowCellHeaders();
			//vMX.getWallChanges(wallid);
			vMX.refreshState(vMX.getWallById(wallid));
		}
		else {
			vMX._log(2, "Error in 'onShowCellHeaders': failed to get layout");
		}
	};

	vMX.clearWallLayout = function() {
		var curLayout = vMX.getWallLayout();
		if(curLayout) curLayout.newLayout();
	};


	/**
	 * Check whether magnifier opened and currently active group is in archive state
	 * return group of players or null
	 */
	vMX.getArchiveGroup = function() {
		var wallLayout = vMX.getWallLayout();
		if(!wallLayout) return null;

		// Check for active magnifier and group
		var magnifier = wallLayout.getMagnifier();
		if(!magnifier) return null;

		var group = magnifier.getGroup();
		if(group.size() == 0) return null;

		var leader = group.get(0).getMonCell();
		if (leader.arc_from == 0 && leader.arc_to == 0) return null;

		return group;
	};


	/**
	 * Is called by Timeline after user moves its sliders. Here we can do various things like
	 * setting new values of date&time in Player Control and restarting magnifier group cells
	 * with new values of 'arc_from' and 'arc_to'
	 */
	vMX.onSetTimelineRange = function(itimeLine, istartTime, ifinishTime, igranularity) {
		// Check for active wall layout
		if (!vMX.getArchiveGroup()) {
			return false;
		}

		if (!istartTime || !ifinishTime)
			return false;

		var magnifier = vMX.getWallLayout().getMagnifier();
		if(itimeLine == "up") {
			appLogger.info('onSetTimelineRange: start: ' + mx.TIME.timestamp2date(istartTime) + ' end: ' + mx.TIME.timestamp2date(ifinishTime));
			magnifier.groupSetStreamPos({'from':istartTime/1000, 'to':ifinishTime/1000, 'pos':istartTime/1000});
		}
	};

	/**
	 * Is called by Timeline for querying current storage coverage
	 */
	vMX.onRequestXML = function(itimeLine, istartTime, ifinishTime, igranularity) {
		var group = vMX.getArchiveGroup();
		if (!group)
			return false;

		if (!istartTime || !ifinishTime)
			return false;

		// Prepare array of objIds
		var objids = [];
		var streamnum = vMX.MonitorCell.STREAM_QUALITY_DEFAULT = -1;
		for(var i = 0; i < group.size(); i++) {
			var cell = group.get(i).getMonCell();
			var id = cell.srcObjid;
			streamnum = cell.streamnum;
			if (id) {
				objids.push(id);
			}
		}
		if (objids.length > 0) {
			// If all OK, update data for Timeline
			mx.TIMELINE.getStorageCoverage({
				timeLine:    itimeLine,
				startTime:   istartTime,
				finishTime:  ifinishTime,
				granularity: igranularity,
				objids:      objids/*,
				streamnumber:streamnum*/
			});
		}
	};

	vMX.onTimeChange = function(time) {

		var group = vMX.getArchiveGroup();
		if (!group) {
			return false;
		}

		var leader = group.get(0).getMonCell();
		var magnifier = vMX.getWallLayout().getMagnifier();
		time = Math.round(time / 1000);

		if (leader.arc_from != 0 && leader.arc_to != 0) {
			if (leader.arc_from > time || leader.arc_to < time)
				return;
		}

		magnifier.groupSetStreamPos({'from':leader.arc_from, 'to':leader.arc_to, 'pos': time + ''});
	};


	/**
	 * Handles vMX tab change events, passes them to the WallLayout objects
	 */
	mx.vMX.onTabChange = function(objTHIS) {
		var newLayout = vMX.getWallLayout();
		if(newLayout != vMX.CUR_LAYOUT) {
			if(newLayout) {
				newLayout.show();
				newLayout.onActivate(vMX.CUR_LAYOUT);
				vMX.updateWallDesc(newLayout.getWall().desc);
			}

			for (var objid in vMX.WALL_LAYOUTS) {
				var layout = vMX.WALL_LAYOUTS[objid];
				if (layout != newLayout) {
					layout.onDeactivate(newLayout);
					layout.hide();
				}
			}
		}
		vMX.CUR_LAYOUT = newLayout;
	};

	mx.vMX.onWallLayoutChange  = function(objTHIS) {
		var newLayout = vMX.getWallLayout();
		if(newLayout && newLayout != vMX.CUR_LAYOUT) {
			newLayout.show();
			newLayout.onActivate(vMX.CUR_LAYOUT);
			vMX.updateWallDesc(newLayout.getWall().desc);

			for (var objid in vMX.WALL_LAYOUTS) {
				var layout = vMX.WALL_LAYOUTS[objid];
				if (layout != newLayout) {
					layout.onDeactivate(newLayout);
					layout.hide();
				}
			}

			vMX.CUR_LAYOUT = newLayout;
		}
	};

	vMX.confSaveDialog = function(display) {
		var curWall = vMX.getCurWall();
		if(!curWall) return;

		if (!display) {
			vMX.AWAIT_CFG_LIST = { "type" : "save", "vmx": true };
			vMX.getLayoutList(curWall.objid);
		}
		else { // Display dialog box
			mx.Dialog.createLayoutDialog(
			'conf_dialog',
			__("SAVE CONFIGURATION"),
			{'top': '60px', 'width':'312px', 'height':'auto'},
			'<div style="margin-bottom: 5px;">' +
				'<input type="text" id="conf_name" placeholder="' + __("input new name") + '" />' +
			'</div>' +
			'<select id="conf_id" class="select">' + vMX.LAYOUTS + '</select>' +
			'<span id="btn_conf_save" class="button save" title="' + __("Save configuration") + '"></span>'
			);
			$('#conf_name').placeholder();
			$('#btn_conf_save').click(
				function() {
					var confName = $('#conf_name').val() == $('#conf_name').attr('placeholder') ?
					'' : $.trim($('#conf_name').val());
					if( !confName || confName == "") {
						alert(__("Invalid configuration name!"));
						return;
					}
					// Check if such configuration already exists
					var confId = null;
					$.each(vMX.LAYOUT_LIST, function() {
						if (this.name == confName) {
							confId = this.obj;
							return false;
						}
					});
					if (confId) {
						if (!confirm(
						  Gettext.strargs(
						  __("Do you really want to overwrite existing configuration '%1'?"),
						  [confName]))
							)
						return;
					}

					$('#conf_dialog').remove();

					vMX.saveLayout(vMX.getCurWall(),confName, confId);
				}
			);
			$('#conf_id').change(
				function() {
					if ( $('#conf_id').val() != 0 ) {
						$('#conf_id option:selected').each(function () {
							$('#conf_name').val($(this).text());
						});

					}
				}
			);
			$('#conf_id').jaoselect();
		}
	};

	vMX.confLoadDialog = function(display) {
		var curWall = vMX.getCurWall();
		if(!curWall) return;

		if (!display) {
			vMX.AWAIT_CFG_LIST = { "type" : "load", "vmx": true };
			vMX.getLayoutList(curWall.objid);
		}
		else {
			mx.Dialog.createLayoutDialog(
				'conf_dialog',
				__("LOAD CONFIGURATION"),
				{'top': '70px', 'width':'312px', 'height':'auto'},
				'<select id="conf_id" class="select">' + vMX.LAYOUTS + '</select>' +
				'<span id="btn_conf_load" class="button load" title="' + __("Load configuration") + '"></span>'
			);
			$('#conf_id').jaoselect();
			$('#btn_conf_load').click(function() {
				var layoutId = parseInt($('#conf_id').val());

				if (!layoutId) {
					alert(__("No configuration selected!"));
					return;
				}

				$('#conf_dialog').remove();
				mx.MATRIX2.mask(__("LOADING CONFIGURATION..."));
				$(document).bind('vMX.loaded', function() {
					mx.MATRIX2.mask(false);
					$(document).unbind('vMX.loaded');
				});

				vMX.recallLayout(layoutId);
			});
		}
	};

	vMX.confManageDialog = function(display) {
		var curWall = vMX.getCurWall();
		if(!curWall) return;

		if (!display) {
			vMX.AWAIT_CFG_LIST = { "type" : "manage", "vmx": true };
			vMX.getLayoutList(curWall.objid);
		}
		else {
			var list = vMX.LAYOUT_LIST;
			var str = '';
			if(list.length == 0) {
				str = __("No configurations available");
			} else {
				$.each(list, function() {
					str +=
					'<li><label>' +
						'<input type="checkbox" obj="' + this.obj + '"/>' +
						$.escape(this.name) +
					'</label></li>';
				});
			}

			var listH = list.length == 0? 20 :
				(list.length <= 4 ? list.length * 22 : 88);
			mx.Dialog.createLayoutDialog(
				'conf_dialog',
				__("MANAGE CONFIGURATIONS"),
				{'top': 60, 'width':312, 'height':listH + 69},
				'<ul id="conf_list" style="height: ' + listH + 'px; width: 250px; overflow: auto; margin: 0 0 10px 20px;">' + str + '</ul>' +
				'<span id="conf_delete" class="dialog_command">' +
				__("DELETE", "Configurations manage dialog") +
				'</span>'
			);

			$('#conf_list').html(str);
			$('#conf_delete').click(function() {
				var Ids = [];
				$('#conf_list input:checked').each(function() {
					Ids.push($(this).attr('obj'))
				});
				if (!confirm(__("Do you want to delete selected configurations?")))
					return;

				mx.MATRIX2.mask(__("DELETING CONFIGURATIONS..."));
				window["vmx_curDeletedLayoutIndex"] = -1;
				var f = function() {
					var ind = window["vmx_curDeletedLayoutIndex"] + 1;
					if (ind < Ids.length) {
						vMX.deleteLayout(Ids[ind]);
						window["vmx_curDeletedLayoutIndex"] = ind;
						window["vmx_deleteLayoutTmt"] = setTimeout(f, 500);
					}
					else {
						delete window["vmx_curDeletedLayoutIndex"];
						clearTimeout(window["vmx_deleteLayoutTmt"]);
						delete window["vmx_deleteLayoutTmt"];
					}
				};
				setTimeout(f, 200);

				setTimeout(function() {
					mx.MATRIX2.mask(false);
					vMX.confManageDialog(true);
				}, Ids.length * 500 + 500);
			}).hide();

			// For all checkboxes add onclick event
			$('#conf_list').on('click', 'input[type="checkbox"]', function () {
				$('#conf_delete').hide();
				var checked = $('#conf_list input:checked');
				if (!checked.length)
					return;
				$('#conf_delete').show();
			});
		}
	};

	vMX.getConfList = function() {
		vMX.getServer().getCache().setDocument("CONF_DATA", vMX.LAYOUTS);
		vMX.getServer().getCache().setDocument("CONF_DATA_PRIVATE", vMX.LAYOUTS);
	};

	vMX.confCheckSave = function(confName, isExists) {
		vMX.saveLayout(vMX.getCurWall(), confName, isExists);
	};

	vMX.playSound = function(objTHIS) {
		var blkCell = objTHIS.getAncestorOfName("cell_blk").getParent();
		var sndObjid = blkCell.getAttribute("assocAudioId");

		if(vMX.DSPS.size() > 0) vMX.assignAudio(vMX.DSPS.get(0).id, sndObjid, "", "Play");

		blkCell.setAttribute("state","playing");
		objTHIS.setImage("jsxuser:///assets/sound32.gif");
		objTHIS.setOverImage("jsxuser:///assets/sound32_over.gif");
		objTHIS.setEvent("mx.vMX.pauseSound(this);", jsx3.gui.Interactive.EXECUTE);
		objTHIS.setTip(__("Disable sound"));
		objTHIS.repaint();
	};

	vMX.pauseSound = function(objTHIS) {
		var blkCell = objTHIS.getAncestorOfName("cell_blk").getParent();
		var sndObjid = blkCell.getAttribute("assocAudioId");

		if(vMX.DSPS.size() > 0) vMX.assignAudio(vMX.DSPS.get(0).id, sndObjid, "", "None");

		blkCell.setAttribute("state","pause");
		objTHIS.setImage("jsxuser:///assets/sound32_pause.gif");
		objTHIS.setOverImage("jsxuser:///assets/sound32_over_pause.gif");
		objTHIS.setEvent("mx.vMX.playSound(this);", jsx3.gui.Interactive.EXECUTE);
		objTHIS.setTip(__("Enable sound"));
		objTHIS.repaint();
	};



	/***************************************************************************
	 *                                                                         *
	 * Communication with vMX daemon: splitting, assigning, controlling, etc.  *
	 *                                                                         *
	 **************************************************************************/

	vMX.splitMonitor = function(objid, schema, cellid) {
		var cdfSplitInfoDoc = jsx3.xml.CDF.Document.newDocument();
		cdfSplitInfoDoc.setAttribute("objid", objid);
		cdfSplitInfoDoc.setAttribute("sid", vMX.SID);

		var bSizeTooSmall = false;

		if(schema instanceof Array) {
			var objMonitor = vMX.getMonitorById(objid);

			for(var i = 0; i < schema.length; i++) {
				//var objCell = objMonitor.getCellById(schema[i].id);
				var touring = schema[i].touring;
				delete schema[i].touring;
				cdfSplitInfoDoc.insertRecord(schema[i]);

				if(touring) {
					var recTouring = {
						jsxid : jsx3.xml.CDF.getKey(), delay : touring.delay,
						sid : vMX.SID, monid : objid, cellid : schema[i].id, pos : touring.pos
					};
					cdfSplitInfoDoc.insertRecord(recTouring, schema[i].jsxid);
					for(var j = 0; j < touring.source.size(); j++)
						cdfSplitInfoDoc.insertRecord(
							{
							  jsxid : jsx3.xml.CDF.getKey(),
							  id : touring.source.get(j).id,
							  options : touring.source.get(j).options,
							  presetid : touring.source.get(j).presetid
							},
							recTouring.jsxid
						);
				}
			}
		}
		else if(schema instanceof Object) {
			var objMonitor = vMX.getMonitorById(objid);
			// Do nothing if we do not know resolution of the monitor
			if(!objMonitor.initialized) return;

			var w, h, w_corr, h_corr, x_start, y_start, id_prefix;
			if(cellid) {
				var c;
				for(var iter = objMonitor.iterator(); iter.hasNext();) {
					var cell = iter.next();
					if(cell.id != cellid) {
						cell.jsxid = jsx3.xml.CDF.getKey();
						cdfSplitInfoDoc.insertRecord(cell);

						// Append records for tour device array
						if(cell.isTour()) {
							var recTouring = {
								jsxid : jsx3.xml.CDF.getKey(), delay : cell.touring.delay,
								sid : vMX.SID, monid : objid, cellid : cell.id, pos : cell.touring.pos
							};
							cdfSplitInfoDoc.insertRecord(recTouring, cell.jsxid);
							for(var j = 0; j < cell.touring.source.size(); j++)
								cdfSplitInfoDoc.insertRecord(
								  {
									jsxid : jsx3.xml.CDF.getKey(),
									id : cell.touring.source.get(j).id,
									options : cell.touring.source.get(j).options,
									presetid : cell.touring.source.get(j).presetid
								  },
								  recTouring.jsxid
								);
						}
					}
					else {
						c = cell;
					}

				}
				w = Math.floor(c.w / schema.cols) - 1;
				h = Math.floor(c.h / schema.rows) - 1;
				w_corr = c.w - schema.cols*w - schema.cols + 1;
				h_corr = c.h - schema.rows*h - schema.rows + 1;
				x_start = c.x; y_start = c.y;
				id_prefix = c.id + " | " ;
			} else {
				w = Math.floor(objMonitor.hres / schema.cols) - 1;
				h = Math.floor(objMonitor.vres / schema.rows) - 1;
				w_corr = objMonitor.hres - schema.cols*w - schema.cols + 1;
				h_corr = objMonitor.vres - schema.rows*h - schema.rows + 1;
				x_start = y_start = 0;
				id_prefix = "";
			}

			// Must handle special kind of split schema: 4x3 with big cell at top left
			var bExclusive = schema.cols == 4 && schema.rows == 3 && schema.exclusive == true;
			var groupid    = (schema.apc == true) ? vMX.getCurWall().objid : 0;

			for(var i = 0; i < schema.rows; i++)
				for(var j = 0; j < schema.cols; j++) {

					if(bExclusive && i == 0 && j == 0) {
						var cell_w = 2*w + 1;
						var cell_h = 2*h + 1;
					} else if(bExclusive && i<= 1 && j <= 1) {
						continue;
					} else {
						var cell_w = j==(schema.cols-1)?w+w_corr:w;
						var cell_h = i==(schema.rows-1)?h+h_corr:h;
					}

					// Check for small cells
					if(cell_w < vMX.MIN_CELL_WIDTH || cell_h < vMX.MIN_CELL_HEIGHT) {
						bSizeTooSmall = true;
					}

					//Empty cells should have srcObjid == 0
					cdfSplitInfoDoc.insertRecord( {
					  'jsxid' : jsx3.xml.CDF.getKey(), 'id' : id_prefix + "Cell " + i + "." + j,
					  'x' : x_start + j*w + j, 'y' : y_start + i*h + i, 'w' : cell_w, 'h' : cell_h,
					  'state' : "Empty", 'scale' : "Cell", 'open_timeout' : 10, 'play_timeout' : 10,
					  'srcObjid' : 0, 'arc_from' : 0, 'arc_to' : 0, 'arc_pos' : 0, 'groupid': groupid,
					  'analytics' : vMX.MonitorCell.NO_ANALYTICS,
					  'streamnum' : vMX.MonitorCell.STREAM_QUALITY_DEFAULT,'refresh' : 0,
                      'audioObjid' : 0, 'volume' : 0
					} );
				}
		}

		var numCells = cdfSplitInfoDoc.selectNodes("//record[@x and @y and @w and @h and @id]").size();
		if( schema != null && !(schema instanceof Array) && numCells > 20 ) {
			alert(__("Cannot create a new cell.\nThe limit of cells on vMX screen (20) is reached"));
		}
		else if(cellid && bSizeTooSmall) {
			alert(__("Cell size isn't enough to be split.\nPlease increase the cell size."))
		} else {
			vMX.getServer().getCache().setDocument("cdfSplitInfoDoc", cdfSplitInfoDoc);
			vMX.doServiceCall("split_monitor");
			//appLogger.error(vMX.SplitMonitorSvc.getOutboundDocument());
		}
	};

	vMX.onSplitMonitorSuccess = function(objEvent) {
		var response = vMX.getServer().getCache().getDocument("cdfSplitMonitorResponseDoc");
		if(!response) {
			vMX._log(2, "onSplitMonitorSuccess: response is NULL!");
			return;
		}
		var errMsg = response.getAttribute("errMsg");
		if(!errMsg) {
			var doc = vMX.getServer().getCache().getDocument("cdfSplitInfoDoc");
			if(doc)  {
				var objid = doc.selectSingleNode("/data[@objid]").getAttribute("objid");
				window.clearTimeout(vMX.QUERY_STATE_TID);
				window.setTimeout(function() { vMX.refreshState(vMX.getMonitorById(objid)); }, 1000);
			}
		} else {
			vMX._log(2, "'split_monitor' returned error: " + errMsg);
		}
	};


	vMX.controlCells = function(ctrlData) {
		var cdfCellControlInDoc = jsx3.xml.CDF.Document.newDocument();
		cdfCellControlInDoc.setAttribute("sid", vMX.SID);
		for(var i = 0; i < ctrlData.length; i++) {
			cdfCellControlInDoc.insertRecord(ctrlData[i]);
		}

		vMX.getServer().getCache().setDocument("cdfCellControlInDoc", cdfCellControlInDoc);
		vMX.doServiceCall("control_cells");
		//alert(vMX.ControlCellsSvc.getOutboundDocument());
	};

	vMX.onControlCellsSuccess = function() {
		var response = vMX.getServer().getCache().getDocument("cdfControlCellsResponseDoc");
		if(response && !response.getAttribute("errMsg")) {
			var doc = vMX.getServer().getCache().getDocument("cdfCellControlInDoc");
			if(doc) {
				var monid = doc.selectNodes("//record[@monid]").get(0).getAttribute("monid");
				window.clearTimeout(vMX.QUERY_STATE_TID);
				window.setTimeout(function() {vMX.refreshState(vMX.getMonitorById(monid))}, 500);
			}
		} else {
			vMX._log(2, "'control_cells' returned error: " +
						(response ? response.getAttribute("errMsg") : "response doc is null!"));
		}

	};


	vMX.getMonitorState = function(objid) {
		var cdfGetMonitorStateDoc = jsx3.xml.CDF.Document.newDocument();
		cdfGetMonitorStateDoc.setAttribute("objid", objid);
		cdfGetMonitorStateDoc.setAttribute("sid", vMX.SID);
		vMX.getServer().getCache().setDocument("cdfGetMonitorStateDoc", cdfGetMonitorStateDoc);
		vMX.doServiceCall("get_monitor_state");
	};


	vMX.onGetMonitorStateSuccess = function(objEvent){
		if(vMX.QUERY_STATE_METHOD == "getMonitorState") vMX.onRefreshStateReady();
		var cdfMonitorStateOutDoc = vMX.getServer().getCache().getDocument("cdfMonitorStateOutDoc");

		var errMsg = cdfMonitorStateOutDoc.getAttribute("errMsg");
		if (errMsg) {
			// Check if session was closed
			if (errMsg.search(/No such session/i) > -1) {
				mx.MATRIX2.guiLogOff("SESSION_EXPIRED");
			}
			return;
		}

		var nodeMonitor = cdfMonitorStateOutDoc.selectSingleNode("/data/record[@monid]");
		if(nodeMonitor == null) return;
		var objid = nodeMonitor.getAttribute("monid");
		var objMonitor = vMX.getMonitorById(objid);
		if(objMonitor == null) {
			/* i18n: %1 is an object id */
			alert(Gettext.strargs(__("No matching monitor found for objid %1"), [objid]));
			return;
		}

		objMonitor.fromXML(nodeMonitor, nodeMonitor.selectNodes("//record[@cellid]"));
	};

	vMX.onGetMonitorStateError = function(objEvent) {
		if(vMX.QUERY_STATE_METHOD == "getMonitorState") vMX.onRefreshStateReady();
	};
	vMX.onGetMonitorStateInvalid = function(objEvent) {
		if(vMX.QUERY_STATE_METHOD == "getMonitorState") vMX.onRefreshStateReady();
	};


	vMX.assignStream = function(
	    srcObjid,
	    monid,
	    cellid,
	    archive_time,
	    state,
	    text,
	    analytics,
	    streamnum,
	    volume)
    {
		var arrRec = [];
		if(arguments.length == 1 && arguments[0] instanceof Array) {
			arrRec = arguments[0];
			for(var i = 0; i < arrRec.length; i++) {
				arrRec[i].jsxid = i;
				arrRec[i].open_timeout = 10;
				arrRec[i].play_timeout = 10;
				if (arrRec[i].analytics == null) {
					arrRec[i].analytics = vMX.getDefaultAnlState(arrRec[i].srcObjid);
				}
				if (arrRec[i].streamnum == null) {
					arrRec[i].streamnum = vMX.getDefaultStreamNumber(arrRec[i].srcObjid);
				}
				if (arrRec[i].volume == null) {
					arrRec[i].volume = vMX.MonitorCell.SOUND_MUTE;
				}

				arrRec[i].audioObjid = vMX.getAudioObjid(arrRec[i].srcObjid);
			}
		} else {
			var rec = {
				'jsxid' : 0, 'cellid' : cellid, 'monid' : monid, 'srcObjid' : srcObjid,
				'text' : text, 'state' : state?state:"Play", 'open_timeout' : 10, 'play_timeout' : 10,
				'analytics' : (analytics==null ? vMX.getDefaultAnlState(srcObjid) : analytics),
				'streamnum' : (streamnum==null ? vMX.getDefaultStreamNumber(srcObjid) : streamnum),
				'audioObjid' : vMX.getAudioObjid(srcObjid),
				'volume' : (volume == null ?  vMX.MonitorCell.SOUND_MUTE : volume),
				'refresh' : 0
			};
			if(archive_time) {
				rec.arc_from = archive_time.from;
				rec.arc_to   = archive_time.to;
				rec.arc_pos  = archive_time.pos;
			} else {
				rec.arc_from = rec.arc_to = rec.arc_pos = 0;
			}
			arrRec[0] = rec;
		}

		var cdfAssignInfoDoc = jsx3.xml.CDF.Document.newDocument();
		cdfAssignInfoDoc.setAttribute("sid", vMX.SID);
		for(var i = 0; i < arrRec.length; i++) {
			cdfAssignInfoDoc.insertRecord(arrRec[i]);
		}
		vMX.getServer().getCache().setDocument("cdfAssignInfoDoc", cdfAssignInfoDoc);
		vMX.doServiceCall("assign_stream");
	};

	vMX.assignURL = function(monid, cellid, url, state, refresh) {
		if (!state) state = "Play";
		var vncurl = "";
		var weburl = "";

		if (String(url).substr(0, 3) == "vnc") {
			vncurl = url;
		}
		else {
			weburl = url;
			if (weburl != vMX.MonitorCell.ALERTLOG_URL)
				vMX.MagnifierCell.PrevWebURL = weburl;
			if (refresh) vMX.MagnifierCell.PrevRefreshInterval = refresh;
		}

		var rec = {
			'jsxid' : 0, 'cellid' : cellid, 'monid' : monid, 'srcObjid' : 0,
			'text' : vncurl, 'state' : state, 'open_timeout' : 10, 'play_timeout' : 10,
			'vncurl' : vncurl, 'analytics' : vMX.MonitorCell.NO_ANALYTICS,
			'streamnum' : vMX.MonitorCell.STREAM_QUALITY_DEFAULT,
			'audioObjid' : 0, 'volume' :  vMX.MonitorCell.SOUND_MUTE,
			'weburl' : weburl, 'refresh' : refresh
		};
		rec.arc_from = rec.arc_to = rec.arc_pos = 0;

		var cdfAssignInfoDoc = jsx3.xml.CDF.Document.newDocument();
		cdfAssignInfoDoc.setAttribute("sid", vMX.SID);
		cdfAssignInfoDoc.insertRecord(rec);
		vMX.getServer().getCache().setDocument("cdfAssignInfoDoc", cdfAssignInfoDoc);
		vMX.doServiceCall("assign_stream");
	};

	vMX.onAssignStreamSuccess = function(objEvent) {
		var response = vMX.getServer().getCache().getDocument("cdfAssignStreamResponseDoc");
		if(response && !response.getAttribute("errMsg")) {
			var doc = vMX.getServer().getCache().getDocument("cdfAssignInfoDoc");
			if(doc)  {
				var monid = doc.selectNodes("//record[@monid]").get(0).getAttribute("monid");
				window.clearTimeout(vMX.QUERY_STATE_TID);
				window.setTimeout(function() { vMX.refreshState(vMX.getMonitorState(monid)) }, 500);
			}
		} else {
			mx.vMX._log(2, "'assign_stream' returned error: " +
					(response ? response.getAttribute("errMsg") : "response doc is null!"));
		}

	};


	vMX.assignAudio = function(dspid, srcObjid, srcOptions, state) {
		var dspListInDoc = jsx3.xml.CDF.Document.newDocument();
		dspListInDoc.setAttribute("sid", vMX.SID);
		dspListInDoc.insertRecord( {
									dspid:dspid, srcObjid:srcObjid, srcOptions:srcOptions,
									state:state, play_timeout:10, open_timeout:10
								} );
		vMX.getServer().getCache().setDocument("dspListInDoc", dspListInDoc);
		vMX.doServiceCall("assign_audio");
	};

	vMX.onAssignAudioSuccess = function(objEvent) {};


	vMX.getWallChanges = function(vntWall) {
		var objWall;
		if(!vntWall)
			objWall = vMX.getCurWall();
		else if(typeof(vntWall) == "string" || typeof(vntWall) == "number")
			objWall = vMX.getWallById(vntWall);
		else if(typeof(vntWall) == "object" && vntWall instanceof vMX.Wall)
			objWall = vntWall;

		if(!objWall) {
			if(vMX.WALLS.size()) objWall = vMX.WALLS.get(0);
			else return;
		}
		var cdfWallStateInDoc = jsx3.xml.CDF.Document.newDocument();
		cdfWallStateInDoc.setAttribute("sid", vMX.SID);
		cdfWallStateInDoc.setAttribute("wallid", objWall.objid);
		cdfWallStateInDoc.setAttribute("wall_state", objWall.state);

		vMX.getServer().getCache().setDocument("cdfWallStateInDoc", cdfWallStateInDoc);
		vMX.doServiceCall("get_wall_changes");
	};


	vMX.onGetWallChangesSuccess = function(objEvent) {
		if(vMX.QUERY_STATE_METHOD == "getWallChanges") vMX.onRefreshStateReady();

		var cdfWallStateOutDoc = vMX.getServer().getCache().getDocument("cdfWallStateOutDoc");
		if(!cdfWallStateOutDoc)  {
			vMX._log(2, "onGetWallChangesSuccess(): cdfWallStateOutDoc is null");
			return;
		} else if(cdfWallStateOutDoc.getAttribute("errMsg")) {
			vMX._log(3, "'get_wall_changes' method returned error! : " + cdfWallStateOutDoc.getAttribute("errMsg"));
			// Check if session was closed
			var errMsg = cdfWallStateOutDoc.getAttribute("errMsg");
			if (errMsg.search(/No such session/i) > -1) {
				mx.MATRIX2.guiLogOff("SESSION_EXPIRED");
			}
			return;
		}
		var bWallChanged = false;
		var nodeWall = cdfWallStateOutDoc.selectSingleNode("/data/record[@wallid]");
		var wall = vMX.getWallById(nodeWall.getAttribute("wallid"));
		if(!wall) {
			vMX._log(2, "onGetWallChangesSuccess(): no  wall found for objid: " + nodeWall.getAttribute("wallid"));
			return;
		}

		var rows = nodeWall.getAttribute("rows");
		var cols = nodeWall.getAttribute("cols");
		if(rows != wall.rows || cols != wall.cols) {
			bWallChanged = true;
			vMX._log(3, "Configuration of wall " + wall.objid + " was changed!");
			return;
		}
		var wall_state = cdfWallStateOutDoc.getAttribute("wall_state");

		for(var iter = cdfWallStateOutDoc.selectNodeIterator("//record[@monid]"); iter.hasNext();) {
			var nodeWallCell = iter.next();

			var monid = nodeWallCell.getAttribute("monid");
			var row = nodeWallCell.getAttribute("row");
			var col = nodeWallCell.getAttribute("col");

			// Let's find matching WallCell and Monitor object
			var wallcell = wall.getCellAt(row, col);

			if(wallcell.monitor == null) continue;
			var nodeCells = nodeWallCell.selectNodes(
				"/data/record[@wallid=" + wall.objid + "]/record[@monid=" + monid + "]/record[@cellid]"
			);

			wallcell.monitor.fromXML(nodeWallCell, nodeCells);
		}

		// If all OK, save new wall state string
		wall.state = wall_state;

		wall.checkHeaders();
	};

	/**
	 * Must provide a non-default handler for service call failure
	 * because we need to restart query loop
	 */
	vMX.onGetWallChangesError = function(objEvent) {
		vMX._log(2, "'getWallChanges' call failed: " + objEvent.target.getRequest().getResponseText());
		if(vMX.QUERY_STATE_METHOD == "getWallChanges") vMX.onRefreshStateReady();
		//if(vMX.QUERY_STATE_ENABLED && vMX.getWallLayout())
		//vMX.QUERY_STATE_TID = window.setTimeout("mx.vMX.getWallChanges();", vMX.QUERY_STATE_TIMEOUT);
	};
	vMX.onGetWallChangesInvalid = function(objEvent) {
		vMX._log(2, "'getWallChanges' call Invalid!: " + objEvent.target.getRequest().getResponseText());
		if(vMX.QUERY_STATE_METHOD == "getWallChanges") vMX.onRefreshStateReady();
	};

	vMX.onGetAudioDevicesSuccess = function(objEvent) {
		var dspListOutDoc = vMX.getServer().getCache().getDocument("dspListOutDoc");
		if(!dspListOutDoc) return;
		var errMsg = dspListOutDoc.getAttribute("errMsg");
		if(errMsg) {
			/* i18n: %1 is an error message */
			alert(Gettext.strargs(__("Error querying audio devices: %1"), [errMsg]));
		} else {
			var dspNodes = dspListOutDoc.selectNodes("/data/record[@dspid]");
			vMX.DSPS.clear();
			for(var i = 0; i < dspNodes.size(); i++) {
				var node = dspNodes.get(i);
				var objDSP = new vMX.DSP(node.getAttribute("dspid"), node.getAttribute("srcObjid"),
										 node.getAttribute("srcOptions"), node.getAttribute("state"));
				vMX.DSPS.add(objDSP);
			}
		}
	};


	vMX.setTouring = function(monid, cellid, delay, devices) {
		var objMonitor = vMX.getMonitorById(monid);
		if(!objMonitor) return;
		var objCell = objMonitor.getCellById(cellid);
		if(!objCell) return;

		// If new touring is created, check whether tour limit(12) is reached
		if(!objCell.isTour()) {
			var arrCells = objMonitor.getCellsArray();
			var numTours = 0;
			for(var i = 0; i < arrCells.length; i++) {
				if(arrCells[i].touring) numTours++;
			}
			if(numTours >= 12) {
				alert(__("Cannot configure a new touring cell\nThe limit of touring cells on vMX screen (12) is reached"));
				return;
			}
		}

		var cdfTouringDoc = jsx3.xml.CDF.Document.newDocument();
		cdfTouringDoc.setAttribute("sid", vMX.SID);
		cdfTouringDoc.setAttribute("monid", monid);
		cdfTouringDoc.setAttribute("cellid", cellid);
		cdfTouringDoc.setAttribute("delay", delay);
		if(devices instanceof Array)
			devices = jsx3.util.List.wrap(devices);
		for(var i = 0; i < devices.size(); i++) {
			var dev = devices.get(i);
			if(!dev.jsxid) dev.jsxid = jsx3.xml.CDF.getKey();
			cdfTouringDoc.insertRecord(dev);
		}
		vMX.getServer().getCache().setDocument("cdfTouringDoc", cdfTouringDoc);
		vMX.doServiceCall("set_touring");
	};

	vMX.onSetTouringSuccess = function(objEvent) {
		var doc = vMX.getServer().getCache().getDocument("cdfTouringDoc");
		if(doc) {
			window.clearTimeout(vMX.QUERY_STATE_TID);
			window.setTimeout(function() {vMX.refreshState(vMX.getMonitorById(doc.getAttribute("monid")))}, 500);
		}
	};

	vMX.setApc = function(bSetRemove, apcStruct) {
		//vMX._log(3, "set APC: start");
		var cdfDoc = jsx3.xml.CDF.Document.newDocument();
		cdfDoc.setAttribute("sid", vMX.SID);
		cdfDoc.setAttribute("actionFlag", bSetRemove ? "true" : "false");
		if (!apcStruct.jsxid) {
			apcStruct.jsxid = jsx3.xml.CDF.getKey();
		}
		cdfDoc.insertRecord(apcStruct);
		vMX.getServer().getCache().setDocument("cdfSetAPCInDoc", cdfDoc);
		vMX.doServiceCall("set_apc");
	};

	vMX.onSetApcSuccess = function() {
		var doc = vMX.getServer().getCache().getDocument("cdfBooleanOutDoc");
	};

	/**
	 * Service call to get event info when opening dialog window
	 */
	vMX.getEventInfo = function(eventId) {

		vMX.CURR_WITNESSES = {};

		var cdfDoc = jsx3.xml.CDF.Document.newDocument();
		cdfDoc.setAttribute("eventid", eventId);
		cdfDoc.insertRecord({jsxid:jsx3.xml.CDF.getKey(),name:"witnesses",value:"true"});
		cdfDoc.insertRecord({jsxid:jsx3.xml.CDF.getKey(),name:"timeformat",value:"UTCTIME"});
		vMX.getServer().getCache().setDocument("cdfEventRequestDoc", cdfDoc);

		var objService = vMX.getServer().loadResource("vmx_elog_xml");
		objService.setOperation("get_events");
		//subscribe
		objService.subscribe(jsx3.net.Service.ON_SUCCESS, vMX.onGetEventInfoSuccess);
		objService.subscribe(jsx3.net.Service.ON_ERROR, function(objEvent) { vMX._log(2, "getEventInfo failed")});
		objService.subscribe(jsx3.net.Service.ON_INVALID, function() { vMX._log(2, "Invalid get_events request!");});

		objService.setEndpointURL(mx.MATRIX2.getServerURL()+"/axis2/services/EventLogService");
		objService.doCall();
	};


	vMX.onGetEventInfoSuccess = function() {
		var cdfDoc = vMX.getServer().getCache().getDocument("cdfEventResponseDoc");

		var objEvent = cdfDoc.selectNodes("//record").get(0);
		mx.ELog2.checkTags(objEvent);

		var event = {};
		var objAttrs = objEvent.getAttributeNames();
		for (var i=0; i < objAttrs.length; i++) {
			event[objAttrs[i].toLowerCase()] = objEvent.getAttribute(objAttrs[i]);
		}
		mx.vMX.CURRENT_EVENT = objEvent;

		// Get icon for device
		var icon = resourceTree.getObject(event.objid);
		if (icon) {
			event.css = icon.css;
		} else {
			// check witnesses to provide correct icon
			var witness = vMX.CURR_WITNESSES[event.objid];
			event.css = resourceTree.getDeviceType(witness).css;
		}

		event.note       = event.note || '';
		event.local_when = mx.TIME.timestamp2date(event.utctime_when * 1000);

		vMX.popupEventDialog(event);
	};

	vMX.parseWitnesses = function(node) {
		var objects = node.getChildNodes();

		for (var i=0; i < objects.size(); i++) {
			var witness = objects.get(i);
			var witnessId = witness.getAttribute("OBJID");
			vMX.CURR_WITNESSES[witnessId] = {
				name:    witness.getAttribute("NAME"),
				otype:   witness.getAttribute("OTYPE"),
				subtype: witness.getAttribute("SUBTYPE")
			};
		}

		var witnessList = [];
		for (i in vMX.CURR_WITNESSES) if (vMX.CURR_WITNESSES.hasOwnProperty(i)) {
			witnessList.push(i);
		}
		vMX.CURR_WITNESSES.witnessList = witnessList.join(' ');
	};


	vMX.popupEventDialog = function(event)
	{
		var dialog = mx.MATRIX2.getServer().getJSXByName('eventInfoDialog');
		dialog.getDescendantOfName("vMXWitnessActions").setVisibility(jsx3.gui.Block.VISIBILITYHIDDEN, true);
		var dimensions = mx.MATRIX2.getServer().getJSXByName('root_block').getAbsolutePosition();
		dialog.setLeft((dimensions.W - dialog.getWidth()) / 2);
		dialog.setTop((dimensions.H - dialog.getHeight()) / 2);

		dialog.getDescendantOfName('eventCaption').setText(
			'<span class="inline ' + event.css + '"> ' + vMX.CURR_WITNESSES[event.objid].name + '</span>'
		);

		var matrixWitnesses = dialog.getDescendantOfName('eventWitnesses');
		matrixWitnesses.resetXmlCacheData();
		var XMLString = mx.ELog2.checkWitnesses(vMX.CURR_WITNESSES.witnessList, event.eventid, event.objid, (event.tag.search(/6/) != -1));
		matrixWitnesses.setXMLString(XMLString);
		matrixWitnesses.repaintData();

		var messages = dialog.getDescendantOfName('messages');
		var disabled = (event.needupd) ? '' : 'disabled = "disabled"';
		messages.setText(
			'<table style="text-align: right;">' +
				'<tr>' +
					'<td>Event:</td>' +
					'<td style="text-align: left;">' + event.eventid + ' - ' + event.local_when + '</td>' +
				'</tr><tr>' +
					'<td>System message:</td>' +
					'<td style="text-align: left;">' + event.message + '</td>' +
				'</tr><tr>' +
					'<td>User note:</td>' +
					'<td><input type="text" id="vmx_event_user_message" value="' + event.note + '" ' + disabled +' style="width: 250px;" onkeydown="mx.vMX.allowEventUpdate();" /></td>' +
				'</tr>' +
			'</table>',
			true
		);

		dialog.getDescendantOfName('blockAckn').setDisplay(jsx3.gui.Block.DISPLAYNONE);
		dialog.getDescendantOfName('blockClr').setDisplay(jsx3.gui.Block.DISPLAYNONE);
		dialog.getDescendantOfName('blockUpd').setDisplay(jsx3.gui.Block.DISPLAYNONE);

		if (event.needacknowlegement) {
			dialog.getDescendantOfName('blockAckn').setDisplay(jsx3.gui.Block.DISPLAYBLOCK);
			dialog.getDescendantOfName('buttonAckn').setEvent(
				"mx.vMX.submitEventAction('ack'," + event.eventid + ");",
				jsx3.gui.Interactive.EXECUTE
			);
		}
		if (event.needclr) {
			dialog.getDescendantOfName('blockClr').setDisplay(jsx3.gui.Block.DISPLAYBLOCK);
			dialog.getDescendantOfName('buttonClr').setEvent(
				"mx.vMX.submitEventAction('clear'," + event.eventid + ");",
				jsx3.gui.Interactive.EXECUTE
			);
		}
		if (event.needupd) {
			dialog.getDescendantOfName('buttonUpd').setEvent(
				"mx.vMX.submitEventAction('update'," + event.eventid + ");",
				jsx3.gui.Interactive.EXECUTE
			);
		}

		dialog.setAttribute('eventId', event.eventid);

		dialog.setVisibility(jsx3.gui.Block.VISIBILITYVISIBLE);
		dialog.repaint();
	};

	vMX.allowEventUpdate = function() {
		var dialog = mx.MATRIX2.getServer().getJSXByName('eventInfoDialog');
		dialog.getDescendantOfName('blockUpd').setDisplay(jsx3.gui.Block.DISPLAYBLOCK).repaint();
		return true;
	};

	/**
	 *
	 * @param {string} action
	 * @param {number} eventid
	 */
	vMX.submitEventAction = function(action, eventid) {

		var name = action;
		var parameters = {
			eventid: eventid,
			note:    document.getElementById('vmx_event_user_message').value
		};

		mx.ELog2.eLogNoter.submitAction(name, parameters)
			.fail(function(code, message){
				var error = ELog2.checkEADAError(message);
				alert("submitAction: " + error);
			})
			.done(function(){
				mx.MATRIX2.getServer().getJSXByName('eventInfoDialog').setVisibility(jsx3.gui.Block.VISIBILITYHIDDEN).repaint();
			});
	};

	vMX.showWitnessMethods = function(objButton, objEVENT) {
		var dialog = mx.MATRIX2.getServer().getJSXByName('eventInfoDialog');
		mx.ELog2.showObjMenu(objButton.emGetSession().matrix,objButton.emGetSession().recordId,objEVENT,dialog.getDescendantOfName('vMXWitnessActions'));
	};

	vMX.checkMonitorInSeveralWalls = function(monitorid) {
		var count = 0;

		for (var i = 0; i < vMX.WALLS.size(); i++) {
			var wall = vMX.WALLS.get(i);
			for (var j = 0; j < wall._cells.size(); j++) {
				var cell = wall._cells.get(j);
				if (cell.monitor && cell.monitor.objid == monitorid ) {
					count++;
				}
			}
		}
		return count > 1;
	};


	vMX.getLayoutList = function(wallid) {

		if(!wallid) try { wallid = vMX.getCurWall().objid; } catch(e) {return}
		if(wallid != vMX.getLayoutList.WALLID) vMX.LAYOUTS = "";
		vMX.getLayoutList.WALLID = wallid;

		var cdfGetLayoutListDoc = jsx3.xml.CDF.Document.newDocument();
		cdfGetLayoutListDoc.setAttribute("objid", wallid);
		cdfGetLayoutListDoc.setAttribute("sid", vMX.SID);

		vMX.getServer().getCache().setDocument("cdfGetLayoutListDoc", cdfGetLayoutListDoc);
		vMX.doServiceCall("get_layout_list");
	};

	vMX.onGetLayoutListSuccess = function(objEvent) {
		var doc = vMX.getServer().getCache().getDocument("cdfLayoutListDoc");
		if(!doc) {
			vMX._log(2, "onGetLayoutListSuccess: cdfLayoutListDoc is null");
			return;
		}
		if(doc.getAttribute("errMsg")) {
			vMX._log(2, "'get_layout_list' returned error: " + doc.getAttribute("errMsg"));
			return;
		}

		var arrIds = doc.getRecordIds();
		vMX.LAYOUTS = '<option value="0">' + __("-Select configuration-") + '</option>';
		vMX.LAYOUT_LIST = new Array;
		for (var i = 0; i < arrIds.length; i++) {
			var rec = doc.getRecord(arrIds[i]);
			vMX.LAYOUTS += '<option value="' + rec.layoutid + '">' +
				$.escape(rec.jsxtext) + '</option>';
			vMX.LAYOUT_LIST.push({'name':rec.jsxtext, 'obj':rec.layoutid});
		}

		if(vMX.AWAIT_CFG_LIST) {
			var type = vMX.AWAIT_CFG_LIST.type;
			var params = new Object;
			params.share = 0;
			params.value = "";
			params[vMX.AWAIT_CFG_LIST.action] = vMX.AWAIT_CFG_LIST.script;
			vMX.AWAIT_CFG_LIST = null;
			vMX.getConfList();
			switch(type) {
				case 'save'  : vMX.confSaveDialog(true);   break;
				case 'load'  : vMX.confLoadDialog(true);   break;
				case 'manage': vMX.confManageDialog(true); break;
			}

		}
	};

	vMX.recallLayout = function(layoutid) {
		var cdfRecallLayoutDoc = jsx3.xml.CDF.Document.newDocument();
		cdfRecallLayoutDoc.setAttribute("sid", vMX.SID);
		cdfRecallLayoutDoc.setAttribute("layoutid", layoutid);
		vMX.getServer().getCache().setDocument("cdfRecallLayoutDoc", cdfRecallLayoutDoc);
		vMX.doServiceCall("recall_layout");
	};

	vMX.onRecallLayoutSuccess = function(objEvent) {
		$(document).trigger('vMX.loaded');
		var doc = vMX.getServer().getCache().getDocument("cdfRecallLayoutResponseDoc");
		var msg = doc.getAttribute("errMsg");
		var curLayout = vMX.getWallLayout();
		if(curLayout.getMagnifier() != null) curLayout.getMagnifier().doClose();
		if(!msg) {
			//alert("Request to load configration was accepted");
		}
		else {
			var result = msg.match(/Warnings:/i);
			if(result != null) {
				//alert(result[0]);
				//alert("Request to load configration was accepted");
			}
			else {
				alert(__("ERROR --> Failed to recall layout!"));
			}
		}

		vMX.getServer().getRootBlock().hideMask();
		vMX.getServer().getRootBlock().setCursor('default', true);
		vMX.getWallChanges(vMX.getCurWall());
	};

	vMX.onRecallLayoutError = function(objEvent) {
		$(document).trigger('vMX.loaded');
		vMX._log(2, "'recall_layout' call failed: " + objEvent.target.getRequest().getResponseText());
		alert(__("ERROR --> Failed to recall layout!"));
		//alert("ERROR --> " + "'recall_layout' call failed");
	};

	vMX.onRecallLayoutInvalid = function(objEvent) {
		$(document).trigger('vMX.loaded');
		vMX._log(2, "'invalid recall_layout' call: " +
			 objEvent.target.getRequest().getResponseText());
	};


	vMX.saveLayout = function(wallid, layout_name, layoutid) {
		if(wallid instanceof vMX.Wall)
			var wall = wallid;
		else
			var wall = vMX.getWallById(wallid);
		if(!wall) {
			vMX._log(2, "Invalid argument passed to saveLayout: " + wallid);
			return;
		}

		var cdfLayoutInDoc = jsx3.xml.CDF.Document.newDocument();
		cdfLayoutInDoc.setAttribute("sid", vMX.SID);
		cdfLayoutInDoc.setAttribute("wallid", wall.objid);

		// Must set 'wallid' to 0 to save new snapshot of current wall layout
		// If 'layoutid' not null, must overwrite layout with that id
		var wRec = { layoutid:(layoutid!=null?layoutid:0), rows:wall.rows, cols:wall.cols,
						name:layout_name, desc:wall.desc, jsxid:jsx3.xml.CDF.getKey() };
		cdfLayoutInDoc.insertRecord(wRec);
		for(var wcIter = wall.iterator(); wcIter.hasNext();) {
			var wallcell = wcIter.next();
			var wcRec = { row:wallcell.row, col:wallcell.col, jsxid:jsx3.xml.CDF.getKey() };
			var monitor = wallcell.monitor;
			if(monitor && monitor.state == "connected" && monitor.getCellsArray().length > 0) {
				wcRec.monid      = monitor.objid;
				wcRec.id         = monitor.objid;
				wcRec.vres       = monitor.vres;
				wcRec.hres       = monitor.hres;
				wcRec.statenum   = monitor.statenum;
				wcRec.cellscount = monitor.cellscount;
				wcRec.state      = monitor.state;
			} else {
				wcRec.monid = 0;
			}
			wcRec.jsxid = jsx3.xml.CDF.getKey();
			cdfLayoutInDoc.insertRecord(wcRec, wRec.jsxid);

			for(var i = (monitor&&monitor.state=="connected") ? monitor.iterator() : null; i && i.hasNext();) {
				var cell = i.next();
				var cRec = {
						cellid:cell.id, x:cell.x, y:cell.y, w:cell.w, h:cell.h,
						srcObjid:cell.srcObjid, arc_from:cell.arc_from,
						arc_to:cell.arc_to, arc_pos:cell.arc_pos,
						open_timeout:cell.open_timeout, play_timeout:cell.play_timeout,
						scale:cell.scale, state:cell.state, text:cell.text,
						groupid:cell.groupid, vncurl:cell.vncurl, analytics:cell.analytics,
						streamnum:cell.streamnum, audioObjid:cell.audioObjid, volume:cell.volume,
                        weburl:cell.weburl, refresh:cell.refresh,
						jsxid:jsx3.xml.CDF.getKey()
				};
				cdfLayoutInDoc.insertRecord(cRec, wcRec.jsxid);
				if(cell.isTour()) {
					var tRec = {
						sid:vMX.SID, monid:monitor.objid, cellid:cell.id, pos:cell.touring.pos,
						jsxid:jsx3.xml.CDF.getKey(), delay:cell.touring.delay
					};
					cdfLayoutInDoc.insertRecord(tRec, cRec.jsxid);
					for(var srcIter = cell.touring.source.iterator(); srcIter.hasNext();) {
						var src = srcIter.next();
						cdfLayoutInDoc.insertRecord(
							{
							  id:src.id,
							  options:src.options,
							  presetid:src.presetid,
							  jsxid:jsx3.xml.CDF.getKey()
							}, tRec.jsxid);
					}
				}
			}
		}

		vMX.getServer().getCache().setDocument("cdfLayoutInDoc", cdfLayoutInDoc);
		vMX.doServiceCall("save_layout");
	};

	vMX.onSaveLayoutSuccess = function(objEvent) {
		var doc = vMX.getServer().getCache().getDocument("cdfSaveLayoutResponseDoc");
		var errMsg = doc.getAttribute("errMsg");
		if(errMsg)
			alert(__("ERROR --> ") + errMsg);

		vMX.getLayoutList(vMX.getCurWall().objid);
	};

	vMX.onSaveLayoutError = function(objEvent) {
		vMX._log(2, "'save_layout' call failed: " + objEvent.target.getRequest().getResponseText());
		alert(__("ERROR --> Failed to save layout!"));
		vMX.getServer().getRootBlock().hideMask();
		vMX.getServer().getRootBlock().setCursor('default', true);
	};


	vMX.deleteLayout = function(layoutid) {
		if (layoutid == null) {
			vMX._log(2, "Empty layout ID passed to 'deleteLayout'");
			return;
		}
		var cdfDeleteLayoutDoc = jsx3.xml.CDF.Document.newDocument();
		cdfDeleteLayoutDoc.setAttribute("sid", vMX.SID);
		cdfDeleteLayoutDoc.setAttribute("layoutid", layoutid);
		vMX.getServer().getCache().setDocument("cdfDeleteLayoutDoc", cdfDeleteLayoutDoc);
		vMX.doServiceCall("delete_layout");
	};

	vMX.onDeleteLayoutSuccess = function(objEvent) {
		var doc = vMX.getServer().getCache().getDocument("cdfDeleteLayoutResponseDoc");
		var errMsg = doc.getAttribute("errMsg");
		if(errMsg)
			alert(__("ERROR --> ") + errMsg);

		vMX.getLayoutList();
	};


	vMX.getSessionId = function() {
		var cdfLoginInfoDoc = jsx3.xml.CDF.Document.newDocument();
		cdfLoginInfoDoc.setAttribute("name", /*"admin"*/window.jsSessionId);
		cdfLoginInfoDoc.setAttribute("password", "100A2C64CB89AFE7DC7B2ABBE04C1393456A7F18");
		cdfLoginInfoDoc.setAttribute("key", "NCHO9067NF");

		vMX.getServer().getCache().setDocument("cdfLoginInfoDoc", cdfLoginInfoDoc);

		var objService = vMX.getServer().loadResource("session_xml");
		objService.setOperation("open_session");
		//subscribe
		objService.subscribe(jsx3.net.Service.ON_SUCCESS, vMX.onGetSessionIdOK);
		objService.subscribe(jsx3.net.Service.ON_ERROR, function() { vMX._log(2, "Login failed!");});
		objService.subscribe(jsx3.net.Service.ON_INVALID, function() { vMX._log(2, "Invalid login request!");});

		objService.setEndpointURL(mx.MATRIX2.getServerURL()+"/axis2/services/Session_svc");

		objService.doCall();
	};

	vMX.onGetSessionIdOK = function() {
		var cdfSidOutDoc = vMX.getServer().getCache().getDocument("cdfSidOutDoc");
		var sid = cdfSidOutDoc.getAttribute("sid");
		var errMsg = cdfSidOutDoc.getAttribute("errMsg");
		if(errMsg != null) {
			vMX._log(2, "Failed to open session: " + errMsg);
			return;
		}
		vMX.SID = sid;
		// Next step is to set 'UID' session attribute
		vMX.setSessionUID();
	};

	/**
	 * Sets 'UID' attribute for current WS session.
	 * UID value is taken from 'jsUserId'
	 */
	vMX.setSessionUID = function() {
		var cdfSessionAttrInDoc = jsx3.xml.CDF.Document.newDocument();
		cdfSessionAttrInDoc.setAttribute("sid", vMX.SID);
		cdfSessionAttrInDoc.setAttribute("name", "UID");
		cdfSessionAttrInDoc.setAttribute("value", window.jsUserId);
		vMX.getServer().getCache().setDocument("cdfSessionAttrInDoc", cdfSessionAttrInDoc);

		var objService = vMX.getServer().loadResource("session_xml");
		objService.setOperation("set_session_attribute");
		//subscribe
		objService.subscribe(jsx3.net.Service.ON_SUCCESS, vMX.onSetSessionUIDSuccess);
		objService.subscribe(jsx3.net.Service.ON_ERROR,
							function() { vMX._log(2, "Failed to set session UID " + window.jsUserId + "!");});
	  objService.subscribe(jsx3.net.Service.ON_INVALID,
							function() { vMX._log(2, "Invalid 'set_session_attribute' request!");});

		objService.setEndpointURL(mx.MATRIX2.getServerURL()+"/axis2/services/Session_svc");
		objService.doCall();
	};

	vMX.onSetSessionUIDSuccess = function() {
		var cdfResponse = vMX.getServer().getCache().getDocument("cdfSetSessionAttributeResponseDoc");
		var errMsg = cdfResponse.getAttribute("errMsg");
		if(errMsg != null) {
			vMX._log(2, "Failed to set session UID: " + errMsg);
			return;
		}

		// Last step is to set session role
		vMX.setSessionRole();
	};

	vMX.setSessionRole = function(roleid) {

		roleid = roleid || resourceTree.getRole().obj;

		if (!roleid) {
			vMX._log(2, "Role isn't set!");
			return;
		}

		var cdfSetRoleDoc = jsx3.xml.CDF.Document.newDocument();
		cdfSetRoleDoc.setAttribute("sid", vMX.SID);
		cdfSetRoleDoc.setAttribute("roleid", roleid);
		vMX.getServer().getCache().setDocument("cdfSetRoleDoc", cdfSetRoleDoc);

		var objService = vMX.getServer().loadResource("session_xml");
		objService.setOperation("set_role");
		//subscribe
		objService.subscribe(jsx3.net.Service.ON_SUCCESS, vMX.onSetSessionRoleSuccess);
		objService.subscribe(jsx3.net.Service.ON_ERROR, function() { vMX._log(2, "Failed to setRole " + roleid + "!");});
		objService.subscribe(jsx3.net.Service.ON_INVALID, function() { vMX._log(2, "Invalid 'set_role' request!");});

		objService.setEndpointURL(mx.MATRIX2.getServerURL()+"/axis2/services/Session_svc");
		objService.doCall();
	};

	vMX.onSetSessionRoleSuccess = function() {
		var cdfSetRoleResponseDoc = vMX.getServer().getCache().getDocument("cdfSetRoleResponseDoc");
		if(!cdfSetRoleResponseDoc || cdfSetRoleResponseDoc.getAttribute("errMsg")) {
			vMX._log(2, "Failed to setRole ! " + cdfSetRoleResponseDoc.getAttribute("errMsg"));
		} else {
			// If OK - start vMX application
			var cdfSetRoleDoc = vMX.getServer().getCache().getDocument("cdfSetRoleDoc");
			vMX.ROLEID = cdfSetRoleDoc.getAttribute("roleid");

			// If this is first call to set_role, run post-init stage
			if(vMX._INIT)
				vMX.vmx_init(true);
		}
	};

	/**
	 * Retrieve information of vMX system objects: walls and monitors
	 */
	vMX.getSystemObjects = function() {
		//vMX._log("getSystemObjects starts");
		if(vMX.QUERY_OBJS_TID) {
			window.clearTimeout(vMX.QUERY_OBJS_TID);
			vMX.QUERY_OBJS_TID = 0;
		}

		vMX.getMonitorList();
	};

	/*
	 * Called after resource tree is refreshed
	 * Here we can update vMX internal structures
	 * for Walls and Monitors
	 */
	 vMX.onRefreshResources = function() {
		if (window.jsDebug) appLogger.info("onRefreshResources() called");

		// First refresh Monitor cache
		var monids = resourceTree.getObjects({type : 'monitor'});
		for (var i = 0; i < monids.length; i++) {
			var objid = monids[i];
			var monRes = resourceTree.getObject(objid);
			var objMonitor = vMX.getMonitorById(objid);

			if(!objMonitor) {
				objMonitor = new vMX.Monitor(objid, monRes.name, monRes.description);
				vMX.MONITORS.push(objMonitor);
				if (window.jsDebug) appLogger.info("NEW MONITOR ADDED: " + objMonitor);
			}
			else {
				objMonitor.setNameDesc(monRes.name, monRes.description);
			}
		}

		// Then update Walls
		var wallids = resourceTree.getObjects({type : 'wall'});
		wallids = vMX.sortByName(wallids);

		var lstProcessed = new jsx3.util.List;
		var lstRemove = new jsx3.util.List;

		var wallTab = mx.CT.wallTab;

		for (var i = 0; i < wallids.length; i++) {
			var wallid = wallids[i];
			var wall = vMX.getWallById(wallid);
			var wallRes = resourceTree.getObject(wallid);

			if (!wallRes)
				continue;

			if(!wall) {// -> Have to add new wall
				if (window.jsDebug) appLogger.info("Have to create new wall!");

				wall = new vMX.Wall(wallRes);
				vMX.WALLS.add(wall);
				if (!wallTab) { // Now create wall tab if still not created
					wallTab = mx.CT.createNewTab("vMX", "vmx");
				}

				var layout = new vMX.WallLayout(wall, wallTab);

				// Add reference into global hash
				vMX.WALL_LAYOUTS[wall.objid] = layout;

				//layout.paintLayout();
				layout.setDirty(); // TP4547

				// Call 'onTabChange' to check active/inactive layouts
				// and make required layout updates
				//vMX.onTabChange();

			} else {
				wall.update(wallRes);
			}

			lstProcessed.add(wall);
		};

		if (wallTab) {
			var menuBlock = wallTab.getDescendantOfName("wall_menu_block");
			// Fill wall selector list
			if (!vMX.WALL_SELECTOR) {
				vMX.WALL_SELECTOR = new jsx3.gui.NativeSelect('wall_menu', 0, 20, 250, 30, __("-Select wall-"));
				vMX.WALL_SELECTOR.setRelativePosition(jsx3.gui.Block.RELATIVE);
				vMX.WALL_SELECTOR.setEvent("mx.vMX.onWallLayoutChange(this);", jsx3.gui.Interactive.SELECT);
				menuBlock.setChild(vMX.WALL_SELECTOR);

				var wallDesc = new jsx3.gui.Block("wall_menu_desc", 250, 20, 500, 30);
				wallDesc.setRelativePosition(jsx3.gui.Block.RELATIVE);
				wallDesc.setMargin("0 0 0 20");
				wallDesc.setDisplay(jsx3.gui.Block.DISPLAYNONE);
				wallDesc.setCSSOverride("white-space: nowrap");
				menuBlock.setChild(wallDesc);
			}

			// Fill wall menu
			var wallXML = '<data jsxid="jsxroot">';
			for(var iter = vMX.WALLS.iterator(); iter.hasNext();) {
				var wall = iter.next();
				wallXML += '<record jsxid="' + wall.objid + '" jsxtext="[' + wall.objid + '] ' + wall.name + '"/>';
			}
			wallXML += '</data>';
			vMX.WALL_SELECTOR.setXMLString(wallXML);
			menuBlock.repaint();
		}

		vMX.onWallLayoutChange();

		for(var iter = vMX.WALLS.iterator(); iter.hasNext();) {
			var wall = iter.next();
			if(!lstProcessed.contains(wall)) {
				lstRemove.add(wall);
			}
		}

		if(lstRemove.size())  {
			var curWall = vMX.getCurWall();
			if(curWall && lstRemove.contains(curWall)) {
				/* i18n: %1 is a name of a wall, %2 is an object id of a wall */
				alert(Gettext.strargs(__("Wall '%1' [id:%2] was removed from configuration"), [curWall.name, curWall.objid]));
			}
			vMX.removeWallTabs(lstRemove);
			for(var iter = lstRemove.iterator(); iter.hasNext();) {
				var wall = iter.next();
				vMX._log("MUST DELETE WALL: " + wall);
				vMX.WALLS.remove(wall);
			}
		}

		if(vMX._INIT) {
			vMX.QUERY_STATE_ENABLED = true;
			vMX._INIT = false;
			for(var iter = vMX.WALLS.iterator(); iter.hasNext();)
				vMX.getWallChanges(iter.next());
			// Start activity guard
			vMX.startGuard();
		}
	};

	/**
	 * Retrieve information about vMX monitors. Several WS methods can provide such info,
	 *  but the fastest way is to call 'get_resources' method, because it doesn't
	 *  query display server and get all data directly from the database
	 */
	vMX.getMonitorList = function() {
		var cdfResourceRequestDoc = jsx3.xml.CDF.Document.newDocument();
		cdfResourceRequestDoc.setAttribute("sid", vMX.SID);
		cdfResourceRequestDoc.setAttribute("objtype", "D"); // 'D' for 'Device' object type
		cdfResourceRequestDoc.setAttribute("objsubtype", "V"); // 'V' for 'vMX Monitor' device subtype

		vMX.getServer().getCache().setDocument("cdfResourceRequestDoc", cdfResourceRequestDoc);
		vMX.doServiceCall("get_resources", "Cred_svc");
	};

	vMX.onGetResourcesSuccess = function(event) {
		var doc = vMX.getServer().getCache().getDocument("cdfObjectInfoListDoc");
		if(!doc || doc.getAttribute("errMsg")) {
			vMX._log(2, "Failed to get monitor list: " + (doc?doc.getAttribute("errMsg"):"doc is null"));
		}
		else {
			var nodes = doc.selectNodes("//record");
			for(var i = nodes.iterator(); i.hasNext();) {
				var node = i.next();
				var objid = node.getAttribute('objid');
				var name = node.getAttribute('name');
				var desc = node.getAttribute('desc');

				var objMonitor = vMX.getMonitorById(objid);
				if(!objMonitor) {
					objMonitor = new vMX.Monitor(objid, name, desc);
					vMX.MONITORS.push(objMonitor);
					vMX._log("NEW MONITOR ADDED: " + objMonitor);
				}
				else {
					objMonitor.setNameDesc(name, desc);
				}
			}
			vMX.getWalls();
		}
	};

	vMX.onGetWallsSuccess = function(event) {
		//vMX._log("onGetWallsSuccess");
		if(vMX.QUERY_OBJS_ENABLED)
			vMX.QUERY_OBJS_TID = window.setTimeout("mx.vMX.getSystemObjects();", vMX.QUERY_OBJS_TIMEOUT);

		var cdfWallListDoc = vMX.getServer().getCache().getDocument("WallListDoc");

		// Check for errors
		var errMsg = cdfWallListDoc.getAttribute("errMsg");
		if(errMsg) {
			vMX._log(2, "getWalls returned error: " + errMsg);
			return;
		}

		// Check whether we have to display warning message if walls configuration
		// was significantly changed. So, we do this if changes were made from Admin
		// GUI, but we don't - if changes were caused by new role chosen
		var bWarn = vMX.onGetWallsSuccess._ROLEID == vMX.ROLEID;
		var strWarnMsg = "";
		vMX.onGetWallsSuccess._ROLEID = vMX.ROLEID;

		var nodeWalls = cdfWallListDoc.selectNodes(".//record[@wallid]");

		nodeWalls = vMX.sortByName(nodeWalls);

		var lstProcessed = new jsx3.util.List;
		var lstRemove = new jsx3.util.List;

		var treeRefreshed = 0;
		for(var iter = nodeWalls.iterator(); iter.hasNext();) {
			var node = iter.next();
			var wallid = node.getAttribute("wallid");
			var wall = vMX.getWallById(wallid);
			var wallcred = resourceTree.getObject(wallid);

			// DE3370
			// Hack for discovering newly created walls
			if (   ! wallcred
				&& ! wall
				&& ! treeRefreshed
				&& vMX.TreeRefreshedForWallid != wallid)
			{
				// Perhaps its a new wall which is still not represented int resource tree
				treeRefreshed = 1;
				vMX.TreeRefreshedForWallid = wallid;
				resourceTree.refresh(0, true); // Refresh async
			}

			if (!wallcred)
				continue;

			if(!wall) {// -> Have to add new wall
				vMX._log("Have to create new wall!");

				wall = new vMX.Wall(node, wallcred.credentials);
				vMX.WALLS.add(wall);

				// Create a tab for new wall
				var th = mx.CT.createNewTab(wall.name, "vmx");

				var layout = new vMX.WallLayout(wall, th);

				// Add reference into global hash
				vMX.WALL_LAYOUTS[wall.objid] = layout;

				layout.paintLayout();
				layout.setDirty(); // TP4546

				// Call 'onTabChange' to check active/inactive layouts
				// and make required layout updates
				//vMX.onTabChange();

			} else {
				wall.fromXML(node, wallcred.credentials);
			}

			lstProcessed.add(wall);
		}

		for(var iter = vMX.WALLS.iterator(); iter.hasNext();) {
			var wall = iter.next();
			if(!lstProcessed.contains(wall)) {
				lstRemove.add(wall);
			}
		}


		if(lstRemove.size())  {
			var curWall = vMX.getCurWall();
			if(curWall && lstRemove.contains(curWall)) {
				/* i18n: %1 is a name of a wall, %2 is an object id of a wall */
				alert(Gettext.strargs(__("Wall '%1' [id:%2] was removed from configuration"), [curWall.name, curWall.objid]));
			}
			vMX.removeWallTabs(lstRemove);
			for(var iter = lstRemove.iterator(); iter.hasNext();) {
				var wall = iter.next();
				vMX._log("MUST DELETE WALL: " + wall);
				vMX.WALLS.remove(wall);
			}
		}

		if(vMX._INIT) {
			vMX.QUERY_STATE_ENABLED = true;
			vMX._INIT = false;
			for(var iter = vMX.WALLS.iterator(); iter.hasNext();)
				vMX.getWallChanges(iter.next());
			// Start activity guard
			vMX.startGuard();
		}
	};

	/**
	* sort walls by name
	**/
	vMX.sortByName = function(walls) {

		var sortedNames = new Array();
		var sortedList = new Array();

		for(var i = 0; i < walls.length; i++) {
			var wall = resourceTree.getObject(walls[i]);
			sortedNames.push(wall.name);
		}

		function compare(a,b) {
			if (a.toLowerCase() < b.toLowerCase()) return -1;
			if (a.toLowerCase() > b.toLowerCase()) return 1;
			return 0;
		}

		sortedNames = sortedNames.sort(function(a,b) { return compare(a, b) });

		for(var i = 0; i < sortedNames.length; i++){
			for(var j = 0; j < walls.length; j++) {
				var wall = resourceTree.getObject(walls[j]);
				if(sortedNames[i] == wall.name){
					sortedList.push(walls[j]);
				}
			}
		}
		return sortedList;
	};

	/*****************************************
	 *
	 *          Utility methods              *
	 *
	 * ***************************************/

	vMX.dump_obj = function(obj) {
		var str = "";
		str += "[";
		for(var key in obj) {
			str += (key + ": " );
			if(obj[key] instanceof jsx3.lang.Object) {
				str += "[jsx3.lang.Object] ";
			}
			else if (typeof obj[key] == "object") {
				str += vMX.dump_obj(obj[key]);
			} else {
				str += (obj[key] +"; ");
			}
		}
		return str + "] ";
	};


	/**
	 * Recurses the monitor layout and analyzes cell geometry looking for splitters
	 * Return value: object that reflects the sequence of splitters and their containers parameters
	 */
	vMX.recursiveScan = function(x, y, w, h, list, obj) {
		var top_cells = list.filter(function(c){return c.y==y&&c.x!=x?true:false});
		var iter = top_cells.iterator();
		var bFound = false;

		obj.x = x; obj.y = y; obj.w = w; obj.h = h; obj.cells = list;
		obj.orientation = "none";

		while(iter.hasNext() && !bFound) {
			var cur = iter.next();
			//Find all sibling cells which have same Y coordinate
			var sibling_cells = list.filter(function(c){return c.x==cur.x&&c!=cur?true:false});
			var total_h = Number(cur.h);
			for(var i = 0; i < sibling_cells.size(); i++)
				total_h += Number(sibling_cells.get(i).h) + 1;


			if(total_h == h) {
				bFound = true;
				//vMX._log("Found vertical splitter in [" + x +"," + y + "," + w + "," + h + "] on x = " + cur.x);


				var x1 = x, y1 = y, w1 = cur.x - x - 1,   h1 = h;
				var x2 = cur.x, y2 = cur.y, w2 = w - w1 - 1, h2 = h;
				var list1 = list.filter( function(c){
					return c.x >= x1 && c.y >= y1 && c.x+c.w <= x1+w1 && c.y+c.h <= y1+h1 ? true : false
				});
				var list2  = list.filter( function(c){
					return c.x >= x2 && c.y >= y2 && c.x+c.w <= x2+w2 && c.y+c.h <= y2+h2 ? true : false
				});

				// Run recursion for both regions
				obj.region1 = {};
				obj.region2 = {};
				obj.orientation = jsx3.gui.Splitter.ORIENTATIONH;//"vertical";
				obj.subcontainer1pct = w1/w * 100;
				var res1 = arguments.callee(x1, y1, w1, h1, list1, obj.region1);
				var res2 = arguments.callee(x2, y2, w2, h2, list2, obj.region2);
				if(res1 == false || res2 == false) return false;

				break;
			}
		}

		var left_cells = list.filter(function(c){return c.x==x&&c.y!=y?true:false});

		iter = left_cells.iterator();

		while(iter.hasNext() && !bFound) {
			var cur = iter.next();
			//Find all sibling cells which have same X coordinate
			var sibling_cells = list.filter(function(c){return c.y==cur.y&&c!=cur?true:false});
			var total_w = Number(cur.w);
			for(var i = 0; i < sibling_cells.size(); i++)
				total_w += Number(sibling_cells.get(i).w) + 1;

			if(total_w == w) {
				bFound = true;
				//vMX._log("Found horizontal splitter in [" + x +"," + y + "," + w + "," + h + "] on y = " + cur.y);
				var x1 = x, y1 = y, w1 = w, h1 = cur.y - y - 1;
				var x2 = cur.x, y2 = cur.y, w2 = w, h2 = h - h1 - 1;
				var list1  = list.filter( function(c){
					return c.x >= x1 && c.y >= y1 && c.x+c.w <= x1+w1 && c.y+c.h <= y1+h1 ? true : false
				});
				var list2  = list.filter( function(c){
					return c.x >= x2 && c.y >= y2 && c.x+c.w <= x2+w2 && c.y+c.h <= y2+h2 ? true : false
				});

				// Run recursion for both regions
				obj.region1 = {};
				obj.region2 = {};
				obj.orientation = jsx3.gui.Splitter.ORIENTATIONV;//"horizontal";
				obj.subcontainer1pct = h1/h * 100;

				var res1 = arguments.callee(x1, y1, w1, h1, list1, obj.region1);
				var res2 = arguments.callee(x2, y2, w2, h2, list2, obj.region2);
				if(res1 == false || res2 == false) return false;
				break;
			}
		}


		if(!bFound) {
			// region is a single cell
			if(list.size() > 1) return false;
			var c = list.get(0);
			obj.orientation = "cell";
		}

		return true;
	};

	/**
	 * Cross-browser handling of opacity (IE/FX only). Unfortunately, TibcoGI doesn't provide
	 * any methods for this :(
	 */
	vMX.setOpacity = function(block, fltOpacity) {
		if(jsx3.CLASS_LOADER.IE) {
			block.setCSSOverride("filter:alpha(opacity=" + parseInt(fltOpacity*100) + ")");
		} else {
			block.setCSSOverride("opacity:" + fltOpacity);
		}
	};

	vMX.getOpacity = function(block) {
		var css = block.getCSSOverride();
		if(!css) return 1.0;

		if(jsx3.CLASS_LOADER.IE) {
			var result = css.match(/filter:alpha\(opacity=(\d+)\)/);
			return result == null ? 1.0 : parseFloat(result[1]/100);
		} else {
			var result = css.match(/opacity\s*:\s*(\d*\.\d*)/);
			return result == null ? 1.0 : parseFloat(result[1]);
		}
		return 1.0;
	};


	vMX.stopRefresh = function() {
		vMX.QUERY_STATE_ENABLED = false;
		if(vMX.QUERY_STATE_TID != 0) window.clearTimeout(vMX.QUERY_STATE_TID);
		vMX.QUERY_STATE_TID = 0;
	};

	vMX.startRefresh = function() {
		vMX.QUERY_STATE_ENABLED = true;
		vMX.refreshState();
	};

	vMX.refreshState = function(objDev) {
		vMX.LAST_REFRESH_TIME = new Date().getTime();
		if(vMX.QUERY_STATE_TID) {
			window.clearTimeout(vMX.QUERY_STATE_TID);
			vMX.QUERY_STATE_TID = 0;
		}
		if(objDev == null) { // Find active object to query (monitor or wall)
			var curLayout = vMX.getWallLayout();
			if(curLayout == null) return;
			if(curLayout.getMagnifier() != null) {
				var objid = curLayout.getMagnifier().getMonitor().objid;
				vMX.QUERY_STATE_METHOD = "getMonitorState";
			}
			else {
				var objid = curLayout.getWall().objid;
				vMX.QUERY_STATE_METHOD = "getWallChanges";
			}
		}
		else {
			if (typeof(objDev) == "object" && objDev instanceof vMX.Wall) {
				var objid = objDev.objid;
				vMX.QUERY_STATE_METHOD = "getWallChanges";
			}
			else if (typeof(objDev) == "object" && objDev instanceof vMX.Monitor) {
				var objid = objDev.objid;
				vMX.QUERY_STATE_METHOD = "getMonitorState";
			}
		}

		if(objid)
			vMX[vMX.QUERY_STATE_METHOD](objid);
		else
			vMX._log(2,"refreshState(): Cannot resolve object ID for " + objDev)
	};

	vMX.onRefreshStateReady = function() {
		vMX.adjustLoadBalance();
		if(vMX.QUERY_STATE_ENABLED) {
			vMX.QUERY_STATE_TID = window.setTimeout("mx.vMX.refreshState()", vMX.QUERY_STATE_TIMEOUT);
		}
	};

	/**
	 * Varies state refresh timeout - increase it if magnifier with multiple cells is active,
	 * and if wall have too much monitors
	 */
	vMX.adjustLoadBalance = function() {
		var curLayout = vMX.getWallLayout();
		if(!curLayout) return;
		var magnifier = curLayout.getMagnifier();
		if(magnifier != null) {
			var nCells = magnifier.getCellsArray().length;
			if(nCells <= 5)
				vMX.QUERY_STATE_TIMEOUT = 3000;
			else if(nCells <= 10)
				vMX.QUERY_STATE_TIMEOUT = 4000;
			else
				vMX.QUERY_STATE_TIMEOUT = 5000;
		}
		else {
			var nMons = 0;
			for(var iter = curLayout.getWall().iterator(); iter.hasNext();) {
				if(iter.next().monitor != null) nMons++;
			}
			if(nMons <= 4)
				vMX.QUERY_STATE_TIMEOUT = 3000;
			else if(nMons <= 6)
				vMX.QUERY_STATE_TIMEOUT = 4000;
			else
				vMX.QUERY_STATE_TIMEOUT = 5000;
		}
	};

	/**
	 * Global guard process. Monitors overall vMX state and activity
	 *
	 */
	 vMX.startGuard = function() {
		vMX.GUARD_TID = window.setTimeout("mx.vMX.guard()", vMX.GUARD_TIMEOUT);
	 };

	 vMX.guard = function() {
		if (vMX.GUARD_TID) window.clearTimeout(vMX.GUARD_TID);
		vMX.GUARD_TID = window.setTimeout("mx.vMX.guard()", vMX.GUARD_TIMEOUT);

		var curTime = new Date().getTime();
		// check refresh state activity
		if (vMX.QUERY_STATE_ENABLED && (curTime - vMX.LAST_REFRESH_TIME > vMX.GUARD_TIMEOUT)) {
			vMX._log(3,"Guard detected refresh activity stopped! Restarting");
			if (vMX.QUERY_STATE_TID) window.clearTimeout(vMX.QUERY_STATE_TID);
			vMX.QUERY_STATE_TID = window.setTimeout("mx.vMX.refreshState()", 1000);
		}
	 };

	 vMX.parseURL = function(url) {
		/* scheme://domain:port/path?query_string#fragment_id */
		var result = {};

		url = url.replace(/^\s*/,'');
		url = url.replace(/\s*$/,'');
		var cmps = url.match( /^(?:(.+?):\/\/)?(.*?)(?::(.+))?@(.*)$/ );
		if (cmps) {
			result['schema']   = cmps[1];
			result['login']    = cmps[2];
			result['password'] = cmps[3];
			result['part']     = cmps[4];
		} else {
			cmps = url.match( /^(?:(.+?):\/\/)?(.*)$/ );
			result['schema'] = cmps[1];
			result['part']   = cmps[2];
		}

		return result;
	 };

	 // Get default value of analytics visualization for target camera
	 vMX.getDefaultAnlState = function(objid) {
		var anlState = vMX.MonitorCell.NO_ANALYTICS;
		var camObj = resourceTree.getObject(objid);
		if (camObj && camObj.attr) {
			var curState = camObj.attr('VAE_ACTIVE');
			if (curState == 'yes') {
				anlState = camObj.attr('VAE_VISUALIZATION');
			}
		}

		return anlState;
	 };

	/*
	 * Get default value for stream number
	 * Avatar cameras should be displayed with low quality stream
	 */
	 vMX.getDefaultStreamNumber = function(objid) {
		 var streamno;
		 var camObj = resourceTree.getObject(objid);
		 if (camObj && camObj.attr) {
			 if (camObj.attr('AVATARID')) {
				streamno = vMX.MonitorCell.STREAM_QUALITY_LOW;
			 } else {
				 streamno = vMX.MonitorCell.STREAM_QUALITY_NORMAL;
			 }
		 }
		 else {
			 streamno = vMX.MonitorCell.STREAM_QUALITY_DEFAULT;
		 }

		 return streamno;
	 };

     vMX.getAudioObjid = function(objid) {
         var audioObj = 0;
         var camObj = resourceTree.getObject(objid);
		 if (camObj && camObj.attr) {
             var assocAudioRec = resourceTree.getObject({'belongs2devid': objid});
			 if (assocAudioRec && camObj.attr('AUDIO') == 'on') {
                 audioObj = assocAudioRec.obj;
			 }
		 }

		 return audioObj;
     };

	 vMX.updateWallDesc = function(desc) {
		 var wallDesc = mx.CT.wallTab.getDescendantOfName("wall_menu_desc");
		 if (wallDesc) {
			 wallDesc.setDisplay(desc ? jsx3.gui.Block.DISPLAYBLOCK : jsx3.gui.Block.DISPLAYNONE, true);
			 wallDesc.setText(desc, true);
		 }
	 };
});
