resourceTree = {

	sortType:  'name', // Sorting type: 'name'|'id'
	sortOrder: 'asc',  // Sorting order: 'asc'|'desc'
	viewMode:  'list', // Mode of list: 'list'|'tile'
	animationSpeed: 'fast', // Animation speed (no animation for IE)

	role: {}, // Current role
	sets: {}, // Current sets
	objects: {}, //
	roles: {}, // All roles available for current user
	setIds: [], // Auxiliary array to sort sets by name

	defaultSettings: {
		'popup_video': 0,
		'popup_override': 0,
		'alert_3d': 1,
		'alert_audio': 0,
		'alert_header': 0,
		'add_into_filter': 0
	},
	// TODO: remove dual code
	// Settings array, used to paint and operate settings branch
	deviceSettings: {
		'popup_video':     {'default': 0, name: __("Popup video")},
		'popup_override':  {'default': 0, name: __("Popup overrides populated cell")},
		'alert_3d':        {'default': 1, name: __("Alert on Map3D")},
		'alert_audio':     {'default': 0, name: __("Audio alert")},
		'alert_header':    {'default': 0, name: __("Alert on cell header")},
		'add_into_filter': {'default': 0, name: __("Include into filter")}
	},

	// Caching jQuery collections of DOM objects for sets and devices, to improve searching.
	// This values must be updated on every repaint of resource tree
	cache: {$sets: null, $objects: null},

	// Auxiliary vars, to remember opened set and device when refreshing or loading RT
	openSet: null,
	openDevice: null,
	dragobjid: null,

	/**
	 * Initialize Resource Tree
	 * @param tibcoObj DOM id of element in which resource tree will be placed
	 */
	init: function(tibcoObj) {

		// Get user id and name
		this.user = { obj: window.jsUserId, name: window.jsUserName, type: 'user'};

		tibcoObj.setText(
			'<div id="resource_tree_root">' +
				'<div id="resource_tree_title">' +
					'<div id="res_title"><img src="images/design1/resources/' + mx.language.code + '/resources.png"></div>' +
					'<div class="toolbar" style="position: relative; right: 0; display: block; text-align: right;">' +
						'<span class="button wait" id="wait"></span>' +
						'<span id="res_btn_refresh" class="button refresh" title="' + __("Refresh event log and device list(s)") + '"></span>' +
						'<span id="res_btn_finder" class="button search" title="' + __("Open search panel") + '"></span>' +
						'<span id="res_btn_system_state" class="button system_state_unknown" title="' + __("Show system status") + '"></span>' +
						'<span id="res_btn_session_manager" class="button session_manager" title="' + __("Session manager") + '"></span>' +
						'<span id="res_btn_settings" class="button panel_collapsed" title="' + __("Resource tree settings") + '"></span>' +
					'</div>' +
				'</div>' +

				'<div id="resource_tree_body">' +
					'<div id="resource_options" style="display: none;">' +

						'<fieldset>' +
							'<legend>' + __("View settings") + '</legend>' +

							'<table style="width: 250px;"><tr><td>' + __("View mode: ") + '</td>' +
							'<td>' +
								'<select id="res_view_mode">' +
									'<option value="list">' + __("List") + '</option>' +
									'<option value="tile">' + __("Thumbnail") + '</option>' +
								'</select>' +
							'</td><td></td></tr><tr><td>' + __("Sort by: ") + '</td>' +
							'<td>' +
								'<select id="res_sort_type">' +
									'<option value="name">' + __("Name") + '</option>' +
									'<option value="id">' + __("Unique id") + '</option>' +
								'</select>' +
							'</td><td>' +
								'<div id="res_sort_order" class="button asc" title="' + __("Sort order") + '"></div>' +
							'</td></tr></table>' +
						'</fieldset>' + '<br>' +
						'<select id="res_role"></select> ' +
						'<button id="res_btn_role">' + __("Change role") + '</button>' +
						'<div id="resource_settings"></div>' +
					'</div>' +
					'<div id="resource_tree_wrap">' +
						'<div id="resource_search_panel">' +
							'<input id="res_finder_input" type="text" title="' + __("Search device") + '" /> ' +
							'<div id="res_btn_finder_next" class="button" title="' + __("Next") + '">&darr;</div>' +
							'<div id="res_btn_finder_prev" class="button" title="' + __("Previous") + '">&uarr;</div>' +
							'<div id="res_btn_finder_close" class="button search_close" title="' + __("Close search panel") + '"></div>' +
						'</div>' +
						'<div id="resource_tree"></div>' +
					'</div>' +
				'</div>' +
			'</div>').repaint();

		// default role, no async
		this.refresh(null, false);
		this.addEventHandlers();
		this.onAfterResizeTree();
	},


	/**
	 * Clear matrix interface and set new user role
	 * @param roleId
	 */
	setRole: function(roleId) {

		if (roleId == this.role.obj)
			return;
		if (confirm(__("Changing role will reset the interface!"))) {
			mx.MATRIX2.clearMatrix(true);
			if(mx.GE.checkGEPresence()){ mx.GE.loadLocations(); }
			this.refresh(roleId, false);
		}
	},


	/**
	 * Refresh resource tree
	 * @param roleId id of role which must be loaded.
	 * If no roleId provided, load default role
	 * @param async bool if true, do async call
	 */
	refresh: function(roleId, async) {
		roleId = roleId || this.role.obj || '';

		this.waitIndicator(true);
		$.ajax({
			url: '/api/call.php',
			data: {'function': 'getResourceTree', 'roleid': roleId},
			dataType: 'json',
			async: Boolean(async),
			cache: false,
			success: function(data) {
				resourceTree.refreshSuccess(data);
				resourceTree.waitIndicator(false);
			}
		});
	},


	/**
	 * Receive resource tree data and refresh resourceTree
	 * @param data
	 */
	refreshSuccess: function(data) {

		this.setIds = [];

		this.role = data.role;
		this.role.type = 'role';
		this.roles = data.roles;
		this.sets = data.sets || {};

		// If resource tree already contains objects, save their settings in new tree
		if (!$.isEmptyObject(this.objects) && data.objects) {
			for(var objId in this.objects) {
				if (this.objects[objId].settings && data.objects[objId]) {
					data.objects[objId].settings = this.objects[objId].settings;
				}
			}
		}
		this.objects = data.objects || {};

		var firstLevelIds = {};
		for(var setId in this.sets) {
			// Sort devices in each set
			this.sets[setId].objects.sort(this._sort);

			if (this.sets[setId].otype== "V" && this.sets[setId].subtype == "*") {
				this.sets[setId].type = 'avatar';
			} else {
				this.sets[setId].type = 'set';
			}

			// Save set ids and ids of first-level devices
			this.setIds.push(setId);
			$.each(this.sets[setId].objects, function() {firstLevelIds[this] = 1});
		}
		this.role.objects = Object.keys(firstLevelIds);

		// Sort sets
		this.setIds.sort(function(a, b) {return naturalSort(resourceTree.sets[a].name, resourceTree.sets[b].name);});

		// Add aux info to devices
		for(objId in this.objects) {
			$.extend(this.objects[objId], this.getDeviceType(objId));
		}

		// Add settings only to first-level objects
		$.each(this.role.objects, function(i, objId) {
			if (!resourceTree.objects[objId].settings) {
				// Set default settings
				resourceTree.objects[objId].settings = $.extend({}, resourceTree.defaultSettings);
			}
		});

		// Repaint roles selector and set current role
		var rolesOptions = [];
		$.each(this.roles, function() {
			this.type = 'role';
			rolesOptions.push('<option value="' + this.obj + '">' + this.name + '</option>');
		});
		$('#res_role').html(rolesOptions.join('')).val(this.role.obj);

		// Repaint admin button depending on current user rights
		// 51 - predefined GUI object, access to control panel
		if(this.objects[51]) {
			jsx3.GO("button_admin").setDisplay(jsx3.gui.Block.DISPLAYBLOCK, true);
		} else {
			jsx3.GO("button_admin").setDisplay(jsx3.gui.Block.DISPLAYNONE, true);
		}

		//update views list
		VIEWS.getViews();

		// Repaint resource tree and settings
		this.paint({});
		this.paintSettings();

		// Update vMX-specific objects: walls and monitors
		mx.vMX.onRefreshResources();

		mx.ELog2.applyFilters();
	},


	/**
	 * Paint resource tree
	 */
	paint: function() {

		var openSet = this.getExpandedSetId() || this.openSet;
		var openDevice = this.getSelectedId() || this.openDevice;

		// Prepare all sets
		var setsHtml = [];

		setsHtml.push(VIEWS.getViewsHtml());

		for (var i = 0; i < this.setIds.length; i++) {
			setsHtml.push(this.getSetHtml(this.setIds[i]));
		}

		$('#resource_tree').html(
			'<span class="inline role drag" obj=' + this.role.obj + '>' + this.user.name + ' (' + this.role.name + ')</span>' +
			'<ul>' + setsHtml.join('') + '</ul>'
		);

		this.cache.$sets = $('#resource_tree > ul > li > span.set');
		this.cache.$objects = $('#resource_tree > ul > li > ul > li > span.editable');

		// Perform things quickly: disable animation
		$.fx.off = true;
		// !important! showDeviceInfo must go before toggleBranch, DE2852
		if (openDevice) {
			this.showDeviceInfo(openDevice);
		}
		if (openSet) {
			this.toggleBranch(openSet);
		}
		$.fx.off = false;

		// Update camera snapshots if in tile mode
		this.updateSnapshots();
	},


	updateSnapshots: function() {
		if (this.viewMode == 'list') {
			var selected = this.cache.$objects.filter('.selected');
			selected.nextAll('div.edit').children('img').attr('src', resourceTree.getSnapshotHref(selected.attr('obj')));
		} else {
			this.cache.$objects.each(function() {
				var objId = $(this).attr('obj');
				$(this).nextAll('img').attr('src', resourceTree.getSnapshotHref(objId));
			});
		}
		setTimeout(function() {resourceTree.updateSnapshots()}, 60000);
	},


	/**
	 * Resize resource tree container when resizing parent element
	 */
	onAfterResizeTree: function() {
		var height = jsx3.GO('resource_wrapper').getAbsolutePosition().H;
		$('#resource_tree_root').height(height);

		height -= $('#resource_tree_title').height() + parseInt($('#resource_tree_title').css('padding-top')) + 16;
		$('#resource_tree_body').height(height);
	},


	/**
	 * Set content of resource tree settings tab
	 */
	paintSettings: function() {
		var html = '';

		html += this.getSettingsHtml(this.role.obj);
		for (var i = 0; i < this.setIds.length; i++) {
			html += this.getSettingsHtml(this.setIds[i]);
		}

		$('#resource_settings').html('<ul>' + html + '</ul>');

		this.repaintSettings();

	},


	/**
	 * Refresh content of settings tab;
	 * Calculate and set correct icons (yes/no/diff) for sets and role
	 */
	repaintSettings: function() {

		// Calculate settings for sets
		$.each(this.sets, function() {
			resourceTree._setSettingsIcons(this);
		});

		// Get settings for role
		resourceTree._setSettingsIcons(this.role);
	},


	/**
	 * set settings icons for given set of devices
	 * @param  obj  set or role to calculate settings icons
	 */
	_setSettingsIcons: function(obj) {

		var Ids = [];

		if (obj.type == 'set' || obj.type == 'role') {
			Ids = obj.objects;
		}

		for (var opt in this.deviceSettings) {
			var count = 0;
			var className;
			$.each(Ids, function(i, v) {
				if (resourceTree.objects[v].settings[opt]) {
					count++;
				}
			});

			if (count == Ids.length) {
				className = 'inline option yes';
			} else if (count == 0) {
				className = 'inline option no';
			} else {
				className = 'inline option diff';
			}
			$('#resource_settings span[obj=' + obj.obj + '][option_id=' + opt + ']')[0].className = className;
		}
	},


	/**
	 * Change clicked setting of device, set or role
	 */
	changeOption: function() {

		var obj = resourceTree.getObject($(this).attr('obj'));
		var opt = $(this).attr('option_id');
		var objIds; // array of ids which must change settings

		if (obj.type == 'role' || obj.type == 'set') {
			objIds = obj.objects;
		} else {
			objIds = [obj.obj];
		}

		if ($(this).hasClass('diff') || $(this).hasClass('no')) {
			$.each(objIds, function() { resourceTree.objects[this].settings[opt] = 1 });
		} else {
			$.each(objIds, function() { resourceTree.objects[this].settings[opt] = 0 });
		}

		if (obj.type != 'role' && obj.type != 'set')
			$(this).toggleClass("no yes");
		else {
			// If set or role changed, we must repaint settings div and selected device
			resourceTree.repaintSettings();
			// If device opened, set correct style for option
			var objId = resourceTree.getSelectedId();
			if (objId) {
				resourceTree.hideDeviceInfo();
				resourceTree.showDeviceInfo(objId);
			}
		}

		//update elog device set
		if(!mx.ELog2.hidden && mx.ELog2.device == 'filtered' && opt == "add_into_filter"){
			mx.ELog2.setFilterValue('device', "filtered");
		}

		if (opt == 'popup_video' && mx.ELog2.hidden) {
			// if popup_video toggled on and elog is inactive, start it
			if($(this).hasClass('yes') && !mx.ELog2.enabled) {
				mx.ELog2.startELog();
			}
			// if there's no popup video left, stop ELog
			else if(!resourceTree.getObjects({'popup_video': true}).length){
				mx.ELog2.stopELog();
			}
		}
	},


	/**
	 * Get HTML representation of one set, with current sorting and view type
	 * @param setId   id of set
	 */
	getSetHtml: function(setId) {

		if (!this.sets[setId])
			throw new Error("incorrect setId: " + setId);

		return '<li class="collapsed">' +
				'<div class="hitarea" />' +
				'<span class="inline set drag" obj="' + setId + '">' + this.sets[setId].name + '</span>' +
				'<ul style="display: none">' + this.getListHtml(this.sets[setId].objects) + '</ul>' +
			'</li>';
	},


	/**
	 * Get HTML representation of given array of devices
	 * @param list       array of obj ids
	 * @param _options   options hash
	 *     editable: true | false - if elements must be editable
	 *     view:     list | tile  - view mode
	 */
	getListHtml: function(list, _options) {

		if (!list.length)
			return '';

		// Process options
		var options = {'editable': true, 'view': this.viewMode};
		$.extend(options, _options);

		var classes = ['drag', 'inline'];
		if (options['editable']) {
			classes.push('editable');
		}

		var html = '';
		for(var i = 0; i < list.length; i++) {

			var objId = list[i];
			var obj = this.objects[objId];

			//Skip incorrect ids and audio devices
			if (!obj || obj.type == 'audio' || obj.type == 'wall' || obj.type == 'monitor')
				continue;

			var udid = obj.udid || objId;
			var udidstr = '';
			if (typeof(obj.udid) == 'undefined' || obj.udid == '' || !obj.udid){
				udidstr = /*'#' + objId*/'';
			}else{
				udidstr = '[' + udid + ']';
			}


			html +=
				'<li class="tile">' +
					'<span class="' + obj.css.concat(classes).join(' ') + '" obj="' + objId + '">' + udidstr + ' ' + obj.name + '</span>' +
					((options['view'] == 'tile' && obj.type == 'camera') ? '<img width="240" height="180" class="drag" src="' + this.getSnapshotHref(objId) + '">' : '') +
				'</li>';
		}

		return html;
	},


	/**
	 * Get reference to camera snapshot
	 * @param objId objid of camera
	 * @param [time = current timestamp] timestamp of snapshot in seconds
	 */
	getSnapshotHref: function(objId, time) {
		if (!this.objects[objId] || this.objects[objId].type != 'camera')
			return null;

		if (time)
			return "/storage/snapshot?objid=" + objId + '&ts=' + time + "&downscale" + '&status' + '&_=' + Math.random();

        return "/storage/snapshot?objid=" + objId + "&downscale" + "&status" + '&_=' + Math.random();
	},


	/**
	 * get HTML representation for device settings
	 * @param objId  object id
	 */
	getSettingsHtml: function(objId) {

		var obj = this.getObject(objId);
		if (!obj)
			throw new Error("Incorrect objId = " + objId);

		var html = '';
		var title, style;

		switch (obj.type) {
			case 'role': title = __("Role Settings"); break;
			case 'set':  title = obj.name; break;
			default: title = __("Device Settings");
		}

		for (var key in this.deviceSettings) {
			// Use dummy 'no' options for role, set and avatar device, we will set them correctly in repaintSettings
			style = (obj.type == 'role' || obj.type == 'set' || obj.type == 'avatar' || (obj.type == 'device' && obj.attributes.DEVICETYPE == 'AVATAR')) ? 'no' : obj.settings[key] ? 'yes' : 'no';
			html += '<li><span class="inline option ' + style + '" option_id="' + key + '" obj="' + objId + '">' + this.deviceSettings[key]['name'] + '</span></li>';
		}

		if (html) {
			html =
				'<li class="collapsed">' +
					'<div class="hitarea"></div>' +
					'<span class="inline settings">' + title + '</span>' +
					'<ul style="display: none;">' + html + '</ul>' +
				'</li>'
		}

		return html;
	},


	/**
	 * get HTML for associated devices list
	 * @param objId
	 */
	getAssociatedDevicesHtml: function(objId) {
		var obj = this.getObject(objId);
		if (!obj)
			throw new Error("Incorrect objId = " + objId);

		var assoc = obj.attr('ASSOCIATE');
		if (!assoc || !assoc.length)
			return '';

		// build associated devices list
		var html = this.getListHtml(assoc.split(/,\s*/), {'editable': false, 'view': 'list'});

		if (html) {
			html =
				'<li class="collapsed"><div class="hitarea" /><span class="inline assoc">' + __("Associated devices") + '</span>' +
					'<ul style="display: none">' + html + '</ul>' +
				'</li>';
		}
		return html;
	},


	/**
	 * Add and delegate all event handlers for branches,
	 * drag/drop objects and management elements
	 */
	addEventHandlers: function() {

		// Header buttons
		$('#res_btn_settings').click(function() {$('#resource_options, #resource_tree_wrap').toggle(); $(this).toggleClass("panel_expanded panel_collapsed"); resourceTree.repaintSettings(); });
		$('#res_btn_system_state').click(function() {mx.STATE.checkSystemState();mx.STATE.showSystemState();});
		$('#res_btn_refresh').click(function() {
			// remember opened set and device
			resourceTree.openSet = resourceTree.getExpandedSetId();
			resourceTree.openDevice = resourceTree.getSelectedId();
			resourceTree.refresh(null, true);
			mx.TIME.updateTimezones();
			mx.ELog2.applyFilters();
			mx.TIME.changeTimezone();
		});
		$('#res_btn_session_manager').click(function() {
			window.open("/sdi/admin/sessionmanager.php", "", "width=800,height=480,toolbar=no,menubar=no,location=no,scrollbars=yes,resizable=yes");
		});
		$('#res_btn_finder').click(function() {
			$('#resource_search_panel').css('display') == 'block'
			? resourceTree.finder.hide()
			: resourceTree.finder.show() });

		// Settings controls
		$('#res_view_mode').change(function() {resourceTree.viewMode = $(this).val(); resourceTree.paint()});
		$('#res_sort_type').change(function() {resourceTree.sort({sortType: this.value});});
		$('#res_sort_order').click(function() { $(this).toggleClass('asc desc'); resourceTree.sort({sortOrder: $(this).hasClass('asc') ? 'asc': 'desc'});});
		$('#res_btn_role').click(function() {resourceTree.setRole( $('#res_role').val());});

		// Finder controls
		$('#res_btn_finder_close').click(resourceTree.finder.hide);

		// Webkit browsers fix - cannot gain focus on input due to strange Tibco behaviour
		$('#res_finder_input').click(function(ev) {ev.stopPropagation();});
		$('#res_finder_input').keyup(function(ev) {resourceTree.finder.search($(this).val(), ev);});
		$('#res_btn_finder_next').click(function() {resourceTree.finder.search($('#res_finder_input').val(), {keyCode: 40}); });
		$('#res_btn_finder_prev').click(function() {resourceTree.finder.search($('#res_finder_input').val(), {keyCode: 38}); });

		// Resizing function, must be changed to $.resize later
		jsx3.GO('resource_wrapper').subscribe(jsx3.gui.Interactive.AFTER_RESIZE_VIEW, this, "onAfterResizeTree");

		// Branches and devices click handlers
		$('#resource_tree_body').on("click", "div.hitarea, div.hitarea + span", function() {resourceTree.toggleBranch(this);});
		$('#resource_tree').on("click", "span.editable", function() {resourceTree.showDeviceInfo(this); });
		$('#resource_tree_body').on("click", "span.option", resourceTree.changeOption);
		$('#resource_tree').click(function() {resourceTree.clearTrash();});

		// Add drag/drop handlers
		$('#resource_tree')
			.on("dragstart", "span.drag, img.drag", function() {
				dragged = true;
				var span = $(this);
				if (this.nodeName.toLowerCase() == 'img') {
					if ($(this).prevAll('span.drag').length) {
						span = $(this).prevAll('span.drag');
					} else {
						span = $(this).parent().prev();
					}
				}

				return span.clone()
					.css({'opacity': '.75', 'position': 'absolute', 'z-index': 6000, 'border': 'none', 'zoom':1})
					.removeClass('selected')
					.appendTo( $('body') );
				//.appendTo( $('#resource_tree') );
			})
			.on("drag", "span.drag, img.drag", function(ev, dd) {
				ev.stopPropagation();

				$( dd.proxy ).css({
					top: ev.pageY + 10,
					left: ev.pageX + 10
				});

				resourceTree.dragobjid = $(dd.proxy).attr('obj');
			})
			.on("dragend", "span.drag, img.drag", function( ev, dd ) {

				// Get all .drop visible divs: lower part and currently active tab
				var lowerBlocks = $('#' + jsx3.GO('pane_elogsmall').getId())
					.find('.drop').addBack();

				var block = $('#' + mx.CT.getActiveTab().getFirstChild().getId())
					.find('.drop').addBack()
					.add(lowerBlocks)
					.filter(':not(:has(div.drop))')
					.filter(':not(:has(span.drop))')
					.filter(function() {
						return contains(this, [ev.pageX, ev.pageY])
					});

				var obj = resourceTree.getObject( resourceTree.dragobjid );
				$( dd.proxy ).remove();
				resourceTree.dragobjid = null;
				dragged = false;

				if (!block.length)
					return;
				block = block[0];

				try{
					// Dirty hack to provide dnd functionality in vMX
					// Must be rewritten later!
					if (window.dndHandlers && dndHandlers[block.id] && mx.vMX.getCurWall()) {
						dndHandlers[block.id]['handler'].apply(dndHandlers[block.id]['obj'], [obj, block]);
						return;
					}

					if (mx.CT.isActiveTabVMX())
						return;

					// if jQuery drop handler exists, use it
					if ($._data(block, 'events') && $._data(block, 'events').drop) {
						$(block).trigger('drop', obj);
					}
					else {
						mx.MATRIX2.handleOnDrop(obj, matrix2.getJSXById(block.id), obj.obj)
					}
				} catch (e) {
					appLogger.info("Error in drop handler: " + e);
				}
			})
            .on("click", "span.role", function() {
                mx.Dialog.createLayoutDialog(
                    'passwd_change_dialog',
                    __("CHANGE PASSWORD"),
                    {'top': '60px', 'width':'312px', 'height':'auto'},
                    '<div class="editor"></div>'
                );
                var passwordEditor = new PasswordEditor();
                passwordEditor.init({
                    selector: "#passwd_change_dialog .editor"
                });
            })
			.on("click", "span.view", function() {
				/* disable view opening due to #5182
				var obj = resourceTree.getObject($(this).attr('obj'));
				var tab = mx.CT.createNewTab(obj.name);
				VIEWS.load(obj.obj, tab);
				*/
			});

			$("div .jsx30tabbedpane_controlbox").on("mouseup",function(event){
				if(resourceTree.dragobjid != null){
					var obj = resourceTree.getObject(resourceTree.dragobjid);
					if(obj.type == "view"){
						var tab = mx.CT.createNewTab(obj.name);
						VIEWS.load(obj.obj, tab);
					}
				}
			});
	},

	clearTrash: function() {
		$('body > span.drag').remove();
	},


	/**
	 * Sort resource tree devices in specified order
	 * @param _options  Sort options
	 *   sortType: 'name|id'
	 *   sortOrder: 'asc|desc'
	 */
	sort: function(_options) {

		var options =  {sortType: this.sortType, sortOrder: this.sortOrder};
		$.extend(options, _options);

		this.sortType = options.sortType;
		this.sortOrder = options.sortOrder;

		for (var setId in this.sets) if (this.sets.hasOwnProperty(setId)) {
			this.sets[setId].objects.sort(this._sort)
		}

		this.paint();

	},


	/**
	 * Sorting function, uses natural sorting
	 * takes into account current sort type and order
	 * @param a
	 * @param b
	 */
	_sort: function(a, b) {
		var result;

		if (resourceTree.sortType == 'name')
			result = naturalSort(resourceTree.objects[a].name, resourceTree.objects[b].name);
		else
			result = naturalSort(resourceTree.objects[a].obj.toString(), resourceTree.objects[b].obj.toString());

		if (resourceTree.sortOrder == 'asc')
			return result;
		else
			return -result;
	},


	/**
	 * Get id of currently expanded set
	 */
	getExpandedSetId: function() {
		var nodes = $('#resource_tree li.expanded > span.set');
		return nodes.length ? nodes.attr("obj") : null;
	},


	/**
	 * Get id of currently selected device
	 */
	getSelectedId: function() {
		if (!this.cache.$objects)
			return null;
		return this.cache.$objects.filter('.selected').attr("obj");
	},


	/**
	 * Update view of expanded set depending on camera health
	 * @param cameras array of cameras in format obj_id => {health, status}
	 */
	updateHealth: function(cameras) {
		for (var i in cameras) if (this.objects[i]) {
			var span = this.cache.$objects.filter('[obj=' + i + ']');
			if (!span.length)
				continue;

			this.objects[i].health = cameras[i].health;
			this.objects[i].attributes.STATUS = cameras[i].status;

			// Remove old and add new camera classes
			span.removeClass(this.objects[i].css.join(' '));
			$.extend(this.objects[i], this.getDeviceType(i));
			span.addClass(this.objects[i].css.join(' '));
		}
	},
	
	/**
	 * Update view of avatar health
	 * @param obj integer avatar obj_id
	 * @param status string avatar status
	 */
	updateAvatarHealth: function(obj, status) {
		var span = this.cache.$sets.filter('[obj=' + obj + ']');
		
		if(status == "ONLINE" && span.hasClass("set_off")){
			span.removeClass("set_off");
		}
		else if(status == "OFFLINE" && !span.hasClass("set_off")){
			span.addClass("set_off");
		}
	},

	/**
	* Update statistic data of expanded set
	* @param cameras array of cameras in format obj_id => {stat_metadata_receiving}
	*/
	updateStats: function(cameras) {
		for (var i in cameras) if (this.objects[i]) {
			var span = this.cache.$objects.filter('[obj=' + i + ']');
			if (!span.length)
				continue;

			this.objects[i].attributes.STAT_METADATA_RECEIVING = cameras[i].stat_metadata_receiving;
		}
	},

	/**
	 * Check credential for object
	 * @param objId  Object id
	 * @param cred   credential letter
	 * @return bool  true if object have credential
	 */
	cred: function(objId, cred) {
		if (!this.objects[objId])
			return null;
		return this.objects[objId].credentials.search(cred) != -1;
	},


	finder: {
		// jQuery collection of objects matching search query
		found: null,
		// index of currently selected object in collection
		foundId: null,

		/**
	     * Show search panel and focus on it
	     */
		show: function() {
			$('#resource_search_panel').show();
			//$('#resource_search_panel').slideDown(this.animationSpeed);
			$('#res_finder_input').focus();
		},


		/**
	     * Hide search panel
	     */
		hide: function() {
			$('#resource_search_panel').slideUp(this.animationSpeed);
		},

		/**
		 * Quick search algorithm:
	     * @param text     search text
	     * @param ev       keyboard event
	     */
		search: function(text, ev) {

			// Search text was changed
			$('#res_finder_input').removeClass('not-found');

			if (!text.length) {
				this.found = null;
				this.foundId = null;
				return;
			}

			// get new array of filtered spans
			text = text.toLowerCase();
			this.found = resourceTree.cache.$objects.filter(function() {
				var objId = $(this).attr('obj');
				return resourceTree.objects[objId].name.toLowerCase().search(text) >= 0 || objId.toLowerCase().search(text) >= 0;
			});

			if (this.found.length) {
				// Begin from first element or find currently selected element in found
				this.foundId = 0;
				this.found.each(function(ind) {
					if ($(this).hasClass('selected')) resourceTree.finder.foundId = ind;
				});
				resourceTree.showDevice(this.found[this.foundId]);
			} else {
				this.found = null;
				this.foundId = null;
				$('#res_finder_input').addClass('not-found');
			}

			// Check for special commands
			switch (ev.keyCode) {
				case 27: // esc
					this.hide();
					break;
				case 13: // enter
				case 40: // down
					if (this.found.length) {
						this.foundId = (this.foundId + 1 < this.found.length) ? this.foundId + 1 : 0;
						resourceTree.showDevice(this.found[this.foundId]);
					}
					break;
				case 38: // up
					if (this.found.length) {
						this.foundId = (this.foundId == 0) ? this.found.length - 1 : this.foundId - 1;
						resourceTree.showDevice(this.found[this.foundId]);
					}
					break;
			}
		}
	},


	/**
	 * Show device in tree: open all nesessary branches,
	 * scroll tree to right position and show device info
	 * @param span
	 */
	showDevice: function(span) {
		span = $(span);

		$.fx.off = true;

		if (!span.hasClass("selected")) {
			resourceTree.showDeviceInfo(span);
		}

		// expand all branches containing this device
		resourceTree.toggleBranch(span.parents(".collapsed"));

		// Scroll tree to selected element
		var tree = $('#resource_tree_body');
		var scroll = span.offset().top + tree[0].scrollTop - tree.offset().top - 50;
		tree.scrollTop(scroll <= 0 ? 0 : scroll);

		$.fx.off = false;
	},


	/**
	 * Toggle branch open or close
	 * @param branches  set id | jQuery array of li elements
	 */
	toggleBranch: function(branches) {

		// if set id given, find this set
		if (/^\d+$/.test(branches)) {
			branches = resourceTree.cache.$sets.filter('[obj=' + branches + ']');
		}

		branches = $(branches);
		if (!branches.hasClass('expanded') && !branches.hasClass('collapsed')) {
			branches = branches.parent();
		}

		if (!branches.length)
			return;

		// Hide branches that were expanded
		branches.filter('.expanded').each(function() { $(this).find(' > ul').slideUp(resourceTree.animationSpeed) });
		// Hide siblings of branches that will be expanded
		branches.filter('.collapsed').siblings().filter('.expanded').toggleClass('expanded collapsed').find('> ul').slideUp(resourceTree.animationSpeed);
		// Expand branches
		branches.filter('.collapsed').find(' > ul').slideDown(this.animationSpeed);
		branches.toggleClass('expanded collapsed');
	},


	/**
	 * Show device info div
	 * @param node  object id | DOM element
	 */
	showDeviceInfo: function(node) {

		// if device id given, find this device
		if (/^\d+$/.test(node)) {
			if (!this.objects[node])
				return;
			node = this.cache.$objects.filter('[obj=' + node + ']');
		}

		node = $(node);

		// If click on already opened device, close info box
		if (node.hasClass("selected")) {
			this.hideDeviceInfo();
			return;
		}

		// Close other info boxes
		this.hideDeviceInfo();

		// Show info box
		var obj = node.attr("obj");
		node.addClass('selected');
		node.after(
			'<div class="edit loadingIndication" style="display: none;" obj="' + obj + '">' +
				((this.viewMode == 'list' && this.objects[obj].type == 'camera') ? '<img width="160" height="120" class="drag" src="' + this.getSnapshotHref(obj) + '" />' : '') +
				'<ul>' +
					resourceTree.getSettingsHtml(obj) +
					resourceTree.getAssociatedDevicesHtml(obj) +
				'</ul>' +
			'</div>'
		);
		node.next("div.edit").slideDown(resourceTree.animationSpeed);
	},


	/**
	 * Hide device info
	 */
	hideDeviceInfo: function() {
		var node = resourceTree.cache.$objects.filter('.selected');
		$(node).removeClass("selected").next("div.edit").remove();
	},


	/**
	 * get type of object, to provide correct style or device icon
	 * @param obj   objid || plain js object
	 * @return object
	 *     css:  css for span element
	 *     type: system object type, camera|audio|door etc.
	 *     image: icon for object
	 */
	getDeviceType: function(obj) {

		var device;
		if ($.isPlainObject(obj)) {
			device = obj;
		} else {
			if (!this.objects[obj])
				throw new Error("incorrect objId: " + obj);
			device = this.objects[obj];
		}

		var deviceTypes = {
			'C': 'camera',
			'R': 'relay',
			'S': 'sensor',
			'A': 'audio',
			'D': 'door',
			'E': 'device',
			'V': 'monitor',
			'W': 'wall'
		};

		var css = ['device'];
		var type = 'device';

		if (device.otype == 'D') {
			css[0] = deviceTypes[device.subtype];
			type = deviceTypes[device.subtype];

			if (css[0] == 'camera') {
				if(device.attributes.CAMERAMODEL == "iStream") {
					css[0] = 'iphone';
				} else if (device.attributes.POSITIONCTL !== 'none') {
					css[0] = 'camera_ptz';
				}
			}

		} else if (device.otype == 'X'){
			if (device.subtype == 'R' || device.subtype == 'S' || device.subtype == 'D') {
				css[0] = deviceTypes[device.subtype];
				type = deviceTypes[device.subtype];
			}
		}

		if (device.attributes && device.attributes.STATUS == 'OFF') {
			css.push(css[0] + '_stopped');
		} else if (device.health && device.health != 2) {
			css.push(css[0] + '_off');
		}

		return {'css': css, 'type': type, 'image': css[0] + '.png'};
	},


	/**
	 * Return object by its id or search it with filter criteria
	 * If object is an device, it returned with two auxiliary functions:
	 * cred() - check object for credential
	 * attr() - return object attribute
	 * @param objId
	 * @return Object
	 */
	getObject: function(objId) {

		if (typeof objId == 'object')
			return this.getObjects(objId, true);

		if (!/^\d+$/.test(objId))
			return null;

		var obj = this.objects[objId] || this.sets[objId] || this.roles[objId] || VIEWS.list[objId] || null;

		if (!obj)
			return null;

		// get role object with properties
		if (obj.type == 'role' && this.role.obj == objId) {
			obj = this.role;
		}

		if (obj) {
			// Clone object
			obj = $.extend({}, obj);
		}

		if (this.objects[objId]) {
			obj.cred = function(cred) {
				return this.credentials.search(cred) != -1;
			};

			obj.attr = function(attr) {
				return this.attributes[attr.toUpperCase()] ? this.attributes[attr.toUpperCase()] : null;
			};
		}

		return obj;
	},


	/**
	 * Get objects by filter criteria
	 * @param options object  filter criteria
	 * keys:
	 *     obj: list of objids to limit filter
	 *     type: object type (camera|audio|door etc.)
	 *     cred: credential
	 *     <setting_name>: one of settings (resourceTree.deviceSettings)
	 *     <attribute_name>: one of attributes
	 * @param [findFirst] bool find first matched element and return it with getObject
	 * @return Array    array of ids | object, if findFirst == true
	 */
	getObjects: function(options, findFirst) {

		if (!options)
			return null;

		var result = [];
		var ids;

		if (options['obj']) {
			// Limit objects to set of ids
			ids = options['obj'];
			delete options['obj'];
		} else {
			// or search whole bunch of objects
			ids = Object.keys(this.objects);
		}

		for (var i = ids.length; i--;) {

			// Don't process incorrect ids
			if (!this.objects[ids[i]])
				continue;

			if (this._find(options, ids[i])) {

				if (findFirst)
					return this.getObject(ids[i]);

				result.push(ids[i]);
			}
		}

		return result;
	},


	/**
	 * Auxiliary function to getObjects, check if object conforms filter criteria
	 * @param options  filter criteria
	 * @param objId
	 * @return bool
	 */
	_find: function(options, objId) {

		for (var opt in options) {
			// Check credentials
			if (opt == 'cred') {
				// Check each credential in string
				for (var c = 0; c < options['cred'].length; c++) {
					if (this.objects[objId].credentials.search(options['cred'][c]) == -1) {
						return false;
					}
				}
				continue;
			}
			if (opt == 'type') {
				if (this.objects[objId].type != options['type']) {
					return false;
				}
				continue;
			}

			//Check settings
			if (this.deviceSettings[opt]) {
				if (!this.objects[objId].settings)
					return false;

				if (Boolean(this.objects[objId].settings[opt]) != Boolean(options[opt]))
					return false;
				continue;
			}

			if (!this.objects[objId].attributes || this.objects[objId].attributes[opt.toUpperCase()] != options[opt]) {
				return false;
			}
		}
		return true;
	},


	/**
	 * Return current role object
	 */
	getRole: function() {
		return $.extend({}, this.role);
	},


	getUser: function() {
		return $.extend({}, this.user);
	},


	/**
	 * Resource tree loading indicator
	 */
	waitIndicator: function(toShow) {
		if (toShow) {
			$('#resource_tree_title .wait').css('display', 'inline-block');
		} else {
			$('#resource_tree_title .wait').hide();
		}
	},


	/**
	 * Get difference between device settings and default values
	 */
	getSettingsDiff: function(objId) {

		var obj = this.objects[objId];

		if (!obj && !obj.settings)
			return {};

		var diff = {};

		for (var i in this.defaultSettings) {
			if (obj.settings[i] != this.defaultSettings[i]) {
				diff[i] = obj.settings[i];
			}
		}
		return $.isEmptyObject(diff) ? null : diff;
	},


	/**
	 * Serialize tree settings into brief object representation
	 */
	serialize: function() {

		appLogger.time('resourceTree.serialize');

		var result = {
			role: this.role.obj,
			sortType:  this.sortType,
			sortOrder: this.sortOrder,
			viewMode:  this.viewMode,
			openSet: this.getExpandedSetId(),
			openDevice: this.getSelectedId(),
			objects: {}
		};

		$.each(this.role.objects, function(i, objId) {
			var diff = resourceTree.getSettingsDiff(objId);
			if (diff) {
				result.objects[objId] = diff;
			}
		});

		appLogger.timeEnd('resourceTree.serialize');

		return result;
	},


	/**
	 * Load tree settings from serialized data
	 * @param data
	 */
	unserialize: function(data) {

		appLogger.time('resourceTree.unserialize');

		this.objects = {};

		if (data.objects) {
			// Load settings into RT objects
			for (var objId in data.objects) {
				this.objects[objId] = {settings: $.extend({}, this.defaultSettings, data.objects[objId])};
			}
		}

		this.sortType = data.sortType;
		this.sortOrder = data.sortOrder;
		this.viewMode = data.viewMode;
		this.openSet = data.openSet;
		this.openDevice = data.openDevice;

		this.refresh(data.role, false);

		// Set correct settings in 'settings' pane
		$('#res_sort_type').val(this.sortType);
		$('#res_sort_order')[0].className = 'button ' + this.sortOrder;
		$('#res_view_mode').val(this.viewMode);

		appLogger.timeEnd('resourceTree.unserialize');
	}
};
