/**
 * Object Manager class
 * Implements Object Manager cell functionality and communication with
 * OM Java applet.
 */

jsx3.lang.Package.definePackage("mx.OM", function(OM) {

	/*----------------------------------------------------------------------
	 *              OM  Package Vars
	 *--------------------------------------------------------------------*/

	OM.clientApplet = null; // reference to the OM applet
	OM.timeoutId = 0; // Id returned by window.setTimeout
	OM.intervalDuration = 15000; // Interval between cameras health checks

	OM._OBJREC_CACHE = {}; // Mapping 'objid' => 'object record list'

	OM.IGNOREPROPS = new jsx3.util.List(// Ignore for 'X' objects
		[
			'NAME', /*'TYPE', 'SUBTYPE',*/ 'NODE_ID', 'LOCATION',
			'PROTECTED','DELETED', 'DESC', 'ASSOCIATE', 'EVENT_POLICY',
			'TIME_ZONE', 'EXT_ID', 'ENABLED'
		]
	);

	OM.lastUpdate = 0; // Timestamp of last call to 'getChanges'
	OM.nextUpdate = 0; // Timestamp of next planned call to 'getChanges'
	OM.updateInterval = 1000; // Min value of interval between two sequental calls to 'getChanges'
	OM.minHighlightedInterval = 15000;  // Minimum time record should stay in highlighted state


	/*----------------------------------------------------------------------
	 *              OM  Package Methods
	 *--------------------------------------------------------------------*/

	OM._getMethodsDoc = function(objid) {
		return mx.MATRIX2.getServer().getCache().getDocument("OBJ_" + objid + "_METHODS");
	};

	OM._getPropertiesDoc = function(objid) {
		return mx.MATRIX2.getServer().getCache().getDocument("OBJ_" + objid + "_PROPERTIES");
	};


	/**
	 * Tries to initialize OM client applet
	 */
	OM.initAppletOK = function() {
		OM.clientApplet = window.ELog2eLogNoter;

		OM.clientApplet.subscribe("objectchange", function(status){
			OM.onObjectChange(status);
		}, "eventLog");
	};


	/**
	 * creates new OM cell in the specified block. If this block was occupied
	 * with the cell of differednt type, it will be destroyed
	 */
	OM.createOMCell = function(blkDest, objRecord) {
		// First, clear destination block
		mx.MATRIX2.clearDestinationBlock(blkDest);
		blkDest.removeAttribute("playerBlock");

		// Create cell view
		//var blkCell = blkDest.loadAndCache("components/om_cell.xml");
		var blkCell = mx.language.load_translate(blkDest, "components/om_cell.xml");

		OM.appendObject(blkCell, objRecord);
		blkDest.repaint();
	};

	/**
	 * add device to the existing OM cell. If device already present, do nothing
	 * @param blkDest  OM cell
	 * @param objRecord   device id | device object
	 */
	OM.appendObject = function(blkDest, objRecord) {

		if (/^\d+$/.test(objRecord)) {
			objRecord = resourceTree.getObject(objRecord);
		}
		if (!objRecord)
			return;

		if (objRecord.type == 'set') {
			objRecord = objRecord.objects;
		}
		if ($.isArray(objRecord)) {
			$.each(objRecord, function(i, v) { mx.OM.appendObject(blkDest, v) });
			return;
		}

		// Check if object already in the list
		var rec = blkDest.getDescendantOfName("objrec");

		while (rec) {
			var next = rec.getNextSibling();
			var objid = rec.getAttribute("objid");
			if (!objid) {
				rec = next;
				continue;
			}
			if (objid == objRecord.obj)
				return;
			rec = next;
		}


		// Retrieve object parameters and subscribe to notifications
		if (OM._getMethodsDoc(objRecord.obj) && OM._getPropertiesDoc(objRecord.obj)) {
			appLogger.info("Object " + objRecord.obj + " already present in cache");
		}
		else {
			appLogger.info("No data for object " + objRecord.obj + " found in cache. Must do OM call");
			var result = OM.addMonitoredObject(objRecord.obj);
			if (!result) {
				alert(__("Cannot add object to the list!"));
				return;
			}
		}

		// Paint record
		var blkTemplate = blkDest.getDescendantOfName("tmpl_rec");
		var blkDev = blkTemplate.doClone(jsx3.app.Model.PERSISTEMBED, 0);
		var srcImage = OM.getStateImage(objRecord.obj);
		blkTemplate.getParent().insertBefore(blkDev, blkTemplate);
		blkDev.setName("objrec");
		blkDev.setAttribute("objid", objRecord.obj);
		blkDev.getDescendantOfName("objid").setText(objRecord.obj);
		blkDev.getDescendantOfName("text").setText(objRecord.description || objRecord.name);
		blkDev.getDescendantOfName("location").setText(objRecord.location);
		blkDev.getDescendantOfName("type").setText(objRecord.type);
		if (srcImage) {
			blkDev.getDescendantOfName("img").setSrc(srcImage);
		} else {
			blkDev.getDescendantOfName("img").setDisplay(jsx3.gui.Block.DISPLAYNONE);
		}
		blkDev.setDisplay(jsx3.gui.Block.DISPLAYBLOCK);
		blkDest.repaint();
		// check if vertical scrollbar appeared
		OM.updateContentView(blkDest);
		// Add object record to cache
		OM.cacheObjRec(blkDev);
	};

	/**
	 * Destroys cell and performs cleanup
	 */
	OM.onCellClose = function(objTHIS) {
		var blkCell = objTHIS.getAncestorOfName("om_cell");
		var rec = blkCell.getDescendantOfName("objrec");
		while (rec) {
			var next = rec.getNextSibling();
			var objid = rec.getAttribute("objid");
			if (!objid) {
				rec = next;
				continue;
			}
			rec = next;
		}
		blkCell.getParent().setAttribute("playerBlock", "0");
		blkCell.getParent().removeChildren();
	};


	/**
	 * Fires when user drops device on OM cell pane
	 */
	OM.handleOnDrop = function(blkDest, objSource, strId) {
		if (!OM.clientApplet) {
			alert(__("Object Manager not initialized"));
			return;
		}
		// Get device record from objSource
		var objRecord = null;
		if (objSource.type) { // Already a device record from resource tree
			objRecord = objSource;
		}
		else if (objSource.getName) {
			switch (objSource.getName()) {
				case 'matrixWitnesses':
				case 'screenshot':
					var eLog2 = jsx3.GO("eLog2Root");
					var dialog = eLog2.getDescendantOfName('eLog2dialog');
					var objIdMain = dialog.getAttribute("objId");
					if (objSource.getName() == 'screenshot') {
						objRecord = resourceTree.getObject(objIdMain);
					}
					else {
						var witnessesDoc = dialog.getDescendantOfName('matrixWitnesses').getXML();
						var objId = witnessesDoc.selectSingleNode("//record[@jsxid='" + strId + "']").getAttribute("objid");
						objRecord = resourceTree.getObject(objId);
					}
					break;
			}
		}
		if (objRecord) OM.appendObject(blkDest, objRecord);
		else if (window.jsDebug) appLogger.info("OM.handelOnDrop: failed to get objRecord for "+objSource);
	};


	/**
	 * Removes object record from cell list
	 */
	OM.onRemoveRecord = function(btnRemove) {
		var blkRecord = btnRemove.getAncestorOfName("objrec");
		var blkCell = btnRemove.getAncestorOfName("om_cell");
		var blkExpanded = blkCell.getDescendantOfName("rec_expanded");
		var objid = blkRecord.getAttribute("objid");

		if (blkExpanded.getAttribute("objid") == objid) {
			// Must collapse expanded block
			blkExpanded.removeAttribute("objid");
			blkExpanded.setDisplay(jsx3.gui.Block.DISPLAYNONE, true);
		}
		btnRemove.getAncestorOfName("object_list").removeChild(blkRecord);
		OM.updateContentView(blkCell);
	};

	/**
	 * Repaints method buttons for an expanded record
	 */
	OM._repaintRecMethods = function(blkExpanded, cdfMethods, bRepaint) {

		var objid = blkExpanded.getAttribute("objid");
		if (!cdfMethods) {
			cdfMethods = OM._getMethodsDoc(objid);
		}
		// Perform credentials check and paint buttons for object methods
		var bDisableButtons = !resourceTree.cred(objid, 'C');
		var lstRec = cdfMethods ? cdfMethods.selectNodes("//record") : new jsx3.util.List;
		var blkMethods = blkExpanded.getDescendantOfName("ObjectMethods");
		blkMethods.removeChildren();
		if (!lstRec.size()) {
			blkExpanded.getDescendantOfName("main_layout").setCols("100%,0", true);
		} else {
			blkExpanded.getDescendantOfName("main_layout").setCols("50%,50%", true);
		}
		for (var i = 0; i < lstRec.size(); i++) {
			var btnCall = new jsx3.gui.Button("callbtn");
			btnCall.setHeight(25);
			btnCall.setWidth("100%").setRelativePosition(jsx3.gui.Block.RELATIVE);
			btnCall.setEnabled(bDisableButtons ? jsx3.gui.Form.STATEDISABLED : jsx3.gui.Form.STATEENABLED);
			btnCall.setText(lstRec.get(i).getAttribute("displayname"));
			btnCall.setTip(lstRec.get(i).getAttribute("desc"));
			btnCall.setEvent("mx.OM.onMethodCall(this," + objid + ",'" + lstRec.get(i).getAttribute("name") + "')",
				jsx3.gui.Interactive.EXECUTE);
			blkMethods.setChild(btnCall);
		}

		if (bRepaint) {
			blkMethods.repaint();
		}
	};


	/**
	 * Expands/collapses extented object information
	 */
	OM.handleOnClick = function(blkRecord) {
		var blkCell = blkRecord.getAncestorOfName("om_cell");
		var blkExpanded = blkCell.getDescendantOfName("rec_expanded");
		var objid = blkRecord.getAttribute("objid");

		if (blkExpanded.getAttribute("objid") == objid) {
			// Already attached to current record. So we only have to
			// switch its visibility
			if (blkExpanded.getDisplay() == jsx3.gui.Block.DISPLAYNONE) {
				blkExpanded.getDescendantOfName("ObjectProperties").repaintData();
				OM._repaintRecMethods(blkExpanded, null, true);
				blkExpanded.setDisplay(jsx3.gui.Block.DISPLAYBLOCK, true);
			}
			else {
				blkExpanded.setDisplay(jsx3.gui.Block.DISPLAYNONE, true);
			}
		}
		else {
			// Expanded block is attached to another record. We need to move
			// it to a new location
			blkExpanded.setDisplay(jsx3.gui.Block.DISPLAYBLOCK);
			blkExpanded.setAttribute("objid", objid);
			blkExpanded.getDescendantOfName("ObjectProperties").
				setXMLId("OBJ_" + objid + "_PROPERTIES");

			// If object has no properties and methods, must shrink its block height
			var cdfProperties = OM._getPropertiesDoc(objid);
			var cdfMethods = OM._getMethodsDoc(objid);

			if ((cdfProperties == null && cdfMethods == null) ||
				(cdfProperties.selectNodes("//record").size() == 0 &&
					cdfMethods.selectNodes("//record").size() == 0)) {
				blkExpanded.setHeight(40);
				blkExpanded.getChild("data").setDisplay(jsx3.gui.Block.DISPLAYNONE);
				blkExpanded.getChild("message").setDisplay(jsx3.gui.Block.DISPLAYBLOCK);
			} else {
				var nRecs = cdfProperties ? cdfProperties.selectNodes("//record").size() : 0;
				var need_height = blkExpanded.getDescendantOfName("label").getHeight() +
					jsx3.gui.Matrix.DEFAULT_HEADER_HEIGHT +
					nRecs * jsx3.gui.Matrix.DEFAULT_ROW_HEIGHT + 2;
				blkExpanded.setHeight(Math.min(250, need_height));
				blkExpanded.getChild("data").setDisplay(jsx3.gui.Block.DISPLAYBLOCK);
				blkExpanded.getChild("message").setDisplay(jsx3.gui.Block.DISPLAYNONE);
			}

			OM._repaintRecMethods(blkExpanded, cdfMethods, false);
			// Update view
			if (blkRecord.getNextSibling()) {
				blkExpanded.getParent().insertBefore(blkExpanded, blkRecord.getNextSibling(), true);
			}
			else {
				blkExpanded.getParent().insertBefore(blkExpanded, blkCell.getDescendantOfName("tmpl_rec"), true);
			}
		}

		OM.updateContentView(blkCell);
	};


	mx.OM.onMethodCall = function(btnCall, objid, name) {
		var cdfDoc = mx.MATRIX2.getServer().getCache().getDocument("OBJ_" + objid + "_METHODS");
		if (!cdfDoc) return;
		var objRecord = cdfDoc.selectSingleNode("//record[@name='" + name + "']");
		if (!objRecord) return;
		// paint wait status
		var statusbar = btnCall.getParent().getPreviousSibling();
		statusbar.getFirstChild().setText("Status", true);
		statusbar.getLastChild().setText("WAIT").setColor("Chocolate").repaint();
		// do applet call
		var status = mx.OM.invokeObjectMethod(objid, name);
		// update view
		var clr = status == "OK" ? "green" : "red";
		statusbar.getLastChild().setText(status).setColor(clr).repaint();
	};


	/**
	 * invokes applet method "addMonitoredObject" which returns properties and
	 * methods of the target object as a JSON string. Result data is parsed and
	 * stored in cache for further use by any OM cell
	 */
	OM.addMonitoredObject = function(objid) {
		if (!OM.clientApplet) {
			appLogger.warn("addMonitoredObject() called but applet still not intialized");
			return false;
		}
		try {
			var strJSON = OM.clientApplet.addMonitoredObject(objid);
		}
		catch(e) {
			appLogger.error("addMonitoredObject() call failed: " + e.description);
			return false;
		}

		var objResponse = $.parseJSON(String(strJSON));
		if (objResponse && objResponse.status != "OK") {
			appLogger.warn("addMonitoredObject() returned status " +
				objResponse.status + ": " + objResponse.message);
			return false;
		}
		var cdfPropertiesDoc = jsx3.xml.CDF.Document.newDocument();
		var cdfMethodsDoc = jsx3.xml.CDF.Document.newDocument();
		var contentObj = objResponse.object;

		if (contentObj && contentObj.properties && contentObj.methods) {
			var lstProperties = new jsx3.util.List(contentObj.properties);
			var lstMethods = new jsx3.util.List(contentObj.methods);
			try {
				var otype = lstProperties.filter(function(e) {
					return e.name == "TYPE"
				}).get(0).value;
			} catch(e) {
				appLogger.warn("object " + objid + " has no 'TYPE' property!")
			}
			for (var iter = lstProperties.iterator(); iter.hasNext();) {
				var prop = iter.next();
				if (otype == 'X' && OM.IGNOREPROPS.contains(prop.name)) continue;
				if (otype == 'D' && prop.name != 'health' && prop.name != 'TYPE') continue;
				cdfPropertiesDoc.insertRecord({
					jsxid : prop.name,
					name  : prop.name,
					value : prop.value
				});
			}
			for (var iter = lstMethods.iterator(); iter.hasNext();) {
				var method = iter.next();
				cdfMethodsDoc.insertRecord({
					jsxid       : method.name,
					name        : method.name,
					desc        : method.desc,
					displayname : method.displayname
				});
			}
		} else {
			appLogger.error("Bad object returned by parseJSON: " + strJSON);
		}

		mx.MATRIX2.getServer().getCache().setDocument("OBJ_" + objid + "_PROPERTIES", cdfPropertiesDoc);
		mx.MATRIX2.getServer().getCache().setDocument("OBJ_" + objid + "_METHODS", cdfMethodsDoc);
		return true;
	};


	/**
	 * Invokes object method via applet call
	 */
	OM.invokeObjectMethod = function(objid, name) {
		if (!OM.clientApplet) {
			appLogger.warn("invokeObjectMethod() called but applet still not intialized");
			return "ERROR";
		}
		try {
			var strJSON = OM.clientApplet.invokeObjectMethod(objid, name, window.jsUserId, window.jsUserName);
		}
		catch(e) {
			appLogger.error("invokeObjectMethod() call failed: " + e.description);
			return "ERROR";
		}

		var objResponse = eval('(' + strJSON + ')');
		return objResponse.status;
	};

	/**
	 * Wrapper for 'getObjects' method of the ELog applet
	 */
	OM.checkCamHealth = function() {
		if (!OM.clientApplet) {
			appLogger.warn("checkCamHealth() called but applet still not intialized");
			return;
		}

		// Filter camera list. Include only cameras from currently opened set
		var setId = resourceTree.getExpandedSetId();
		if (!setId)
			return;

		var objIds = resourceTree.getObjects({'obj': resourceTree.getObject(setId).objects, 'type': 'camera'});
		if (!objIds.length)
			return;

		var arrHealth = $.parseJSON(
			OM.clientApplet.getObjects({'objids' : objIds, 'properties' : ['health'], 'methods' : false})
		);

		if (arrHealth !== null)
			arrHealth = arrHealth.objects;
		if (!arrHealth)
			return;

		// Construct hash, which has obj ids as keys, and contains info about camera health and status
		// Retrieve camera health
		var cameras = {};
		for (var i = 0; i < arrHealth.length; i++) {
			if (arrHealth[i].objid && arrHealth[i].properties) {
				cameras[arrHealth[i].objid] = { health: arrHealth[i].properties[0] ? arrHealth[i].properties[0].value : 2 }
			}
		}

		// Retrieve camera status
		api.getObjectList({
			setid: setId,
			type: 'camera',
			withAttributes: true
		})
			.fail(function(code, message){

			})
			.done(function(response){
				var list = response.list;
				for (var i = 0; i < list.length; i++) {
					if (cameras[list[i].obj]) {
						cameras[list[i].obj].status = list[i].attributes.STATUS;
						cameras[list[i].obj].stat_metadata_receiving = list[i].attributes.STAT_METADATA_RECEIVING;
					}
				}

				// Update resource tree view
				resourceTree.updateHealth(cameras);
				resourceTree.updateStats(cameras);
			});
			
		// check avatar too
		OM.checkAvatarHealth();
	};

	/**
	 * Check avatar health
	 */
	OM.checkAvatarHealth = function() {
		$.each(resourceTree.sets, function() {
			if(this.type == 'avatar'){
				var avatarObjId = this.obj;
				api.getAvatarStatus({
					obj: avatarObjId
				})
				.fail(function(code, message){

				})
				.done(function(response){
					if(response.error == ""){
						// Update resource tree view
						resourceTree.updateAvatarHealth(avatarObjId, response.status);	
					}
				});
			}
		});
	};

	/**
	 * Implementation for camera health monitoring in real-time
	 * Periodically calls 'getObject' method of the applet and
	 * checks health state of the cameras. If camera's state
	 * is changed, its icon in Resource tree in the left of the
	 * Matrix2 layout is updated
	 */
	OM.startHealthMonitor = function() {
		OM.timeoutId = window.setTimeout(
			mx.OM._healthMonitorImpl,
			OM.intervalDuration
		);
	};

	OM.stopHealthMonitor = function() {
		window.clearTimeout(OM.timeoutId);
		OM.timeoutId = 0;
	};

	OM._healthMonitorImpl = function() {
		OM.checkCamHealth();
		OM.clearHighlightedRecs();
		//	appLogger.error("Health monitor error: " + e.description);
		if (OM.timeoutId) {
			OM.timeoutId = window.setTimeout(
				mx.OM._healthMonitorImpl,
				OM.intervalDuration
			);
		}
	};

	/**
	 * Hack for updating object list block when cell is resized or
	 * a record inside it is expanded, or new record is added to it.
	 * The only purpose of this method is to resize 'object_list' block
	 * in order to avoid painting of horizotal scrollbar. So, we need to
	 * track the moment when a vertical scrollbar appears and decrease
	 * width of the list block by the width of the scrollbar
	 */
	OM.updateContentView = function(blkCell) {
		if (!blkCell) {
			appLogger.warn("OM.updateContentView : cell is null");
			return;
		}
		var blkMainPane = blkCell.getDescendantOfName("main_pane");
		if (!blkMainPane) return;
		var pane = blkMainPane.getRendered();
		var blkList = blkMainPane.getChild("object_list");
		var scrollbarWidth = pane.scrollWidth - pane.clientWidth;
		var posW = blkList.getAbsolutePosition().W;
		if (posW == 0) {
			appLogger.warn("error in updateContentView(): OM cell hasn't been rendered yet!");
			return;
		}
		if (scrollbarWidth > 0 && posW > pane.clientWidth) {
			blkList.setWidth(pane.clientWidth, true)
		}
		else if (scrollbarWidth <= 0 && posW < pane.clientWidth) {
			blkList.setWidth("100%", true);
		}
	};


	/**
	 * Callback method, called by applet when its cache data for
	 * a monitored object got changed. Method receives single parameter
	 * reporting current cache status. Status '3' means that javascript
	 * should query applet for recent changes and process them
	 */
	OM.onObjectChange = function(status) {
		if (status == 3) { // New data available
			var curTime = new Date().getTime();
			if (curTime - OM.lastUpdate < OM.updateInterval) {
				if (OM.nextUpdate == 0) { // Schedule call
					OM.nextUpdate = OM.lastUpdate + OM.updateInterval;
					window.setTimeout(mx.OM.getChanges, OM.nextUpdate - curTime);
				}
			}
			else {
				OM.getChanges();
			}
		}
	};

	/**
	 * Wrapper for applet method. Retrieves recent changes
	 * for all monitored objects and updates view for all
	 * OM corresponding OM cells
	 */
	OM.getChanges = function() {
		if (!OM.clientApplet) {
			appLogger.warn("getChanges() called but applet still not intialized");
			return;
		}

		OM.nextUpdate = 0;
		OM.lastUpdate = new Date().getTime();

		try {
			var strJSON = OM.clientApplet.getChanges();
		}
		catch(e) {
			appLogger.error("getChanges() call failed: " + e.description);
			return;
		}

		try {
			var objResponse = JSON.parse(strJSON);
		}
		catch(e) {
		}
		if (objResponse == null || objResponse.status != "OK" || objResponse.changes == null) {
			appLogger.warn("getChanges() applet call returned inconsistent data!");
		}
		else {
			// outer: loop through all changed objects
			for (var i = 0; i < objResponse.changes.length; i++) {
				var obj = objResponse.changes[i];
				if ((!obj.properties || obj.properties.length == 0) &&
					(!obj.methods || obj.methods.length == 0)) continue;

				var bNeedUpdateProps = false;
				var bNeedUpdateMethods = false;
				var cdfPropertiesDoc = OM._getPropertiesDoc(obj.objid);
				var cdfMethodsDoc = OM._getMethodsDoc(obj.objid);
				var lstProperties = new jsx3.util.List(obj.properties);
				var lstMethods = new jsx3.util.List(obj.methods);
				var obj_otype = cdfPropertiesDoc.getRecord("TYPE");
				var otype = obj_otype ? obj_otype.value : "";

				// inner: loop through object properties
				for (var prop_iter = lstProperties.iterator(); prop_iter.hasNext();) {
					var prop = prop_iter.next();
					// Skip ignored properties according to device type
					if (otype == 'X' && OM.IGNOREPROPS.contains(prop.name)) {
						prop_iter.remove();
						continue;
					}
					if (otype == 'D' && prop.name != 'health' && prop.name != 'TYPE') {
						prop_iter.remove();
						continue;
					}

					var xmlRec = cdfPropertiesDoc.getRecordNode(prop.name);
					if (xmlRec) {
						if (xmlRec.getAttribute("value") != prop.value) bNeedUpdateProps = true;
						xmlRec.setAttribute("value", prop.value);
					}
					else {
						bNeedUpdateProps = true;
						cdfPropertiesDoc.insertRecord({
							'jsxid' : prop.name,
							'name'  : prop.name,
							'value' : prop.value
						});
					}
				}
				mx.MATRIX2.getServer().getCache().setDocument("OBJ_" + obj.objid + "_PROPERTIES", cdfPropertiesDoc);

				// loop through object methods
				if (obj.methods && lstMethods.size() > 0) {
					// Find out if methods list was changed
					if (cdfMethodsDoc.selectNodes("//record").size() != lstMethods.size()) {
						bNeedUpdateMethods = true;
					}
					else {
						for (var meth_iter = lstMethods.iterator(); meth_iter.hasNext();) {
							var m = meth_iter.next();
							var xmlRec = cdfMethodsDoc.getRecordNode(m.name);
							if (!xmlRec) {
								bNeedUpdateMethods = true;
								break;
							}
							// At last, compare method's params
							if (   xmlRec.getAttribute("desc") != m.desc
							    || xmlRec.getAttribute("displayname") != m.displayname)
							{
								bNeedUpdateMethods = true;
								break;
							}
						}
					}

					cdfMethodsDoc = jsx3.CDF.Document.newDocument();
					for (var meth_iter = lstMethods.iterator(); meth_iter.hasNext();) {
						var method = meth_iter.next();
						cdfMethodsDoc.insertRecord({
							jsxid       : method.name,
							name        : method.name,
							desc        : method.desc,
							displayname : method.displayname
						});
					}
					mx.MATRIX2.getServer().getCache().setDocument("OBJ_" + obj.objid + "_METHODS", cdfMethodsDoc);
				}

				// Update associated object records if any
				var lstAssocRecs = OM._OBJREC_CACHE[obj.objid];
				if (lstAssocRecs && (bNeedUpdateProps || bNeedUpdateMethods)) {
					var curTab = mx.CT.getActiveTab();
					for (var iter = lstAssocRecs.iterator(); iter.hasNext();) {
						var objrecStruct = iter.next();
						var objRec = objrecStruct.block;
						// Skip recs which belong to inactive tabs
						if (objrecStruct.tab != curTab) continue;

						try {
							var objSibling = objRec.getNextSibling();
							if (objSibling && objSibling.getDisplay() == jsx3.gui.Block.DISPLAYBLOCK &&
								objSibling.getAttribute("objid") == obj.objid) {
								if (bNeedUpdateProps) {
									var matrProp = objSibling.getDescendantOfName("ObjectProperties");
									// loop through changed properties once more
									for (var k = 0; k < lstProperties.size(); k++) {
										try {
											matrProp.insertRecordProperty(lstProperties.get(k).name, "value",
												lstProperties.get(k).value, true);
										} catch (e) {
										}
									}
								}
								if (bNeedUpdateMethods) {
									OM._repaintRecMethods(objSibling, cdfMethodsDoc, true);
								}
							}
							objRec.getFirstChild().setBackgroundColor("yellow", true);
							objrecStruct.highlightedAt = OM.lastUpdate;
							// update state image if needed
							var img = objRec.getDescendantOfName("img");
							var newImg = OM.getStateImage(obj.objid);
							var curImg = img.getSrc();
							if (newImg != curImg) {
								img.setSrc(newImg);
								img.setDisplay(newImg ? jsx3.gui.Block.DISPLAYBLOCK : jsx3.gui.Block.DISPLAYNONE);
								img.repaint();
							}

						} catch(e) {
							continue
						}
					}
				}

			}
		}
	};

	/**
	 * Store object record block ref in the package-wide cache in order to update its view
	 * according to applet notifications
	 */
	OM.cacheObjRec = function(blkRec) {
		var objid = blkRec.getAttribute("objid");
		if (!objid) return;
		var objrecStruct = {
			'block' : blkRec,
			'tab'   : blkRec.getAncestorOfType("jsx3.gui.Tab"),
			'highlightedAt' : 0
		};
		// Test function: finds occurances of current block in the list
		var f_test = function(e) {
			return e.block == blkRec
		};
		// Add new element to cache if not present
		if (OM._OBJREC_CACHE[objid] == null)
			OM._OBJREC_CACHE[objid] = new jsx3.util.List([objrecStruct]);
		else if (OM._OBJREC_CACHE[objid].filter(f_test).size() == 0)
			OM._OBJREC_CACHE[objid].add(objrecStruct);

		blkRec.subscribe(jsx3.gui.Interactive.DESTROY, OM.onRecordDestroy);
	};

	/**
	 * walks through cached object records and clears
	 * highlighted ones
	 */
	OM.clearHighlightedRecs = function() {
		for (var objid in OM._OBJREC_CACHE) {
			var lstRecs = OM._OBJREC_CACHE[objid];
			if (!lstRecs) continue;
			for (var iter = lstRecs.iterator(); iter.hasNext();) {
				var objrecStruct = iter.next();
				var rec = objrecStruct.block;
				var curTime = new Date().getTime();

				// Check whether 'minHighlightedInterval' has expired for the record
				if (curTime - objrecStruct.highlightedAt < OM.minHighlightedInterval) {
					continue;
				}
				// Reset background color if expired
				if (rec.getFirstChild().getBackgroundColor() != "white") {
					rec.getFirstChild().setBackgroundColor("white", true);
				}
			}
		}
	};

	/**
	 * Handles removing object record from Matrix2 view. Removes associated object
	 * from cache
	 */
	OM.onRecordDestroy = function(event) {
		var lst = OM._OBJREC_CACHE[event.target.getAttribute("objid")];
		if (lst) {
			var f_test = function(e) {
				return e.block == event.target
			};
			var objElem = lst.filter(f_test).get(0);
			if (objElem) lst.remove(objElem);
		}
		event.target.unsubscribe(jsx3.gui.Interactive.DESTROY, OM.onRecordDestroy);

		// Finally must unsubscribe from object notifications
		// if this is the last record view for this object
		if ( (!lst || lst.size() == 0) && OM.clientApplet) {
		    try {
			    OM.clientApplet.removeMonitoredObject(event.target.getAttribute("objid"));
		    }
		    catch (e) {
			    appLogger.warn(
				"Error when trying to unsubscribe from object notifications: " +
				e.description
			    );
		    }
		}
	};

	OM.checkDeviceState = function (objProperties, isCDFDoc) {
		var TYPE = false;
		var SUBTYPE = false;
		var state = false;
		//parse properties
		//if CDF document
		if (isCDFDoc) {
			var propNodes = objProperties.selectNodes("//record");
			for (var i = 0; i < propNodes.size(); i++) {
				if (propNodes.get(i).getAttribute("name") == "TYPE") {
					TYPE = propNodes.get(i).getAttribute("value");
				}
				if (propNodes.get(i).getAttribute("name") == "SUBTYPE") {
					SUBTYPE = propNodes.get(i).getAttribute("value");
				}
				if (propNodes.get(i).getAttribute("name") == "state") {
					state = propNodes.get(i).getAttribute("value");
				}
			}
		}
		else { //if object
			for (var i = 0; i < objProperties.length; i++) {
				if (objProperties[i].name == "TYPE") {
					TYPE = objProperties[i].value;
				}
				if (objProperties[i].name == "SUBTYPE") {
					SUBTYPE = objProperties[i].value;
				}
				if (objProperties[i].name == "state") {
					state = objProperties[i].value;
				}
			}
		}

		//appLogger.info("checkDeviceState: TYPE="+TYPE+", SUBTYPE="+SUBTYPE+", state="+state);

		//check state
		//0 - do not show
		//1 - red
		//2 - green
		if (TYPE == "X" && SUBTYPE == "A") {
			if (state == false) {
				return 0;
			} else if (state == 'Armed' || state == 'Maintenance') {
				return 2;
			} else {
				return 1;
			}
		} else if (TYPE == "X" && SUBTYPE != "A") {
			if (state == false) {
				return 0;
			} else if (state == 'Normal' || state == 'Maintenance') {
				return 2;
			} else {
				return 1;
			}
		} else {
			return 0;
		}
	};

	OM.getStateImage = function(objid) {
		var defImg = "";
		var cdfProperties = mx.MATRIX2.getServer().getCache().
			getDocument("OBJ_" + objid + "_PROPERTIES");
		if (!cdfProperties) return defImg;
		var state = OM.checkDeviceState(cdfProperties, true);
		switch (state) {
			case 2:  return "images/design1/eLog2/eLog2_witness_status_ok.png";
			case 1:  return "images/design1/eLog2/eLog2_witness_status_notok.png";
			default: return defImg;
		}
	};

});
