
function motionSearch(tibcoBlock, initData) {

	/* cache DOM elements for quick access to cell elements and timeline */
	this.cell = null;
	this.timeline = null;
	this.zone_editor = null;

	this.obj = 0; // Current camera objId

	/* Filter options */
	this.objSize = null;
	this.granularity = null;
	this.prebuffer = null;
	this.postbuffer = null;
	this.archive_granularity = null;

	/* Current search mode */
	this.mode = 'archive'; // motion | archive
	this.modes = ['archive', 'motion'];
	this.motionAllowed = false; // is motion search mode allowed for current camera

	var block = new jsx3.gui.Block('motionSearch', 0, 0, '100%', '100%',
		'<div style="height: 160px; float: left;" class="timeline"></div>' +

			'<div class="wrapper" style="overflow: auto;">' +

			'<div class="controls" style="float: right; width: 515px;">' +
				'<div style="width: 10px; height: 570px; float: left;" class="show_hide layout_separator open"></div>' +
				'<div class="controls_wrap" style="height: 570px; margin-left: 10px; background-color: #E4E4E4; overflow: hidden">' +
					'<div style="float: left; padding-left: 10px;">' +
						__("Filters: ") +
						'<button class="motion_apply">' + __("Apply") + '</button> ' +
						'<button class="motion_default">' + __("Default") + '</button>' +
					'</div>' +
					'<div class="tabs" style="float: right;">' +
						'<span class="archive">' + __("Archive search") + '</span>' +
						'<span class="motion">'  + __("Motion search")  + '</span>' +
					'</div>' +
					'<form name="filters_form" onsubmit="return false;">' +
					'<div class="panes" style="float: left; width: 500px; text-align: center; padding: 3px;">' +
						'<div class="archive_search" mode="archive">' +
							'<h5>' + __("Archive search") + '</h5>' +
								__("Granularity (sec)") + '<input type="range" min="5" max="300" name="archive_granularity">' +
						'</div>' +
						'<div class="motion_search" mode="motion">' +
							'<h5>' + __("Motion search") + '</h5>' +
							'<table style="text-align: center;"><tr>' +
								'<td>' + __("Granularity (sec)") + '<input type="range" min="5" max="300" name="granularity"></td>' +
								'<td>' + __("Object size (cells)") + '<br><input type="range" min="1" max="100" name="objSize"></td>' +
							'</tr><tr>' +
								'<td>' + __("Pre-motion (sec)")   + '<input type="range" min="5" max="30" name="prebuffer"></td>' +
								'<td>' + __("Post-motion (sec)")  + '<input type="range" min="10" max="60" name="postbuffer"></td>' +
							'</tr></table>' +
							'<div class="zone_editor_wrap"></div>' +
							'<span style="font-size: 9px; color: gray;">' +
								__("By default, entire picture is included into the search.") + ' ' +
								__("You can EXCLUDE/INCLUDE individual cells by mouse click.") + ' ' +
								__("Grey cells are EXCLUDED, transparent cells are INCLUDED.") + ' ' +
								__("Drag to include rectangular form.") +
								__("Shift+Drag to exclude rectangular form.") + ' ' +
							'</span>' +
						'</div>' +
					'</div>' +
					'</form>' +
				'</div>' +
			'</div>' +

			'<div style="margin-right: 515px;" class="player_wrap">' +
				'<div class="header">' +
					'<span class="camera_name">' + __("&lt;Empty&gt;") + '</span>' +
					'<div style="float: right; margin-top: -3px;">' +
						'<button class="motion_prev" title="' + __("Previous page") + '">&laquo;</button> ' +
						'<button class="motion_next" title="' + __("Next page") + '">&raquo;</button> ' +
						'<button class="motion_calendar" title="' + __("Calendar") + '">C</button> ' +
					'</div>' +
				'</div>' +
			'</div>' +

			'</div>'
	);
	tibcoBlock.setAttribute('motionSearch', '1');
	tibcoBlock.setChild(block).repaint();
	block.subscribe(jsx3.gui.Interactive.AFTER_RESIZE_VIEW, this, "onAfterResizeLayout");
	var self = this;
	var cell = $('#' + block.getId());
	this.cell = cell;
	this.id = cell.attr('id');
	cell.data('model', this);

	// add drop handler
	cell.addClass('drop')
		.bind('drop', function(event, obj) { self.setObj(obj); });

	cell.find("div.tabs").tabs("div.panes > div", {
		tabs: 'span',
		onClick: function(event, tabIndex) {
			self.mode = self.modes[tabIndex];

			var motionSearchCalibratorPlayer = self.cell.find('.motionSearchCalibratorPlayer');
			if (self.mode == 'archive') {
				if(motionSearchCalibratorPlayer){
					motionSearchCalibratorPlayer.css('visibility', 'hidden')
				}
			}
			else if(self.mode == 'motion'){
				if(motionSearchCalibratorPlayer){
					motionSearchCalibratorPlayer.css('visibility', 'visible')
				}
			}

			if (self.timeline) {
				setTimeout(function() {self.timeline.onGetImages('up');}, 100);
			}
		},
		onBeforeClick: function(event, tabIndex) {
			// Make "motion" tab disabled with no-motion camera
			if (!self.motionAllowed && self.modes[tabIndex] == 'motion')
				return false;
		}
	});
	cell.find(":range").rangeinput({
		css: {
			input:  'range-input',
			slider: 'range-slider',
			progress: 'range-progress',
			handle: 'range-handle'}}
	);
	cell.find('form[name=filters_form]').link(this, {
		granularity: {convertBack: "range"},
		objSize: {convertBack: "range"},
		prebuffer: {convertBack: "range"},
		postbuffer: {convertBack: "range"},
		archive_granularity: {convertBack: "range"}
	});


	cell.mouseover(function() {if (dragged) {cell.find('.player').css('visibility', 'hidden');}})
		.mouseleave(function() {cell.find('.player').css('visibility', 'visible');});

	cell.find('div.show_hide').click(function() {
		cell.find('div.controls_wrap').toggle();
		$(this).toggleClass('open');
		self.onAfterResizeLayout();
	});

	cell.find('.motion_apply').click(function() {
		self.updateValues();
		self.timeline.onGetImages('up');
	});

	cell.find('.motion_default').click(function() {
		self.setDefaultValues();
		self.timeline.onGetImages('up');
	});
	cell.find('.motion_prev').click(function() { self.timeline.prevImagesPage(); });
	cell.find('.motion_next').click(function() { self.timeline.nextImagesPage(); });
	cell.find('button.motion_calendar').click(function() {
		if ($('#motion_calendar').length) {
			$('#motion_calendar').remove();
		} else {

			$('body').append('<div id="motion_calendar" style="position: absolute; top: ' + ($(this).offset().top + 25) + 'px; left: ' + ($(this).offset().left + 33) + 'px; z-index: 100;"></div>');
			$('#motion_calendar').jCal({
				day: new Date(),
				days: 1,
				showMonths: 1,
				monthSelect: false,
				callback: function (day) {
					var time = new Date(self.currentTime);
					groupControls.setTime(new Date(day.getFullYear(), day.getMonth(), day.getDate(), time.getUTCHours(), time.getUTCMinutes(), time.getUTCSeconds()).getTime());
					$('#motion_calendar').remove();
				}
			});
		}
	});

	this.setDefaultValues();
	this.focus();
	this.initTimeline(function() {
		if (initData)
			self.unserialize(initData);
	});
}

motionSearch.prototype = {
	/**
	 * Set default values to search filters
	 */
	setDefaultValues: function() {
		$(this)
			.setField('granularity', 15)
			.setField('prebuffer', 10)
			.setField('postbuffer', 5)
			.setField('objSize', 5)
			.setField('archive_granularity', 15);

		$('input[name="granularity"]').val(15);
		$('input[name="prebuffer"]').val(10);
		$('input[name="postbuffer"]').val(5);
		$('input[name="objSize"]').val(5);
		$('input[name="archive_granularity"]').val(15);
	},

	/**
	 * Update values for search filters from GUI
	 */
	updateValues: function() {
		this.granularity = parseInt($('input[name="granularity"]').val());
		this.prebuffer = parseInt($('input[name="prebuffer"]').val());
		this.postbuffer = parseInt($('input[name="postbuffer"]').val());
		this.objSize = parseInt($('input[name="objSize"]').val());
		this.archive_granularity = parseInt($('input[name="archive_granularity"]').val());
	},

	/**
	 * Add camera to search tab
	 * @param obj   ResourceTree camera object
	 */
	setObj: function(obj) {

		if (!obj.type || obj.type != 'camera')
			return;

		// Check if user has archive credentials for camera, and camera has MD settings
		if (!obj.cred('A')) {
			alert(__("You must have archive credentials for camera to use motion/archive search"));
			return;
		}

		//this.motionAllowed = true;
		
		// by default always disabled due to 6674
		// TODO restore backend functionality
		this.motionAllowed = false;
		
		if (obj.attr('VAE_ACTIVE') == 'no'
			&& obj.attr('VAE_MOTION_ACTIVE') == 'no'
			&& obj.attr('STORAGE_POLICY') == "1") {

			if (this.mode == 'motion') {
				alert(__("Camera must have correct storage and motion detection settings to use motion search. You can use archive mode instead."));
				return;
			} else {
				this.motionAllowed = false;
			}
		}
		if (obj.attr('STORAGE_POLICY') == "-7") {
			alert(__("Camera must have correct storage and motion detection settings to use motion search."));
			return;
		}

		this.removeObj();
		this.timeline.clearImages();

		this.obj = obj.obj;
		var now = mx.MATRIX2.getServerTime();

		// add player and camera name
		var plWrap = this.cell.find('div.player_wrap');
		var h = plWrap.height() - this.cell.find('div.header').height() - 6;
		var w = plWrap.width();
		var startTS = now - 300*1000;
		var endTS = now;
		var start = mx.TIME.formatUTCDate(startTS, 'YYYY-MM-DD HH-II-SS');
		var end = mx.TIME.formatUTCDate(endTS, 'YYYY-MM-DD HH-II-SS');

		plWrap
			.append(
				'<iframe class="player" style="width: ' + w + 'px; height: ' + h + 'px; border: 0" frameborder="no" scrolling="no" src="cell_player.php?' +
					'name=' + this.id +
					'&camera=' + this.obj +
					'&width=' + w +
					'&height=' + h +
					'&ts_from=' + start +
					'&ts_to=' + end +
					'"></iframe>')
			.find('span.camera_name').html(obj.name);

		// set default values and reset zone editor with new objid
		this.setDefaultValues();
		
		var cloud = obj.attr('CLOUD_SEPARATOR_TS')
		if(cloud){
			this.timeline.setCloudTimestamp(parseInt(cloud)*1000);
		}
		else{
			this.timeline.resetCloudTimestamp();
		}
		this.timeline.setTime(startTS, true);

		this.addMDplayer(start, end);

		this.setPlayerActive();
	},

	/**
	 * add motion detection player
	 */
	addMDplayer: function(start, end) {
		this.cell.find('div.zone_editor_wrap').html('<iframe class="motionSearchCalibratorPlayer" style="width: 480px; height: 360px; border: 0" frameborder="no" scrolling="no" src="cell_player.php?' +
					'name=' + this.id +
					'&camera=' + this.obj +
					'&width=480' +
					'&height=360' +
					'&ts_from=' + start +
					'&ts_to=' + end +
					'"></iframe>');
	},


	/**
	 * remove camera from search cell
	 */
	removeObj: function() {
		if (!this.obj)
			return;

		activeGroup_Cell = mx.MATRIX2.getObjFromIFrame('groupIFAPlayer');
		activeGroup_Cell.removePlayer(this.getPlayer());
		groupControls.unsetGroup(null);
		activeGroup_Cell.reset();
		groupControls.setAccess(true,true,true,true);

		this.obj = null;

		// remove player and set name to empty
		this.cell.find('div.player_wrap')
			.find('iframe.player').remove().end()
			.find('span.camera_name').html('&lt;Empty&gt;');
	},


	focus: function() {
		activeGroup_Cell = mx.MATRIX2.getObjFromIFrame('groupIFAPlayer');
		if((activeGroup_Cell.getGroupSize()-activeGroup_Cell.getNumOfAudioDevices())==1) {

			var prevActiveCell = mx.MATRIX2.getServer().getJSXById(activeGroup_Cell.getPlayer(0).id);
			try {
				prevActiveCell.getDescendantOfName('cell_dev_name').setColor("5F696D");
				prevActiveCell.getDescendantOfName('cell_header').setBackground('background-image:url(./images/design1/gui_cell/cell_header_bg.gif);background-repeat:repeat-x;').repaint();
			} catch(e) {}
			activeGroup_Cell.removePlayer(activeGroup_Cell.getPlayer(0));
			groupControls.unsetGroup(null);
			activeGroup_Cell.reset();
			groupControls.setAccess(true,true,true,true);
		}
		groupControls.checkTimeline();

		this.cell.find('object, iframe').css('visibility', 'visible');

		this.setPlayerActive();
	},


	blur: function() {

		this.cell.find('object, iframe').css('visibility', 'hidden');
		$('#motion_calendar').remove();
		activeGroup_Cell = mx.MATRIX2.getObjFromIFrame('groupIFAPlayer');
		if((activeGroup_Cell.getGroupSize()-activeGroup_Cell.getNumOfAudioDevices())==1) {
			activeGroup_Cell.removePlayer(activeGroup_Cell.getPlayer(0));
			groupControls.unsetGroup(null);
			activeGroup_Cell.reset();
			groupControls.setAccess(true,true,true,true);
		}
	},


	/**
	 * Connect motionSearch player with groupControls to provide all player commands
	 */
	setPlayerActive: function() {
		if (!this.obj)
			return;

		var player = this.getPlayer();
		var playerMotion = this.getMotionPlayer();

		if (!player || !playerMotion) {
			var self = this;
			setTimeout(function() {self.setPlayerActive();}, 500);
			return;
		}

		activeGroup_Cell.reset();
		activeGroup_Cell.addPlayer(player);
		groupControls.setGroup(activeGroup_Cell,false,false);
		groupControls.play();

		this.initCalibrator();
		playerMotion.play();
	},

	/**
	 * Initialize motion detection calibrator
	 */
	initCalibrator: function() {
		var attrs = resourceTree.getObject(this.obj).attributes;
		var MDplayer = this.getMotionPlayer().playerImpl;

		MDplayer.initMDCalibrator();
		MDplayer.mdcalibrator.horizontalCells = attrs.VAE_MOTION_CELLS_H;
		MDplayer.mdcalibrator.verticalCells = attrs.VAE_MOTION_CELLS_V;
		MDplayer.mdcalibrator.activeMap = attrs.VAE_MOTION_ZONEMAP;
		MDplayer.mdcalibrator.activate();

		this.zone_editor = MDplayer.mdcalibrator;
	},


	/**
	 * Calculate timestamps (with current granularity) and push them into timeline
	 * Used in archive mode only
	 */
	setArchiveData: function() {
		var timestampList = [];

		// midnight of current day
		var startPoint = (new Date()).setUTCHours(0, 0, 0, 0);
		var granularity = this.archive_granularity * 1000;
		
		var dev = resourceTree.getObject(this.obj);
		var cloud = dev.attr('CLOUD_SEPARATOR_TS');

		// Find first timestamp less than beginTime taking granularity into account
		var temp = Math.round(Math.abs(startPoint - this.beginTime) / granularity);
		var currTime = startPoint > this.beginTime ? startPoint - (temp + 1) * granularity : startPoint + temp * granularity;

		var now = mx.MATRIX2.getServerTime();
		var i = 0;

		while (currTime < this.endTime && currTime < now) {
			var timestamp = currTime + Math.round(granularity) / 2;
			var imgUrl = "/storage/snapshot?objid=" + this.obj + "&ts=" + Math.round(timestamp / 1000) + "&downscale";
			
			//console.log("setArchiveData " + cloud +", "+ parseInt(cloud)*1000 +", "+ timestamp);
			if(cloud && parseInt(cloud)*1000 >= timestamp){
				imgUrl = "/TibcoGI/JSXAPPS/mx/images/design1/resources/cloud_content.png";
			}

			timestampList.push({
				Fn: i++, // fragment number
				Tn: 0, // timestamp number in fragment
				timestamp: timestamp, // Unix time in ms
				start_time: currTime, // start time of fragment, Unix time in ms
				end_time: currTime + granularity, // end time of fragment, Unix time in ms
				url: imgUrl
			});

			currTime += granularity;
		}

		this.timeline.setImages('up', timestampList);

	},


	/**
	 * Make storage request to get motion fragments and push them into timeline to view snapshots
	 * Used in motion mode only
	 */
	setMotionData: function() {
		var self = this;

		if(this.zone_editor == null){
			if(window['motion_zone_editor']){ clearTimeout(window['motion_zone_editor']); }
			window['motion_zone_editor'] = window.setTimeout(function() { self.setMotionData(); }, 100);
			return;
		}

		$.ajax({
			url: "/storage/motionsearch",
			dataType: "json",
			async: true,
			cache: false,
			data: {
				objid: this.obj,
				start_time: Math.round(this.beginTime / 1000),
				end_time: Math.round(this.endTime / 1000),
				gridwidth:  this.zone_editor.horizontalCells,
				gridheight: this.zone_editor.verticalCells,
				gridmap: '' + this.zone_editor.activeMap, // cannot use simply activeMap because of Safari bug*/
				objsize: this.objSize,
				pre_motion: this.prebuffer,
				post_motion: this.postbuffer,
				granularity: this.granularity
			},
			success: function(json)
			{
				if (json.error !== "" || !self.obj)
					return;
					
				var dev = resourceTree.getObject(self.obj);
				var cloud = dev.attr('CLOUD_SEPARATOR_TS');

				var fragment = json.obj[self.obj].fragment;

				var timestampList = [];

				for (var i = 0; i < fragment.length; i++)
				{
					var fragmentTimestampList = fragment[i].timestamp;
					var start_time = fragment[i].start_time * 1000;
					var end_time = fragment[i].end_time * 1000;

					for (var j = 0; j < fragmentTimestampList.length; j++)
					{
						var timestamp = fragmentTimestampList[j];					
						var imgUrl = "/storage/snapshot?objid=" + self.obj + "&ts=" + Math.round(timestamp / 1000) + "&downscale";
									
						if(cloud && parseInt(cloud)*1000 >= timestamp){
							imgUrl = "/TibcoGI/JSXAPPS/mx/images/design1/resources/cloud_content.png";
						}

						timestampList.push({
							Fn: i, // fragment number
							Tn: j, // timestamp number in fragment
							timestamp: timestamp * 1000, // Unix time in ms
							start_time: start_time, // start time of fragment, Unix time in ms
							end_time: end_time, // end time of fragment, Unix time in ms
							url: imgUrl
						});
					}
				}

				self.fragments = fragment;
				self.timeline.setImages('up', timestampList);
			}
		});
	},


	/**
	 * Get media player instance of motion search cell
	 */
	getPlayer: function() {
		var player = this.cell.find('iframe.player');
		if (player.length && player[0].contentWindow)
			return player[0].contentWindow['_mplayer_' + this.id];
		else
			return null;
	},

	/**
	 * Get motion player instance of motion search cell
	 */
	getMotionPlayer: function() {
		var player = this.cell.find('iframe.motionSearchCalibratorPlayer');
		if (player.length && player[0].contentWindow)
			return player[0].contentWindow['_mplayer_' + this.id];
		else
			return null;
	},


	/**
	 * Timeline callback, it is called when clicking on snapshot in timeline
	 * @param time       actual time of snapshot
	 * @param beginTime  start time of fragment represented by snapshot
	 * @param endTime    end time of fragment represented by snapshot
	 */
	onImageClick: function(time, beginTime, endTime) {
		if (!this.obj)
			return;

		//groupControls.group.setContext(3, beginTime, endTime);
		groupControls.setTime(time);

		var motionPlayer = this.getMotionPlayer();
		motionPlayer.setContext(this.obj, 3, beginTime, endTime);
		motionPlayer.play();
	},


	/**
	 * Make request to get storage motion info or archive data
	 * @param line
	 * @param beginTime  lower border for motions/snapshots
	 * @param endTime    upper border for motions/snapshots
	 */
	onGetImages: function(line, beginTime, endTime) {

		// remember limits of timeline
		this.beginTime = beginTime || this.beginTime;
		this.endTime = endTime || this.endTime;

		if (!this.obj)
			return;

		switch (this.mode) {
			case 'archive':
				this.setArchiveData();
			break;
			case 'motion':
				this.setMotionData();
			break;
		}

	},


	/**
	 * Make service request to get storage data for current camera in motionSearch
	 * @param line
	 * @param startTime
	 * @param finishTime
	 * @param granularity
	 */
	onGetData: function(line, startTime, finishTime, granularity) {

		if (!this.obj)
			return;

		var streamnumber = this.getPlayer().getParameter('STREAMNUM') || 1;

		mx.TIMELINE.getStorageCoverage({
			timeline: this.timeline,
			timeLine: line,
			startTime: startTime,
			finishTime: finishTime,
			granularity: granularity,
			objids: [this.obj],
			streamnumber: streamnumber
		});
	},


	/**
	 * Adjust controls, player width and height when resizing window or layout
	 */
	onAfterResizeLayout: function() {
		var size= jsx3.GO(this.cell.parent()[0].id).getAbsolutePosition();
		size.H -= this.cell.find('object').height() + 10;

		var plWrap = this.cell.find('.player_wrap');

		this.cell.find('.wrapper, .player_wrap').css('height', size.H + 'px');
		size.H -= 3;

		if (this.cell.find('.controls_wrap').css('display') == 'none') {
			this.cell.find('.layout_separator').css('height', size.H + 'px');
			this.cell.find('.controls').css('width', '10px');
			plWrap.css('margin-right', '10px');
		} else {
			var h = Math.max(size.H, 570);
			this.cell.find('.layout_separator, .controls_wrap').css('height', h + 'px');
			this.cell.find('.controls').css('width', '515px');
			plWrap.css('margin-right', '515px');
		}

		if (this.getPlayer()) {
			var plW = plWrap.width();
			var plH = plWrap.height() - this.cell.find('div.header').height() - 6;
			this.cell.find('iframe.player').css({height: plH, width: plW});
			if (plW > 0 && plH > 0) {
				this.getPlayer().setSize(plW, plH);
			}
		}
	},


	/**
	 * Init flash timeline component, add callbacks
	 */
	initTimeline: function(callback) {

		var id = this.cell.parent()[0].id;
		this.cell.find('.timeline').attr('id', id + '_timeline');

		this.timeline = new TimeLine({
			id: '#' + id + '_timeline',
			variableName: "motion_timeline_" + id,
			isDebug: false,
			withImages: true
		});
		window['motion_timeline_' + id] = this.timeline;
		var self = this;

		this.timeline.init();
		this.timeline.onGetImages = function(line, beginTime, endTime) {self.onGetImages(line, beginTime, endTime)};
		this.timeline.onImageClick = function(time, beginTime, endTime) {self.onImageClick(time, beginTime, endTime)};
		this.timeline.onGetData = function(line, beginTime, endTime, granularity) {self.onGetData(line, beginTime, endTime, granularity);};
		this.timeline.onTimeChange = function(time) {
			self.currentTime = time;
			if (self.obj) {
				groupControls.setTime(time);
			}
		};
		this.timeline.onSelectionChange = function(line, startTime, endTime) {
			if(line == "down"){
				return;
			}

			var cell = jsx3.GO(self.id);

			if (!startTime && !endTime){
				cell.removeAttribute("markA");
				cell.removeAttribute("markB");
			} else {
				cell.setAttribute("markA", startTime);
				cell.setAttribute("markB", endTime);
			}
		};

		this.timeline.onLoad = function() {
			self.onAfterResizeLayout();
			
			var doc = mx.MATRIX2.getServer().Cache.getDocument('TIME_timezones');
			var node = doc.selectSingleNode("//record[@jsxid='"+mx.MATRIX2.getServer().getJSXByName("selectTimezone").getValue()+"']");
			var offset = node.getAttribute("offset");
			var name = node.getAttribute("jsxtext");
			var paramTimezoneOffset = {};
			for(var i = 0; i < mx.TIME.timezones.length; i++){
				if(name == mx.TIME.timezones[i].name){
					var curYear = (new Date().getFullYear());
					paramTimezoneOffset[curYear] = mx.TIME.timezones[i].years[curYear];
				}
			}
			
			this.setParameters({
				selection: {fixed: false}, 
				scale: {granularity: "min"},
				dateFormat: mx.MATRIX2.getIdentityValue("NLS_LONG_DATE_FORMAT"),
				timezoneOffset: paramTimezoneOffset
			});
			
			this.setTime((new Date()).getTime(), true);

			if ($.isFunction(callback))
				callback();
		};
	},


	/**
	 * Serialize motionSearch object to store it into configuration
	 */
	serialize: function() {

		var result = {
			type: 'motionSearch',
			obj: this.obj,
			objSize: this.objSize,
			granularity: this.granularity,
			prebuffer: this.prebuffer,
			postbuffer: this.postbuffer,
			archive_granularity: this.archive_granularity,
			mode: this.mode
		};
		return result;
	},


	/**
	 * Unserialize motionSearch from previously stored configuration
	 * @param data
	 */
	unserialize: function(data) {
		if (data.obj) {
			var obj = resourceTree.getObject(data.obj);
			if (obj) {
				this.setObj(obj);
			}
		}

		$(this)
			.setField('granularity', data.granularity)
			.setField('prebuffer', data.prebuffer)
			.setField('postbuffer', data.postbuffer)
			.setField('objSize', data.objSize)
			.setField('archive_granularity', data.archive_granularity);

		this.cell.find('div.tabs span.' + data.mode).click();
	}

};
