/**
 * Load and Save Configuration Class
 */
LSC = {
	list: {}, // list of configurations
	curConfId: null, // currently loaded configuration
	geHack: null,


	/**
	 * Initialize configurations list, load default configuration if exists
	 */
	init: function() {
		LSC.getConfList();
		$.each(LSC.list, function() {
			if (this.name == 'default' && this.owner == resourceTree.getUser().obj) {
				appLogger.info("Found default configuration for role " + resourceTree.getRole().name + ", loading..");
				LSC.load(this.obj);
				return false;
			}
		});
	},


	/**
	 * Load private and shared configurations ans save it into cache
	 */
	getConfList: function() {

		LSC.list = {};

		$.ajax({
			data: {'function': 'getObjectList', 'type': 'configuration'},
			async: false,
			success: function(data) {
				$.each(data['list'], function() {
					this.name = $('<div/>').text(this.attributes.NAME).html();
					this.shared = parseInt(this.attributes.SHARED) ? 1 : 0;
					this.type = parseInt(this.attributes.SHARED) ? 'shared' : 'private';
					this.owner = parseInt(this.attributes.OWNER);
					this.owner_name = this.attributes.OWNER_NAME;
					this.role_id = parseInt(this.attributes.ROLEID);
					this.img = this.shared ? 'images/design1/dialog/shared.png' : 'images/design1/dialog/private.png';

					LSC.list[this.obj] = this;
				});
			}
		});
	},


	/**
	 * get list of configurations
	 * @param getOwn         bool  if we want to get only confs belonging to user
	 * @param getOptionList  bool  if we want to get html for select options
	 */
	getList: function(getOwn, getOptionList, addPlaceholder) {

		if (addPlaceholder === undefined) {
			addPlaceholder = true;
		}

		// Get sorted list of conf ids
		var confIds = Object.keys(LSC.list).sort(function(a, b) {return naturalSort(LSC.list[a].name, LSC.list[b].name);});
		var userId = resourceTree.getUser().obj;
		var list = [];

		// Get list of configurations
		$.each(confIds, function() {
			var conf = LSC.list[this];

			// If getting info for saving, process only those confs that belong to current user
			if (getOwn && conf.owner != userId)
				return;
			list.push(conf);
		});

		if (!getOptionList)
			return list;

		// Return option list
		var html = addPlaceholder ? '<option value="0">' + __("-Select configuration-") + '</option>' : '';
		$.each(list, function() {
			html += '<option value="' + this.obj + '" class="typed ' + this.type + '">' + '[' + this.owner_name + '] ' + $('<div/>').text(this.name).html() + '</option>';
		});
		return html;
	},


	/**
	 * Save existing configuration or create new
	 * @param confName - name of configuration
	 * @param isShared - is configuration shared
	 * @param confId   - conf id, if configuration already exists
	 */
	save: function(confName, isShared, confId) {

		// Serialize elogsmall cell, elog settings and resource tree settings
		appLogger.time('save configuration');
		appLogger.time('save resource tree & elog');
		var data = {
			version: mx.MATRIX2.getSystemInfo().version,
			tabs: [],
			elogsmall: {
				'contents':     LSC._serialize(jsx3.GO('pane_elogsmall')),
				'sub1pct': jsx3.GO('pane_elogsmall').getParent().getSubcontainer1Pct(), // position of splitter
				'sub2min': jsx3.GO('pane_elogsmall').getParent().getSubcontainer2Min()  // min size of subcontainer-2
			},
			tree: resourceTree.serialize(),
			elog: mx.ELog2.serialize()
		};
		appLogger.timeEnd('save resource tree & elog');

		// Serialize tabs
		appLogger.time('save tabs');
		var tabs = mx.MATRIX2.getServer().getJSXByName('tabbedpaneMain').getChildren();
		var activeTab = mx.CT.getActiveTab();
		for(var i=0;i<tabs.length;i++) {

			// continue if hidden or VMX
			if(tabs[i].getDisplay()==jsx3.gui.Block.DISPLAYNONE || tabs[i].getAttribute("curType") == "vmx")
				continue;

			data.tabs.push({
				// don't save name if it is a number
				'name':     i == tabs[i].getAttribute("initText") ? '' : tabs[i].getAttribute("initText"),
				'active':   Boolean(activeTab == tabs[i]),
				'contents': LSC._serialize(tabs[i].getFirstChild())
			});
		}
		appLogger.timeEnd('save tabs');

		var confData = {
			'NAME': confName,
			'SHARED': isShared ? 1 : 0,
			'DATA': data,
			'OWNER': resourceTree.getUser().obj,
			'ROLEID': data.tree.role
		};

		// id of current conf or id of created conf
		appLogger.time('send data to server');
		var returnId = confId;
		if (!confId) {
			$.ajax({
				type: "POST",
				async: false,
				data: {'function': 'addObject',	'type': 'configuration', 'attributes': JSON.stringify(confData)},
				success: function(data) {returnId = data.obj;}
			});
		} else {
			$.ajax({
				type: "POST",
				async: false,
				data: {'function': 'setAttributes', 'obj': confId, 'attributes': JSON.stringify(confData)}
			});
		}
		appLogger.timeEnd('send data to server');
		appLogger.timeEnd('save configuration');
		return returnId;
	},


	/**
	 * Loading of configuration
	 * @param confId
	 */
	load: function(confId) {

		appLogger.time('load configuration');

		var data;
		this.geHack = null;
		mx.MATRIX2.clearMatrix(true);

		appLogger.time('get configuration from server');
		$.ajax({
			data: {'function': 'getAttribute', 'obj': confId, 'attribute': 'DATA'},
			async: false,
			success: function(json) {data = $.parseJSON(json.value);}
		});
		appLogger.timeEnd('get configuration from server');

		// Load tree
		resourceTree.unserialize(data.tree);

		// Load ELog
		mx.ELog2.unserialize(data.elog);

		appLogger.time('load elogsmall panel');
		var spl = jsx3.GO('pane_elogsmall').getParent();
		spl.setSubcontainer2Min(data.elogsmall.sub2min);
		spl.setSubcontainer1Pct(data.elogsmall.sub1pct, true);
		mx.MATRIX2.splitterResize(spl);
		LSC._unserialize(jsx3.GO('pane_elogsmall'), data.elogsmall.contents);
		appLogger.time('load elogsmall panel');

		// Load tabs;
		appLogger.time('load tabs');
		var tabs = jsx3.GO('tabbedpaneMain').getChildren();
		var activeTab = tabs[0];
		var curTab;
		for (var i = 0; i<data.tabs.length; i++) {
			if (i == 0) {
				curTab = tabs[0];
			} else {
				// special case if motionSearch tab
				if (data.tabs[i].contents['type'] == 'motionSearch') {
					curTab = mx.CT.createNewTab(__('Archive search'), 'motion');
				} else {
					curTab = mx.CT.createNewTab(data.tabs[i].name, "general");
				}
			}

			LSC._unserialize(curTab.getFirstChild(), data.tabs[i].contents);
			if (data.tabs[i].active) {
				activeTab = curTab;
			}
		}
		appLogger.timeEnd('load tabs');

		// Hack to provide correct GE loading in IE:
		// IE crashes when load GE block in invisible layer,
		// so we activete GE tab, wait until GE loads and activate correct tab
		if (this.geHack) {
			mx.CT.onTabSelect(this.geHack.block.getAncestorOfType(jsx3.gui.Tab));

			mx.GE.addGE(this.geHack.block);
			if(this.geHack.fix_obj_id) {
				mx.GE.lockDeviceOnMap(this.geHack.fix_obj_id);
			}
		}

		this.finalizeLoad(activeTab);

		appLogger.timeEnd('load configuration');

		return confId;
	},


	/**
	 * Auxiliary finalizing function
	 * @param activeTab   tab to set as active when GE plugin loads
	 */
	finalizeLoad: function(activeTab) {

		var ge_iframe = jsx3.GO('ge_iframe') ? mx.MATRIX2.getIFrame(jsx3.GO('ge_iframe')) : null;

		// if GE map is absent or fully loaded
		if(!ge_iframe || (ge_iframe.ge || ge_iframe.ge_map_key=='none')){ //DE2162
			mx.CT.onTabSelect(activeTab);
			$(document).trigger('LSC.loaded');
		}
		else {
			setTimeout(function() { LSC.finalizeLoad(activeTab); }, 100);
		}
	},


	/**
	 * Share or privatize list of confs
	 * @param Ids      array of conf ids
	 * @param isShared boolean is conf shared
	 */
	share: function(Ids, isShared) {
		$.each(Ids, function(i, v) {
			$.ajax({
				type: 'POST',
				async: false,
				data: {'function': 'setAttributes', 'obj': v, 'attributes': JSON.stringify({'SHARED': isShared ? 1 : 0})}
			});
		});
	},


	/**
	 * Delete list of confs
	 * @param Ids  array of conf ids
	 */
	drop: function(Ids) {
		$.each(Ids, function(i, v) {
			$.ajax({
				type: 'POST',
				async: false,
				data: {'function': 'deleteObject', 'objList': JSON.stringify([v])}
			});
		});
	},


	/**
	 * Load configuration dialog setup
	 */
	loadDialog: function() {
		if(!mx.CT.isControlPanelOpened()){
			mx.MATRIX2.showMessage(__("Profile"), __("Open resource bar first."))
			return false;
		}

		LSC.getConfList();

		mx.Dialog.createLayoutDialog(
			'conf_dialog',
			__("LOAD CONFIGURATION"),
			{'top': '70px', 'width':'312px', 'height':'auto'},
			'<select id="conf_id" class="select">' + LSC.getList(false, true) + '</select>' +
			'<span id="btn_conf_load" class="button load" title="' + __("Load configuration") + '"></span>'
		);
		$('#conf_id').jaoselect();
		$('#btn_conf_load').click(LSC.loadController);
	},


	/**
	 * Handler of load button
	 */
	loadController: function() {
		var confId = parseInt($('#conf_id').val());

		if (!confId) {
			alert(__("No configuration selected!"));
			return;
		}

		$('#conf_dialog').remove();
		mx.MATRIX2.mask(__("LOADING CONFIGURATION..."));
		$(document).bind('LSC.loaded', function() {
			mx.MATRIX2.mask(false);
			$(document).unbind('LSC.loaded');
		});

		// Timeout for correct mask in IE
		setTimeout(function() {
			LSC.curConfId = LSC.load(confId);
		}, 10);

	},


	/**
	 * Save configuration dialog setup
	 */
	saveDialog: function() {
		if(!mx.CT.isControlPanelOpened()){
			mx.MATRIX2.showMessage(__("Profile"), __("Open resource bar first."))
			return false;
		}

		LSC.getConfList();

		mx.Dialog.createLayoutDialog(
			'conf_dialog',
			__("SAVE CONFIGURATION"),
			{'top': '60px', 'width':'312px', 'height':'auto'},
			'<div style="margin-bottom: 5px;">' +
				'<input type="text" id="conf_name" placeholder="' + __("input new name") + '" />' +
				'<label title="' + __("Share configuration with other users") + '"><input type="checkbox" id="conf_shared"/>' + __("Share") + '</label>' +
			'</div>' +
			'<select id="conf_id" class="select">' + LSC.getList(true, true) + '</select>' +
			'<span id="btn_conf_save" class="button create" title="' + __("Save configuration") + '"></span>'
		);
		$('#conf_name').placeholder();
		$('#conf_name').change(function() {$('#btn_conf_save').prop('disabled', !$(this).val());});
		$('#conf_id').change(function() {
			if (!parseInt(this.value)) {
				$('#conf_name').val('');
				$('#conf_shared').prop('checked', false);
				$('#btn_conf_save').prop('disabled', true);
				return;
			}
			$('#conf_name').val(LSC.list[this.value].name);
			$('#conf_shared').prop('checked', !!LSC.list[this.value].shared);
			$('#btn_conf_save').prop('disabled', false);
		});
		$('#btn_conf_save').click(LSC.saveController);
		$('#conf_id').jaoselect();

		// If we have current conf, we must offer it as default, allow saving and set right share value
		if (LSC.curConfId && LSC.list[LSC.curConfId] && LSC.list[LSC.curConfId].owner == resourceTree.getUser().obj) {
			$('#conf_id').data('jaoselect').value(LSC.list[LSC.curConfId].obj);
		}
	},


	/**
	 * Handler of save button
	 */
	saveController: function() {

		var confName = $.trim($('#conf_name')[0].value);
		var shared = Boolean($('#conf_shared')[0].checked);

		if( !confName || confName == ""){
			alert(__("Invalid configuration name!"));
			return;
		}

		// Check if such configuration already exists
		var confId = 0;
		$.each(LSC.list, function() {
			if (this.name == confName) {
				confId = this.obj;
				return false;
			}
		});

		if (confId) {
			if (!confirm(Gettext.strargs(__("Do you really want to overwrite existing configuration '%1'?"), [confName])))
				return;
		}

		$('#conf_dialog').remove();
		mx.MATRIX2.mask(__("SAVING CONFIGURATION..."));
		setTimeout(function() {
			LSC.curConfId = LSC.save(confName, shared, confId);
			mx.MATRIX2.mask();
		}, 10);
	},


	/**
	 * Manage configuration dialog setup
	 */
	manageDialog: function() {
		if(!mx.CT.isControlPanelOpened()){
			mx.MATRIX2.showMessage(__("Profile"), __("Open resource bar first."))
			return false;
		}

		// Get list of user configurations
		LSC.getConfList();
		var list = LSC.getList(true, false);

		var str = '';
		if(list.length == 0) {
			str = __("No configurations available");
		} else {
			str = '<select id="conf_id" multiple style="height: 88px;">' + this.getList(true, true, false) + '</select>';
		}

		mx.Dialog.createLayoutDialog(
			'conf_dialog',
			__("MANAGE CONFIGURATIONS"),
			{'top': 60, 'width':312, 'height': 157},
			str +
			'<span id="conf_delete" class="dialog_command">' +
				__("DELETE", "Configurations manage dialog") +
			'</span>' +
			'<span id="conf_share" class="dialog_command">' +
				__("SHARE", "Configurations manage dialog") +
			'</span>' +
			'<span id="conf_private" class="dialog_command">' +
				__("PRIVATIZE", "Configurations manage dialog") +
			'</span>'
		);

		$('#conf_delete, #conf_share, #conf_private').click(LSC.manageController).hide();

		// For all checkboxes add onclick event
		$('#conf_id').change(function () {
			if ($(this).val().length) {
				$('#conf_delete, #conf_share, #conf_private').show();
			} else {
				$('#conf_delete, #conf_share, #conf_private').hide();
			}
		}).jaoselect();
	},


	/**
	 * Handler of delete / share / privatize buttons
	 * @param event
	 */
	manageController: function(event) {

		var Ids = $('#conf_id').val();

		switch (event.target.id) {
			case 'conf_share':
			case 'conf_private':
				mx.MATRIX2.mask(__("UPDATING CONFIGURATION..."));
				LSC.share(Ids, Boolean(event.target.id == 'conf_share'));
				mx.MATRIX2.mask();
			break;
			case 'conf_delete':

				var prompt;
				if (Ids.length == 1) {
					/* i18n: %1 is a configuration name */
					prompt = Gettext.strargs(__("Do you want to delete configuration '%1'?"), [LSC.list[Ids[0]].name]);
				} else {
					// Several configurations to delete
					prompt = __("Do you want to delete selected configurations?");
				}

				if (!confirm(prompt))
					return;

				mx.MATRIX2.mask(__("DELETING CONFIGURATION..."));
				LSC.drop(Ids);
				mx.MATRIX2.mask();

			break;
		}
		LSC.manageDialog();
	},


	/**
	 * Serialization helper, recursively goes through object structure and append info into serialization data
	 * @param liveOnly       convert all events to live video
	 */
	_serialize: function(matrixObject) {
		// Avatar cell
		if(matrixObject.jsxtext && matrixObject.jsxtext.indexOf("avatar_cell") != -1){
			return {
				'type': 'avatar_cell',
				'objid': matrixObject.getRendered().firstChild.attributes['objid'].value
			}
		}

		var child = matrixObject.getFirstChild();
		if (!child) {
			return {'type': 'empty_block'}
		}

		var nodes, save_nodes, i;

		if (child.getClass().toString() == 'jsx3.gui.Splitter') {
			return {
				'type':         'splitter',
				'orientation':  child.getOrientation(),
				'subcont_size': child.getSubcontainer1Pct(),
				'children':     [
					LSC._serialize(child.getFirstChild()),
					LSC._serialize(child.getLastChild())
				]
			};
		}

		if (child.getClass().toString() == 'jsx3.gui.Block') {

			// Player cell
			if (child.getDescendantOfName('iframe_player')) {

				// Ordinary player cell
				if (child.getAttribute("eventid") <= 0) {
					return {
						'type':   'camera',
						'obj': child.getAttribute('objectId')
					};
				}

				// Popup event cell
				if (child.getAttribute("eventid") > 0) {
					return {
						'type':          'event',
						'obj':           child.getAttribute("objectId"),
						'event_id':      child.getAttribute('eventid'),
						'event_ts_from': child.getAttribute("event_ts_from"),
						'event_ts_to':   child.getAttribute("event_ts_to"),
						'time_zone':     child.getAttribute("time_zone")
					};
				}
			}

			// Touring cell
			if (child.getDescendantOfName('list_tour')) {

				// Collect all camera ids
				nodes = child.getDescendantOfName('list_tour').getXML().selectNodes("//record").toArray();
				save_nodes = [];
				var save_presets = [];

				for (i = 0; i < nodes.length; i++) {
					save_nodes.push(nodes[i].getAttribute('obj_id'));
					save_presets.push(nodes[i].getAttribute('preset'));
				}

				return {
					'type':    'touring',
					'objects': save_nodes,
					'presets': save_presets,
					'delay':   child.getDescendantOfName("delay_txt_tour").getText(),
					'state':   child.getDescendantOfName("cell_btn_tour").getAttribute('tourListOpen') == "true"  // list open/close
				}
			}

			// URL cell
			if (child.getName() == 'main_url_block') {
				return {
					'type':         'url',
					'url':          child.getDescendantOfName("adressUrl").getValue(),
					'refresh':      (child.getDescendantOfName("checkboxRefresh").getRendered().children[0].checked)
						? child.getDescendantOfName("intervalSec").getValue()
						: 0
				};
			}

			// GE cell
			if (child.getName() == 'ge_cell') {
				var fix_button = child.getDescendantOfName("ge_fixoncamera");

				return {
					'type':       'ge',
					'fix_obj_id': fix_button.getAttribute("fixed_objid") || 0
				}
			}
			
			// M2D cell
			if (child.getName() == 'map_cell') {				
				var cur_position = mx.MAP.getCurrentPosition(child);

				return {
					'type':       'm2d',
					'cur_position': cur_position || ""
				}
			}

			// ELog cell
			if (child.getName() == 'eLog2Root')
				return { 'type': 'elog' };

			// OM cell
			if (child.getName() == 'om_cell') {
				nodes = child.getDescendantOfName('object_list').getChildren();
				save_nodes = [];

				for (i = 0; i < nodes.length; i++) {
					if (nodes[i].getName() != 'objrec')
						continue;
					save_nodes.push(nodes[i].getAttribute('objid'));
				}
				return {
					'type':    'om',
					'objects': save_nodes
				}
			}

			if (child.getName() == 'motionSearch') {
				return $('#' + child.getId()).data('model').serialize();
			}
			appLogger.info('Unknown block type in LSC._serialize(), block type/name: ' + child.getClass().toString() + '/' + child.getName());
		}
	},


	/**
	 * Loading helper, recursively goes through object structure and loads all splitters, cameras etc.
	 * @param dest_block     object, which is currently in process
	 * @param struct         structure of object
	 * @param live_only       convert all events to live video
	 */
	_unserialize: function(dest_block, struct, live_only) {

		if (!struct || struct.type == 'empty_block')
			return;

		if(live_only === undefined){
			live_only = false;
		}

		if(struct.type == 'event' && live_only == true){
			struct.type = 'camera';
		}

		var device;

		switch (struct.type) {
			case 'splitter':
				// switch off size checks when adding splitters
				mx.MATRIX2.addSplitter(dest_block, struct.orientation, struct.subcont_size, false);
				LSC._unserialize(dest_block.getFirstChild().getFirstChild(), struct.children[0], live_only);
				LSC._unserialize(dest_block.getFirstChild().getLastChild(),  struct.children[1], live_only);
			break;

			case 'camera':
				device = resourceTree.getObject(struct.obj);

				if(device){
					// emulate single cell drop
					mx.MATRIX2.createCell(device.obj, device.name, dest_block);
				} else {
					appLogger.info("Loading camera error: camera with objid = " + struct.obj + " doesn't exist in resource tree sets");
				}
			break;

			case 'event':
				device = resourceTree.getObject(struct.obj);
				if(device){
					// emulate event cell drop
					var event_cell = mx.MATRIX2.createCell(struct['obj'], __("Event ID: ") + struct.event_id, dest_block, false, struct.event_ts_from, struct.event_ts_to);
					event_cell.setAttribute("time_zone", struct.time_zone);

					if(!mx.ELog2.isEventExists(struct.event_id)){
						event_cell.messageSet(__("Event is missing"));
						event_cell.messageShow();
					}
				}
				else{
					appLogger.info("Loading event error: camera with obj = " + struct.obj + "doesn't exist in resource tree sets");
				}
			break;

			case 'touring':
				mx.TOUR.addTourCell(dest_block);
				if(!struct.state) {
					// TODO: correct behaviour when list is closed!
					mx.TOUR.listOpenClose(dest_block.getDescendantOfName("cell_btn_tour"));
				}

				var matrix = dest_block.getDescendantOfName('list_tour');

				//emulate touring drop
				for (var i=0; i < struct.objects.length; i++) {
					device = resourceTree.getObject(struct.objects[i]);
					if (!device)
						continue;

					mx.TOUR.listTourOnDrop(matrix, device, struct.presets[i]);
				}

				dest_block.getDescendantOfName("delay_slider_tour").setValue(struct.delay-3).repaint();
				dest_block.getDescendantOfName("delay_txt_tour").setText(struct.delay).repaint();
			break;

			case 'url':
				mx.MATRIX2.addUrlCell(dest_block, {'url': struct.url, 'refresh': struct.refresh});
			break;

			case 'ge':
				LSC.geHack = {
					block: dest_block,
					fix_obj_id: struct.fix_obj_id
				};

			break;
			
			case 'm2d':
				mx.MAP.addMap(dest_block, "position=" + JSON.stringify(struct.cur_position));
			break;

			case 'om':
				mx.OM.createOMCell(dest_block, struct.objects);
			break;

			case 'elog':
				mx.ELog2.elDrop(dest_block);
			break;

			case 'motionSearch':
				new motionSearch(dest_block, struct);
			break;

			case 'avatar_cell':
				var source = resourceTree.getObject(struct.objid);
				if(source){
					var avatarCell = new AvatarCell(dest_block, source);
					avatarCell.render();
				}
				else{
					appLogger.info('Avatar ' + struct.objid + 'not found.');
				}
			break;
		}
	},

	showConfMenu: function() {
		jsx3.GO('menuconfblock').setVisibility(jsx3.gui.Block.VISIBILITYVISIBLE, true);
	},

	hideConfMenu: function() {
		jsx3.GO('menuconfblock').setVisibility(jsx3.gui.Block.VISIBILITYHIDDEN, true);
	}
};
