
(function() {
	"use strict";

	window.AvatarOnDemand = function(block) {
		this.dom = {
			'dialog': $('#avatar_on_demand'),
			'root': block
		};
		this.requests = [];

		this._init();
	};

	window.AvatarOnDemand.prototype = {

		_init: function() {
			this.dom.root.html('<ul></ul>');
			this.dom.list = this.dom.root.find('ul');

			var self = this;
			this.dom.list
			.on('click', '.pause', function() {
				self.getRequestById($(this).closest('div.delivery_request').data('requestid')).pause();
			}).on('click', '.resume', function() {
				self.getRequestById($(this).closest('div.delivery_request').data('requestid')).resume();
			}).on('click', '.cancel', function() {
				self.getRequestById($(this).closest('div.delivery_request').data('requestid')).cancel();
			}).on('click', '.submit', function() {
				self.getRequestById($(this).closest('div.delivery_request').data('requestid')).submit();
			}).on('click', '.delete', function() {
				var requestId = $(this).closest('div.delivery_request').data('requestid');

				for (var i=0; i<self.requests.length; i++) {
					if (self.requests[i].data.requestid == requestId) {
						delete self.requests[i];
						$(this).closest('div.delivery_request').remove();
						return;
					}
				}
			});
		},

		checkStreamExistence: function(cell, player, start, end) {
			var objid = player.devObjId;

			$.ajax({
				url: '/cxf/services/storagestat/getMissedChunks',
				context: this,
				data: {
					objid: objid,
					start: Math.round(start / 1000),
					end: Math.round(end / 1000)
				},
				dataType:'json',
				async: false,
				success: function(data) {
					if (!data.missedChunks || !data.missedChunks.value || !parseInt(data.missedChunks.value, 10))
						return;

					this.show();
					groupControls.pause();


					var request = new DeliveryRequest(objid, Math.round(start / 1000), Math.round(end / 1000), function() {
						mx.MATRIX2.setSelectedGroup(cell);
						groupControls.play();
					});
					this._appendConfirmRequest(request);
				}
			});
		},

		getRequestById: function(requestId) {
			for (var i=0; i<this.requests.length; i++) {
				if (this.requests[i].data.requestid == requestId)
					return this.requests[i];
			}
		},

		show: function() {
			this.dom.dialog.show();
			this._refreshList();
		},

		hide: function() {
			this.dom.dialog.hide();
		},

		_appendConfirmRequest: function(request) {
			this.requests.push(request);
			this._renderRequest(request);
		},


		_refreshList: function() {
			var self = this;
			$.when(this.getDeliveryRequests()).done(function() {
				self._clearList();
				self._renderList();
			})
		},

		_clearList: function() {
			this.dom.list.html('');
		},

		_renderList: function() {
			for (var i = 0; i< this.requests.length; i++) {
				this._renderRequest(this.requests[i]);
			}
		},

		_renderRequest: function(request) {
			var li = $('<li data-requestid></li>');
			request.setBlock(li);
			this.dom.list.prepend(li);
		},

		getDeliveryRequests: function() {
			return $.ajax('/cxf/services/aod/get', {
				context: this,
				async: false,
				data: {
					created_from: Math.round((new Date()).getTime() / 1000) - 2*24*60*60
				},
				success: function(data) {
					this.requests = this._processRequestsList(data.requests ? data.requests : []);
				}
			});
		},

		_processRequestsList: function(requests) {
			if ($.isPlainObject(requests)) {
				requests = [requests];
			}
			var result = [];
			for (var i=0; i<requests.length; i++) {
				result.push(new DeliveryRequest(requests[i]));
			}
			return result;
		}
	};


	var DeliveryRequest = function(data, startTs, endTs, onComplete) {
		// data example:
		// {"avatarid":101,"createdTs":1335359059098,"endTs":1335356338,"nodeid":100,"objid":104,"requestid":1,"startTs":1335356000,"status":"QUEUED","userid":0}

		if ($.isPlainObject(data)) {
			this.data = data;
		} else {
			this.data = {
				requestid: 'confirm' + Math.round(),
				status: 'NOT CONFIRMED',
				objid: data,
				startTs: startTs,
				endTs: endTs
			};
			this.onComplete = onComplete;
		}

		this.dom = {block: null};
	};

	DeliveryRequest.prototype = {

		status: ['NOT CONFIRMED', 'QUEUED', 'PROCESSING', 'COMPLETED', 'CANCELED', 'PAUSED', 'FAILED'],

		getHTML: function() {
			return '<div class="delivery_request" data-requestid="' + this.data.requestid + '">' +
					((this.data.status == 'NOT CONFIRMED') ? __('The video selection requires delivery from a remote site, delivery will take some time.') + '<br />' : '') +

					'<div class="name">' + resourceTree.getObject(this.data.objid).name + '</div>' +
					'<span class="status">&lt;' + this.data.status + '&gt;</span> ' +
					'<div class="right_pane">' + this.getCommandsHTML() + '</div>' +
					'<div class="time">' + (mx.TIME.timestamp2date(this.data.startTs * 1000)) + ' - ' + (mx.TIME.timestamp2date(this.data.endTs * 1000)) + '</div>' +

				'</div>';
		},

		getCommandsHTML: function() {
			var commands = {
				'submit': '<a class="dialog_command submit">submit</a>',
				'delete': '<a class="dialog_command delete">cancel</a>',
				'pause' : '<a class="dialog_command pause">pause</a>',
				'resume': '<a class="dialog_command resume">resume</a>',
				'cancel': '<a class="dialog_command cancel">cancel</a>'
			};

			switch (this.data.status) {
				case 'NOT CONFIRMED':
					return commands['submit'] + commands['delete'];
				case 'QUEUED':
				case 'PROCESSING':
					return commands['pause'] + commands['cancel'];
				case 'PAUSED':
					return commands['resume'] + commands['cancel'];
				default:
					return '';
			}
		},

		setBlock: function(block) {
			this.block = block;
			this.repaint();
		},

		repaint: function() {
			if (!this.block)
				return;
			this.block.html(this.getHTML());
		},

		update: function() {
			if (!$.inArray(this.data.status, ['COMPLETED', 'CANCELED', 'FAILED']))
				return;

			var self = this;

			$.ajax('/cxf/services/aod/get', {
				context: this,
				data: {requestid: this.data.requestid},
				dataType: 'json',
				async: false,
				cache: false,
				success: function(data) {

					if (!data.requests || !$.isPlainObject(data.requests))
						return;

					this.data = data.requests;
					this.checkCompleted();
					this.repaint();
					setTimeout(function() {self.update()}, 2000);
				}
			});
		},

		checkCompleted: function() {
			if (this.data.status == 'COMPLETED' && this.onComplete) {
				this.onComplete.apply();
				this.onComplete = null;
			}
		},

		submit: function() {
			$.ajax('/cxf/services/aod/submit', {
				context: this,
				data: {
					objid: this.data.objid,
					start: this.data.startTs,
					end: this.data.endTs
				},
				dataType: 'json',
				async: false,
				success: function(data) {
					if (!data.requests || !$.isPlainObject(data.requests))
						return;

					this.data = data.requests;
					this.repaint();
					this.update();
				}
			});
		},

		pause: function() {
			$.ajax('/cxf/services/aod/pause', {
				cache: false,
				async: false,
				data: {requestid: this.data.requestid}
			});
		},

		resume: function() {
			$.ajax('/cxf/services/aod/resume', {
				cache: false,
				async: false,
				data: {requestid: this.data.requestid}
			});
		},

		cancel: function() {
			$.ajax('/cxf/services/aod/cancel', {
				cache: false,
				async: false,
				data: {requestid: this.data.requestid}
			});
		}
	};

})();
