
(function() {
	"use strict";

	/**
	 * @class
	 * Maintains table view of events
	 *
	 * @triggers AlertLogView.selectRow [{AlertLogEvent[]} eventList] is triggered when new events receivedAlertLogView.selectRow
	 *
	 * @param block root jquery block of view
	 * @constructor
	 */
	window.AlertLogView = function(block) {
		this.blocks = {
			root: block,
			list: null
		};

		this._render();
		this._addCallbacks();

	};

	window.AlertLogView.prototype = {

		/**
		 * Max number of rows in view
		 * @field
		 */
		MAX_ROWS: 100,
		/**
		 * Timeout of fading when event is removed from view
		 * @field
		 */
		REMOVE_TIMEOUT: 10000, // Must be defined in Identity

		/**
		 * Render initial html skeleton
		 */
		_render: function() {
			this.blocks.root.html(
				'<table id="AlertLogList">' +
				'<thead>' +
					'<tr>' +
						'<th class="expand"></th>' +
						'<th class="id">ID</th>' +
						'<th class="play"></th>' +
						'<th class="priority" title="Priority">P</th>' +
						'<th class="time">Time</th>' +
						'<th class="device">Device: UDID/Name</th>' +
						'<th class="system">System Message</th>' +
						'<th class="notes">Operator Notes</th>' +
					'</tr>' +
				'</thead>' +
				'<tbody></tbody>' +
				'</table>'
			);

			this.blocks.list = this.blocks.root.find('tbody');
		},

		/**
		 * Add nesessary callbacks.
		 */
		_addCallbacks: function() {
			var self = this;

			this.blocks.list.on('click', 'tr', function() {
				self._selectRow($(this));
				$.Topic('AlertLogView.selectEvent').publish($(this).data('event'));
			});

			this.blocks.list.on('dblclick', 'tr', function() {
				self._selectRow($(this));
				$.Topic('AlertLogView.dblSelectEvent').publish($(this).data('event'));
			});

			this.blocks.list.on('click', 'td.play', function(evt) {
				$.Topic('AlertLogView.selectPlay').publish($(this).parent().data('event'));
			});

			this.blocks.list.on('click', 'td.expand', function(evt) {
				$(this).parent().data('event').toggleNotes();
			});
		},

		/**
		 * Move holded event to the end of list
		 * @param event
		 */
		holdEvent: function(event) {
			var tr = this._getEventRow(event).detach();
			this.blocks.list.append(tr);
		},

		/**
		 * Clear list of events
		 */
		clearList: function() {
			this.blocks.list.html('');
		},

		appendEvents: function(eventList) {
			for (var i=0; i<eventList.length; i++) {
				this._appendEvent(eventList[i]);
			}
		},

		/**
		 * Remove event from list with timeout defined by REMOVE_TIMEOUT
		 * @param {AlertLogEvent} event
		 */
		removeEvent: function(event) {
			var self = this;
			setTimeout(function() {
				var tr = self._getEventRow(event);
				if (!tr.size())
					return;

				$.Topic('AlertLogView.removeEvent').publish(tr.data('event'));
				tr.fadeOut('slow', function() {$(this).remove();});
			}, this.REMOVE_TIMEOUT);

		},

		/**
		 * Append one event at head of list
		 * @param {AlertLogEvent} event
		 */
		_appendEvent: function(event) {
			this.blocks.list.prepend(event.render());
			if (this.blocks.list.find('tr').size() > this.MAX_ROWS) {
				var tr = this.blocks.list.find('tr:last');
				$.Topic('AlertLogView.removeEvent').publish(tr.data('event'));
				tr.remove();
			}
		},

		/**
		 * Set style of particular row in table to be selected
		 * @param row
		 */
		_selectRow: function(row) {
			this.blocks.list.find('tr.selected').removeClass('selected');
			row.addClass('selected');
		},

		/**
		 * Get row element representing event object
		 * @param {AlertLogEvent} event
		 * @return {jQuery}
		 */
		_getEventRow: function(event) {
			return this.blocks.list.find('tr[data-eventid="' + event.data.id + '"]');
		}

	};

})();
