
(function() {
	"use strict";

	window.AlertLogPlayer = function(block) {

		this.block = block;
		this.currentEvent = null;
		this.player = null;
		this.updateTimeHandler = null;

		var self = this;

		block.html(
			'<table>' +
				'<tr class="header">' +
					'<td class="title"></td>' +
					'<td class="priority"></td>' +
					'<td class="eventid"></td>' +
				'</tr>' +
				'<tr class="content">' +
					'<td colspan="3" class="player"></td>' +
				'</tr>' +
				'<tr class="footer">' +
					'<td colspan="3">' +
						'<div class="controls">' +
							'<div class="button play"></div>' +
						'</div>' +
						'<div class="timeline">' +
							'<input type="text" min="0" max="100" value="0" />' +
							'<div class="from"></div>' +
							'<div class="to"></div>' +
						'</div>' +
						'<div class="when"></div>' +
						'<div class="time"></div>' +
					'</td>' +
				'</tr>' +
			'</table>'
		);

		this.blocks = {
			player: this.block.find('.player'),
			title: this.block.find('.title'),
			priority: this.block.find('.priority'),
			eventid: this.block.find('.eventid'),
			controls: this.block.find('.controls'),
			timeline: this.block.find('div.timeline input'),
			from: this.block.find('div.from'),
			to: this.block.find('div.to'),
			when: this.block.find('div.when'),
			time: this.block.find('div.time')
		};

		this.clear();
		this.addCallbacks();
		this.blocks.timeline.rangeinput({
			css: {
				input:  'range-input',
				slider: 'range-slider',
				progress: 'range-progress',
				handle: 'range-handle'},
			precision: 0.01
		}).change(function(event, value) {
			self.onTimelineChange(value);
		});

		return this;
	};

	window.AlertLogPlayer.prototype = {

		TS_UPDATE_INTERVAL: 500,

		STATUS: {
			0: 'PAUSE',
			1: 'PLAY',
			2: 'PLAY',
			3: 'STOP'
		},

		setEvent: function(event) {

			var h = this.blocks.player.innerHeight() - 4,
				w = this.blocks.player.innerWidth(),
				self = this;

			this.currentEvent = event;

			this.blocks.title.text(event.getEventName());
			this.blocks.priority.css('backgroundColor', event.getPriorityBGColor()).text(event.getPriorityName().slice(0, 1));
			this.blocks.eventid.text(event.data.shortId);

			this.blocks.player.html(
				'<iframe style="width: ' + w + 'px; height: ' + h + 'px; border: 0" frameborder="no" scrolling="no" src="cell_player.php?' +
					'name=alertlog_player' +
					'&camera=' + event.data.obj.objid +
					'&width=' + w +
					'&height=' + h +
					'&imagecallbacks=1' +
					'&eventid=' + event.data.id +
					'&ts_from=' + core.time.formatUTCDate(event.data.utc.from, 'YYYY-MM-DD HH-II-SS') +
					'&ts_to=' + core.time.formatUTCDate(event.data.utc.to, 'YYYY-MM-DD HH-II-SS') +
				'"></iframe>'
			);

			this.blocks.from.html(core.time.timestamp2date(this.currentEvent.data.utc.from).replace(' ', '<br />'));
			this.blocks.to.html(core.time.timestamp2date(this.currentEvent.data.utc.to).replace(' ', '<br />'));
			this.blocks.when.text('0:00/0:00');

			$.when(this.playerLoaded(this.blocks.player.find('iframe')[0])).done(
				function() {
					self.player = self.blocks.player.find('iframe')[0].contentWindow._mplayer_alertlog_player;

					self.player.listener.onNewImage = function(player, ts) {self.updateTS(ts);};
					self.player.listener.onStatusChange = function() {self.onStatusChange(arguments[2]);};

					//self.callUpdateTS();

					self.toggleControls(true);
				}
			)
		},

		clear: function() {
			this.blocks.title.html('&lt;Empty&gt;');
			this.blocks.priority.css('backgroundColor', 'transparent').text('');
			this.blocks.eventid.text('');

			this.block.find('.from, .to, .when, .clock').text('');

			if (this.updateTimeHandler) {
				clearTimeout(this.updateTimeHandler);
			}

			this.blocks.player.find('iframe').remove();
			this.toggleControls(false);
		},

		getEventId: function() {
			if (!this.currentEvent)
				return null;

			return this.currentEvent.data.id;
		},

		toggleControls: function(active) {
			var buttons = this.blocks.controls.find('div.button');

			if (active) {
				buttons.removeClass('disabled');
				this.blocks.timeline.prop('disabled', false);
			} else {
				buttons.addClass('disabled');
				this.blocks.timeline.prop('disabled', true);
			}
		},

		togglePlayPause: function(button) {
			if (button == 'play') {
				this.blocks.controls.find('div.button.pause').removeClass('pause').addClass('play');
			} else {
				this.blocks.controls.find('div.button.play').removeClass('play').addClass('pause');
			}
		},

		addCallbacks: function() {

			var self = this;
			this.blocks.controls.find('div.button').click(function() {

				if ($(this).hasClass('disabled'))
					return;

				if ($(this).hasClass('play')) {
					self.togglePlayPause('play');

					if (self.player.getCurrentTS() >= self.currentEvent.data.utc.to) {
						self.player.play(1, 1, self.currentEvent.data.utc.to + 15);
					} else {
						self.player.play(1, 1);
					}
				}

				if ($(this).hasClass('pause')) {
					self.togglePlayPause('play');
					self.player.pause();
				}
			});
		},

		/**
		 * Update current time in tineline
		 */
		updateTS: function(ts) {
			var eTime = this.currentEvent.data.utc;
			//ts = this.player.getCurrentTS();

			if (!ts) {
				this.callUpdateTS();
				return;
			}

			if (!this.blocks.timeline.data('isDragged')) {
				this.blocks.timeline.data('rangeinput').setValue(Math.round((ts - eTime.from) / eTime.duration * 100));
			}

			this.blocks.when.text(core.time.formatUTCDate(ts - eTime.from, 'HH?:II:SS') + '/' + core.time.formatUTCDate(eTime.duration, 'HH?:II:SS'));
			this.blocks.time.html(core.time.timestamp2date(ts).replace(' ', '<br />'));

			//this.callUpdateTS();
		},

		callUpdateTS: function() {
			var self = this;
			this.updateTimeHandler = setTimeout(function() {self.updateTS();}, this.TS_UPDATE_INTERVAL);
		},

		onTimelineChange: function(value) {
			var eTime = this.currentEvent.data.utc,
				ts = Math.round(eTime.duration * value / 100 / 1000 * 1000) + eTime.from;

			//this.player.playOneFrameTS(ts);
			this.player.stop();
			this.player.play(1, 1, ts);

			this.waitForPlay = ts;
		},

		onStatusChange: function(status) {
			switch (this.STATUS[status]) {
				case 'PAUSE':
					this.togglePlayPause('play');
					if (this.waitForPlay) {
						this.player.play(1, 1, this.waitForPlay);
						this.waitForPlay = 0;
					}
					break;
				case 'PLAY':
					this.togglePlayPause('pause');
					break;
				case 'STOP':
					break;
			}
		},


		playerLoaded: function(iframe, d) {
			if (!d) {
				d = new $.Deferred();
			}

			if (!iframe || !iframe.contentWindow || !iframe.contentWindow._mplayer_alertlog_player) {
				var self = this;
				setTimeout(function() {self.playerLoaded(iframe, d)}, 100);
			} else {
				d.resolveWith(this);
			}

			return d.promise();
		}
	};
})();
