(function() {

	"use strict";

	/**
	 * @class AlertLogModel
	 * Bridge for Elog applet in Matrix, providing convenient way to filter events from applet
	 *
	 * @triggers AlertLogModel.newEvents [{AlertLogEvent[]} eventList] is triggered when new events received
	 * @triggers AlertLogModel.statusChange is triggered when Elog applet status changed
	 *
	 * @constructor
	 * @param {Node} elogModel DOM node with Elog Java applet
	 */
	window.AlertLogModel = function(elogModel) {

		/**
		 * Link to the Elog applet
		 * @field
		 */
		this._model = elogModel;

		/**
		 * events cache by their id
		 * @field
 		 */
		this.events = {};

		/**
		 * events cache by their shortId
		 * @field
		 */
		this.eventsByShortId = {};

		/**
		 * timer handler to control requests process
		 * @field
		 */
		this.timerHandle = null;

		return this;
	};


	window.AlertLogModel.prototype = {

		/**
		 * Interval between continuous update requests for the ELog, ms.
		 * @default 1000
		 */
		UPDATE_INTERVAL: 1000,

		/** PUBLIC **/

		/**
		 * Set events filter and start making requests for events
		 * @param {Object} filter filter object as given by AlertLogFilter.getFilterSettings
		 */
		start: function(filter) {
			this._setEventFilter(filter);
			this._startUpdateCycle();
		},

		/**
		 * Stop all requests and clear event cache
		 */
		stop: function() {
			this._stopUpdateCycle();
			this.events = {};
			this.eventsByShortId = {};
		},

		/**
		 * Set new event filter and restart model
		 * @param {Object} filter new event filter object as given by AlertLogFilter.getFilterSettings
		 */
		applyFilter: function(filter) {
			this.stop();
			this.start(filter);
		},

		/**
		 * get event object by its id
		 * @param {Integer} eId event id
		 */
		getEvent: function(eId) {
			return this.events[eId];
		},

		/**
		 * get event object by its shortid
		 * @param {Integer} eId short event id from event.data.shortId
		 */
		getEventByShortId: function(eId) {
			return this.eventsByShortId[eId];
		},

		/**
		 * remove event from cache
		 * @param {AlertLogEvent} event
		 */
		removeEvent: function(event) {
			if (this.events[event.data.id]) {
				delete this.events[event.data.id];
			}
			if (this.eventsByShortId[event.data.shortId]) {
				delete this.eventsByShortId[event.data.shortId];
			}
		},

		/**
		 * Submit request for elog applet
		 * @param {Object} data data in special format to be submitted
		 */
		submitData: function(data) {
			this._model.submitAction(data.name, dara.parameters, data.properties, data.actions)
				.fail(function(code, message){
					this._checkError(message);
				});
		},

		/* PRIVATE */
		/**
		 * Set event filters for elog applet
		 * @param s object representing filter settings
		 */
		_setEventFilter: function(s) {
			this._model.setEventFilter(s.tag, s.tagOnFlag, s.eventType, s.priority, s.objids, s.message, s.startTs, s.endTs, s.pageNum, s.pageSize);
		},

		/**
		 * Start update cycle;
		 * first call queryEvents(true)
		 * all other calls queryEvents(false) to receive only increment changes
		 */
		_startUpdateCycle: function() {
			var self = this;

			// Need to set timeout because if we call queryEvents(true) immediately, it will cause errors
			this.timerHandle = setTimeout(function() {
				self._queryEvents(true);

				self.timerHandle = setInterval(function() {
					self._queryEvents(false);
				}, self.UPDATE_INTERVAL);
			}, self.UPDATE_INTERVAL);
		},

		/**
		 * Stop sending update requests to applet
		 */
		_stopUpdateCycle: function() {
			clearInterval(this.timerHandle);
			this.timerHandle = null;
		},

		/**
		 * Make request to applet to get new events or changes for existing events
		 * @param getAll boolean if true then get complete set of events; otherwise get incremental changes
		 */
		_queryEvents: function(getAll) {
			var alerts = $.parseJSON(String(this._model.getEvents(getAll))),
				event, newEventList = [], i;

			// TODO: check status and Events field
			alerts = alerts.Events;

			for (i = 0; i < alerts.length; i++) {

				if (this.events[alerts[i].id]) {
					// Event exists and changed
					this.events[alerts[i].id].update(alerts[i]);
					continue;
				}

				event = new AlertLogEvent(alerts[i], this);

				this.events[event.data.id] = this.eventsByShortId[event.data.shortId] = event;

				// Aggregate all new events into array to publish all of them at the same time
				newEventList.push(event);
			}

			if (newEventList.length) {
				$.Topic('AlertLogModel.newEvents').publish(newEventList);
			}
		},

		/**
		 * Check applet answer for errors
		 * @param response applet response
		 * @throws Error
		 */
		_checkError: function(response) {

			var alter = {}, i;
			response = $.parseJSON(String(response)).response;

			for (i = 0; i < response.length; i++) {
				alter[response[i].name] = response[i].value;
			}

			if (alter['STATUS'] == 'ERROR') {
				throw new Error('AlertLogModel: ' + alter['MSG']);
			}
		}
	};

})();
