
(function() {
	"use strict";

	/**
	 * @class
	 * Incapsulates logic connected with eventlog filtering
	 *
	 * @triggers AlertLogFilter.update [{Object} filter] triggers when filter value is changed
	 *
	 * @constructor
	 * @param {Object} [settings = {priority: 1, message: '', state: 'all', device: 'full'}] settings which define initial filter
	 */
	window.AlertLogFilter = function(settings) {

		/**
		 * Internal filter object; describes all filtering parameters needed for ELog applet
		 */
		this._modelFilter = {
			tag: 0,
			tagOnFlag: false,
			eventType: 1,
			objids: [],
			startTs: 0,
			endTs: 0,
			pageNum: 0,
			pageSize: 100,
			priority: 1,
			message: ''
		};

		/**
		 * Simplified filter object; describes simplified schema of filtering, which can be translated to internal representation
		 * This object is used to represent filter for user
		 */
		this.filter = $.extend({}, this.DEFAULT_FILTER);
		this.setFilter(settings);

		return this;
	};

	window.AlertLogFilter.prototype = {

		/**
		 * Default filter value
		 */
		DEFAULT_FILTER: {
			state: 'all',
			priority: 1,
			device: 'full',
			message: ''
		},

		/**
		 * @param {Object} [settings = {priority: 1, message: '', state: 'all', device: 'full'}] settings which define initial filter
		 */
		setFilter: function(settings) {
			settings = this._validateInput(settings);

			$.extend(this.filter, settings);
			this._translateFilter();

			$.Topic('AlertLogFilter.update').publish($.extend({}, this._modelFilter));
		},

		/**
		 * Get inner filter value
		 * @return {Object}
		 */
		getModelFilter: function() {
			return $.extend({}, this._modelFilter);
		},

		/**
		 * Validates input object for setFilter()
		 * @param {Object} input
		 * @return {object} purified and reformed input object
		 * @throws Error
		 */
		_validateInput: function(input) {
			// Check input values
			if (input.priority) {
				input.priority = parseInt(input.priority, 10);
				if (!input.priority)
					throw new Error('Incorrect value of settings.priority: ' + input.priority);
			}

			if (input.state && $.inArray(input.state, ['all', 'not_acknowledged', 'not_cleared', 'cleared']) == -1)
				throw new Error('Incorrect value of settings.state: ' + input.state);

			if (input.device && $.inArray(input.device, ['full', 'filtered', 'selected']) == -1)
				throw new Error('Incorrect value of settings.device: ' + input.device);

			// Delete incorrect fields
			for (var key in input) {
				if ($.inArray(key, ['state', 'device', 'priority', 'message']) == -1) {
					delete input[key];
				}
			}

			return input;
		},


		/**
		 * translate external filter into internal model
		 */
		_translateFilter: function() {
			this._translateMessage();
			this._translatePriority();
			this._translateDevice();
			this._translateState();
		},

		/**
		 * Translate user filter priority into inner representation
		 */
		_translatePriority: function() {
			this._modelFilter.priority = this.filter.priority;
		},

		/**
		 * Translate user filter device into inner filter objids
		 */
		_translateDevice: function() {
			switch (this.filter.device) {
				case 'full':
					this._modelFilter.objids = [];
					break;
				case 'filtered':
					// Not implemented yet
					break;
				case 'selected':
					// Not implemented yet
					break;
			}
		},

		/**
		 * Translate user filter state into inner eventType, tag, tagOnFlag filters
		 */
		_translateState: function() {
			switch (this.filter.state) {
				case 'all':
					this._modelFilter.eventType = 777;
					this._modelFilter.tag = 0;
					this._modelFilter.tagOnFlag = false;
					break;
				case 'not_cleared':
					this._modelFilter.eventType = 1;
					this._modelFilter.tag = 8;
					this._modelFilter.tagOnFlag = false;
					break;
				case 'not_acknowledged':
					this._modelFilter.eventType = 1;
					this._modelFilter.tag = 4;
					this._modelFilter.tagOnFlag = false;
					break;
				case 'cleared':
					this._modelFilter.eventType = 1;
					this._modelFilter.tag = 8;
					this._modelFilter.tagOnFlag = true;
					break;
			}
		},

		/**
		 * Translate user filter message into inner message filter
		 */
		_translateMessage: function() {
			this._modelFilter.message = this.filter.message;
		},

		/**
		 * Check whether event fits in current filter
		 * @param {AlertLogEvent} event
		 * @return Boolean
		 */
		fit: function(event) {

			//if device not in filter
			if (this._modelFilter.objids.length > 0 && this._modelFilter.objids.indexOf(event.data.objid) == -1)
				return false;

			//if alerts mode but event has informational type
			if (this.filter.state != 'all' && event.data.eventtype == 0)
				return false;

			//if event cleared
			if (this.filter.state == 'not_cleared' && event.checkTag('Cleared'))
				return false;

			//if event acknowledged
			if (this.filter.state == 'not_acknowledged' && (event.checkTag('Cleared') || event.checkTag('Acknowledged')))
				return false;

			//if event not cleared
			if (this.filter.state == 'cleared' && !event.checkTag('Cleared'))
				return false;

			return true;
		}
	}

})();
