
(function() {
	"use strict";

	/**
	 * @class
	 * Incapsulates logic of one event in EventLog
	 *
	 * @triggers AlertLogEvent.update [{AlertLogEvent} event] triggers when event is updated from its model
	 *
	 * @constructor
	 * @param {Object} data
	 * @param {AlertLogModel} model
	 */
	window.AlertLogEvent = function(data, model) {

		this.mapData(data);

		/**
		 * Event data as it given by applet
		 * @field
		 */
		this.data = data;

		/**
		 * Link with AlertLogModel object; this is necessary when saving event data
		 * @field
		 */
		this._model = model;

		return this;
	};

	AlertLogEvent.prototype = {

		// quantity of symbols in short representation of message and operator notes in row
		CUT_QTY: 100,

		/**
		 * Correlation with tag ids and their human interpretations
		 */
		TAGS: {
			'needannouncement':   1,
			'needsupervisor':     2,
			'needack':            3, // need acknowledgement
			'ack':                4, // acknowledged
			'updatable':          5,
			'eada':               6,
			'clearable':          7,
			'cleared':            8,
			'final':              9,
			'popup':              10,
			'marked':             11
		},

		PRIORITIES: {
			1: {colorcode: "#00FF00", colorname: "Green", description: "Low Priority", eventpriority: 1, name: 'Low'},
			2: {colorcode: "#FFFF00", colorname: "Yellow", description: "Normal Priority", eventpriority: 2, name: 'Normal'},
			3: {colorcode: "#F88017", colorname: "Orange", description: "High Priority", eventpriority: 3, name: 'High'},
			4: {colorcode: "#FF0000", colorname: "Red", description: "Critical Priority", eventpriority: 1, name: 'Critical'}
		}, // this dictionary updates form server

		/**
		 * Check whether tag passed by its human description (prototype.TAGS) is set for current event
		 * @param {String} tagDescr human description of tag, one of the AlertLogEvent.prototype.TAGS keys
		 * @return boolean whether tag is set for current event
		 */
		checkTag: function(tagDescr) {
			tagDescr = tagDescr.toLowerCase();
			return this.TAGS[tagDescr] && $.inArray(this.TAGS[tagDescr], this.data.tags) !== -1;
		},


		/**
		 * Maps data given by elog model into inner representation;
		 * converts timestamps from s to ms
		 */
		mapData: function(data) {
			var i, props = {};

			// Convert s to ms for convenient work with dates
			data.utc = {
				from: data.utc.from * 1000,
				to:   data.utc.to * 1000,
				when: data.utc.when * 1000,
				duration: (data.utc.to - data.utc.from) * 1000
			};

			// compute shortId
			data.shortId = this.calculateShortId(data.id);

			// Convert properties to convenient form
			for (i = 0; i<data.properties.length; i++) {
				props[data.properties[i].name] = data.properties[i].value;
			}
			data.properties = props;
		},

		update: function(changes) {

			this.mapData(changes);

			$.Topic('AlertLogEvent.update').publish([this, this.getChanges(changes)]);
			this.data = $.extend(true, this.data, changes);

			if (!this.block)
				return;

			this.block.html(this.getHTML());
			this.updateBackground();
		},

		getChanges: function(changes) {

			var result = $.extend({
				id: this.data.id, shortId: this.data.shortId
			}, this.getDiff(this.data, changes));

			if (result.priority) {
				result.priorityName = this.getPriorityName(result.priority);
			}

			return result;
		},

		/**
		 * @param {Integer} [priority = this.data.priority] code of priority
		 * @return {String} priority name
		 */
		getPriorityName: function(priority) {
			priority = priority || this.data.priority;
			return this.PRIORITIES[priority].name;
		},

		getPriorityBGColor: function(priority) {
			priority = priority || this.data.priority;
			return this.PRIORITIES[priority].colorcode;
		},

		getModel: function() {
			return this._model;
		},

		/**
		 * Get priorities dictionary from server, must be run at initialization
		 */
		updatePrioritiesDictionary: function() {
			$.ajax({
				url: "/api/call.php",
				context: AlertLogEvent.prototype,
				async: false,
				cache: false,
				dataType: "json",
				data: {'function': 'getEventPriorities'},
				success: function(data) {
					var i, pr;

					if (data.error)
						throw new Error('Error initializing priorities dictionary: ' + data.error);



					this.PRIORITIES = {};
					for (i = 0; i < data.list.length; i++) {
						pr = data.list[i];
						this.PRIORITIES[pr.eventpriority] = pr;
					}

					debugger;
				}
			});
		},

		calculateShortId: function(id) {
			return this.calculateHash(id) + String(id).substr(-3);
		},

		getWitnessesList: function() {
			var result = [this.data.objId];

			for (var i=0; i<this.data.witnesses.length; i++) {
				result.push(this.data.witnesses[i].objid);
			}

			return result;
		},

		getWitnesses: function() {
			var result = {};
			result[this.data.objId] = this.data.obj;

		    for (var i=0; i<this.data.witnesses.length; i++) {
		        result[this.data.witnesses[i].objid] = this.data.witnesses[i];
		    }

		    return result;
		},

		getEventName: function() {
			return this.data.obj.type + this.data.obj.subtype + this.data.obj.objid + ' ' + this.data.obj.name;
		},

		/**
		 * Get hash of string
		 * @param k - string to be hashed
		 * @return hash
		 */
		calculateHash: function(k) {
			var h = 0;

			// make string
			k = String(k);

			for (var i=0; i<k.length; i++) {
				h += parseInt(k[i]);
				h += ( h << 10 );
				h ^= ( h >> 6 );
			}

			h += ( h << 3 );
			h ^= ( h >> 11 );
			h += ( h << 15 );

			// Returning only one digit for "checksum"
			return String(h)[5];
		},


		/**
		 * View functions
		 */
		render: function() {
			this.block = $(
				'<tr class="AlertLogEvent" data-eventid="' + this.data.id + '">' + this.getHTML() + '</tr>'
			).data('event', this);

			this.updateBackground();

			return this.block;
		},

		getHTML: function() {
			return  '<td class="expand">+</td>' +
				'<td class="shortId">' + this.data.shortId + '</td>' +
				'<td class="play">&rArr;</td>' +
				'<td class="priority" style="background-color: ' + this.getPriorityBGColor() + '">' + this.getPriorityName().slice(0, 1) + '</td>' +
				'<td>' + core.time.timestamp2date(this.data.utc.when) + '</td>' +
				'<td>' + this.data.obj.name+ '</td>' +
				'<td class="message">' + this.cutNotes(this.data.msg) + '</td>' +
				'<td class="note">' + this.cutNotes(this.data.note) + '</td>';
		},


		updateBackground: function() {

			var tags = ['Cleared', 'Ack', 'NeedAck', 'Marked'];

			this.block
				.css('background-color', '').removeClass(tags.join(' ')).removeClass('hold');

			if (this.data.bgcolor) {
				this.block.css('background-color', this.data.bgcolor);
				return;
			}

			if (this.data.eventtype == 0)
				return;

			if (this.data.properties.hold) {
				this.block.addClass('hold');
				return;
			}

			for (var key in tags) {
				if (this.checkTag(tags[key])) {
					this.block.addClass(tags[key].toLowerCase());
					return;
				}
			}
		},

		/**
		 * @param {Integer} [priority = this.event.data.priority] code of priority
		 */
		getPriorityHtml: function(priority) {
			priority = priority || this.data.priority;
			var obj = this.PRIORITIES[priority];

			if (!obj)
				return null;

			return '<span style="background-color: ' + obj.colorcode + ';">' + obj.name.slice(0,1) + '</span>';
		},

		toggleNotes: function() {
			var expand = this.block.find('td.expand');
			if (expand.text() == '+') {
				expand.text('-');
				this.block.find('td.message').text(this.data.msg);
				this.block.find('td.note').text(this.data.note);
			} else {
				expand.text('+');
				this.block.find('td.message').text(this.cutNotes(this.data.msg));
				this.block.find('td.note').text(this.cutNotes(this.data.note));
			}
		},

		cutNotes: function(text) {
			if (text.length > this.CUT_QTY)
				return text.substr(0, this.CUT_QTY) + '...';
			else
				return text;
		},


		getDiff: function(oldObj, newObj) {
			var diff = {};

			for (var k in oldObj) {
				if (!(k in newObj))
					diff[k] = undefined;  // property gone so explicitly set it undefined
				else if (oldObj[k] !== newObj[k]) {
					if (($.isPlainObject(oldObj[k]) || $.isArray(oldObj[k])) && ($.isPlainObject(newObj[k]) || $.isArray(oldObj[k]))) {
						var tmpObj = this.getDiff(oldObj[k], newObj[k]);
						if (!$.isEmptyObject(tmpObj)) {
							diff[k] = tmpObj;
						} else
							continue;
					}
					diff[k] = newObj[k];  // property in both but has changed
				}
			}

			for (k in newObj) {
				if (!(k in oldObj))
					diff[k] = newObj[k]; // property is new
			}

			return diff;
		}
	};

	//AlertLogEvent.prototype.initPrioritiesDictionary();

	/**
	 * @class EventMapper
	 * adapter for passing event changes to Elog applet
	 * Method calls can be chained
	 *
	 * @example
	 * var mapper = new EventMapper(event);
	 * mapper
	 *   .setPriority(2)
	 *   .setNote('new user note')
	 *   .setAction('clear')
	 *   .save();
	 *
	 * @constructor
	 * @param {AlertLogEvent} event
	 */
	window.EventMapper = function(event) {

		this._event = event;

		this.changes = {};
		this._flushChanges();

		return this;
	};

	EventMapper.prototype = {

		/**
		 * List of allowed actions.
		 * Every action is associated with event tags for it to be allowed
		 */
		ACTIONS: {
			'update': 'updatable',
			'clear': 'clearable',
			'ack': 'needack'
		},
		// there are two special actions: 'eada' and 'create' to call eada actions and create new event;

		/**
		 * @private
		 * set object state to initial state; clear all previous changes
		 */
		_flushChanges: function() {
			this.changes = {
				name: 'update',
				parameters: {eventid: parseInt(this._event.data.id)},
				actions: [],
				properties: {}
			};
		},

		/**
		 * Set action to be performed
		 * @param {String} action string representing action; can be one of this.ACTIONS keys
		 */
		setAction: function(action) {

			if (this.ACTIONS[action] && this._event.checkTag(this.ACTIONS[action])) {
				this.changes.name = action;
			} else {
				// todo: raise exception?
			}
			return this;
		},

		/**
		 * set event priority
		 * @param priority
		 */
		setPriority: function(priority) {
			this.changes.parameters.priority = priority;
			return this;
		},

		/**
		 * set event note
		 * @param note
		 */
		setNote: function(note) {
			this.changes.parameters.note = note;
			return this;
		},

		/**
		 * set 'hold' parameter of event
		 */
		hold: function(userId) {
			this.changes.properties.hold = userId || 1;
			return this;
		},

		/**
		 * submit changes to Elog applet
		 */
		save: function() {
			if (!$.isEmptyObject(this.changes) && this.changes.name) {
				try {
					this._event.getModel().submitData(this.changes);
				} catch(e) {

				}
			}
			this._flushChanges();
			return this;
		}
	}

})();