
(function() {
	"use strict";

	/**
	 * @class
	 * Incapsulates logic connected with command line parsing and processing changes to events
	 */

	/**
	 *
	 * @param {jQuery} block block in which commander must accomodate
	 * @param {AlertLogModel} model
	 * @constructor
	 */
	window.AlertLogCommander = function(block, model) {
		/**
		 * @field
		 */
		this.root = block;
		this.input = null;

		/**
		 * @field
		 */
		this.model = model;

		this.render();
		this.addCallbacks();

	};

	window.AlertLogCommander.prototype = {

		// list of command-parsing regexps
		REGEXP: {
			ID: /^\d+/,              // find an integer at the beginning of command
			PRIORITY: /\*\s*([1234])/, // find priority command, supposed to be exactly one digit 1-4 after the asterisk
			HOLD:  /\-$/, // supposed to be at a tail of command
			CLEAR: /\/$/, // suppose that clear symbol always must be at the tail of command
			MESSAGE: /\+\s*(.*)/
		},

		/**
		 * Init commander in specified block
		 */
		render: function() {

			this.root.html(
				'Command Input Area: ' +
				'<div class="toolbar">? Help</div>' +
				'<textarea style="width: 100%;"></textarea>'
			);

			this.input = this.root.find('textarea');

		},

		addCallbacks: function() {
			var self = this;
			this.input.keydown(function(e) {
				switch (e.keyCode) {
					case 13: self.processCommand();
						e.preventDefault();
						break;
					case 27: self.clearInput();
						break;
				}
				e.stopPropagation();
			});
		},

		/**
		 * Clear command line
		 */
		clearInput: function() {
			this.input.val('');
		},

		setEventId: function(eId) {
			// Do not overwrite existing command if setting the same eventid
			if (this.getEventIdFromCommand(this.input.val()) == eId)
				return;

			this.setCommand(eId);
		},

		/**
		 * Fill command line with given command
		 * @param {String} command command line
		 */
		setCommand: function(command) {
			this.input.val(command);
			this.input.focus();
		},


		/**
		 * Parse and process current command in input field
		 */
		processCommand: function() {
			var command = $.trim(this.input.val()),
				event = this.getCommandEvent(command),
				clearInput = false;

			if (!event)
				return;

			var mapper = new EventMapper(event);
			mapper.setAction('update');

			// Process priority
			var priority = this.REGEXP.PRIORITY.exec(command);
			if (priority) {
				command = command.replace(this.REGEXP.PRIORITY, '');
				mapper.setPriority(priority[1]);
			}

			// Process hold and clear
			if (this.REGEXP.HOLD.test(command)) {
				mapper.hold(resourceTree.getUser().obj);
				command = $.trim(command.slice(0, -1));
				clearInput = true;
			}

			if (this.REGEXP.CLEAR.test(command)) {
				mapper.setAction('clear');
				command = $.trim(command.slice(0, -1));
				clearInput = true;
			}

			var message = this.REGEXP.MESSAGE.exec(command);
			if (message) {
				// TODO: prepend with time
				mapper.setNote(message[1]);
			}

			mapper.save();
			if (clearInput) {
				this.clearInput();
			} else {
				this.setCommand(event.data.shortId);
			}
		},

		/**
		 *
		 * @param {String} command
		 * @return {Integer} Null if event id was not found
		 */
		getEventIdFromCommand: function(command) {
			var result = this.REGEXP.ID.exec(command);
			if (result)
				return result[0];
			return null;
		},


		/**
		 * Get event object by its shortId
		 */
		getCommandEvent: function(command) {
			var eId = this.getEventIdFromCommand(command);
			if (!eId)
				return null;

			return this.model.getEventByShortId(eId);
		}
	};

})();
