package SM::Spindle;

use 5.008008;
use strict;
use warnings;

require Exporter;

our @ISA = qw(Exporter);

# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.

# This allows declaration	use SM::Spindle ':all';
# If you do not need this, moving things directly into @EXPORT or @EXPORT_OK
# will save memory.
our %EXPORT_TAGS = ( 'all' => [ qw(
 spawn	
) ] );

our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

our @EXPORT = qw(
 spawn
);

our $VERSION = '0.01';


# Preloaded methods go here.
use Carp qw(croak carp);
use POE qw/Wheel::FollowTail/;

# CONS ------------------------------------------------------------------------
my $SMSTAT;
my $LOG;

# FINITE STATE MASHINE ========================================================
sub spawn {    # constructor
 my ($class, %args) = @_;
 my $uuid = delete $args{ID}          || croak "ERR:ID spidle uuid required";
 $LOG     = delete $args{LOG}         || croak "ERR:LOG handler required";
 $SMSTAT  = delete $args{SMSTAT}      || croak "ERR:SMSTAT directory required";
 my $mount= delete $args{ACT_MOUNT}   || croak "ERR:ACT_MOUNT required";
 my $start= delete $args{ACT_START}   || croak "ERR:ACT_START required";
 my $stop = delete $args{ACT_STOP}    || croak "ERR:ACT_STOP  required";
 my $pid  = delete $args{ACT_PID}     || croak "ERR:ACT_PID   required";
 my $info = delete $args{ACT_INFO}    || croak "ERR:ACT_INFO  required";
 my $empty= delete $args{ACT_ISEMPTY} || croak "ERR:ACT_ISEMPTY required";
 my @states = qw( _start _stop Pid Start Stop Mount Degraded Status 
                  Tight Weak Cool Try Failed IsEmpty
                  Online Offline Suspend
                  Confound Full Broken Alien Unmarked
                  Unload Empty );
 POE::Session->create(
 #----------------------------------
  package_states =>[ (__PACKAGE__) =>\@states],
  heap => {ID=>$uuid,OST=>'INIT',CST=>'INIT',FAILED_LIMIT=>3,FAILED_COUNT=>0,
           ACT_MOUNT=>$mount,ACT_START=>$start,ACT_STOP=>$stop,ACT_PID=>$pid,
           ACT_INFO=>$info,ACT_ISEMPTY=>$empty,sid=>'['.substr($uuid,0,8).']'
  },
 );

}

#----------------------------------
# spindle session supplementary 

sub Prf  { "$_[HEAP]->{sid} $_[HEAP]->{CST}/".$_[HEAP]->{OST}.'->'.$_[STATE] }

sub sinfo{ 
  return if  $_[HEAP]->{CST} eq 'INIT';   # do not report INIT actions
  my $msg="$_[HEAP]->{OST} -(".$_[HEAP]->{CST}.")-> $_[STATE]";
  $LOG->info($msg);
  $_[HEAP]->{ACT_INFO}->($_[HEAP]->{ID},$msg);
}

sub read_OST {             # reads operation state from file
                           # returns Offline if ost does not exist
                           # returns Online if ost was 'Degraded' or 'Full'
  my $id=shift;
  return 'Offline' if not open (OST,"$SMSTAT/$id.ost");
  my $ost=<OST>;
  close OST;
  chomp $ost;
  return 'Offline' if not $ost=~/^(Degraded|Online|Start|Stop|Mount|Failed|Full|Suspend|Empty)$/;
  return 'Online'  if     $ost=~/^(Degraded|Full)$/;   # also online states
  return $ost;
}

sub set_OST {              # set heap->OST and write into $ID.ost
   my ($heap,$ost)=@_;
   $heap->{OST}=$ost;
   open (OST,">$SMSTAT/$heap->{ID}.ost");  # TBD: errors
   print OST "$ost\n";
   close OST;
   system("touch $SMSTAT/.updated");       # set update flag
}

#------------------------------------
# SPINDLE STATES

sub _default { $LOG->info("ERR: unexpected '$_[ARG0]'") }

sub _start {
    $LOG->info(Prf(@_)." for ID=$_[HEAP]->{ID}");
    my $last_OST = read_OST($_[HEAP]->{ID});
    $_[HEAP]->{cst_wheel} =
      POE::Wheel::FollowTail->new(
       Filename   => "$SMSTAT/$_[HEAP]->{ID}.cst",
       InputEvent => 'Status',
       SeekBack   => 80
      );
    $_[KERNEL]->alias_set('SPINDLE');
    $_[KERNEL]->yield('Pid'); 
}

sub _stop {}                              # actually never happens

sub Status {	# file with configuration state is updated
    my ($heap,$kernel,$cst)=@_[HEAP,KERNEL,ARG0];
    $LOG->debug(Prf(@_)." check");
    if($heap->{OST} eq 'INIT')     {      # try to go into last known state 
      $LOG->debug(Prf(@_)." inital");
      my $last_OST = read_OST($_[HEAP]->{ID});
      if($last_OST=~/^(Online|Offline)$/) {# to last monitored state
         $_[KERNEL]->yield($last_OST);
      }else {
         $_[KERNEL]->yield('Stop');
      }
      $_[KERNEL]->yield('Status',$cst);
      
      return;
    }
    $heap->{CST}=uc($heap->{OST})  if $heap->{CST} eq 'INIT';# and uc($heap->{OST}) eq $cst;
    $LOG->debug(Prf(@_)." check1");
    if ($heap->{CST} ne $cst) {   # conf-state is changed [ONLINE|OFFLINE|SUSPEND]
        $LOG->debug(Prf(@_)." check2 $cst");
        if(uc($heap->{OST}) ne $cst) { # need effort for a target CST
          $LOG->info(Prf(@_)."\t============ new target: $cst");
          if(    $heap->{OST}=~/^(Online|Degraded|Full)$/){
              $LOG->debug(Prf(@_)." check3 ost=$heap->{OST} newcst=$cst");
              $_[KERNEL]->yield("Stop")    if $cst eq 'OFFLINE';
              $_[KERNEL]->yield("Suspend") if $cst eq 'SUSPEND';
          }elsif($heap->{OST} eq 'Offline'){
              $_[KERNEL]->yield("Start")   if $cst=~/^(ONLINE|SUSPEND|EMPTY)$/;
          }elsif($heap->{OST} eq 'Suspend'){
              $_[KERNEL]->yield("Stop")    if $cst eq 'OFFLINE';
              $_[KERNEL]->yield("Online")  if $cst eq 'ONLINE';
              $_[KERNEL]->yield("Unload")  if $cst eq 'EMPTY';
          }elsif($heap->{OST} eq 'Unload'){
              $_[KERNEL]->yield("Stop")    if $cst eq 'OFFLINE';
              $_[KERNEL]->yield("Empty")   if $cst eq 'EMPTY';
              $_[KERNEL]->yield("Suspend") if $cst eq 'SUSPEND';
          }else {                          # long way to go
              $_[KERNEL]->yield("Stop");
          }
        }
        $heap->{CST}=$cst;
    }
}

sub Start {
    return if not $_[HEAP]->{OST}=~/^(INIT|Offline)$/;# only INIT|Offline
    sinfo(@_);				      # show state info 
    $LOG->debug(Prf(@_));
    set_OST($_[HEAP],$_[STATE]);
    if($_[HEAP]->{FAILED_COUNT} > $_[HEAP]->{FAILED_LIMIT}) {
      $_='Broken';                            # Force Broken if FAILED over LIMIT
      $LOG->info(Prf(@_)." force Broken since FAILED over LIMIT ". 
                 " $_[HEAP]->{FAILED_COUNT} > $_[HEAP]->{FAILED_LIMIT}\n"); 
    } else {
#     $_=act_start($_[HEAP]->{ID});
      $_=$_[HEAP]->{ACT_START}->($_[HEAP]->{ID});
    }
    s/^SUCCESS$/Mount/;
    $_[KERNEL]->call($_[SESSION],$_);            # LAST code in sub
}

sub Stop { 
    $LOG->debug(Prf(@_));
    sinfo(@_);				      # show state info 
    if($_[HEAP]->{OST} eq 'Failed') {          #count how many
       $_[HEAP]->{FAILED_COUNT}++;             #times FAILED in the row
       $LOG->info(Prf(@_)." FAILED count=$_[HEAP]->{FAILED_COUNT}");
    } else {                                   # see also Start & Cool
       $_[HEAP]->{FAILED_COUNT}=0;             #reset if stop not from FAILED 
    }
    set_OST($_[HEAP],$_[STATE]);
    $_[KERNEL]->call('MONITOR','Freeze');
#    act_stop($_[HEAP]->{ID});
    $_[HEAP]->{ACT_STOP}->($_[HEAP]->{ID});
    $_[KERNEL]->call($_[SESSION],'Offline');          # LAST code in sub
}

sub Mount {
    $LOG->debug(Prf(@_));
    sinfo(@_);				      # show state info 
    set_OST($_[HEAP],$_[STATE]);
#    $_=act_mount($_[HEAP]->{ID});
    $_=$_[HEAP]->{ACT_MOUNT}->($_[HEAP]->{ID});
    s/^SUCCESS$/Online/;       
    $_='Suspend' if $_ eq 'Online' and $_[HEAP]->{CST} eq 'SUSPEND';
    $_='Unload'  if $_ eq 'Online' and $_[HEAP]->{CST} eq 'EMPTY';
    $_[KERNEL]->call($_[SESSION],$_);                   # LAST code in sub
}

sub Suspend {
   $LOG->debug(Prf(@_));
   sinfo(@_);				      # show state info 
   set_OST($_[HEAP],$_[STATE]);
   $_[KERNEL]->call('MONITOR','Freeze');                # stop monitor process
} 

sub Unload {
   $LOG->debug(Prf(@_));
   sinfo(@_);                                 # show state info
   set_OST($_[HEAP],$_[STATE]);
   $_[KERNEL]->call($_[SESSION],'IsEmpty');   # LAST code in sub
} 

sub Empty {
   $LOG->debug(Prf(@_));
   sinfo(@_);                                 # show state info
   set_OST($_[HEAP],$_[STATE]);
}

sub Failed {
   return if not $_[HEAP]->{OST}=~/^(Online|Degraded|Full)$/; #valid incoming
   $LOG->debug(Prf(@_));
   sinfo(@_);				      # show state info 
   set_OST($_[HEAP],$_[STATE]);
   $_[KERNEL]->call($_[SESSION],'Stop');                # LAST code in sub
} 

sub Degraded {
   $LOG->debug(Prf(@_));
   sinfo(@_);				      # show state info 
   set_OST($_[HEAP],$_[STATE]);
}    

sub Alien {
   $LOG->debug(Prf(@_));
   sinfo(@_);				      # show state info 
   set_OST($_[HEAP],$_[STATE]);
}

sub Confound {
   $LOG->debug(Prf(@_));
   sinfo(@_);				      # show state info 
   set_OST($_[HEAP],$_[STATE]);
}

sub Unmarked {
   $LOG->debug(Prf(@_));
   sinfo(@_);				      # show state info 
   set_OST($_[HEAP],$_[STATE]);
}

sub  Broken  { 
   $LOG->debug(Prf(@_));
   sinfo(@_);				      # show state info 
   set_OST($_[HEAP],$_[STATE]); 
   $_[KERNEL]->delay(Try=>300);   # try in 5 minutes
}

sub  Online  { 
   $LOG->debug(Prf(@_));
   sinfo(@_);				      # show state info 
   $_[KERNEL]->post('MONITOR','Unfreeze') if $_[HEAP]->{OST}=~/^(Mount|INIT|Suspend|Empty)$/;
   set_OST($_[HEAP],$_[STATE]); 
}

sub  Full  {
   $LOG->debug(Prf(@_));
   sinfo(@_);				      # show state info 
   set_OST($_[HEAP],$_[STATE]);
}

sub Offline   { 
   $LOG->debug(Prf(@_));
   sinfo(@_);				      # show state info 
   set_OST($_[HEAP],$_[STATE]); sleep 3;
   $_[KERNEL]->yield('Start') if not $_[HEAP]->{CST}=~/^(OFFLINE|INIT)$/;
}

sub Pid {
  $LOG->debug(Prf(@_)." writing pid ");
  $_[HEAP]->{ACT_PID}->($_[HEAP]->{ID});
  $_[KERNEL]->delay(Pid=>30);   # call again in 30 seconds
}

#---------- semi-state: Try,Week,Cool uncomment set_OST for debug only

sub Try {
   return if $_[HEAP]->{OST} ne 'Broken'; #ignore if we in the different state
   $LOG->debug(Prf(@_));
   set_OST($_[HEAP],$_[STATE]);
   $_[KERNEL]->call($_[SESSION],'Stop'); # LAST code in sub
}

sub IsEmpty {
   return if $_[HEAP]->{OST} ne 'Unload';     #ignore if we in the different state
   $LOG->debug(Prf(@_));
#   sinfo(@_);                                 # show state info
#   set_OST($_[HEAP],$_[STATE]);
   if($_[HEAP]->{ACT_ISEMPTY}->($_[HEAP]->{ID}) eq 'Empty') {
     $_[KERNEL]->call($_[SESSION],'Empty');   # LAST code in sub
   }else{
     $_[KERNEL]->delay(IsEmpty=>120);         # do IsEmpty in 2 minutes
   }
} 

sub Weak {          # swith to Degraded if ost=Online|Full
   return if not $_[HEAP]->{OST}=~/^(Online|Full)$/; 
#   $LOG->info(Prf(@_));
#   set_OST($_[HEAP],$_[STATE]);
   $_[KERNEL]->call($_[SESSION],'Degraded'); # LAST code in sub
}

sub Tight {          # swith to Full if ost=Online|Degraded
   return if not $_[HEAP]->{OST}=~/^(Online|Degraded)$/;
#   $LOG->info(Prf(@_));
#   set_OST($_[HEAP],$_[STATE]);
   $_[KERNEL]->call($_[SESSION],'Full'); # LAST code in sub
}

sub Cool {          # swith to Online if ost=Degraded|Full
   $_[HEAP]->{FAILED_COUNT}=0;              #reset FAILED count since cool
   return if not $_[HEAP]->{OST}=~/^(Degraded|Full)$/;
#   $LOG->info(Prf(@_));
#   set_OST($_[HEAP],$_[STATE]);
   $_[KERNEL]->call($_[SESSION],'Online'); # LAST code in sub    
}

# ACTIONS =====================================================================
# suppouse to be redefined 

sub act_mount   { sleep 2; 'SUCCESS' }    # possible 'Broken'
sub act_start   { sleep 2; 'SUCCESS' }    # possible Unmarked,Alien,Confound
sub act_stop    { sleep 2; 'SUCCESS' }    # always retrun SUCCESS


1;
__END__
# Below is stub documentation for your module. You'd better edit it!

=head1 NAME

SM::Spindle - Perl extension for blah blah blah

=head1 SYNOPSIS

  use SM::Spindle;
  blah blah blah

=head1 DESCRIPTION

Stub documentation for SM::Spindle, created by h2xs. It looks like the
author of the extension was negligent enough to leave the stub
unedited.

Blah blah blah.

=head2 EXPORT

None by default.



=head1 SEE ALSO

Mention other useful documentation such as the documentation of
related modules or operating system documentation (such as man pages
in UNIX), or any relevant external documentation such as RFCs or
standards.

If you have a mailing list set up for your module, mention it here.

If you have a web site set up for your module, mention it here.

=head1 AUTHOR

A. U. Thor, E<lt>apl@localdomainE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2008 by A. U. Thor

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.8 or,
at your option, any later version of Perl 5 you may have available.


=cut
