package SM::Config;

use 5.008008;
use strict;
use warnings;
use Data::Dumper;
use File::Basename qw(basename);
use Log::Log4perl "get_logger";
require Exporter;

our @ISA = qw(Exporter);

# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.

# This allows declaration	use SM::Config ':all';
# If you do not need this, moving things directly into @EXPORT or @EXPORT_OK
# will save memory.
our %EXPORT_TAGS = ( 'all' => [ qw(
 SM_Declared SM_Wheels SM_Unassigned SM_Devices SM_Objects SM_DateSplit SM_WritePid SM_Options
 mtime shortid 
 SM_info SM_warn SM_error SM_cinfo SM_cwarn SM_cerror SM_SpaceInfo SM_StopAll
 SM_StopSpindle
 SM_CONF SM_DEVCONF SM_MNT SM_STAT SM_LOG SM_VER SM_PID	SM_VAR SM_DBLOG SM_OS
) ] );

our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

our @EXPORT = qw(
  SM_Wheels SM_LOG
);

our $VERSION = '0.01';

my $APL       =$ENV{APL};
my $APL_VAR   =$ENV{APL_VAR};
my $SMCONF    ="$APL_VAR/conf/sm";
my $DEVCONF   ="$APL_VAR/conf";                   #device configuration
my $DEVCONFUPD="$APL_VAR/conf/conf";              #indicate:devcfg is updated
my $SMSTATUPD ="$APL_VAR/sm/stat/.updated";       #sm status is updated
my $PIDPATH   ="$APL_VAR/sm/pids";                #dir for pid files
my $SMVAR     ="$APL_VAR/sm";                     #location of var for sm

my $MOUNTPOINT="/vasm";
my $SMSTAT    ="$APL_VAR/sm/stat";                #location for stat and status
my $SMLOG     ="$APL_VAR/log/sm/stat";            #location for stat log files
my $DBLOG     ="$APL_VAR/log/sm/dblog";           #location for dblog files
my $OPTIONS   ="$APL_VAR/conf/sm/options";        # defines options
my $VER       =$ENV{APL_SM_VER};                  #version ex: va-2.6.2
my $LOGCFG_NAME = "$APL/sm/etc/sm_logger.conf";   #logger config file
my $LOGCFG_INTERVAL = 60;                         #logger refresh interval
my $SM_OS     =`uname`; chomp $SM_OS;             #operation system (Linux|Darwin)
my $DF        ='/bin/df -P -k';                   #portable free disk space util
Log::Log4perl::init_and_watch($LOGCFG_NAME,$LOGCFG_INTERVAL);
my $LOG       =get_logger('NEXTCAM::SM');
# SUBS ------------------------------------------------------------------------
sub SM_CONF    () { $SMCONF     }
sub SM_DEVCONF () { $DEVCONF    }
sub SM_MNT     () { $MOUNTPOINT }
sub SM_STAT    () { $SMSTAT     }
sub SM_LOG     () { $LOG        }
sub SM_VER     () { $VER        }
sub SM_PID     () { $PIDPATH    }
sub SM_VAR     () { $SMVAR      }
sub SM_DBLOG   () { $DBLOG      }
sub SM_OS      () { $SM_OS      }

sub mtime { (stat(shift))[9] }
#-------------------------------------------------------
# write pid file to correct location
#-------------------------------------------------------
sub SM_WritePid {
  my $name=shift;
  open(PID,">$PIDPATH/$name.pid");
  print PID "$$\n";
  close PID;
}

#----------------------------------------------------------------------------
# SM_StopAll check if any sm_proc (except sm_engine) is running and stop it
# called from sm_engine
#----------------------------------------------------------------------------
sub pid_by_name {
    my $name=shift;
    open(NAME,$name) || return -1;
    my $pid=<NAME>;
    close NAME;
    return $pid+0;
}

sub is_running {
    my ($pid,$name)=@_;
    $name=~s/\./ /;
    system("ps -o pid,command $pid | grep -q '$name'");
    return 1 if not $?;
    return 0;
}

sub SM_StopAll {
    my $found=0;
    opendir(DIR,$PIDPATH);
    my %pids = map {/(\w+\S*)\.pid/; ($1,pid_by_name("$PIDPATH/$1.pid"))} grep {/\w+\S*\.pid/} grep {!/sm_engine/} readdir(DIR);
    closedir DIR;
    #print Dumper(\%pids);
    foreach my $proc (keys %pids) {
      if(is_running($pids{$proc},$proc)){
         $found++;
         #print "killing $pids{$proc},$proc\n";
         kill 15, $pids{$proc};
      }
    }
    return if not $found;
    sleep 1;
    foreach my $proc (keys %pids) {
      if(is_running($pids{$proc},$proc)){
         kill 9, $pids{$proc};
      }
    }
}

sub SM_StopSpindle {
    my $uuid=shift;
    my $pid=pid_by_name("$PIDPATH/sm_spindle.$uuid.pid");
    if ($pid >0 and is_running($pid,"sm_spindle $uuid")) {
       kill 15, $pid;
    }
}

sub SM_Options {
   my %options=(MOUNT_TIMEOUT=>50,MODE=>'MULTIWEELS',VALID_LABEL=>'^va-(default|\d+)$'); # defaults
   if(-f $OPTIONS) {
     if(open OPTIONS,$OPTIONS) {
       my %newoptions=map{/(^\w+)=(.+)/} grep {/^\w+=.+/} <OPTIONS>;
       close OPTIONS;
       %options=(%options,%newoptions); # overwrite defaults
     }
   }
   return \%options;
}

#-------------------------------------------------------
# write messages for posting to DB log
#-------------------------------------------------------
sub dblog {
   my ($level,$term,$id,$msg)=@_;
   my ($pack, $file, $line) = caller(1);
   chomp $msg;
   print("$msg\n")  if $term;  # print to console if requested
   my $message=time .'|'.basename($file)."|$line|$level|$id|$msg\n";
   if(open (DBLOG,">>$DBLOG/dblog")) {
      print DBLOG $message;
      close DBLOG;
   } else {
      $LOG->warn("Cannot post message into dblog");
   }
}

sub SM_info   { dblog('INFO' ,0,@_) } # info  message to log
sub SM_error  { dblog('ERROR',0,@_) } # error message to log
sub SM_warn   { dblog('WARN' ,0,@_) } # warn  message to log
sub SM_cinfo  { dblog('INFO' ,1,@_) } # info  message to log+console
sub SM_cerror { dblog('ERROR',1,@_) } # error message to log+console
sub SM_cwarn  { dblog('WARN' ,1,@_) } # warn  message to log+console

#-----------------------------------------------------------
#  get free_space (MB) on mount point
#  uses 'df -P' for getting space results
#  retrun:  {SIZE,USED,FREE,RATE}
#-----------------------------------------------------------
sub SM_SpaceInfo {
   my $mpoint=shift;
   my $info={SIZE=>1,USED=>1,FREE=>0,RATE=>99};
   eval {
     open(DF, "$DF $mpoint 2>/dev/null|") || die ("Cannot get df -P $mpoint");
     my @rows=<DF>;
     close DF;
     foreach(@rows) {
       $info={SIZE=>int($1/1024),USED=>int($2/1024),FREE=>int($3/1024),RATE=>$4}
                      if m|\s(\d+)\s+(\d+)\s+(\d+)\s+(\d)+%\s+$mpoint|;
     }
   };
   if($@) {
      SM_LOG->error("$@"); # TBD Error handling
   } 
   SM_LOG->debug("$mpoint ".join(' ',%$info));
   return $info;
}


#-------------------------------------------------------
# read declared devices
#-------------------------------------------------------
sub SM_Declared {
  my %declared; # the result will be placed into
  eval {
    opendir(DIR,"$SMCONF/declared") || die "ERR-KPR002: can't opendir $SMCONF/declared: $!";
    my @dir= grep{! /^\.\.?$/} readdir(DIR);
    closedir DIR;
    foreach my $id (@dir) { # reading actual conf files
      my $confname="$SMCONF/declared/$id";
      next if not -f $confname;
      if (open(CONF,$confname)) {
        my %conf=map{/(^\w+)=(.+)/} grep {/^\w+=.+/} <CONF>;
        close CONF;
        $conf{conf_file} =$confname;
        $conf{conf_mtime}=mtime $confname;
        $declared{$id}=\%conf;
      } else {
         $LOG->error("Cannot open $confname\n"); # TBD. what to do with empty conf
      }
    }
  }; # eval end
 print "ERROR: $@\n"  if $@;
 return \%declared;
}

#-------------------------------------------------------
# read unassigned [unused] devices
#-------------------------------------------------------
sub SM_Unassigned {
  my %unassigned; # the result will be placed into
  eval {
    opendir(DIR,"$SMCONF/unassigned") || die "ERR-KPR002: can't opendir $SMCONF/unassigned: $!";
    my @dir= grep{/^\w{8}-(\w{4}-){3}\w{12}$/} readdir(DIR);
    closedir DIR;
    foreach my $id (@dir) { # reading actual conf files
      my $confname="$SMCONF/unassigned/$id";
      next if not -f $confname;
      if (open(CONF,$confname)) {
        my %conf=map{/(^\w+)=(.+)/} grep {/^\w+=.+/} <CONF>;
        close CONF;
        $conf{conf_mtime}=mtime $confname;
      #------------------------ get stat information
        my $statname="$SMSTAT/$id.stat";
        my %stat=(WRITE=>-1,SIZE=>-1,FREE=>-1,USED=>-1,RATE=>-1,AWAIT=>-1,SVCTM=>-1,UTIL=>-1);
        if(open(STAT,$statname)) {
           %stat=(%stat,map{/(^\w+)=(.+)/} grep {/^\w+=.+/} <STAT>);
           close STAT;
        } else{  
          $LOG->error("Cannot open $statname");
        }
        $conf{'stat_'.$_}=$stat{$_} foreach(keys %stat);
        $conf{stat_freeprc}=($stat{SIZE}>0)?int(100*$stat{FREE}/($stat{SIZE}+1)):-1;
        $conf{stat_mtime}=mtime $statname;
        #------------------------ get mount information
        $unassigned{$id}=\%conf;
      } else {
         $LOG->error("Cannot open $confname\n"); # TBD. what to do with empty conf
      }
    }
  }; # eval end
 print "ERROR: $@\n"  if $@;
 return \%unassigned;
}

#-------------------------------------------------------
#  read spindle configuration
#  add _space from stat{FREE} - TBD-reserve
#
#-------------------------------------------------------
sub SM_Wheels {
 my %wheels;        # the result will be placed into
 eval {
  opendir(DIR,"$SMCONF/wheels") || die "ERR-KPR002: can't opendir $SMCONF/wheels: $!";
  my @dir= grep{/^\w{8}-(\w{4}-){3}\w{12}$/} readdir(DIR);
  closedir DIR;
  foreach my $id (@dir) { # reading actual conf files
    my $confname="$SMCONF/wheels/$id";
    #print "$confname\n";
    next if not -f $confname;
    if (open(CONF,$confname)) {
      my %conf=map{/(^\w+)=(.+)/} grep {/^\w+=.+/} <CONF>;
      close CONF;
      $conf{conf_mtime}=mtime $confname;
      #------------------------ get stat information
      my $statname="$SMSTAT/$id.stat";
      my %stat=(WRITE=>-1,SIZE=>-1,FREE=>-1,USED=>-1,RATE=>-1,AWAIT=>-1,SVCTM=>-1,UTIL=>-1);
      if(open(STAT,$statname)) {
         %stat=(%stat,map{/(^\w+)=(.+)/} grep {/^\w+=.+/} <STAT>);
         close STAT;
      } else{  
         $LOG->error("Cannot open $statname");
      }
      $conf{'stat_'.$_}=$stat{$_} foreach(keys %stat);
      $conf{stat_freeprc}=($stat{SIZE}>0)?int(100*$stat{FREE}/($stat{SIZE}+1)):-1;
      $conf{stat_mtime}=mtime $statname;
      #------------------------ get mount information
      my $mname="$SMSTAT/$id.minfo";
      my %minfo=(ALIAS=>'NONE',DEV=>'NONE');
      if(open(MINFO,$mname)) { 
         %minfo= map{/(^\w+)=(.+)/} grep {/^\w+=.+/} <MINFO>;
         close MINFO; 
      }else {
        #$LOG->warn("Cannot open $mname");
      }
      $conf{minfo_ALIAS}=$minfo{ALIAS};
      $conf{minfo_DEV}=$minfo{DEV};
      $conf{minfo_mtime}=mtime $mname; 
      #---------------------- get ost
      my $stname="$SMSTAT/$id.ost";
      if(open(OST,$stname)) {
         $conf{ost}=<OST>; chomp $conf{ost};
         close OST;
      }else{
         $LOG->error("Cannot open $stname");
         $conf{ost}='undef';
      }
      $conf{ost_mtime}=mtime $stname;
      #---------------------- get cst 
      $stname="$SMSTAT/$id.cst";
      if(open(CST,$stname)) {
         $conf{cst}=<CST>; chomp $conf{cst};
         close CST;
      }else{
         $LOG->error("Cannot open $stname");
         $conf{cst}='undef';
      }
      $conf{cst_mtime}=mtime $stname;
      $wheels{$id}=\%conf;
    } else {
      $LOG->error("Cannot open $confname\n"); # TBD. what to do with empty conf
    }
  }
 }; # eval end
 if($@) {
   print "ERROR: $@\n";
 }
 return \%wheels;
}

#-------------------------------------------------------------------------------
# split date to array: (yymmdd,hh,mi,ss)
#-------------------------------------------------------------------------------
sub SM_DateSplit {
  my $tm=shift;
  my ($tsec,$tmin,$thour,$tday,$tmon,$tyear)=gmtime($tm);
  my $yymmdd=sprintf("%02d%02d%02d",$tyear-100,$tmon+1,$tday);
  my $hh=sprintf("%02d",$thour);
  my $mi=sprintf("%02d",$tmin);
  my $ss=sprintf("%02d",$tsec);
  return ($yymmdd,$hh,$mi,$ss);
}

#sub shortid {my $uuid=shift; substr($uuid,0,3).'-'.substr($uuid,-3,3) }
sub shortid {my $uuid=shift; substr($uuid,0,8)}

#-------------------------------------------------------
# estimate writepower for device
# function is used if stat is not collected
#-------------------------------------------------------
sub wp_estimate {
  my $conf=shift;   # reference to device configuration
  return 4   if($conf->{DEVICETYPE} eq 'AUDIO');# no details estimation for aac 4MB/sec
  return 0   if($conf->{DEVICETYPE} ne 'CAMERA'); # it is not a space consumer
  if ($conf->{MEDIA_FORMAT}=~ /^(mpeg4|h264)$/) { #---- mpeg4 or h264
    my $brate=($conf->{RC_MODE} eq 'vbr')?$conf->{RC_MAXBITRATE}:$conf->{RC_TARGETBITRATE};
    $brate=2048 if $brate=~/^(unlimited|cam-defined)$/; # cannot be estimated..
    return $brate/8;
  }
  #-------------------------------------------- mjpg TBD: REDO!!
  return 50;         # do not know
  my $BS=27;         # this is base size for lo-compression CIF image
  my $ksize=1;       # for CIF
  $ksize=3           if $conf->{IMAGESIZE} eq 'hugesize';
  $ksize=1/3         if $conf->{IMAGESIZE} eq 'halfsize';
  $conf->{CAMCOMPRESSION}=1 if not defined $conf->{CAMCOMPRESSION} or ! $conf->{CAMCOMPRESSION};
  return $conf->{ARCFRAMERATE}*$BS*$ksize/$conf->{CAMCOMPRESSION};
}

#-------------------------------------------------------
#  read dev configuration
sub SM_Devices {
 my %dev;                   # hash for results
 eval {
  opendir(DIR,$DEVCONF) || die "ERR-KPR001: can't opendir $DEVCONF: $!";
  my @dir= grep { /^a?\d+$/ } readdir(DIR);
  closedir DIR;
  foreach my $id (@dir) { # reading actual conf files
    my $confname="$DEVCONF/$id/conf";
    my $statname="$DEVCONF/$id/stat";
    my $statwipe="$DEVCONF/$id/stat.wipe";
#    print "$confname\n";
    next if not -f $confname;
    if (open(CONF,$confname)) {
      my %conf=map{/(^\w+)=(.+)/} grep {/^\w+=.+/} <CONF>;
      close CONF;
      next if not defined $conf{DEVID}; # ignore config without devid
      if(open(STAT,$statname)){ #--------------------- stat
         my %stat=map{/(^\w+)=(.+)/} grep {/^\w+=.+/} <STAT>;
         close STAT;
         $conf{_status}=$stat{STATUS} if exists $stat{STATUS};
      }
      $conf{_status}='DOWN' if not $conf{_status}; 
      if(open(STATW,$statwipe)){#-------------------- stat.wipe
         my %stat=map{/(^\w+)=(\d+)/} grep {/^\w+=\d+/} <STATW>;
         close STATW;
         if(defined $stat{VA_RATE} and $stat{VA_RATE}>0) {
            $conf{stat_VA_RATE}=$stat{VA_RATE};
         }else {
            $conf{stat_VA_RATE}=wp_estimate(\%conf); 
         }
      }else{                     # stat is not avaiable
         $conf{stat_VA_RATE}=wp_estimate(\%conf); #
         #$LOG->warn("Cannot get stat for device id=$id [ $conf{OBJID} ]. use estimation");
      }
      $dev{$id}=\%conf;
    } else {
      $LOG->error("Cannot open $confname\n"); # TBD. what to do with empty conf
    }
  } 
 }; # eval end
 if($@) {
   print "ERROR: $@\n";
   $LOG->error('Cannot read device configuration Err: $@');
 } 
 \%dev;
} 

#-------------------------------------------------------
#  read dev configuration and convert to obj configuration
sub SM_Objects {
  my %objs;                   # hash for results
  my $devs=SM_Devices();
  foreach my $dev (keys %$devs) {
    my $obj=$devs->{$dev}->{OBJID};
    $objs{$obj}=$devs->{$dev};
  }
  \%objs;
}

1;
__END__
# Below is stub documentation for your module. You'd better edit it!

=head1 NAME

SM::Config - Perl extension for blah blah blah

=head1 SYNOPSIS

  use SM::Config;
  blah blah blah

=head1 DESCRIPTION

Stub documentation for SM::Config, created by h2xs. It looks like the
author of the extension was negligent enough to leave the stub
unedited.

Blah blah blah.

=head2 EXPORT

None by default.



=head1 SEE ALSO

Mention other useful documentation such as the documentation of
related modules or operating system documentation (such as man pages
in UNIX), or any relevant external documentation such as RFCs or
standards.

If you have a mailing list set up for your module, mention it here.

If you have a web site set up for your module, mention it here.

=head1 AUTHOR

A. U. Thor, E<lt>apl@localdomainE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2008 by A. U. Thor

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.8 or,
at your option, any later version of Perl 5 you may have available.


=cut
