/**
 * @version $Id: timeline.js 32802 2015-07-24 15:25:48Z astarostin $
 *
 * TimeLine interface class for working with flash TimeLine
 *
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2010
 * ------------------------------------------------------------------------------
 */

(function (root, factory) {
    if (typeof define === 'function' && define.amd) {
        // AMD. Register as an anonymous module.
	    define(["log", "swfobject", "jquery"], factory);
    } else {
        // Browser globals
        root.TimeLine = factory(root.Log, root.swfobject, root.$);
    }
}(this, function(Log, swfobject, $){
	"use strict";

	/**
	 * create new timeline on page
	 *
	 * @param {Object.<String, *>} parameters
	 * @constructor
	 */
	function TimeLine(parameters)
	{
		this._parameters = {
			variableName: parameters.variableName || "",
			width: parameters.width || "100%",
			isDebug: !!parameters.isDebug,
			oneLine: !!parameters.oneLine,
			withImages: !!parameters.withImages
		};

		this._id = parameters.id || null;

		this._timeline = null;

		/**
		 * selection sticked to timeline
		 * @const {Number}
		 */
		this.SELECTION_STICKY_TO_TIMELINE = 1;
		/**
		 * selection sticked to view
		 * @const {number}
		 */
		this.SELECTION_STICKY_TO_VIEW = 2;
	}

	/**
	 * initialize TimeLine
	 */
	TimeLine.prototype.init = function()
	{
		this._parameters.isDebug && Log.info("init");

		if (!swfobject || $(this._id).length == 0)
			return;

		var TimeLineParameters = {
			external_objectTimeline: this._parameters.variableName,
			isDebug: this._parameters.isDebug,
			oneLine: this._parameters.oneLine,
			withImages: this._parameters.withImages,
			sliderUpCloseImagePath: "/sdi/timeline/img/slider_up_close.png"
		};

		var params = {
			allowScriptAccess: "always",
			quality: "high",
			scale: "noscale",
			bgcolor: "#8B8B8B"
		};
		var attributes = {};
		var id = this._id.match(/#(.*)/)[1];
		var height = this._parameters.oneLine ? 30 : 60;

		if (this._parameters.withImages) height += 100;

		var self = this;
		swfobject.embedSWF("/sdi/timeline/timeline.swf", id, this._parameters.width, height, "8.0.0", false, TimeLineParameters, params, attributes,
			function(e)
			{
				self._parameters.isDebug && Log.info("onEmbed" + e);

				self._timeline = e.ref;
			}
		);
	};

	/**
	 * event handler for load event
	 */
	TimeLine.prototype.onLoad = function()
	{
		this._parameters.isDebug && Log.info("onLoad");
	};

	/**
	 * event handler for selection change event
	 *
	 * @param {String} line 'up'|'down'
	 * @param {Number} beginTime int|null begin of the time interval in Unix time in ms
	 * @param {Number} endTime int|null end of the time interval in Unix time in ms
	 */
	TimeLine.prototype.onSelectionChange = function(line, beginTime, endTime)
	{
		this._parameters.isDebug && Log.info("onSelectionChange");
	};

	/**
	 * event handler for clear selection event
	 */
	TimeLine.prototype.onSelectionClear = function()
	{
		this._parameters.isDebug && Log.info("onSelectionClear");
	};

	/**
	 * event handler for debug message event
	 */
	TimeLine.prototype.onDebug = function(message, type)
	{
		this._parameters.isDebug && console.log(message);
	};

	/**
	 * event handler for requesting data event
	 * @param {String} line
	 * @param {Number} beginTime begin of the time interval in Unix time in ms
	 * @param {Number} endTime end of the time interval in Unix time in ms
	 * @param {String} granularity 'year'|'mon'|'day'|'hour'|'min'|'second'
	 */
	TimeLine.prototype.onGetData = function(line, beginTime, endTime, granularity)
	{
		this._parameters.isDebug && Log.info("onGetData");
		this.clearColorScale();
	};

	/**
	 * event handler for requesting images
	 *
	 * @param {String} line
	 * @param {Number} beginTime begin of the time interval in Unix time in ms
	 * @param {Number} endTime end of the time interval in Unix time in ms
	 */
	TimeLine.prototype.onGetImages = function(line, beginTime, endTime)
	{
		this._parameters.isDebug && Log.info("onGetImages");
	};

	/**
	 * event handler for time change event
	 *
	 * @param {Number} time Unix time in ms
	 */
	TimeLine.prototype.onTimeChange = function(time)
	{
		this._parameters.isDebug && Log.info("onTimeChange");
	};

	/**
	 * event handler for image click event
	 *
	 * @param {Number} time Unix time in sec
	 * @param {Number} beginTime begin of the of fragment in Unix time in ms
	 * @param {Number} endTime end of the of fragment in Unix time in ms
	 */
	TimeLine.prototype.onImageClick = function(time, beginTime, endTime)
	{
		this._parameters.isDebug && Log.info("onImageClick");
	};

	/**
	 * set xml with data for drawing intervals
	 *
	 * @param {String} line
	 * @param {Object.<Number, Object.<String, Number>>} timestampList {"1289988104000": {"percentage": 53}}
	 */
	TimeLine.prototype.setData = function(line, timestampList)
	{
		this._parameters.isDebug && Log.info("setData");

		var badArgs = (line == null || timestampList == null);
		if (badArgs) return;

		if (this._timeline)
		{
			this._timeline.setData(line, timestampList);
		}
	};

	/**
	 * set data for drawing images
	 *
	 * @param {String} line
	 * @param {Array.<Object>} timestampList
	 * 	[{
	 * 		Fn: 0, // fragment number
	 *		Tn: 0, // timestamp number in fragment
	 *		timestamp: 1289988104000, // Unix time in ms
	 *		start_time: 1289988103000, // start time of fragment, Unix time in ms
	 *		end_time: 1289988105000, // end time of fragment, Unix time in ms
	 *		url: "/storage/snapshot?objid=117&ts=timestamp&downscale" // url to timestamp image
	 *	}]
	 */
	TimeLine.prototype.setImages = function(line, timestampList)
	{
		this._parameters.isDebug && Log.info("setImages");

		var badArgs = (line == null || timestampList == null);
		if (badArgs) return;

		if (this._timeline)
		{
			this._timeline.setImages(line, timestampList);
		}
	};

	TimeLine.prototype.clearImages = function()
	{
		this._parameters.isDebug && Log.info("clearImages");

		if (this._timeline)
		{
			this._timeline.clearImages();
		}
	};

	/**
	 * go to next timeinterval with images
	 *
	 * @returns {Boolean} true if success
	 */
	TimeLine.prototype.nextImagesPage = function()
	{
		this._parameters.isDebug && Log.info("nextImagesPage");

		var result = false;

		if (this._timeline)
		{
			result = this._timeline.nextImagesPage();
		}

		return result;
	};

	/**
	 * go to previous timeinterval with images
	 *
	 * @returns {Boolean} true if success
	 */
	TimeLine.prototype.prevImagesPage = function()
	{
		this._parameters.isDebug && Log.info("prevImagesPage");

		var result = false;

		if (this._timeline)
		{
			result = this._timeline.prevImagesPage();
		}

		return result;
	};

	/**
	 * set time to time
	 *
	 * @param {Number} time Unix time in ms
	 * @param {Boolean} isCallback
	 * @returns {Boolean}
	 */
	TimeLine.prototype.setTime = function(time, isCallback)
	{
		this._parameters.isDebug && Log.info("setTime");

		isCallback = !!isCallback;

		var badArgs = (!time && time != 0);
		if (badArgs) return null;

		var result = false;
		if (this._timeline)
		{
			result = this._timeline.setTime(time, isCallback);
		}
		return result;
	};

	/**
	 * @deprecated for now
	 * @returns {Number} time Unix time in ms
	 */
	TimeLine.prototype.getTime = function()
	{
		var time = null;

		if (this._timeline)
		{
			time = this._timeline.getTime();
		}

		return time;
	};

	/**
	 * set timeline parameters
	 *
	 * @param {Object.<String, ?(Object|String)>} timelineParameters
	 * default {
	 *  selection: {fixed: false, stickyToTimeLine: true, stickyToView: false},
	 *  scale: {granularity: "hour"},
	 *  dateFormat: "MM/DD/YYYY",
	 *  timezoneOffset: null
	 * }
	 * fixed - can user change (create new) selection borders or not
	 * sticky - SELECTION_STICKY_TO_TIMELINE, SELECTION_STICKY_TO_VIEW
	 * granularity - 'year'|'mon'|'day'|'hour'|'min'|'second'
	 * dateFormat - "YYYY-MM-DD"|"MM/DD/YYYY"|"DD/MM/YYYY"
	 * timezoneOffset: {2010: {"startDST": 1206838800, "endDST": 1224982800, "offset": 7200, "offsetDST": 10800}}
	 */
	TimeLine.prototype.setParameters = function(timelineParameters)
	{
		this._parameters.isDebug && Log.info("setParameters");

		var selection = timelineParameters.selection || {};

		if (typeof selection.fixed != "undefined")
		{
			selection.fixed = !!selection.fixed;
		}

		if (typeof selection.visible != "undefined")
		{
			selection.visible = !!selection.visible;
		}

		if (typeof selection.sticky != undefined)
		{
			// can be only one sticky option
			if (selection.sticky == this.SELECTION_STICKY_TO_TIMELINE)
			{
				selection.stickyToTimeLine = true;
				selection.stickyToView = false;
			} else
			if (selection.sticky == this.SELECTION_STICKY_TO_VIEW)
			{
				selection.stickyToView = true;
				selection.stickyToTimeLine = false;
			}
		}

		var scale = timelineParameters.scale ? timelineParameters.scale : null;

		var dateFormat = timelineParameters.dateFormat ? timelineParameters.dateFormat : null;

		var timezoneOffset = timelineParameters.timezoneOffset ? timelineParameters.timezoneOffset : null;

		if (this._timeline)
		{
			this._timeline.setSelectionParameters(selection);

			if (scale && scale.granularity)
			{
				this._timeline.setScale(scale.granularity);
			}

			if (dateFormat)
			{
				this._timeline.setDateFormat(dateFormat);
			}

			if (timezoneOffset)
			{
				this._timeline.setTimezoneOffset(timezoneOffset);
			}
		}
	};

	/**
	 * get timeline selection parameters
	 *
	 * @returns {Object}
	 */
	TimeLine.prototype.getSelectionParameters = function()
	{
		var selectionParameters = null;

		if (this._timeline)
		{
			selectionParameters = this._timeline.getSelectionParameters();
		}

		return selectionParameters;
	};

	/**
	 * Set selected interval
	 *
	 * @param {Number} beginTime begin of the time interval in Unix time in ms
	 * @param {Number} endTime end of the time interval in Unix time in ms
	 * @param {Boolean} isCallback if true send callback of changing upper interval, or if false not send
	 */
	TimeLine.prototype.setSelection = function(beginTime, endTime, isCallback)
	{
		this._parameters.isDebug && Log.info("setSelection");

		isCallback = !!isCallback;

		var badArgs = (beginTime == null || endTime == null);
		if (badArgs) return;

		if (this._timeline)
		{
			// swap values
			if (beginTime > endTime)
			{
				var tmp = beginTime;
				beginTime = endTime;
				endTime = tmp;
			}
			this._timeline.setSelection(beginTime, endTime, isCallback);
		}
	};

	/**
	 * clear selection in timeline
	 */
	TimeLine.prototype.clearSelection = function()
	{
		this._parameters.isDebug && Log.info("clearSelection");

		if (this._timeline)
		{
			this._timeline.clearSelection();
		}
	};

	/**
	 * draw fixed disabled selection under selection with style like selection
	 *
	 * @param {Number} beginTime
	 * @param {Number} endTime
	 */
	TimeLine.prototype.setDisabledSelection = function(beginTime, endTime)
	{
		this._parameters.isDebug && Log.info("setDisabledSelection");

		var badArgs = (beginTime == null || endTime == null);
		if (badArgs) return;

		if (this._timeline)
		{
			// swap values
			if (beginTime > endTime)
			{
				var tmp = beginTime;
				beginTime = endTime;
				endTime = tmp;
			}
			this._timeline.setFixedSelection(beginTime, endTime);
		}
	};

	/**
	 * clear fixed disabled selection in timeline
	 */
	TimeLine.prototype.clearDisabledSelection = function()
	{
		this._parameters.isDebug && Log.info("clearDisabledSelection");

		if (this._timeline)
		{
			this._timeline.clearFixedSelection();
		}
	};

	/**
	 * Draw red dot on timeline
	 * @param {Boolean} isVisible if true draw, if false not draw
	 * @param {Number} dotSize size of dot
	 */
	TimeLine.prototype.drawRedDot = function(isVisible, dotSize)
	{
		this._parameters.isDebug && Log.info("drawRedDot");

		var badArgs = (isVisible == null || dotSize == null);
		if (badArgs) return;

		isVisible = !!isVisible;

		if (this._timeline)
		{
			this._timeline.drawRedDot(isVisible, dotSize);
		}
	};

	/**
	 * clear color scale on timeline
	 */
	TimeLine.prototype.clearColorScale = function()
	{
		this._parameters.isDebug && Log.info("clearColorScale");

		if (this._timeline)
		{
			this._timeline.clearColorScale();
		}
	};

	/**
	 * set cloud border timestamp
	 *
	 * @param {Number} time Unix time in ms
	 * @returns {Boolean}
	 */
	TimeLine.prototype.setCloudTimestamp = function(time)
	{
		this._parameters.isDebug && Log.info("setCloudTimestamp");

		var badArgs = (time == null);
		if (badArgs) return null;

		var result = false;
		if (this._timeline)
		{
			result = this._timeline.setCloudTimestamp(time);
		}
		return result;
	};

	/**
	 * reset cloud border timestamp
	 *
	 * @returns {Boolean}
	 */
	TimeLine.prototype.resetCloudTimestamp = function()
	{
		this._parameters.isDebug && Log.info("resetCloudTimestamp");

		var result = false;
		if (this._timeline)
		{
			result = this._timeline.resetCloudTimestamp();
		}
		return result;
	};

	return TimeLine;
}));
