/**
 * @version $id:$
 * ------------------------------------------------------------------------------
 * description
 * ------------------------------------------------------------------------------
 * @author Andrey Starostin
 * @QA
 * @copyright videoNEXT Network Solutions, Inc, 2013
 * ------------------------------------------------------------------------------
 */

define("pointlist", [], function(){
	"use strict";

	/**
	 * @constructor
	 */
	function PointList()
	{
		/**
		 * @type {Object}
		 * @private
		 */
		this._config = {};

		/**
		 * @type {number}
		 * @private
		 */
		this._currentConfigObj = null;

		/**
		 * @type {number}
		 * @private
		 */
		this._selectedPointColor = 0xFF00FF00;

		/**
		 * @type {MediaPlayer}
		 * @private
		 */
		this._associatedPlayer = null;
	}

	/**
	 * @param {number} x
	 * @param {number} y
	 * @param {number} width
	 * @param {number} height
	 * @param {number} color
	 * @returns {number}
	 * @protected
	 */
	PointList.prototype._addPoint = function(x, y, width, height, color)
	{
		var pointList = this._getCurrentPointList();
		var id = this._genNewPointId();
		pointList[id] = {
			position: null,
			x: x,
			y: y,
			width: width,
			height: height,
			color: color,
			primitiveList: []
		};

		var pointX = Math.round(x - width / 2);
		var pointY = Math.round(y - height / 2);

		pointList[id].primitiveList[pointList[id].primitiveList.length] = {
			id: this._addEllipse(null, pointX, pointY, width, height, color)
		};
		pointList[id].primitiveList[pointList[id].primitiveList.length] = {
			id: this._addText(null, String(id), pointX + width, pointY + height, height, color)
		};

		return id;
	};

	/**
	 * @returns {Object}
	 * @protected
	 */
	PointList.prototype._getCurrentPointList = function()
	{
		return this._getCurrentConfig().pointList;
	};

	/**
	 * @returns {number}
	 * @private
	 */
	PointList.prototype._genNewPointId = function()
	{
		var pointList = this._getCurrentPointList();
		var id = Math.round(Math.random() * 100);
		if (!pointList[id])
		{
			return id;
		} else {
			return this._genNewPointId();
		}
	};

	/**
	 * @param {boolean} [isEmpty]
	 * @returns {Object}
	 * @protected
	 */
	PointList.prototype._getCurrentConfig = function(isEmpty)
	{
		var obj = this._currentConfigObj;
		isEmpty = !!isEmpty;
		if (!this._config[obj] || isEmpty)
		{
			/**
			 * @type {{pointList: {}, positionLimit: {pan: {min: number, max: number}, tilt: {min: number, max: number}}}}
			 */
			this._config[obj] = {
				pointList: {},
				positionLimit: {
					pan: {
						min: 180,
						max: -180
					},
					tilt: {
						min: 90,
						max: -90
					},
					hfov: {
						min: 180,
						max: 0
					}
				}
			};
		}
		return this._config[obj];
	};

	/**
	 * @param {string} id
	 * @param {number} x
	 * @param {number} y
	 * @param {number} width
	 * @param {number} height
	 * @param {number} color
	 * @returns {boolean}
	 * @private
	 */
	PointList.prototype._updatePoint = function(id, x, y, width, height, color)
	{
		var pointList = this._getCurrentPointList();
		if (!pointList[id])
		{
			return false;
		}

		var point = pointList[id];
		var pointEllipseId = point.primitiveList[0].id;
		var pointTextId = point.primitiveList[1].id;

		this._addEllipse(pointEllipseId, x, y, width, height, color);
		this._addText(pointTextId, String(id), x + width, y + height, height, color);

		return true;
	};

	/**
	 * @param {string} id
	 * @protected
	 */
	PointList.prototype._removePoint = function(id)
	{
		var pointList = this._getCurrentPointList();
		if (pointList[id])
		{
			var point = pointList[id];
			for (var i = 0; i < point.primitiveList.length; i++)
			{
				this._associatedPlayer.removePrimitive(point.primitiveList[i].id);
			}

			delete pointList[id];

			if (Object.keys(pointList).length == 0)
			{
				this._getCurrentConfig(true);
			}
		}
	};

	/**
	 * @param {string} id
	 * @protected
	 */
	PointList.prototype._selectPoint = function(id)
	{
		var pointList = this._getCurrentPointList();
		for (var pointId in pointList)
		{
			if (!pointList.hasOwnProperty(pointId))
				continue;

			var point = pointList[pointId];
			var color = point.color;
			if (pointId == id)
			{
				color = this._selectedPointColor;
			}

			var pointX = Math.round(point.x - point.width / 2);
			var pointY = Math.round(point.y - point.height / 2);
			this._updatePoint(pointId, pointX, pointY, point.width, point.height, color);
		}
	};

	/**
	 * @param {string|null} id
	 * @param {number} x
	 * @param {number} y
	 * @param {number} width
	 * @param {number} height
	 * @param {number} color
	 * @returns {string|null}
	 * @protected
	 */
	PointList.prototype._addEllipse = function(id, x, y, width, height, color)
	{
		return this._associatedPlayer.addEllipse(id, x, y, width, height, color);
	};

	/**
	 * @param {string|null} id
	 * @param {Array} points
	 * @param {number} color
	 * @returns {string|null}
	 * @protected
	 */
	PointList.prototype._addPolyline = function(id, points, color)
	{
		return this._associatedPlayer.addPolyline(id, points, color);
	};

	/**
	 * @param {string|null} id
	 * @param {string} text
	 * @param {number} x
	 * @param {number} y
	 * @param {number} size
	 * @param {number} color
	 * @returns {string|null}
	 * @protected
	 */
	PointList.prototype._addText = function(id, text, x, y, size, color)
	{
		return this._associatedPlayer.addText(id, text, x, y, size, color);
	};

	return PointList;
});
